# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliMatcher
import CliCommand
import CliParser
import BasicCli
import BasicCliUtil
from CliPlugin import IpAddrMatcher, Ip6AddrMatcher
from CliPlugin.TrafficPolicyCliLib import ( PolicyConfigCmdBase )
from CliPlugin.TrafficPolicyCli import ( matchRuleName,
                                         ActionsConfigCmdBase,
                                         DefaultActionConfigCmdBase,
                                         DscpConfigCmd,
                                         MatchRuleConfigCmdBase,
                                         PrefixFieldSetIpv4BaseV2,
                                         PrefixFieldSetIpv6CmdV2Base )
from CliPlugin.ClassificationCliLib import ( ProtocolIPv4SingleMatchConfigCmd,
                                             PrefixCmdMatcher,
                                             PrefixFieldSetCmdMatcher,
                                             PrefixCmdBaseV2,
                                             FieldSetIpPrefixBaseConfigCmd,
                                             FieldSetPrefixConfigCmdsBase,
                                             FieldSetSourceConfigCmdBase,
                                             SourceMatcher,
                                             protectedFieldSetNamesRegex )
import CliPlugin.IntfCli as IntfCli # pylint: disable=consider-using-from-import
# pylint: disable-next=consider-using-from-import
import CliPlugin.PolicyMapCliLib as PolicyMapCliLib
from CliPlugin.ClassificationCliLib import CommitAbortModelet
from CliPlugin.VrfCli import (
   VrfExprFactory,
   VrfNameExprFactory,
)
import CliPlugin.VrfSelectionPolicyCliLib as cliLib
from CliMode.VrfSelectionPolicyMode import (
   VrfSelectionConfigMode,
   VrfSelectionPolicyConfigMode,
   VrfSelectionMatchIpv4RuleConfigMode,
   VrfSelectionMatchIpv6RuleConfigMode,
   VrfSelectionDefaultIpv4ActionConfigMode,
   VrfSelectionDefaultIpv6ActionConfigMode,
   VrfSelectionActionRuleConfigMode,
   VrfSelectionIntfModelet,
   VrfSelectionPolicyVrfConfigMode,
   VrfSelectionFieldSetIpPrefixConfigMode,
   VrfSelectionFieldSetIpv6PrefixConfigMode,
   FEATURE, )
import Toggles.VrfSelectionPolicyToggleLib as vspToggleLib
import Toggles.TrafficPolicyToggleLib as tpToggleLib
from TypeFuture import TacLazyType
import Tac

allIntfStatusDir = None

def vrfSelectionPolicySupported( mode, token ):
   if ( cliLib.vrfSelPolicyHwCapabilities and
        cliLib.vrfSelPolicyHwCapabilities.vrfSelectionSupported ):
      return None
   return CliParser.guardNotThisPlatform

def secondaryVrfSelectionSupported( mode, token ):
   if ( cliLib.vrfSelPolicyHwCapabilities and
        cliLib.vrfSelPolicyHwCapabilities.secondaryVrfSelectionSupported ):
      return None
   return CliParser.guardNotThisPlatform

def vrfSelectionDecapNhFallbackSupported( mode, token ):
   if ( cliLib.vrfSelPolicyHwCapabilities and
        cliLib.vrfSelPolicyHwCapabilities.vrfSelectionDecapNhFallbackSupported ):
      return None
   return CliParser.guardNotThisPlatform

def vrfSelectionDecapNhSupported( mode, token ):
   if ( cliLib.vrfSelPolicyHwCapabilities and
        cliLib.vrfSelPolicyHwCapabilities.vrfSelectionDecapNhSupported ):
      return None
   return CliParser.guardNotThisPlatform

ActionType = TacLazyType( "PolicyMap::ActionType" )
vrfKw = CliMatcher.KeywordMatcher( 'vrf', helpdesc='Configure VRFs' )
destinationVrfKwForSetAction = CliMatcher.KeywordMatcher( 'vrf',
   helpdesc='Set destination VRF' )
policyKw = CliCommand.guardedKeyword( 'policy',
      helpdesc='VRF selection policy', guard=vrfSelectionPolicySupported )
selectionIntfKw = CliCommand.guardedKeyword(
      'selection',
      helpdesc='Specify VRF selection policy for interface',
      guard=vrfSelectionPolicySupported )
setKw = CliMatcher.KeywordMatcher( 'set',
                                   helpdesc='Set packet header or property' )

# [ no ] vrf selection policy
class VrfSelectionConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''vrf selection policy'''
   noOrDefaultSyntax = syntax
   data = {
         'vrf': vrfKw,
         'selection': 'VRF selection policy',
         'policy': policyKw,
      }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( VrfSelectionConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      for policyName in cliLib.vrfSelPolicyConfig.pmapType.pmap:
         # pylint: disable-next=protected-access
         VrfSelectionPolicyConfigCmd._removePolicy( mode, policyName )
      cliLib.vrfSelPolicyFallbackConfig.fallbackVrf.clear()
      for name in cliLib.vrfSelPolicyFieldSetConfig.fieldSetIpPrefix:
         # pylint: disable-next=protected-access
         VrfSelectionFieldSetIpPrefixConfigCmd._removeFieldSet( mode, name, 'ipv4' )
      if vspToggleLib.toggleJ2Ipv6VrfSelectionPolicyForSipMatchEnabled():
         for name in cliLib.vrfSelPolicyFieldSetConfig.fieldSetIpv6Prefix:
            # pylint: disable-next=protected-access
            VrfSelectionFieldSetIpv6PrefixConfigCmd._removeFieldSet( mode, name,
                                                                     'ipv6' )
      cliLib.vrfSelPolicyConfig.decapNhEnabled = False
      cliLib.vrfSelPolicyConfig.decapNhFallbackEnabled = False

def getPolicyNames( mode ):
   return cliLib.vrfSelPolicyConfig.pmapType.pmap

nextHopNode = CliCommand.guardedKeyword(
   'next-hop', helpdesc='Nexthop feature',
   guard=vrfSelectionDecapNhFallbackSupported )

# [ no ] next-hop fallback decapsulation vrf
class NexthopFallbackDecapConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'next-hop fallback decapsulation vrf'
   noOrDefaultSyntax = syntax
   data = {
      'next-hop': nextHopNode,
      'fallback': 'Nexthop fallback feature',
      'decapsulation': 'Nexthop fallback with decapsulation feature',
      'vrf': 'Enable Nexthop fallback with decapsulation in VRF',
   }

   @staticmethod
   def handler( mode, args ):
      cliLib.vrfSelPolicyConfig.decapNhFallbackEnabled = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cliLib.vrfSelPolicyConfig.decapNhFallbackEnabled = False

nhDecapNode = CliCommand.guardedKeyword(
   'decapsulation', helpdesc='Nexthop decapsulation feature',
   guard=vrfSelectionDecapNhSupported )

warningPrompt = """
WARNING!
This command is not compatible with ingress/egress traffic-policy
or segment security configuration and the corresponding feature
agent will be disabled.

The Layer 3 Forwarding agent should be restarted for this
configuration to take effect.
"""

def checkConfirmation( mode ):
   promptText = "Proceed [y/n]"
   mode.addWarning( warningPrompt )
   if not BasicCliUtil.confirm( mode, promptText, answerForReturn=False ):
      mode.addError( "Command aborted by user" )
      return False
   return True

# [ no ] next-hop decapsulation vrf
class NexthopDecapConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'next-hop decapsulation vrf'
   noOrDefaultSyntax = syntax
   data = {
      'next-hop': nextHopNode,
      'decapsulation': nhDecapNode,
      'vrf': 'Enable Nexthop with decapsulation in VRF',
   }

   @staticmethod
   def handler( mode, args ):
      if checkConfirmation( mode ):
         cliLib.vrfSelPolicyConfig.decapNhEnabled = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cliLib.vrfSelPolicyConfig.decapNhEnabled = False

policyNameMatcher = CliMatcher.DynamicNameMatcher( getPolicyNames, "Policy Name" )

# [ no ] policy < policy-name >
class VrfSelectionPolicyConfigCmd( PolicyConfigCmdBase ):
   syntax = 'policy POLICY_NAME'
   noOrDefaultSyntax = syntax
   data = {
         'policy': 'Configure VRF selection policy',
         'POLICY_NAME': policyNameMatcher,
   }

   @staticmethod
   def _feature():
      return FEATURE

   @classmethod
   def _context( cls, name, mode ):
      return cliLib.VrfSelPolicyContext( cliLib.vrfSelPolicyConfig,
            cliLib.vrfSelPolicyStatusReqDir, cliLib.vrfSelPolicyStatus, name )

# [ no ] match <match-rule-name> ( ipv4 | ipv6 )
class VrfSelectionMatchRuleConfigCmd( MatchRuleConfigCmdBase ):
   syntax = ( 'match RULE_NAME ( ipv4 | ipv6 )'
              '[ ( before | after ) REFERENCE_RULE ]' )
   noOrDefaultSyntax = 'match RULE_NAME [ ipv4 | ipv6 ]'
   data = {
      'match': 'Configure match rule',
      'RULE_NAME': matchRuleName,
      'ipv4': 'IPv4 match rule',
      'ipv6': 'IPv6 match rule',
      'before': 'Add this rule immediately before another rule',
      'after': 'Add this rule immediately after another rule',
      'REFERENCE_RULE': matchRuleName,
   }

   @staticmethod
   def _feature():
      return FEATURE

   @staticmethod
   # pylint: disable-next=arguments-renamed
   def _context( policyContext, ruleName, matchOption ):
      return cliLib.VrfSelPolicyMatchRuleContext( policyContext, ruleName,
            matchOption )

# -----------------------------------------------------------------------------------
# "actions (ipv4|ipv6) default" sub-prompt mode command
# -----------------------------------------------------------------------------------
class VrfSelectionDefaultActionConfigCmd( DefaultActionConfigCmdBase ):
   @staticmethod
   def _feature():
      return FEATURE

   @staticmethod
   def _context( policyContext, matchOption ): # pylint: disable=arguments-renamed
      return cliLib.VrfSelPolicyDefaultActionsContext( policyContext, matchOption )

#------------------------------------------------------------------------------------
# The "actions" sub-prompt mode command
#------------------------------------------------------------------------------------
class VrfSelActionsConfigCmd( ActionsConfigCmdBase ):
   data = ActionsConfigCmdBase.data.copy()

   @staticmethod
   def _feature():
      return FEATURE

# [ no ] set vrf VRF_NAME
class VrfSelectionSetVrfActionConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''set VRF'''
   noOrDefaultSyntax = '''set vrf ...'''
   data = {
         'set': setKw,
         'VRF': VrfExprFactory( helpdesc='Set destination VRF',
                                inclDefaultVrf=True ),
         'vrf': destinationVrfKwForSetAction,
         }

   @staticmethod
   def handler( mode, args ):
      vrfName = args[ 'VRF' ]
      mode.context.setAction( ActionType.setVrf, vrfName,
            clearActions=[ ActionType.setVrf ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context.setAction( ActionType.setVrf, no=True )

# [ no ] set vrf secondary VRF_NAME
class VrfSelectionSetVrfSecondaryActionConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'set vrf secondary VRF_NAME'
   noOrDefaultSyntax = 'set vrf secondary ...'
   data = {
      'set': setKw,
      'vrf': destinationVrfKwForSetAction,
      'secondary': CliCommand.guardedKeyword(
          'secondary',
          helpdesc='Set secondary destination VRF',
          guard=secondaryVrfSelectionSupported ),
      'VRF_NAME': VrfNameExprFactory( inclDefaultVrf=True ),
   }

   @staticmethod
   def handler( mode, args ):
      vrfName = args[ 'VRF_NAME' ]
      mode.context.setAction( ActionType.setVrfSecondary, vrfName,
            clearActions=[ ActionType.setVrfSecondary ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context.setAction( ActionType.setVrfSecondary, no=True )

# [ no ] set vrf decapsulation VRF_NAME fallback VRF_NAME forward VRF_NAME
class VrfSelectionSetVrfDecapsulationActionCmd( CliCommand.CliCommandClass ):
   syntax = 'set vrf DECAP_VRF FALLBACK_VRF FORWARD_VRF'
   noOrDefaultSyntax = 'set vrf decapsulation ...'
   data = {
      'set': setKw,
      'vrf': destinationVrfKwForSetAction,
      'decapsulation': 'decapsulation vrf feature',
      'DECAP_VRF': VrfExprFactory( keyword='decapsulation',
         helpdesc='Set forward/fallback VRF based on decapsulation VRF lookup',
         inclAllVrf=False ),
      'FALLBACK_VRF': VrfExprFactory( keyword='fallback',
         helpdesc='Set fallback VRF to use if decapsulation VRF lookup fails',
         inclAllVrf=False ),
      'FORWARD_VRF': VrfExprFactory( keyword='forward',
         helpdesc='Set forward VRF to use if decapsulation VRF lookup succeeds',
         inclAllVrf=False ),
   }

   @classmethod
   def handler( cls, mode, args ):
      if not cliLib.vrfSelPolicyConfig.decapNhEnabled:
         mode.addError( "`next-hop decapsulation vrf` must be specified first" )
         return
      decapsulation = args[ 'DECAP_VRF' ]
      fallback = args[ 'FALLBACK_VRF' ]
      forward = args[ 'FORWARD_VRF' ]
      mode.context.setAction( ActionType.setDecapVrf,
                              ( decapsulation, fallback, forward ),
                              no=False )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      mode.context.setAction( ActionType.setDecapVrf, no=True )

# ---------------------------------------------------------
# source prefix <A.B.C.D/E> [ <A.B.C.D/E> ... ]
# ---------------------------------------------------------
class VrfSelectionPrefixIpv4Cmd( PrefixCmdBaseV2 ):
   data = {
      "PREFIX_EXPR": PrefixCmdMatcher( addrType="ipv4", allowMultiple=True,
                                       allowSource=True, allowDestination=False )
   }

# ---------------------------------------------------------
# source prefix <A:B:C:D:E:F:G:H/I> [ A:B:C:D:E:F:G:H/I ..... ]
# ---------------------------------------------------------
class VrfSelectionPrefixIpv6Cmd( PrefixCmdBaseV2 ):
   data = {
      "PREFIX_EXPR": PrefixCmdMatcher( addrType="ipv6", allowMultiple=True,
                                       allowSource=True, allowDestination=False )
   }

# ---------------------------------------------------------
# source prefix field-set FIELD_SET_NAME
# ---------------------------------------------------------
def getIpPrefixFieldSetNames( mode ):
   return cliLib.vrfSelPolicyFieldSetConfig.fieldSetIpPrefix

ipPrefixFieldSetNameMatcher = CliMatcher.DynamicNameMatcher(
   getIpPrefixFieldSetNames,
   "IP prefix field-set name",
   pattern=protectedFieldSetNamesRegex( 'prefix' ),
   priority=CliParser.PRIO_LOW )

class VrfSelectionPrefixFieldSetIpv4Cmd( PrefixFieldSetIpv4BaseV2 ):
   data = {
      "PREFIX_EXPR": PrefixFieldSetCmdMatcher( allowMultiple=True,
                                        fieldSetMatcher=ipPrefixFieldSetNameMatcher,
                                        allowSource=True,
                                        allowDestination=False )
   }

# ---------------------------------------------------------
# field-set ipv4 prefix FIELD_SET_NAME
# ---------------------------------------------------------
class VrfSelectionFieldSetIpPrefixConfigCmd( FieldSetIpPrefixBaseConfigCmd ):
   syntax = 'field-set ipv4 prefix FIELD_SET_NAME'
   noOrDefaultSyntax = syntax
   _feature = FEATURE
   data = {
      'FIELD_SET_NAME': ipPrefixFieldSetNameMatcher,
   }
   data.update( FieldSetIpPrefixBaseConfigCmd._baseData )

   @classmethod
   def _getContextKwargs( cls, fieldSetIpPrefixName, setType, mode=None ):
      return {
         'fieldSetIpPrefixName': fieldSetIpPrefixName,
         'fieldSetConfig': cliLib.vrfSelPolicyFieldSetConfig,
         'setType': setType,
         'childMode': VrfSelectionFieldSetIpPrefixConfigMode,
         'featureName' : cls._feature,
      }

class VrfSelectionFieldSetIpPrefixConfigCmds( FieldSetPrefixConfigCmdsBase ):
   data = {
      'PREFIXES': IpAddrMatcher.ipPrefixExpr(
         'Prefix address',
         'Prefix mask',
         'Prefix',
         overlap=IpAddrMatcher.PREFIX_OVERLAP_REJECT,
         maskKeyword=False,
         raiseError=True )
   }
   data.update( FieldSetPrefixConfigCmdsBase._baseData )

# ---------------------------------------------------------
# source prefix field-set FIELD_SET_NAME
# ---------------------------------------------------------
def getIpv6PrefixFieldSetNames( mode ):
   return cliLib.vrfSelPolicyFieldSetConfig.fieldSetIpv6Prefix

ipv6PrefixFieldSetNameMatcher = CliMatcher.DynamicNameMatcher(
   getIpv6PrefixFieldSetNames,
   "IPv6 prefix field-set name",
   pattern=protectedFieldSetNamesRegex( 'prefix' ),
   priority=CliParser.PRIO_LOW )

class VrfSelectionPrefixFieldSetIpv6Cmd( PrefixFieldSetIpv6CmdV2Base ):
   data = {
      "PREFIX_EXPR": PrefixFieldSetCmdMatcher( allowMultiple=True,
                                       fieldSetMatcher=ipv6PrefixFieldSetNameMatcher,
                                       allowSource=True,
                                       allowDestination=False )
   }

# ---------------------------------------------------------
# field-set ipv6 prefix FIELD_SET_NAME
# ---------------------------------------------------------
class VrfSelectionFieldSetIpv6PrefixConfigCmd( FieldSetIpPrefixBaseConfigCmd ):
   syntax = 'field-set ipv6 prefix FIELD_SET_NAME'
   noOrDefaultSyntax = syntax
   _feature = FEATURE
   data = {
      'FIELD_SET_NAME': ipv6PrefixFieldSetNameMatcher,
   }
   data.update( FieldSetIpPrefixBaseConfigCmd._baseData )

   @classmethod
   def _getContextKwargs( cls, fieldSetIpPrefixName, setType, mode=None ):
      return {
         'fieldSetIpPrefixName': fieldSetIpPrefixName,
         'fieldSetConfig': cliLib.vrfSelPolicyFieldSetConfig,
         'setType': setType,
         'childMode': VrfSelectionFieldSetIpv6PrefixConfigMode,
         'featureName': cls._feature,
      }

class VrfSelectionFieldSetIpv6PrefixConfigCmds( FieldSetPrefixConfigCmdsBase ):
   data = {
      'PREFIXES': Ip6AddrMatcher.ip6PrefixExpr(
         'Prefix address',
         'Prefix mask',
         'Prefix',
         overlap=IpAddrMatcher.PREFIX_OVERLAP_REJECT,
         raiseError=True )
   }
   data.update( FieldSetPrefixConfigCmdsBase._baseData )

#---------------------------------------------------------
# [ no|default ] vrf selection policy <policy-name>
# command, in config-if mode.
#---------------------------------------------------------
class VrfSelectionIntfConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''vrf selection policy POLICY_NAME'''
   noOrDefaultSyntax = '''vrf selection policy ...'''
   data = {
         'vrf': 'Specify VRF for interface',
         'selection': selectionIntfKw,
         'policy': 'Specify VRF selection policy for interface',
         'POLICY_NAME': policyNameMatcher,
         }

   @staticmethod
   def handler( mode, args ):
      policyName = args.get( 'POLICY_NAME' )
      intfName = mode.intf.name

      def maybeAddWarning( mode, allIntfStatus, intfName ):
         """
         Function to check if the interface is a routed port and display a warning
         that the configuration will be ignored if it is not a routed port
         """
         intfStatus = allIntfStatus.intfStatus.get( intfName )
         if intfStatus and intfStatus.forwardingModel != 'intfForwardingModelRouted':
            # pylint: disable-next=consider-using-f-string
            mode.addWarning( 'Configuration will be ignored while interface %s '
                             'is not a routed port.' % intfName )

      if policyName != cliLib.vrfSelPolicyIntfConfig.intf.get( intfName, None ):
         maybeAddWarning( mode, cliLib.allIntfStatusDir, intfName )

      PolicyMapCliLib.handleServicePolicy( mode, False, policyName,
                  cliLib.vrfSelPolicyConfig, cliLib.vrfSelPolicyStatusReqDir,
                  cliLib.vrfSelPolicyStatus, fallback=False,
                  intfConfig=cliLib.vrfSelPolicyIntfConfig )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      PolicyMapCliLib.handleServicePolicy( mode, True, None,
                  cliLib.vrfSelPolicyConfig, cliLib.vrfSelPolicyStatusReqDir,
                  cliLib.vrfSelPolicyStatus, fallback=False,
                  intfConfig=cliLib.vrfSelPolicyIntfConfig )

class VrfSelectionIntfJanitor( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      PolicyMapCliLib.delIntfPolicy( cliLib.vrfSelPolicyIntfConfig,
                                       self.intf_.name )

#--------------------------------------------------------------------------------
# [ no | default ] vrf VRF
#--------------------------------------------------------------------------------
class VrfSelectionPolicyVrfCmd( CliCommand.CliCommandClass ):
   syntax = 'VRF'
   noOrDefaultSyntax = syntax
   data = {
      'VRF': VrfExprFactory( helpdesc='Enter VRF submode',
                             inclDefaultVrf=True ),
   }

   @staticmethod
   def handler( mode, args ):
      ''' Function to go into vrf selection policy vrf submode'''
      vrfName = args[ 'VRF' ]
      childMode = mode.childMode( VrfSelectionPolicyVrfConfigMode, vrfName=vrfName )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vrfObj = Tac.Value( 'L3::VrfName', args[ 'VRF' ] )
      del cliLib.vrfSelPolicyFallbackConfig.fallbackVrf[ vrfObj ]

#----------------------------------------------
# "[ no | default ] fallback vrf <vrf-name>"
# in 'vrf selection policy vrf' mode
#----------------------------------------------
class VrfSelectionFallbackVrfCmd( CliCommand.CliCommandClass ):
   syntax = 'fallback VRF'
   noOrDefaultSyntax = 'fallback vrf ...'
   data = {
      'fallback': 'Specify fallback route lookup option',
      'VRF': VrfExprFactory( helpdesc='Specify fallback vrf',
                             inclDefaultVrf=True ),
      'vrf': 'Specify fallback vrf',
   }

   @staticmethod
   def handler( mode, args ):
      vrfObj = Tac.Value( 'L3::VrfName', mode.vrfName )
      fallbackVrf = Tac.Value( 'L3::VrfName', args[ 'VRF' ] )
      cliLib.vrfSelPolicyFallbackConfig.fallbackVrf[ vrfObj ] = fallbackVrf

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vrfObj = Tac.Value( 'L3::VrfName', mode.vrfName )
      del cliLib.vrfSelPolicyFallbackConfig.fallbackVrf[ vrfObj ]

# --------------------------------------------------------------------------
# The "[ ( no | default ) ] source { static | bgp }" command
# --------------------------------------------------------------------------
class FieldSetPrefixSourceConfigCmd ( FieldSetSourceConfigCmdBase ):
   _field = 'prefixes'
   data = {
      'SOURCE_EXPR': SourceMatcher( supportsBgp=True, supportsUrl=False ),
      'REMOVE_SOURCE_EXPR': SourceMatcher( supportsBgp=False, supportsUrl=False,
                                            isRemove=True )
   }

for _mode in [ VrfSelectionPolicyConfigMode,
               VrfSelectionMatchIpv4RuleConfigMode,
               VrfSelectionMatchIpv6RuleConfigMode,
               VrfSelectionActionRuleConfigMode,
               VrfSelectionFieldSetIpPrefixConfigMode,
               VrfSelectionFieldSetIpv6PrefixConfigMode, ]:
   _mode.addModelet( CommitAbortModelet )

for _mode in [ VrfSelectionDefaultIpv4ActionConfigMode,
               VrfSelectionDefaultIpv6ActionConfigMode, ]:
   _mode.addModelet( CommitAbortModelet )

if vspToggleLib.toggleVrfSelectionPolicyConfigEnabled():
   BasicCli.GlobalConfigMode.addCommandClass( VrfSelectionConfigCmd )
   VrfSelectionConfigMode.addCommandClass( NexthopFallbackDecapConfigCmd )
   VrfSelectionConfigMode.addCommandClass( NexthopDecapConfigCmd )
   VrfSelectionConfigMode.addCommandClass( VrfSelectionPolicyConfigCmd )
   VrfSelectionConfigMode.addCommandClass( VrfSelectionPolicyVrfCmd )
   VrfSelectionConfigMode.addCommandClass( VrfSelectionFieldSetIpPrefixConfigCmd )
   VrfSelectionPolicyConfigMode.addCommandClass( VrfSelectionMatchRuleConfigCmd )
   VrfSelectionMatchIpv4RuleConfigMode.addCommandClass( DscpConfigCmd )
   VrfSelectionMatchIpv4RuleConfigMode.addCommandClass( VrfSelectionPrefixIpv4Cmd )
   VrfSelectionMatchIpv4RuleConfigMode.addCommandClass(
         VrfSelectionPrefixFieldSetIpv4Cmd )
   VrfSelectionMatchIpv4RuleConfigMode.addCommandClass(
         ProtocolIPv4SingleMatchConfigCmd )
   VrfSelectionMatchIpv4RuleConfigMode.addCommandClass( VrfSelActionsConfigCmd )
   VrfSelectionMatchIpv6RuleConfigMode.addCommandClass( DscpConfigCmd )
   if vspToggleLib.toggleJ2Ipv6VrfSelectionPolicyForSipMatchEnabled():
      VrfSelectionMatchIpv6RuleConfigMode.addCommandClass(
         VrfSelectionPrefixIpv6Cmd )
      VrfSelectionMatchIpv6RuleConfigMode.addCommandClass(
         VrfSelectionPrefixFieldSetIpv6Cmd )
      VrfSelectionConfigMode.addCommandClass(
         VrfSelectionFieldSetIpv6PrefixConfigCmd )
      VrfSelectionFieldSetIpv6PrefixConfigMode.addCommandClass(
         VrfSelectionFieldSetIpv6PrefixConfigCmds )
      VrfSelectionFieldSetIpv6PrefixConfigMode.addCommandClass(
         FieldSetPrefixSourceConfigCmd )
   VrfSelectionMatchIpv6RuleConfigMode.addCommandClass( VrfSelActionsConfigCmd )
   VrfSelectionActionRuleConfigMode.addCommandClass(
         VrfSelectionSetVrfActionConfigCmd )
   if tpToggleLib.toggleTrafficPolicySetDecapActionEnabled():
      VrfSelectionActionRuleConfigMode.addCommandClass(
         VrfSelectionSetVrfDecapsulationActionCmd )
   IntfCli.IntfConfigMode.addModelet( VrfSelectionIntfModelet )
   VrfSelectionIntfModelet.addCommandClass( VrfSelectionIntfConfigCmd )
   IntfCli.Intf.registerDependentClass( VrfSelectionIntfJanitor )
   VrfSelectionPolicyVrfConfigMode.addCommandClass( VrfSelectionFallbackVrfCmd )
   VrfSelectionFieldSetIpPrefixConfigMode.addCommandClass(
         VrfSelectionFieldSetIpPrefixConfigCmds )
   VrfSelectionFieldSetIpPrefixConfigMode.addCommandClass(
      FieldSetPrefixSourceConfigCmd )

   VrfSelectionPolicyConfigMode.addCommandClass(
      VrfSelectionDefaultActionConfigCmd )
   for _defaultActionMode in ( VrfSelectionDefaultIpv4ActionConfigMode,
                               VrfSelectionDefaultIpv6ActionConfigMode, ):
      _defaultActionMode.addCommandClass( VrfSelectionSetVrfActionConfigCmd )
      _defaultActionMode.addCommandClass(
         VrfSelectionSetVrfSecondaryActionConfigCmd )
      _defaultActionMode.addCommandClass(
         VrfSelectionSetVrfDecapsulationActionCmd )
