# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import sys
from itertools import groupby
from operator import itemgetter

import BasicCli
import CliCommand
import CliMatcher
import ConfigMount
import CliPlugin.ControllerdbLib
import CliPlugin.TechSupportCli
from CliPlugin import IpAddrMatcher
from CliPlugin import Ip6AddrMatcher
from CliPlugin import MacAddr
from CliPlugin.VxlanControllerModel import (
                                   VxlanControllerAddressTableReceivedModel,
                                   VxlanControllerStatus,
                                   SwitchCapability,
                                   VxlanControllerCapability,
                                   VxlanControllerSwitchCapability,
                                   VxlanControllerLogicalRouterList,
                                   VxlanControllerRouteList,
                                   VxlanControllerLogicalRouterMap,
                                   VniRange,
                                   VniList,
                                   SwitchList,
                                   VxlanVniStatusList,
                                   SwitchVniListReceivedModel,
                                   SwitchVniListAdvertisedModel,
                                   VxlanFdbSetModel,
                                   VxlanVtepSetModel )
from CliPlugin.ControllerdbLib import ( controllerNotReady,
                                        controllerGuard,
                                        switchIdCache )
from CliPlugin.ControllerCli import ( addNoCvxCallback,
                                      CvxConfigMode,
                                      serviceKwMatcher,
                                      serviceAfterShowKwMatcher )
import LazyMount
import Plugins
import ShowCommand
import SmashLazyMount
import Tac
import Toggles.VxlanControllerToggleLib

from CliMode.Vxlan import VxlanMode
from Tracing import t8
from VxlanVniLib import ( VniFormat,
                          VniMatcher )

controllerConfig = None
controllerStatus = None
switchPublishStatus = None
vxlanCtrlConfig = None
vxlanCtrlStatus = None
vcsStateV2 = None
globalSwitchDir = None
globalHscToVcsVniStatusDir = None
globalVcsToHscVniStatusDir = None
globalMssVniStatusDir = None
globalEvpnToVcsHostTable = None
globalEvpnToVcsFloodTable = None
globalVcsToEvpnHostTable = None
globalVcsToEvpnFloodTable = None
globalVniDir = None
serviceCfgDir = None
vcsStateClientViewDir = None
lRStatusDir = None

clientViewType = Tac.Type( "VxlanController::VcsStateClientViewV2" )

# To box vxlanCtrlConfig 
vxlanCtrlCfgBox = []

def makeIpGenAddr( ipStr ):
   return Tac.Value( 'Arnet::IpGenAddr', ipStr )

def makeIpGenAddrV6( ip6Addr ):
   return Tac.Value( 'Arnet::IpGenAddr', ip6Addr.stringValue )

def makeDefaultIpGenAddr():
   return Tac.newInstance( "Arnet::IpGenAddr" )

def allLRNames( mode ):
   if lRStatusDir is not None:
      # pylint: disable-next=unnecessary-comprehension
      return [ key for key in lRStatusDir ]
   else:
      return []

def forceControllerdbMount( mount ):
   if controllerConfig.enabled and controllerStatus.enabled:
      return LazyMount.force( mount )
   else:
      return None

def forceSmashMount( mount ):
   # The preconditions of which VCS being enabled is (1) vxlanCtrlConfig.enable
   # == True, (2) controllerStatus.enabled == True, and (3) the running node
   # being the leader of the CVX cluster ( refer to StartSm::handleEnable() in
   # VxlanCntrlrAgent.tin ).
   # Conditions (1) and (2) being true is enough to know that the VCS has
   # mounted hte Smash tables; condition (3) is present to just ensure that
   # only one node is active and reacts to changes corresponding to the VCS.
   if vxlanCtrlConfig.enable and controllerStatus.enabled and \
      hasattr( mount, '__tac_object__' ):
      return mount.__tac_object__()
   else:
      return None

# All show commands calling this function are using a deferred cli model which is
# why the model is not populated in the show command function or here.  The
# TEXT/JSON rendering is done by the Tac type returned by the instantiate function
# passed in here.
def processDeferredModel( mode, mount, model, instantiate, auxMountList=None ):
   fd = sys.stdout.fileno()
   revision = mode.session_.requestedModelRevision()
   fmt = mode.session_.outputFormat()

   # comparing the types instead of checking if 'mount' is None because the 'if'
   # condition will evaluate 'mount' and that'd trigger lazyMount
   if mount is not None:
      forceControllerdbMount( mount )

      if auxMountList:
         for srv, mnt in auxMountList:
            if srv == "EVPN":
               for mnt_ in mnt:
                  forceSmashMount( mnt_ )   
            else:
               forceControllerdbMount( mnt )
         instance = instantiate( mount, auxMountList )
      else:
         instance = instantiate( mount )

      # Copy from sysdb to C++ cli model happens outside the activityLock since
      # genericif is not involved. The copy can take multiple system ticks. The
      # chance of someone modifying a collection we are accessing is high. We
      # avoid that problem by grabbing the activity lock.
      with Tac.ActivityLockHolder():
         instance.render( fd, fmt, revision )
   else:
      t8( "mount is None" )

   return model

def getLRNameFilter( lRName ):
   return lRName if lRName else ''

def getVtepFilter( vteps ):
   vtepFilter = Tac.newInstance( "VxlanController::Cli::VtepList" )

   for vtep in vteps:
      vtepFilter.vtep[ vtep ] = True

   return vtepFilter

def getVniStatusDirList( vniStatusDirs ):
   vniStatusDirList = Tac.newInstance( "VxlanController::Cli::VniStatusDirList" )
   
   for svcName, vniStatusDir in vniStatusDirs:
      vniStatusDirList.vniStatusDir[ svcName ] = vniStatusDir

   return vniStatusDirList   

def getHostFloodTableList( hostFloodTables ):
   hostFloodTableList = \
         Tac.newInstance( "VxlanController::Cli::HostFloodTableList" )

   for svc, table in hostFloodTables:
      hostFloodTable = \
            Tac.newInstance( "VxlanController::Cli::HostFloodTable" )
      ht, ft = table
      hostFloodTable.hostTable = ht
      hostFloodTable.floodTable = ft
      hostFloodTableList.hostFloodTable[ svc ] = hostFloodTable

   return hostFloodTableList

#-------------------------------------------------------------------------------
# Service Vxlan mode
#-------------------------------------------------------------------------------
class VxlanConfigMode ( VxlanMode, BasicCli.ConfigModeBase ):
   name = 'Vxlan configuration'

   def __init__( self, parent, session, name=None ):
      VxlanMode.__init__( self, name )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def disableVxlanService( mode, args=None ):
   vxlanCtrlConfig.reset()
   shutServiceVxlan( mode )
 
def shutServiceVxlan( mode, args=None ):
   vxlanCtrlConfig.enable = False
   vxlanServiceConfig = serviceCfgDir.service[ "Vxlan" ]
   vxlanServiceConfig.enabled = False

def enableDatapathLearning( mode, args ):
   vxlanCtrlConfig.dataPathLearning = True

def disableDatapathLearning( mode, args ):
   vxlanCtrlConfig.dataPathLearning = False

def enablePortDot1qVniMapping( mode, args ):
   vxlanCtrlConfig.portDot1qVniMapping = True

def disablePortDot1qVniMapping( mode, args ):
   vxlanCtrlConfig.portDot1qVniMapping = False

vtepKwMatcher = CliMatcher.KeywordMatcher( 'vtep',
                                   helpdesc='VTEP configuration parameters' )
vniKwMatcher = CliMatcher.KeywordMatcher( 'vni', helpdesc='VNI keyword' )
macLearningKwMatcher = CliMatcher.KeywordMatcher( 'mac-learning',
      helpdesc='MAC learning mode on vtep' )
mappingKwMatcher = CliMatcher.KeywordMatcher( 'mapping',
      helpdesc='Mapping keyword' )
macKwMatcher = CliMatcher.KeywordMatcher( 'mac',
      helpdesc='Ethernet address' )
ipKwMatcher = CliMatcher.KeywordMatcher( 'ip',
      helpdesc='IP address' )
matcherVxlan = CliMatcher.KeywordMatcher( 'vxlan',
      helpdesc='VXLAN control services' )

#--------------------------------------------------------------------------------
# [ no | default ] service vxlan
#--------------------------------------------------------------------------------
class VxlanConfigModeCmd( CliCommand.CliCommandClass ):
   syntax = 'service vxlan'
   noOrDefaultSyntax = syntax
   data = {
      'service' : serviceKwMatcher,
      'vxlan' : matcherVxlan,
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( VxlanConfigMode )
      mode.session_.gotoChildMode( childMode )

   noOrDefaultHandler = disableVxlanService

CvxConfigMode.addCommandClass( VxlanConfigModeCmd )

#--------------------------------------------------------------------------------
# [ no | default ] shutdown
#--------------------------------------------------------------------------------
class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown' : 'Disable VXLAN service',
   }

   handler = shutServiceVxlan
   defaultHandler = handler

   @staticmethod
   def noHandler( mode, args ):
      if not controllerConfig.enabled:
         mode.addError( "Vxlan service cannot be enabled until the CVX server is "
                        "enabled" )
         return

      vxlanCtrlConfig.enable = True
      vxlanServiceConfig = serviceCfgDir.service[ "Vxlan" ]
      vxlanServiceConfig.enabled = True

VxlanConfigMode.addCommandClass( ShutdownCmd )

#--------------------------------------------------------------------------------
# [ no | default ] vtep mac-learning control-plane
#--------------------------------------------------------------------------------
class VtepMacLearningControlPlaneCmd( CliCommand.CliCommandClass ):
   syntax = 'vtep mac-learning control-plane'
   noOrDefaultSyntax = syntax
   data = {
      'vtep' : vtepKwMatcher,
      'mac-learning' : macLearningKwMatcher,
      'control-plane' : 'Enable MAC learning from VXLAN Control Service',
   }

   handler = disableDatapathLearning
   noHandler = enableDatapathLearning
   defaultHandler = disableDatapathLearning

VxlanConfigMode.addCommandClass( VtepMacLearningControlPlaneCmd )

#--------------------------------------------------------------------------------
# [ no | default ] vtep mac-learning data-plane
#--------------------------------------------------------------------------------
class VtepMacLearningDataPlaneCmd( CliCommand.CliCommandClass ):
   syntax = 'vtep mac-learning data-plane'
   noOrDefaultSyntax = syntax
   data = {
      'vtep' : vtepKwMatcher,
      'mac-learning' : macLearningKwMatcher,
      'data-plane' : 'Enable MAC learning on vteps over VXLAN tunnels',
   }

   handler = enableDatapathLearning
   noOrDefaultHandler = disableDatapathLearning

VxlanConfigMode.addCommandClass( VtepMacLearningDataPlaneCmd )

#--------------------------------------------------------------------------------
# [ no | default ] vtep vni mapping port-dot1q-tag
#--------------------------------------------------------------------------------
class VtepVniMappingPortDot1QTagCmd( CliCommand.CliCommandClass ):
   syntax = 'vtep vni mapping port-dot1q-tag'
   noOrDefaultSyntax = 'vtep vni mapping ...'
   data = {
      'vtep' : vtepKwMatcher,
      'vni' : vniKwMatcher,
      'mapping' : mappingKwMatcher,
      'port-dot1q-tag' : 'Enable VNI mapping to port, dot1q tag pairs',
   }

   handler = enablePortDot1qVniMapping
   noOrDefaultHandler = disablePortDot1qVniMapping

VxlanConfigMode.addCommandClass( VtepVniMappingPortDot1QTagCmd )

#--------------------------------------------------------------------------------
# vtep vni mapping vlan
#--------------------------------------------------------------------------------
class VtepVniMappingVlanCmd( CliCommand.CliCommandClass ):
   syntax = 'vtep vni mapping vlan'
   data = {
      'vtep' : vtepKwMatcher,
      'vni' : vniKwMatcher,
      'mapping' : mappingKwMatcher,
      'vlan' : 'Enable VNI mapping to VLANs',
   }

   handler = disablePortDot1qVniMapping

VxlanConfigMode.addCommandClass( VtepVniMappingVlanCmd )

#--------------------------------------------------------------------------------
# [ no | default ] resync-period PERIOD
#--------------------------------------------------------------------------------
class RestartSyncPeriodCmd( CliCommand.CliCommandClass ):
   syntax = 'resync-period PERIOD'
   noOrDefaultSyntax = 'resync-period ...'
   data = {
      'resync-period' : ( 'Grace period for completion of synchronization between '
                          'VCS and clients after CVX restart' ),
      'PERIOD' : CliMatcher.IntegerMatcher( 30, 4800, helpdesc='Time period' ),
   }

   @staticmethod
   def handler( mode, args ):
      vxlanCtrlConfig.restartSyncPeriod = args[ 'PERIOD' ]

   @staticmethod
   def noHandler( mode, args ):
      vxlanCtrlConfig.restartSyncPeriod = 0

   @staticmethod
   def defaultHandler( mode, args ):
      vxlanCtrlConfig.restartSyncPeriod = vxlanCtrlConfig.defaultRestartSyncPeriod

VxlanConfigMode.addCommandClass( RestartSyncPeriodCmd )

#--------------------------------------------------------------------------------
# [ no | default ] redistribute bgp evpn vxlan
#--------------------------------------------------------------------------------
class RedistributeBgpEvpnVxlanCmd( CliCommand.CliCommandClass ):
   syntax = 'redistribute bgp evpn vxlan'
   noOrDefaultSyntax = syntax
   data = {
      'redistribute' : 'Redistribute MAC-VTEP bindings in to VXLAN Control Service',
      'bgp' : 'Redistribute MAC-VTEP bindings from BGP into VXLAN Control Service',
      'evpn' : ( 'Redistribute MAC-VTEP bindings from BGP EVPN into VXLAN Control '
                  'Service' ),
      'vxlan' : ( 'Redistribute MAC-VTEP bindings from BGP EVPN (Vxlan Encap) into '
                  'VXLAN Control Service' ),
   }

   @staticmethod
   def handler( mode, args ):
      vxlanCtrlConfig.bgpEvpnRedist = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vxlanCtrlConfig.bgpEvpnRedist = False

VxlanConfigMode.addCommandClass( RedistributeBgpEvpnVxlanCmd )

#--------------------------------------------------------------------------------
# [ no | default ] multicast membership-list
#--------------------------------------------------------------------------------
class MulticastDistributionCmd( CliCommand.CliCommandClass ):
   syntax = 'multicast membership-list'
   noOrDefaultSyntax = syntax
   data = {
      'multicast' : 'Enable multicast membership-list distribution',
      'membership-list' : 'membership-list keyword',
   }

   @staticmethod
   def handler( mode, args ):
      vxlanCtrlConfig.multicastMembershipList = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vxlanCtrlConfig.multicastMembershipList = False

if Toggles.VxlanControllerToggleLib.toggleVCSMulticastKnobEnabled():
   VxlanConfigMode.addCommandClass( MulticastDistributionCmd )

def vcsNoCvx( mode ):
   disableVxlanService( mode )

addNoCvxCallback( vcsNoCvx )

#--------------------------------------------------------------------------------
# [ no | default ] vxlan vni notation dotted
#--------------------------------------------------------------------------------
class VxlanVniNotationDottedCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan vni notation dotted'
   noOrDefaultSyntax = syntax
   data = {
      'vxlan' : 'Configure global VXLAN parameters',
      'vni' : 'Vni keyword',
      'notation' : 'Notation keyword',
      'dotted' : 'Dotted keyword',
   }

   @staticmethod
   def handler( mode, args ):
      vxlanCtrlConfig.vniInDottedNotation = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vxlanCtrlConfig.vniInDottedNotation = False

BasicCli.GlobalConfigMode.addCommandClass( VxlanVniNotationDottedCmd )

#--------------------------------------------------------------------------------
# show commands
#--------------------------------------------------------------------------------
def setCapabilities( source, dest ):
   for capability in source.attributes:
      if capability != 'capability':
         dest.__setattr__( capability, source.__getattribute__( capability ) )

vniMatcher = VniMatcher( 'VXLAN Network Identifier', vxlanCtrlCfgBox )
matcherAll = CliMatcher.KeywordMatcher( 'all',
      helpdesc='All switches' )
switchKwMatcher = CliMatcher.KeywordMatcher( 'switch',
      helpdesc='Filter on a switch spec' )
nodeLogicalRouter = CliCommand.guardedKeyword( 'logical-router',
      helpdesc='Logical router information',
      guard=controllerGuard )
matcherName = CliMatcher.KeywordMatcher( 'name',
      helpdesc='Specify logical router name' )
lRNameMatcher = CliMatcher.DynamicNameMatcher( allLRNames,
      'Logical router namess', pattern='[A-Za-z0-9_-]+' )
nodeArp = CliCommand.guardedKeyword( 'arp', helpdesc='ARP entries',
      guard=controllerGuard )
nodeNd = CliCommand.guardedKeyword( 'nd', helpdesc='Neighbor Discovery entries',
      guard=controllerGuard )
matcherAddress = CliMatcher.KeywordMatcher( 'address',
      helpdesc='Filter by MAC address' )
nodeAddressTable = CliCommand.guardedKeyword( 'address-table',
      helpdesc='MAC forwarding table', guard=controllerGuard )
switchIdMatcher = CliMatcher.DynamicNameMatcher( switchIdCache.allNames,
      'Hostname, IP address or ID of the switch',
      pattern=r'[A-Za-z0-9_\.:{}\[\]-]+' ) # We need '.' to match IP address
vniKwMatcher = CliMatcher.KeywordMatcher( 'vni',
      helpdesc='VNI information' )
vtepAddrMatcher = IpAddrMatcher.IpAddrMatcher( 'IP address of remote VTEP' )

#--------------------------------------------------------------------------------
# show service vxlan status
#--------------------------------------------------------------------------------
class ServiceVxlanStatusCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show service vxlan status'
   data = {
      'service' : serviceAfterShowKwMatcher,
      'vxlan' : matcherVxlan,
      'status' : 'VxlanController agent status',
   }
   cliModel = VxlanControllerStatus

   @staticmethod
   def handler( mode, args ):
      vxlcStatus = VxlanControllerStatus()

      configEnable = vxlanCtrlConfig.enable
      statusEnable = vxlanCtrlStatus.enable

      vxlcStatus.status = "running" if statusEnable else "stopped"
      if not configEnable and statusEnable:
         vxlcStatus.status = "shutting down"

      vxlcStatus.macLearning = 'dataPlane' if vxlanCtrlStatus.dataPathLearning else \
          'controlPlane'
      vxlcStatus.resyncPeriod = vxlanCtrlStatus.restartSyncPeriod
      vxlcStatus.vniMapping = 'portDot1qTag' if vxlanCtrlStatus.portDot1qVniMapping \
                              else 'vlan'
      vxlcStatus.multicastDistributionEnabled = \
            vxlanCtrlConfig.multicastMembershipList

      if vxlanCtrlStatus.restartSyncStartTime:
         vxlcStatus.resyncStartedAt = vxlcStatus.toUtc( 
            vxlanCtrlStatus.restartSyncStartTime )
      if controllerNotReady() or not statusEnable:
         vxlcStatus.resyncInProgress = False
      else:
         vxlcStatus.resyncInProgress = vcsStateV2.convergenceInProgress

      vxlcStatus.bgpEvpnRedistEnabled = vxlanCtrlConfig.bgpEvpnRedist
      vxlcStatus.capability = VxlanControllerCapability()
      setCapabilities( vxlanCtrlStatus.vcsCapability, vxlcStatus.capability )
      for ip in vxlanCtrlConfig.arpReplyRelayVtep:
         if ip in vxlanCtrlStatus.switchInfoDir.vtepMap:
            vxlcStatus.arpRelayVteps.append( ip.stringValue )
      return vxlcStatus

BasicCli.addShowCommandClass( ServiceVxlanStatusCmd )

#--------------------------------------------------------------------------------
# show service vxlan switch ( all | SWITCH_ID ) capability
#--------------------------------------------------------------------------------
class ServiceVxlanSwitchCapabilityCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show service vxlan switch ( all | SWITCH_ID ) capability'
   data = {
      'service' : serviceAfterShowKwMatcher,
      'vxlan' : matcherVxlan,
      'switch' : switchKwMatcher,
      'all' : matcherAll,
      'SWITCH_ID' : switchIdMatcher,
      'capability' : 'Switch capabilities',
   }
   cliModel = VxlanControllerSwitchCapability

   @staticmethod
   def handler( mode, args ):
      switchFilter = args.get( 'SWITCH_ID', 'all' )
      model = VxlanControllerSwitchCapability()

      def addSwitch( switchInfo ):
         # pylint: disable-msg=unsupported-assignment-operation
         switch = model.switchCapability[ switchInfo.switchId ] = SwitchCapability()
         for key, vtepMap in switchInfo.vtepMap.items():
            if vtepMap.role == 'roleFdbContributor':
               if key.vtepKey.endswith( '-mlag' ):
                  switch.mlagVtepIp = vtepMap.vtepAddr
               else:
                  switch.vtepIp = vtepMap.vtepAddr
         setCapabilities( switchInfo.vtepCapability, switch )

         hostname = switchIdCache.getHost( switchInfo.switchId )
         if hostname:
            switch.hostname = hostname
         else:
            switch.hostname = ''

      if switchFilter == 'all':
         for switchInfo in vxlanCtrlStatus.switchInfoDir.switchInfo.values():
            addSwitch( switchInfo )
      else:
         switchId = switchIdCache.getSwitch( switchFilter )
         if not switchId:
            mode.addError( 'Switch ' + switchFilter + ' not found.' )
            return model

         switchInfo = vxlanCtrlStatus.switchInfoDir.switchInfo.get( switchId )

         if switchInfo is None:
            mode.addError( 'Switch ID ' + switchFilter + ' not found.' )
            return model

         addSwitch( switchInfo )

      return model

BasicCli.addShowCommandClass( ServiceVxlanSwitchCapabilityCmd )

#--------------------------------------------------------------------------------
# show service vxlan logical-router [ name NAME ]
#--------------------------------------------------------------------------------
class ServiceVxlanLogicalRouterCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show service vxlan logical-router [ name NAME ]'
   data = {
      'service' : serviceAfterShowKwMatcher,
      'vxlan' : matcherVxlan,
      'logical-router' : nodeLogicalRouter,
      'name' : matcherName,
      'NAME' : lRNameMatcher,
   }
   cliModel = VxlanControllerLogicalRouterList

   @staticmethod
   def handler( mode, args ):
      lRName = args.get( 'NAME' )
      LazyMount.force( vxlanCtrlStatus )
      forceControllerdbMount( switchPublishStatus )

      def instantiate( mount ):
         vniDotted = vxlanCtrlConfig.vniInDottedNotation
         
         return Tac.newInstance( 'VxlanController::Cli::VcsLR', switchPublishStatus,
                                 vxlanCtrlStatus.switchInfoDir, mount,
                                 vniDotted, getLRNameFilter( lRName ) )

      return processDeferredModel( mode, lRStatusDir,
                                   VxlanControllerLogicalRouterList, instantiate )


BasicCli.addShowCommandClass( ServiceVxlanLogicalRouterCmd )

#--------------------------------------------------------------------------------
# show service vxlan logical-router [ name NAME ] route [ ROUTE ]
#--------------------------------------------------------------------------------
class ServiceVxlanLogicalRouterRouteCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show service vxlan logical-router [ name NAME ] route [ ROUTE ]'
   data = {
      'service' : serviceAfterShowKwMatcher,
      'vxlan' : matcherVxlan,
      'logical-router' : nodeLogicalRouter,
      'name' : matcherName,
      'NAME' : lRNameMatcher,
      'route' : 'Show routes for the specified logical router',
      'ROUTE' : IpAddrMatcher.IpPrefixMatcher( helpdesc='Filter by route prefix' )
   }
   cliModel = VxlanControllerRouteList

   @staticmethod
   def handler( mode, args ):
      lRName = args.get( 'NAME' )
      prefix = args.get( 'ROUTE' )
      def instantiate( mount ):
         prefixFilter = Tac.newInstance( 'Arnet::IpGenPrefix',
               prefix if prefix is not None else '' )

         return Tac.newInstance( 'VxlanController::Cli::VcsLRRoute', mount,
                                 getLRNameFilter( lRName ) , prefixFilter )

      try:
         return processDeferredModel( mode, lRStatusDir,
               VxlanControllerRouteList, instantiate )
      except IndexError:
         mode.addError( 'Invalid route prefix' )
         return VxlanControllerRouteList

BasicCli.addShowCommandClass( ServiceVxlanLogicalRouterRouteCmd )

#--------------------------------------------------------------------------------
# show service vxlan logical-router [ name NAME ]
#                              ( ( vtep { VTEP } ) | ( switch ( all | SWITCH_ID ) ) )
#--------------------------------------------------------------------------------
class ServiceVxlanLogicalRouterMapCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show service vxlan logical-router [ name NAME ] '
                           '( ( vtep { VTEP } ) | ( switch ( all | SWITCH_ID ) ) )' )
   data = {
      'service' : serviceAfterShowKwMatcher,
      'vxlan' : matcherVxlan,
      'logical-router' : nodeLogicalRouter,
      'name' : matcherName,
      'NAME' : lRNameMatcher,
      'vtep' : 'Logical router to VTEP information',
      'VTEP' : IpAddrMatcher.IpAddrMatcher( helpdesc='VTEP address' ),
      'switch' : 'Logical router to switch information',
      'all' : matcherAll,
      'SWITCH_ID' : switchIdMatcher,
   }
   cliModel = VxlanControllerLogicalRouterMap

   @staticmethod
   def handler( mode, args ):
      lRName = args.get( 'NAME' )
      LazyMount.force( vxlanCtrlStatus )
      LazyMount.force( switchPublishStatus )
  
      switchFilterArg = args.get( 'SWITCH_ID', 'all' ) if 'switch' in args else None
      switchFilter = None
      if switchFilterArg and switchFilterArg != 'all':
         switchFilter = switchIdCache.getSwitch( switchFilterArg )

         if not switchFilter:
            mode.addError( 'Switch ' + switchFilterArg + ' not found.' )
            return VxlanControllerLogicalRouterMap

      def instantiate( mount ):
         return Tac.newInstance( 'VxlanController::Cli::VcsLRMap', mount,
                                 vxlanCtrlStatus.switchInfoDir,
                                 getLRNameFilter( lRName ),
                                 switchFilter if switchFilter else '',
                                 getVtepFilter( args.get( 'VTEP', [] ) ) )
      return processDeferredModel( mode, switchPublishStatus,
                            VxlanControllerLogicalRouterMap, instantiate )

BasicCli.addShowCommandClass( ServiceVxlanLogicalRouterMapCmd )

#------------------------------------------------------------------------------------
# show service vxlan address-table advertised [ DEST ] [ vni VNI ]
#                                              [ address MAC_ADDR ] [ vtep { VTEP } ]
#------------------------------------------------------------------------------------
class ServiceVxlanAddressTableAdvertisedCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show service vxlan address-table advertised [ DEST ] [ vni VNI ] '
                                          '[ address MAC_ADDR ] [ vtep { VTEP } ]' )
   data = {
      'service' : serviceAfterShowKwMatcher,
      'vxlan' : matcherVxlan,
      'address' : matcherAddress,
      'address-table' : nodeAddressTable,
      'advertised' : 'Advertised MAC addresses',
      'DEST': CliMatcher.EnumMatcher( {
         'hsc' : 'Filter by entries advertised to HSC',
         'evpn' : 'Filter by entries advertised to EVPN',
      } ),
      'vni' : vniKwMatcher,
      'VNI' : vniMatcher,
      'MAC_ADDR' : MacAddr.macAddrMatcher,
      'vtep' : vtepKwMatcher,
      'VTEP' : vtepAddrMatcher,
   }
   cliModel = VxlanVniStatusList

   @staticmethod
   def handler( mode, args ):
      dest = args.get( 'DEST' )
      vniDottedFormat = args.get( 'VNI' )
      macAddr = args.get( 'MAC_ADDR' )
      vteps = args.get( 'VTEP' )
      def instantiate( vniDir, auxMountPoints ):
         vniDotted = vxlanCtrlConfig.vniInDottedNotation
         vniFilter = VniFormat( vniDottedFormat ).vniNum if vniDottedFormat else \
                        Tac.Value( 'Vxlan::VniOrNone' )
         macFilter = macAddr if macAddr else Tac.Type( "Arnet::EthAddr" ).ethAddrZero
         # pylint: disable-next=consider-using-in
         destFilter = dest if dest == "hsc" or dest == "evpn" else ''

         vniStatusDirs = []
         hostFloodTables = []
         for svc, mnt in auxMountPoints:
            if svc == "EVPN":
               hostFloodTables.append( ( svc, mnt ) )
            else:
               vniStatusDirs.append( (svc, mnt ) )

         return Tac.newInstance( "VxlanController::Cli::VcsVniAdvertised", vniDir,
                  getVniStatusDirList( vniStatusDirs if vniStatusDirs else [] ),
                 getHostFloodTableList( hostFloodTables if hostFloodTables else [] ),
                  vniDotted, vniFilter, macFilter, destFilter,
                  getVtepFilter( vteps if vteps else [] ) )
      vniStatusDirs = [ ( "HSC", globalVcsToHscVniStatusDir ) ]
      vniStatusDirs.append( ( "EVPN",
                        ( globalVcsToEvpnHostTable, globalVcsToEvpnFloodTable ) ) )
      return processDeferredModel( mode, globalVniDir,
                                   VxlanVniStatusList, instantiate, vniStatusDirs )


BasicCli.addShowCommandClass( ServiceVxlanAddressTableAdvertisedCmd )

#------------------------------------------------------------------------------------
# show service vxlan address-table received
#                           [ ( hsc | mss | evpn | ( switch ( all | SWITCH_ID ) ) ) ]
#                           [ vni VNI ] [ address MAC_ADDR ] [ vtep { VTEP } ]
#------------------------------------------------------------------------------------
class ServiceVxlanAddressTableReceivedCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show service vxlan address-table received '
                         '[ ( hsc | mss | evpn | ( switch ( all | SWITCH_ID ) ) ) ] '
                         '[ vni VNI ] [ address MAC_ADDR ] [ vtep { VTEP } ]' )
   data = {
      'service' : serviceAfterShowKwMatcher,
      'vxlan' : matcherVxlan,
      'address-table' : nodeAddressTable,
      'address' : matcherAddress,
      'received' : 'Received MAC addresses',
      'hsc' : 'Filter by entries received from HSC',
      'mss' : 'Filter by entries received from MSS',
      'evpn' : 'Filter by entries received from EVPN',
      'switch' : switchKwMatcher,
      'all' : matcherAll,
      'SWITCH_ID' : switchIdMatcher,
      'vni' : vniKwMatcher,
      'VNI' : vniMatcher,
      'MAC_ADDR' : MacAddr.macAddrMatcher,
      'vtep' : vtepKwMatcher,
      'VTEP' : vtepAddrMatcher,
   }
   cliModel = VxlanControllerAddressTableReceivedModel

   @staticmethod
   def handler( mode, args ):
      vniDottedFormat = args.get( 'VNI' )
      macAddr = args.get( 'MAC_ADDR' )
      vteps = args.get( 'VTEP' )
      def instantiate( switchDir, auxMountPoints ):
         vniDotted = vxlanCtrlConfig.vniInDottedNotation
         vniFilter = VniFormat( vniDottedFormat ).vniNum if vniDottedFormat else \
                        Tac.Value( 'Vxlan::VniOrNone' )
         macFilter = macAddr if macAddr else Tac.Type( "Arnet::EthAddr" ).ethAddrZero
         switchFilter = ''
         if 'hsc' in args:
            switchFilter = 'hsc'
         elif 'mss' in args:
            switchFilter = 'mss'
         elif 'evpn' in args:
            switchFilter = 'evpn'
         elif 'switch' in args:
            if 'all' in args:
               switchFilter = 'all'
            else:
               switchFilter = switchIdCache.getSwitch( args[ 'SWITCH_ID' ] )

         # switchFilter == False means that a specific switchId was provided but
         # it was not found in the switchIdCache
         if switchFilter is False:
            return Tac.newInstance( "VxlanController::Cli::VcsVniReceived", None,
                        Tac.newInstance( "VxlanController::Cli::VniStatusDirList" ),
                       Tac.newInstance( "VxlanController::Cli::HostFloodTableList" ),
                        False, Tac.Value( 'Vxlan::VniOrNone' ),
                        Tac.Type( "Arnet::EthAddr" ).ethAddrZero, '',
                        Tac.newInstance( "VxlanController::Cli::VtepList" ) )

         vniStatusDirs = []
         hostFloodTables = []
         for svc, mnt in auxMountPoints:
            if svc == "EVPN":
               hostFloodTables.append( ( svc, mnt ) )
            else:
               vniStatusDirs.append( ( svc, mnt ) )


         return Tac.newInstance( "VxlanController::Cli::VcsVniReceived", switchDir,
                  getVniStatusDirList( vniStatusDirs if vniStatusDirs else [] ),
                 getHostFloodTableList( hostFloodTables if hostFloodTables else [] ),
                  vniDotted, vniFilter, macFilter, switchFilter, 
                  getVtepFilter( vteps if vteps else [] ) )

      auxMountPoints = [ ( "HSC", globalHscToVcsVniStatusDir ) ]
      auxMountPoints.append( ( "MSS", globalMssVniStatusDir ) )
      auxMountPoints.append( ( "EVPN", 
                        ( globalEvpnToVcsHostTable, globalEvpnToVcsFloodTable ) ) )
      return processDeferredModel( mode, globalSwitchDir, 
                                   VxlanControllerAddressTableReceivedModel,
                                   instantiate, auxMountPoints )

BasicCli.addShowCommandClass( ServiceVxlanAddressTableReceivedCmd )

#--------------------------------------------------------------------------------
# show service vxlan arp advertised [ vni VNI ] [ mac MACADDR ] [ ip IPADDR ]
#--------------------------------------------------------------------------------
class ServiceVxlanArpAdvertisedCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show service vxlan arp advertised [ vni VNI ] [ mac MAC_ADDR ] '
                                                                  '[ ip IP_ADDR ]' )
   data = {
      'service' : serviceAfterShowKwMatcher,
      'vxlan' : matcherVxlan,
      'arp' : nodeArp,
      'advertised' : 'Advertised ARP entries',
      'vni' : vniKwMatcher,
      'VNI' : vniMatcher,
      'mac' : macKwMatcher,
      'MAC_ADDR' : MacAddr.macAddrMatcher,
      'ip' : ipKwMatcher,
      'IP_ADDR' : IpAddrMatcher.ipAddrMatcher,
   }
   cliModel = VxlanFdbSetModel

   @staticmethod
   def handler( mode, args ):
      vniDottedFormat = args.get( 'VNI' )
      macAddr = args.get( 'MAC_ADDR' )
      ipAddr = args.get( 'IP_ADDR' )
      def instantiate( vniDir ):
         vniDotted = vxlanCtrlConfig.vniInDottedNotation
         vniFilter = VniFormat( vniDottedFormat ).vniNum if vniDottedFormat else \
                        Tac.Value( 'Vxlan::VniOrNone' )
         macFilter = macAddr if macAddr else Tac.Type( "Arnet::EthAddr" ).ethAddrZero
         ipFilter = makeIpGenAddr( ipAddr ) if ipAddr else makeDefaultIpGenAddr()

         return Tac.newInstance( "VxlanController::Cli::ArpNdAdvertised", vniDir,
                                 vniDotted, vniFilter, macFilter, ipFilter,
                                 Tac.Type( 'Arnet::AddressFamily' ).ipv4 )

      return processDeferredModel( mode, globalVniDir, VxlanFdbSetModel,
                                   instantiate )

BasicCli.addShowCommandClass( ServiceVxlanArpAdvertisedCmd )

#--------------------------------------------------------------------------------
# show service vxlan nd advertised [ vni VNI ] [ mac MACADDR ] [ ip IPADDR ]
# When toggle is no longer needed, it should be merged to
# ServiceVxlanArpAdvertisedCmd to reduce duplication
#--------------------------------------------------------------------------------
class ServiceVxlanNdAdvertisedCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show service vxlan nd advertised [ vni VNI ] [ mac MAC_ADDR ] '
                                                                  '[ ip IP_ADDR ]' )
   data = {
      'service' : serviceAfterShowKwMatcher,
      'vxlan' : matcherVxlan,
      'nd' : nodeNd,
      'advertised' : 'Advertised Neighbor Discovery entries',
      'vni' : vniKwMatcher,
      'VNI' : vniMatcher,
      'mac' : macKwMatcher,
      'MAC_ADDR' : MacAddr.macAddrMatcher,
      'ip' : ipKwMatcher,
      'IP_ADDR' : Ip6AddrMatcher.ip6AddrMatcher,
   }
   cliModel = VxlanFdbSetModel

   @staticmethod
   def handler( mode, args ):
      vniDottedFormat = args.get( 'VNI' )
      macAddr = args.get( 'MAC_ADDR' )
      ipAddr = args.get( 'IP_ADDR' )
      def instantiate( vniDir ):
         vniDotted = vxlanCtrlConfig.vniInDottedNotation
         vniFilter = VniFormat( vniDottedFormat ).vniNum if vniDottedFormat else \
                        Tac.Value( 'Vxlan::VniOrNone' )
         macFilter = macAddr if macAddr else Tac.Type( "Arnet::EthAddr" ).ethAddrZero
         ipFilter = makeIpGenAddrV6( ipAddr ) if ipAddr else makeDefaultIpGenAddr()

         return Tac.newInstance( "VxlanController::Cli::ArpNdAdvertised", vniDir,
                                 vniDotted, vniFilter, macFilter, ipFilter,
                                 Tac.Type( 'Arnet::AddressFamily' ).ipv6 )

      return processDeferredModel( mode, globalVniDir, VxlanFdbSetModel,
                                   instantiate )

if Toggles.VxlanControllerToggleLib.toggleVxlanVccNdProxySupportedEnabled():
   BasicCli.addShowCommandClass( ServiceVxlanNdAdvertisedCmd )

#------------------------------------------------------------------------------------
# show service vxlan arp received [ switch SWITCH ] [ vni VNI ] [ mac MAC_ADDR ]
#                                                                      [ ip IP_ADDR ]
#------------------------------------------------------------------------------------
class ServiceVxlanArpReceivedCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show service vxlan arp received [ switch SWITCH_ID ] [ vni VNI ] '
                                                '[ mac MAC_ADDR ] [ ip IP_ADDR ]' )
   data = {
      'service' : serviceAfterShowKwMatcher,
      'vxlan' : matcherVxlan,
      'arp' : nodeArp,
      'received' : 'Received ARP entries',
      'switch' : switchKwMatcher,
      'SWITCH_ID' : switchIdMatcher,
      'vni' : vniKwMatcher,
      'VNI' : vniMatcher,
      'mac' : macKwMatcher,
      'MAC_ADDR' : MacAddr.macAddrMatcher,
      'ip' : ipKwMatcher,
      'IP_ADDR' : IpAddrMatcher.ipAddrMatcher,
   }
   cliModel = VxlanVtepSetModel

   @staticmethod
   def handler( mode, args ):
      switchId = args.get( 'SWITCH_ID' )
      vniDottedFormat = args.get( 'VNI' )
      macAddr = args.get( 'MAC_ADDR' )
      ipAddr = args.get( 'IP_ADDR' )

      def instantiate( switchDir ):
         vniDotted = vxlanCtrlConfig.vniInDottedNotation
         vniFilter = VniFormat( vniDottedFormat ).vniNum if vniDottedFormat else \
                        Tac.Value( 'Vxlan::VniOrNone' )
         macFilter = macAddr if macAddr else Tac.Type( "Arnet::EthAddr" ).ethAddrZero
         ipFilter = makeIpGenAddr( ipAddr ) if ipAddr else makeDefaultIpGenAddr()
         switchFilter = switchIdCache.getSwitch( switchId ) if switchId else ''
         if switchFilter is False:
            return Tac.newInstance( "VxlanController::Cli::ArpNdReceived", None,
                                    False, 0,
                                    Tac.Type( "Arnet::EthAddr" ).ethAddrZero,
                                    makeDefaultIpGenAddr(), '',
                                    Tac.Type( 'Arnet::AddressFamily' ).ipv4 )

         return Tac.newInstance( "VxlanController::Cli::ArpNdReceived", switchDir,
                                 vniDotted, vniFilter, macFilter, ipFilter,
                                 switchFilter,
                                 Tac.Type( 'Arnet::AddressFamily' ).ipv4 )

      return processDeferredModel( mode, globalSwitchDir, VxlanVtepSetModel,
                                   instantiate )

BasicCli.addShowCommandClass( ServiceVxlanArpReceivedCmd )

#------------------------------------------------------------------------------------
# show service vxlan nd received [ switch SWITCH ] [ vni VNI ] [ mac MAC_ADDR ]
#                                                                      [ ip IP_ADDR ]
# When toggle is no longer needed, it should be merged to
# ServiceVxlanArpReceivedCmd to reduce duplication
#------------------------------------------------------------------------------------
class ServiceVxlanNdReceivedCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show service vxlan nd received [ switch SWITCH_ID ] [ vni VNI ] '
                                             '[ mac MAC_ADDR ] [ ip IP_ADDR ]' )
   data = {
      'service' : serviceAfterShowKwMatcher,
      'vxlan' : matcherVxlan,
      'nd' : nodeNd,
      'received' : 'Received Neighbor Discovery entries',
      'switch' : switchKwMatcher,
      'SWITCH_ID' : switchIdMatcher,
      'vni' : vniKwMatcher,
      'VNI' : vniMatcher,
      'mac' : macKwMatcher,
      'MAC_ADDR' : MacAddr.macAddrMatcher,
      'ip' : ipKwMatcher,
      'IP_ADDR' : Ip6AddrMatcher.ip6AddrMatcher,
   }
   cliModel = VxlanVtepSetModel

   @staticmethod
   def handler( mode, args ):
      switchId = args.get( 'SWITCH_ID' )
      vniDottedFormat = args.get( 'VNI' )
      macAddr = args.get( 'MAC_ADDR' )
      ipAddr = args.get( 'IP_ADDR' )

      def instantiate( switchDir ):
         vniDotted = vxlanCtrlConfig.vniInDottedNotation
         vniFilter = VniFormat( vniDottedFormat ).vniNum if vniDottedFormat else \
                        Tac.Value( 'Vxlan::VniOrNone' )
         macFilter = macAddr if macAddr else Tac.Type( "Arnet::EthAddr" ).ethAddrZero
         ipFilter = makeIpGenAddrV6( ipAddr ) if ipAddr else makeDefaultIpGenAddr()
         switchFilter = switchIdCache.getSwitch( switchId ) if switchId else ''
         if switchFilter is False:
            return Tac.newInstance( "VxlanController::Cli::ArpNdReceived", None,
                                    False, 0,
                                    Tac.Type( "Arnet::EthAddr" ).ethAddrZero,
                                    makeDefaultIpGenAddr(), '',
                                    Tac.Type( 'Arnet::AddressFamily' ).ipv6 )

         return Tac.newInstance( "VxlanController::Cli::ArpNdReceived", switchDir,
                                 vniDotted, vniFilter, macFilter, ipFilter,
                                 switchFilter,
                                 Tac.Type( 'Arnet::AddressFamily' ).ipv6 )

      return processDeferredModel( mode, globalSwitchDir, VxlanVtepSetModel,
                                   instantiate )

if Toggles.VxlanControllerToggleLib.toggleVxlanVccNdProxySupportedEnabled():
   BasicCli.addShowCommandClass( ServiceVxlanNdReceivedCmd )

# Convert list of Vnis to list of VniRange()
def vnisToVniRanges( vnis ):
   vniRanges = []
   for _, g in groupby( enumerate( vnis ), lambda x: x[ 0 ] - x[ 1 ] ):
      vniRangeModel = VniRange()
      iList = list( map( itemgetter( 1 ), g ) )
      if len( iList ) > 1:
         vniRangeModel.minVni = iList[ 0 ]
         vniRangeModel.maxVni = iList[ -1 ]
      else:
         vniRangeModel.minVni = iList[ 0 ]
         vniRangeModel.maxVni = iList[ 0 ]
      vniRanges.append( vniRangeModel )
   return vniRanges

#-------------------------------------------------------------------------------
# show service vxlan vni received [ ( switch SWITCH_ID ) | VNI ]
#-------------------------------------------------------------------------------
class ShowVxlanVniReceviedCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show service vxlan vni received [ ( switch SWITCH_ID ) | VNI ]'
   data = { 'service': serviceAfterShowKwMatcher,
            'vxlan': matcherVxlan,
            'vni': vniKwMatcher,
            'received': 'Received VNIs',
            'switch': switchKwMatcher,
            'SWITCH_ID': switchIdMatcher,
            'VNI': vniMatcher,
   }
   cliModel = SwitchVniListReceivedModel

   # We take only one VNI as a filter today, this will change to a range of
   # VNIs in the future. Code must takes care of that.
   @staticmethod
   def _showReceivedFrom( vniList, switchVniList ):
      switchVniList._vniArg = True # pylint: disable=protected-access
      vniFilterList = set( vniList )

      # We have to invert switch -> VNI relation and group by VNI
      for switchId, switch in globalSwitchDir.items():
         # Find VNIs which are in filter list and published by this switch
         vnisReceived = { str( vni ) for vni in switch.vniStatusV2 }

         for vni in vniFilterList.intersection( vnisReceived ):
            if not vni in switchVniList.vniList:
               switchVniList.vniList[ vni ] = SwitchList()
            switchVniList.vniList[ vni ].switches.append( switchId )

      return switchVniList

   @staticmethod
   def handler( mode, args ):
      switchVniList = SwitchVniListReceivedModel()
      # If no switchDir or ctrldb not enabled return empty
      if controllerNotReady() or not globalSwitchDir:
         return switchVniList

      switchVniList.vniDotted = vxlanCtrlConfig.vniInDottedNotation
      vni = args.get( 'VNI' )
      if vni:
         return ShowVxlanVniReceviedCmd._showReceivedFrom( [ vni ], switchVniList )

      switchId = switchIdCache.getSwitch( args.get( "SWITCH_ID" ) )
      if switchId is False:
         return switchVniList

      # Switch(es) to iterate through built separately to accomodate future
      # use case of multi-switch filters.
      switchIter = [ switchId ] if switchId else sorted( globalSwitchDir )

      for switchId in switchIter:
         switch = globalSwitchDir.get( switchId )
         clientView = vcsStateClientViewDir.get( switchId )
         if not switch or not isinstance( clientView, clientViewType ):
            continue

         vniListModel = VniList()
         vniListModel.vnis = vnisToVniRanges( sorted( switch.vniStatusV2.keys() ) )
         vniListModel.converging = clientView.convergenceInProgress
         switchVniList.switchList[ switchId ] = vniListModel

      return switchVniList

#-------------------------------------------------------------------------------
# show service vxlan vni advertised [ switch SWITCH_ID ]
#-------------------------------------------------------------------------------
class ShowVxlanVniAdvertisedCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show service vxlan vni advertised [ switch SWITCH_ID ]'
   data = { 'service': serviceAfterShowKwMatcher,
            'vxlan': matcherVxlan,
            'vni': vniKwMatcher,
            'advertised': 'Advertised VNIs',
            'switch': switchKwMatcher,
            'SWITCH_ID': switchIdMatcher }
   cliModel = SwitchVniListAdvertisedModel

   @staticmethod
   def _getVniList( switch ):
      """Return list VNIs published to this switch"""
      if not switch:
         return []

      # For all published paths for this vniStatusV2, get the last path
      # component, the VNI.
      vnis = [ int( path.split( '/' )[ -1 ] )
               for path, mountDesc in switch.publishedMount.items()
               if ( mountDesc.serviceName == 'Vxlan' and
                  path.startswith( 'vxlancontroller/version2/vni/' ) ) ]

      return vnis

   @staticmethod
   def handler( mode, args ):
      switchVniList = SwitchVniListAdvertisedModel()
      switchVniList.vniDotted = vxlanCtrlConfig.vniInDottedNotation
      switchId = switchIdCache.getSwitch( args.get( 'SWITCH_ID' ) )
      if controllerNotReady() or switchId is False:
         return switchVniList

      # Switch(es) to iterate through built separately to accomodate future
      # use case of multi-switch filters.
      switchIter = [ switchId ] if switchId else sorted( switchIdCache.switchIds() )

      for switch in switchIter:
         clientView = vcsStateClientViewDir.get( switch )
         if not isinstance( clientView, clientViewType ):
            continue

         vniListModel = VniList()

         systemId = Tac.Value( "Controller::SystemId", switch )
         switchSts = switchPublishStatus.system.get( systemId )
         vniList = sorted( ShowVxlanVniAdvertisedCmd._getVniList( switchSts ) )
         vniListModel.vnis = vnisToVniRanges( vniList )
         vniListModel.converging = clientView.convergenceInProgress
         switchVniList.switchList[ switch ] = vniListModel

      return switchVniList

# Register the 'show service vxlan vni ...' command class
BasicCli.addShowCommandClass( ShowVxlanVniReceviedCmd )
BasicCli.addShowCommandClass( ShowVxlanVniAdvertisedCmd )

#-------------------------------------------------------------------------------
# CVX flood vtep list configuration
#-------------------------------------------------------------------------------
# flood vtep { VTEPS } - replace the entire default vtep list
# flood vtep add { VTEPS } - add to default vtep list
# flood vtep remove { VTEPS } - remove from default vtep list
# no flood vtep - remove all vtep lists
# vni { VNIS } flood vtep { VTEPS } -
#                       replace the entire vtep list for { VNIS }
# vni { VNIS } flood vtep add { VTEPS } - add to existing list for { VNIS }
# vni { VNIS } flood vtep remove { VTEPS } -
#                       remove from exisitng list for { VNIS }
# no vni { VNIS } flood vtep - remove entire flood vtep list for { VNIS }
# no vni flood vtep - remove all vtep lists for all { VNIS }
#-------------------------------------------------------------------------------

class ServiceVxlanFloodVtepCmd( CliCommand.CliCommandClass ):
   syntax = ( '[ ( vni { VNIS } ) ] flood vtep '
              '[ ACTION ] '
              '[ { VTEPS } ]' )
   noOrDefaultSyntax = '[ vni ] [ { VNIS } ] flood vtep ...'

   data = { 'vni': vniKwMatcher,
            'VNIS': vniMatcher,
            'flood': 'Configure flood VTEP list',
            'vtep': vtepKwMatcher,
            'ACTION': CliMatcher.EnumMatcher( {
                  'add': 'Add VTEP(s)',
                  'remove': 'Remove VTEP(s)',
            } ),
            'VTEPS' : vtepAddrMatcher,
   }

   @staticmethod
   def _configVniToVtepList( vni, action, vteps ):
      # handle "no" command
      if not vteps:
         if vni in vxlanCtrlConfig.vniToVtepList:
            macAddr = Tac.Type( "Arnet::EthAddr" ).ethAddrZero
            del vxlanCtrlConfig.vniToVtepList[ vni ].remoteVtep[ macAddr ]
            # for future expansion - non-zero bum mac address allowed
            if not vxlanCtrlConfig.vniToVtepList[ vni ].remoteVtep:
               del vxlanCtrlConfig.vniToVtepList[ vni ]
         return

      # obtain current flood list for (vni,macAddr) pair
      if vni in vxlanCtrlConfig.vniToVtepList:
         vtepList = vxlanCtrlConfig.vniToVtepList[ vni ]
      else:
         vtepList = vxlanCtrlConfig.vniToVtepList.newMember( vni )
      macAddr = Tac.Type( "Arnet::EthAddr" ).ethAddrZero
      if macAddr in vtepList.remoteVtep:
         remoteVtep = vtepList.remoteVtep[ macAddr ]
      else:
         remoteVtep = vtepList.remoteVtep.newMember( macAddr )

      # handle add/remove/replace
      if action == "add":
         for v in vteps:
            if v not in remoteVtep.vtepIpFloodList:
               remoteVtep.vtepIpFloodList[ v ] = True
      elif action == "remove":
         for v in vteps:
            del remoteVtep.vtepIpFloodList[ v ]
      else:
         for v in remoteVtep.vtepIpFloodList:
            if v not in vteps:
               del remoteVtep.vtepIpFloodList[ v ]
         for v in vteps:
            if v not in remoteVtep.vtepIpFloodList:
               remoteVtep.vtepIpFloodList[ v ] = True

      # clean up if all vteps removed
      if not remoteVtep.vtepIpFloodList:
         del vtepList.remoteVtep[ macAddr ]
         if not vtepList.remoteVtep:
            del vxlanCtrlConfig.vniToVtepList[ vni ]

   @staticmethod
   def _configDefaultFloodList( action, vteps ):
      # handle "no" command
      if not vteps:
         if action not in ( 'add', 'remove' ):
            vxlanCtrlConfig.floodVtepList.clear()
         return

      # add vtep to default vtep list
      if action == "add":
         for v in vteps:
            if v not in vxlanCtrlConfig.floodVtepList:
               vxlanCtrlConfig.floodVtepList[ v ] = True
      # remove vtep from default vtep list
      elif action == "remove":
         for v in vteps:
            del vxlanCtrlConfig.floodVtepList[ v ]
      # replace default vtep list
      else:
         for v in vxlanCtrlConfig.floodVtepList:
            if v not in vteps:
               del vxlanCtrlConfig.floodVtepList[ v ]
         for v in vteps:
            if v not in vxlanCtrlConfig.floodVtepList:
               vxlanCtrlConfig.floodVtepList[ v ] = True

   @staticmethod
   def handler( mode, args ):
      # define action as replace list/ add vtep/ remove vtep
      action = args.get( 'ACTION', 'default' )
      vteps = args.get( 'VTEPS', [] )

      badAddrs = []
      # Check for valid IP addr
      for vtep in vteps:
         addr = makeIpGenAddr( vtep )
         if ( addr.isLoopback or
              addr.isMulticast or
              addr.isUnspecified or
              ( addr.af == 'ipv4' and vtep == '255.255.255.255' ) ):
            badAddrs.append( vtep )
      if badAddrs:
         # pylint: disable-next=consider-using-f-string
         mode.addError( "Invalid vtep address(es) - %s" % ','.join( badAddrs ) )
         return

      # config vni specific list / default flood vtep list
      vniSet = args.get( 'VNIS' )
      if vniSet:
         for key in vniSet:
            vni = VniFormat( key ).toNum()
            ServiceVxlanFloodVtepCmd._configVniToVtepList( vni, action, vteps )
      else:
         ServiceVxlanFloodVtepCmd._configDefaultFloodList( action, vteps )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      action = "default"
      if 'vni' in args:
         vniSet = args.get( 'VNIS' )
         if vniSet: # delete flood list for designated vnis
            for key in vniSet:
               vni = VniFormat( key ).toNum()
               ServiceVxlanFloodVtepCmd._configVniToVtepList( vni, action, [] )
         else: # delete vni specific list for all vnis
            for key in vxlanCtrlConfig.vniToVtepList:
               ServiceVxlanFloodVtepCmd._configVniToVtepList( key, action, [] )
      # delete default vtep list
      else:
         ServiceVxlanFloodVtepCmd._configDefaultFloodList( action, [] )

VxlanConfigMode.addCommandClass( ServiceVxlanFloodVtepCmd )

#--------------------------------------------------------------------------------
# [ no | default ] arp reply relay vtep { VTEP }
#--------------------------------------------------------------------------------
class ArpReplyRelayVtepCmd( CliCommand.CliCommandClass ):
   syntax = 'arp reply relay vtep { VTEP }'
   noOrDefaultSyntax = 'arp reply relay vtep ...'
   data = {
      'arp' : 'Configure ARP feature',
      'reply' : 'Configure ARP Reply',
      'relay' : 'ARP Relay',
      'vtep' : vtepKwMatcher,
      'VTEP' : vtepAddrMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      vteps = args[ 'VTEP' ]
      for vtepIp in vteps:
         ip = Tac.Value( "Arnet::IpGenAddr", vtepIp )
         vxlanCtrlConfig.arpReplyRelayVtep[ ip ] = True
      for vtepIp in vxlanCtrlConfig.arpReplyRelayVtep:
         if vtepIp.stringValue not in vteps:
            del vxlanCtrlConfig.arpReplyRelayVtep[ vtepIp ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vxlanCtrlConfig.arpReplyRelayVtep.clear()

VxlanConfigMode.addCommandClass( ArpReplyRelayVtepCmd )

#------------------------------------------------------------------------------
# show tech-support commands
#------------------------------------------------------------------------------
def _showTechVcsGuard():
   return controllerConfig and controllerConfig.enabled and vxlanCtrlConfig.enable

CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2015-06-11 09:00:00',
   cmds=[ 'show service vxlan status',
          'show service vxlan address-table received',
          'show service vxlan address-table advertised',
          'show service vxlan arp advertised',
          'show service vxlan arp received',
          'show service vxlan nd advertised',
          'show service vxlan nd received',
          'show service vxlan switch all capability' ],
   cmdsGuard=_showTechVcsGuard )

def doVxlanControllerMounts( controllerdbEm ):
   t8( "doVxlanControllerMounts",
       "enabled" if controllerdbEm else "disabled" )

   global globalSwitchDir
   global globalHscToVcsVniStatusDir
   global globalVcsToHscVniStatusDir
   global globalMssVniStatusDir
   global globalVniDir
   global vcsStateV2
   global vcsStateClientViewDir
   global lRStatusDir

   if controllerdbEm:
      globalVniDir = LazyMount.mount( controllerdbEm,
                                      "vxlancontroller/version2/vni",
                                      "Tac::Dir", "ri" )

      globalSwitchDir = LazyMount.mount( controllerdbEm,
                                         "vxlan/version2/vniStatusDir/switch",
                                         "Tac::Dir", "ri" )

      globalHscToVcsVniStatusDir = LazyMount.mount( controllerdbEm,
                                               "hsc/version2/vniStatusDir",
                                               "VxlanController::VniStatusDirV2",
                                               "r" )

      globalVcsToHscVniStatusDir = LazyMount.mount( controllerdbEm,
                                       "vxlancontroller/version2/hsc/vniStatusDir",
                                       "VxlanController::VniStatusDirV2", "r" )

      globalMssVniStatusDir = LazyMount.mount( controllerdbEm,
                                               "mss/version2/vniStatusDir",
                                               "VxlanController::VniStatusDirV2",
                                               "r" )

      vcsStateV2 = LazyMount.mount( controllerdbEm,
                                    "vxlancontroller/version2/vcsState",
                                    "VxlanController::VcsStateV2", "r" )

      vcsStateClientViewDir = LazyMount.mount( controllerdbEm,
                                    "vxlan/version2/vcsStateClientView/switch",
                                    "Tac::Dir", "ri" )

      lRStatusDir = LazyMount.mount( controllerdbEm,
                                    "vxlancontroller/version2/logicalRouter",
                                    "Tac::Dir", "ri" )
   else:
      globalSwitchDir = None
      globalHscToVcsVniStatusDir = None
      globalVcsToHscVniStatusDir = None
      globalMssVniStatusDir = None
      globalVniDir = None
      vcsStateV2 = None
      vcsStateClientViewDir = None
      lRStatusDir = None


@Plugins.plugin( requires=( "ControllerdbMgr", ) )
def Plugin( entityManager ):
   global controllerConfig
   global controllerStatus
   global switchPublishStatus
   global vxlanCtrlConfig
   global vxlanCtrlStatus
   global serviceCfgDir
   global globalEvpnToVcsHostTable
   global globalEvpnToVcsFloodTable
   global globalVcsToEvpnHostTable
   global globalVcsToEvpnFloodTable

   controllerConfig = LazyMount.mount( entityManager,
                                       "controller/config",
                                       "Controllerdb::Config", "r" )

   controllerStatus = LazyMount.mount( entityManager,
                                       "controller/status",
                                       "Controllerdb::Status", "r" )

   switchPublishStatus = LazyMount.mount( entityManager,
                                       "controller/switchpublish/status",
                                       "Controllerdb::PublishStatus", "r" )

   vxlanCtrlConfig = ConfigMount.mount( entityManager,
                                        "vxlancontroller/config",
                                        "VxlanController::Config", "w" )

   # vxlanCtrlConfig is immutable. But VniMatcher needs it at mod-load.
   # Passing it inside an array to make it mutable (boxing).
   vxlanCtrlCfgBox.append( vxlanCtrlConfig )

   vxlanCtrlStatus = LazyMount.mount( entityManager,
                                      "vxlancontroller/status",
                                      "VxlanController::Status", "r" )
   # To let the CVX infrastructure to know that Vxlan service is enabled/disabled
   serviceCfgDir = ConfigMount.mount( entityManager,
                                      "controller/service/config",
                                      "Controller::ServiceConfigDir", "w" )

   # Lazy mounting smash tables used for learning/advertising MAC entries to EVPN
   globalEvpnToVcsHostTable = SmashLazyMount.mount( entityManager,
                                          "bgpToVcs/hostTable",
                                          "VcsEvpn::HostTable",
                                          SmashLazyMount.mountInfo( 'reader' ) )

   globalEvpnToVcsFloodTable = SmashLazyMount.mount( entityManager,
                                          "bgpToVcs/floodTable",
                                          "VcsEvpn::FloodTable",
                                          SmashLazyMount.mountInfo( 'reader' ) )

   globalVcsToEvpnHostTable = SmashLazyMount.mount( entityManager,
                                          "vcsToBgp/hostTable",
                                          "VcsEvpn::HostTable",
                                          SmashLazyMount.mountInfo( 'reader' ) )

   globalVcsToEvpnFloodTable = SmashLazyMount.mount( entityManager,
                                          "vcsToBgp/floodTable",
                                          "VcsEvpn::FloodTable",
                                          SmashLazyMount.mountInfo( 'reader' ) )

   CliPlugin.ControllerdbLib.registerNotifiee( doVxlanControllerMounts )
