#!/usr/bin/env python3
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string
from ArnetModel import Ip4Address
from ArnetModel import IpGenericAddress
from ArnetModel import IpGenericPrefix
from ArnetModel import MacAddress
from CliModel import Bool
from CliModel import Dict
from CliModel import Enum
from CliModel import Float
from CliModel import Int
from CliModel import List
from CliModel import Model
from CliModel import Submodel
from CliModel import DeferredModel
from CliModel import Str
from VxlanVniLib import vniToString, vniRangesToString, vniRangesToVnis
import Ark
import Tac
from TableOutput import createTable, Format
from IntfModels import Interface
import Toggles.VxlanControllerToggleLib

class VxlanMacVtepPair( Model ):
   macAddr = MacAddress( help="Host Mac address" )
   vtepIp = Ip4Address( help="Vtep Ip address" )
   moveCount = Int( help="Number of local and remote Mac moves", optional=True )

class VxlanMacVtepListPair( Model ):
   macAddr = MacAddress( help="Destination MAC address" )
   vtepIpListType = Str( help="BUM forwarding mode" )
   vtepIpList = List( valueType=Ip4Address, help="List of VTEPs" )

class VxlanVniStatus( Model ):
   vlan = Int( help="VLAN", optional=True )
   vni = Int( help="VNI", optional=True )
   macAddress = MacAddress( help="A MAC address", default="ff:FF:ff:FF:ff:FF" )
   unicastHostTable = List( valueType=VxlanMacVtepPair,
                              help="Unicast address table" )
   bumVtepListTable = List( valueType=VxlanMacVtepListPair,
                              help="Flood table" )

class VniRange( Model ):
   minVni = Int( help="Minimum value for VNI range" )
   maxVni = Int( help="Maximum value for VNI range" )

class VniList( Model ):
   vnis = List( valueType=VniRange,
                help="List of VNIs in dotted or plain notation" )
   converging = Bool( default=False,
                      help="True if Vxlan Controller Service sync in progress" )

class SwitchList( Model ):
   switches = List( valueType=str,
                    help="List of Switches, hostname, IP or default ID" )

vcsCapabilityDescription = {
      'routing': 'VXLAN v4 overlay routing',
      'indirectRouting': 'VXLAN v4 overlay indirect routing',
}

class VxlanControllerCapability( Model ):
   routing = Bool( help="Indicates routing support" )
   indirectRouting = Bool( help="Indicates indirect routing support" )

class VxlanControllerStatus( Model ):
   def toUtc( self, timestamp ):
      utcTime = timestamp + Tac.utcNow() - Tac.now()
      return utcTime

   def utcToStr( self, utcThen, relative=False ):
      timeDiff = Tac.utcNow() - utcThen
      tacTimeThen = Tac.now() - timeDiff if utcThen else 0
      return Ark.timestampToStr( tacTimeThen, relative )

   status = Enum( values=( "running", "shutting down", "stopped" ),
                  help="Status of VxlanController agent" )
   macLearning = Enum( values=( "controlPlane", "dataPlane" ),
                  help="MAC learning. Defaults to controlPlane" )
   vniMapping = Enum( values=( "vlan", "portDot1qTag" ),
                      help="VNI mapping. Defaults to vlan" )
   resyncPeriod = Int( help="Sync period in seconds for Vxlan Controller "
                            "Service after CVX restart. Value of 0 means "
                            "finish sync immediately after CVX reboot without "
                            "waiting" )
   resyncStartedAt = Float( help="UTC Time at which sync of Vxlan Controller "
                               "Service started after CVX restart. 0 signifies "
                               "never", optional=True )
   resyncInProgress = Bool( help="When set to True it signifies Vxlan Controller "
                                 "Service sync, after CVX restart, is in progress",
                                optional=True )
   multicastDistributionEnabled = Bool( help="When set to True it signifies "
                                             "multicast floodlist distribution "
                                             "happens on Vxlan Controller", 
                                             optional=True )
   bgpEvpnRedistEnabled = Bool( help="When set to True it means "
                                      "MAC-VTEP bindings generated by BGP EVPN "
                                      "are being redistributed in to Vxlan "
                                      "Control Service",
                                             optional=True )

   capability = Submodel( valueType=VxlanControllerCapability, optional=True,
         help="Vxlan Controller capability" )
   arpRelayVteps = List( valueType=str,
                         help="List of VTEPs relaying ARP replies" )

   def render( self ):
      macLearningText = { "controlPlane" : "Control plane",
                          "dataPlane": "Data plane" }
      vniMappingText = { "vlan" : "VLAN",
                         "portDot1qTag": "Port, dot1q tag pair" }
      print( "%-30s: %s" % ( "Vxlan Controller Service is", self.status ) )
      print( "%-30s: %s" % ( "Mac learning", macLearningText[ self.macLearning ] ) )
      print( "%-30s: %s" % ( "VNI mapping", vniMappingText[ self.vniMapping ] ) )
      print( "%-30s: %s" % ( "Resync period", str( self.resyncPeriod ) +
                            " seconds" ) )
      print( "%-30s: %s" % ( "Resync in progress", "Yes" if 
                                    self.resyncInProgress else "No" ) )
      print( "%-30s: %s" % ( "BGP EVPN Redistribution", "Yes" if 
                                    self.bgpEvpnRedistEnabled else "No" ) )
      if Toggles.VxlanControllerToggleLib.toggleVCSMulticastKnobEnabled():
         print( "%-30s: %s" % ( "Multicast Distribution", "Yes" if 
                                    self.multicastDistributionEnabled else "No" ) )
      if self.resyncInProgress and self.resyncStartedAt is not None:
         print( "%-30s: %s" % ( "Resync started at",
                               self.utcToStr( self.resyncStartedAt ) ) )

      def printFirstCapability( desc ):
         print( "%-30s: %s" % ( 'Capability', desc ) )

      first = True
      for attribute in self.capability.__attributes__:
         if self.capability.__getattribute__( attribute ):
            desc = vcsCapabilityDescription[ attribute ]

            if first:
               first = False
               printFirstCapability( desc )
            else:
               print( " " * 32 + desc )

      if first:
         printFirstCapability( '' )

      if len( self.arpRelayVteps ) > 0:
         print( "%-30s: %s" % ( "VTEPs relaying ARP replies",
                               " ".join( sorted( self.arpRelayVteps ) ) ) )

switchCapabilityDescription = {
      'bfd': 'BFD over VXLAN',
      'routing': 'VXLAN v4 overlay routing',
      'overlayEcmp': 'Overlay layer3 ECMP',
      'createLocalIpPortsOnly' : 'Create LR local ip ports only',
      'portDot1qVniMapping' : 'VNI mapping to port, dot1q tag pairs'
}

class SwitchCapability( Model ):
   __public__ = False

   vtepIp = IpGenericAddress( help="VTEP IP address" )
   mlagVtepIp = IpGenericAddress( help="MLAG VTEP IP address", optional=True )
   hostname = Str( help='Switch hostname' )
   bfd = Bool( help="Indicates VXLAN over BFD support" )
   routing = Bool( help="Indicates VXLAN v4 overlay routing support" )
   overlayEcmp = Bool( help="Indicates overlay layer3 ECMP support" )
   createLocalIpPortsOnly = Bool( help=
           "Indicates support to create local logical-router ip ports only" )
   portDot1qVniMapping = Bool( help=
         "Indicates support for VNI mapping to port, dot1q tag pairs" )

class VxlanControllerSwitchCapability( Model ):
   __public__ = False

   switchCapability = Dict( keyType=MacAddress, valueType=SwitchCapability,
         help="A mapping of switch ID to capabilities" )

   def render( self ):
      attributes = None

      for switchId, model in self.switchCapability.items():
         print( 'Switch %s' % ( switchId, ) )

         cols = [ ( 'Hostname', model.hostname ),
                  ( 'VTEP IP', model.vtepIp ) ]
         if model.mlagVtepIp:
            cols.append( ( 'MLAG VTEP IP', model.mlagVtepIp ) )
         cols.append( ( 'Capability', '' ) )

         for col in cols:
            print( '  %-12s : %s' % col )

         if attributes is None:
            attributes = [ attribute for attribute in model.__attributes__
                  if attribute not in [ 'vtepIp', 'hostname', 'mlagVtepIp' ] ]

         output = ''

         for attribute in attributes:
            if model.__getattribute__( attribute ):
               output += '    %s\n' % ( switchCapabilityDescription[ attribute ], )

         print( output )

class LogicalRouterIpPort( Model ):
   __public__ = False

   vni = Int( help="Vni", optional=True )
   mac = MacAddress( help='Router port MAC address' )
   vtepIp = IpGenericAddress( help='The VTEP address the router IP port is on' )

class LogicalRouterUplinkPort( Model ):
   __public__ = False

   vlan = Int( help="Vlan", optional=True )
   mac = MacAddress( help='Uplink port MAC address' )
   ports = List( valueType=Interface, help='A list of uplink ports' )
   switchId = MacAddress( help='The switch ID the uplink port is on' )
   vtepIp = IpGenericAddress( help='The VTEP address the uplink port is on' )

class LogicalRouter( Model ):
   __public__ = False

   ipPort = Dict( valueType=LogicalRouterIpPort,
         help='A mapping between IP address and router IP ports' )

   uplinkPort = Dict( valueType=LogicalRouterUplinkPort,
         help='A mapping between IP address and uplink ports' )

class VxlanControllerLogicalRouterList( DeferredModel ):
   __public__ = False

   router = Dict( valueType=LogicalRouter,
         help='A mapping between logical router names and their IP and uplink '
         'ports' )

class LogicalRouterNexthop( Model ):
   __public__ = False

   nexthop = List( valueType=IpGenericAddress,
         help='The list of nexthop addresses' )

class LogicalRouterRoute( Model ):
   __public__ = False

   prefix = Dict( keyType=IpGenericPrefix, valueType=LogicalRouterNexthop,
         help='A mapping between prefixes and nexthop addresses' )

class VxlanControllerRouteList( DeferredModel ):
   __public__ = False

   route = Dict( valueType=LogicalRouterRoute,
         help='A mapping between logical router names and their routes' )

class SwitchInfo( Model ):
   __public__ = False

   switchId = MacAddress( help='Switch identifier' )
   vtepIp = IpGenericAddress( help='VTEP IP Address' )

class LogicalRouterSwitchList( Model ):
   __public__ = False

   switches = List( valueType=SwitchInfo,
         help='The list of switches to which the logical router applies' )

class VxlanControllerLogicalRouterMap( DeferredModel ):
   __public__ = False

   logicalRouterMap = Dict( valueType=LogicalRouterSwitchList,
         help='A mapping between logical router names and the switches to which '
         'they apply' )

class VxlanVniStatusList( DeferredModel ):
   __revision__ = 2

   vniInDottedNotation = Bool( help="VNI display format", default=False )
   vnis = List( valueType=VxlanVniStatus,
               help="list of Vxlan Vni Status, one per VNI" )

class VxlanControllerAddressTableReceivedModel( DeferredModel ):
   """
   This is a deferred Cli model and serves only to publish the types
   and help strings. Rendering is done in C++ by the show method.
   """
   __revision__ = 2

   switches = Dict( valueType=VxlanVniStatusList,
                     help="Mapping from switch id to VniStatus" )

   nonSwitchSources = Dict( valueType=VxlanVniStatusList,
                           help="Mapping from non-switch source to VniStatus" )

class SwitchVniListReceivedModel( Model ):
   vniDotted = Bool( help="VNI display format", default=False )
   switchList = Dict( valueType=VniList,
                      help="List of switches and VNIs received from each" )
   vniList = Dict( valueType=SwitchList,
                   help="List of VNIs and switches they were received from" )
   _vniArg = Bool( help="True if model data was filtered by VNI(s)" )

   def renderSwitchVniList( self ):
      print( "VNIs received per switch by Vxlan Control Service" )
      print( '-------------------------------------------------\n' )

      table = createTable( ( "Switch", "VNIs" ) )
      table.formatColumns( Format( justify='left', maxWidth=18, wrap=True ),
                           Format( justify='left', maxWidth=60, wrap=True ) )
      switchCount = 0
      vniSet = set()

      # Max switches is small  - in 100s, straight sorting is good enough
      for switch in sorted( self.switchList.keys() ):
         switchEntry = self.switchList[ switch ]
         table.newRow( ( "*" if switchEntry.converging else ' ' ) + switch,
                       vniRangesToString( switchEntry.vnis, self.vniDotted ) )
         switchCount += 1
         vniSet.update( vniRangesToVnis( switchEntry.vnis ) )

      print( table.output() )
      print( f"{len( vniSet )} VNI(s), {switchCount} Switch(es)" )
      if True in [ switch.converging for switch in self.switchList.values() ]:
         print( "* Vxlan Controller Service sync, "
                "after CVX restart, is in progress" )

   def renderVniBySwitch( self ):
      print( "List of switch(es) receiving given VNI(s)" )
      print( '-------------------------------------------\n' )

      table = createTable( ( "VNI", "Switch(es)" )  )
      table.formatColumns( Format( justify='left', maxWidth=12, wrap=True ),
                           Format( justify='left', maxWidth=66, wrap=True ) )
      switchSet = set()
      vniCount = 0

      # VNIs are in thousands and are ints, simple sort will do.
      for vni in sorted( self.vniList.keys() ):
         switchList = sorted( self.vniList[ vni ].switches )
         table.newRow( vniToString( vni, self.vniDotted ),
                       " ".join( switchList ) )
         vniCount += 1
         switchSet.update( self.vniList[ vni ].switches )

      print( table.output() )

      print( "Total {} VNI(s) across {} Switch(es)\n".format(
         vniCount, len( switchSet ) ) )

   def render( self ):
      if self._vniArg:
         self.renderVniBySwitch()
      else:
         self.renderSwitchVniList()


class SwitchVniListAdvertisedModel( Model ):
   vniDotted = Bool( help="VNI display format", default=False )
   switchList = Dict( valueType=VniList,
                    help="List of VNIs advertised per switch" )

   def render( self ):
      print( "VNIs Advertised per switch by Vxlan Control Service" )
      print( '---------------------------------------------------\n' )
      switchCount = 0
      vniSet = set()

      table = createTable( ( "Switch", "VNIs" )  )
      table.formatColumns( Format( justify='left', maxWidth=18, wrap=True ),
                           Format( justify='left', maxWidth=60, wrap=True ) )

      for switch in sorted( self.switchList.keys() ):
         switchEntry = self.switchList[ switch ]
         table.newRow( ( "*" if switchEntry.converging else ' ' ) + switch,
                       vniRangesToString( switchEntry.vnis, self.vniDotted ) )
         switchCount += 1
         vniSet.update( vniRangesToVnis( switchEntry.vnis ) )
      print( table.output() )

      print( f"{len( vniSet )} VNI(s), {switchCount} Switch(es)" )
      if True in [ switch.converging for switch in self.switchList.values() ]:
         print( "* Vxlan Controller Service sync, "
                "after CVX restart, is in progress" )

class VxlanArpEntry( Model ):
   macAddress = MacAddress( help="MAC address of an ARP entry" )
   preference = Int( help="ARP entry preference value" )
   changes = Int( help="Number of times the ARP entry changed" )

class VxlanArpTable( Model ):
   addresses = Dict( keyType=IpGenericAddress, valueType=VxlanArpEntry,
                     help="A mapping from IP address to ARP entry" )

class VxlanFdbSetModel( DeferredModel ):
   vnis = Dict( keyType=int, valueType=VxlanArpTable,
                help="A mapping from VNI to ARP table" )

class VxlanVtep( DeferredModel ):
   vnis = Dict( keyType=str, valueType=VxlanArpTable,
                help="A mapping from VNI to ARP table" )

class VxlanVtepSetModel( DeferredModel ):
   switches = Dict( keyType=str, valueType=VxlanVtep,
                    help="A mapping from switch ID to vtep" )
