#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliMode import McastVpn
from CliPlugin import ConfigConvert
from CliPlugin import IntfCli
from CliPlugin import IpAddrMatcher, Ip6AddrMatcher
from CliPlugin import VlanCli
from CliPlugin import VxlanCli
import Arnet
import BasicCli
import CliCommand
import CliMatcher
import CliParser
import ConfigMount
import LazyMount
import McastCommonCliLib
import Tac
import Toggles.McastVpnLibToggleLib
import Tracing

t0 = Tracing.trace0

# Global variables for state mounted and shared across command
# classes.
ipTunnelGroupConfig = None
remoteDomainIpTunnelGroupConfig = None
vtiConfigDir = None
bridgingHwCapabilities = None
UnderlayRouteType = Tac.Type( "Routing::Multicast::UnderlayRouteType::RouteType" )

def getIpTunnelGroupConfig( mode ):
   if isinstance( mode, RemoteDomainMode ):
      return remoteDomainIpTunnelGroupConfig
   else:
      return ipTunnelGroupConfig

matcherGroup = CliMatcher.KeywordMatcher(
   'group', helpdesc="Underlay multicast group address" )
groupNode = CliCommand.Node( matcherGroup,
      guard=VxlanCli.vxlanUnderlayMcastSupportedGuard )
matcherMulticast = CliMatcher.KeywordMatcher( 'multicast',
      helpdesc='Multicast vxlan commands' )
multicastVxlanIntfNode = CliCommand.Node( matcherMulticast,
      guard=VxlanCli.isVxlan1InterfaceGuard )
ipFamilyAlias = "AF"
matcherIpv4 = CliMatcher.KeywordMatcher( 'ipv4', helpdesc='Overlay multicast ipv4' )
matcherIpv6 = CliMatcher.KeywordMatcher( 'ipv6', helpdesc='Overlay multicast ipv6' )
ipv4Node = CliCommand.Node( matcherIpv4,
      guard=McastCommonCliLib.mcastRoutingSupportedGuard,
      alias=ipFamilyAlias )
ipv6Node = CliCommand.Node( matcherIpv6,
      guard=VxlanCli.vxlanMcastIpv6OverlaySupportedGuard,
      alias=ipFamilyAlias )

addrMatcher = IpAddrMatcher.IpAddrMatcher( helpdesc="IP multicast group address" )

addr6Matcher = CliCommand.Node(
   matcher=Ip6AddrMatcher.Ip6AddrMatcher( 'IPv6 multicast group address' ),
   guard=VxlanCli.vxlanMcastIpv6UnderlaySupportedGuard )

class VxlanMcastIpFamilyExpr( CliCommand.CliExpression ):
   expression = "ipv4 | ipv6"
   data = {
      'ipv4': ipv4Node,
      'ipv6': ipv6Node
   }

def maybeCleanupIntfConfig( mode ):
   intfName = mode.intf.name
   ipTunGrpCfg = getIpTunnelGroupConfig( mode )
   groupConfig = ipTunGrpCfg.intfConfig
   if groupConfig.get( intfName ) and groupConfig[
         intfName ].isConfigEmpty:
      t0( "Cleanup tunnel intf config", intfName )
      del groupConfig[ intfName ]

groupNodeFlood = CliCommand.Node( matcherGroup,
      guard=VxlanCli.vxlanFloodOverUnderlayMcastSupportedGuard )
#-------------------------------------------------------------------------------
# config-interface-vxlan: vxlan vlan <vlanRange> flood group <ip-addr>
#                         vxlan vlan add <vlanRange> flood group <ip-addr>
#                         vxlan vlan remove <vlanRange> flood group <ip-addr>
#                         [ no | default ] vxlan vlan <vlanRange> flood group
#
# sets the vlan to flood group mappings under interface vxlan
#-------------------------------------------------------------------------------
def vlanToFloodGroupAdd( mode, ipTunnelGroupIntfConfig, vlanList, group ):
   if len( vlanList ) > 1:
      vtiConfigDir.vlanFloodGroupRangeSyntax = True
   for vlan in vlanList:
      defaultUnderlayGroup = Tac.Value( 'Routing::Multicast::UnderlayAddr', group )
      configEntry = Tac.Value( 'Routing::Multicast::IpTunnelFloodConfigEntry',
                               vlan, 'tunnelTypeStatic', defaultUnderlayGroup )
      ipTunnelGroupIntfConfig.vlanToFloodGroup.addMember( configEntry )

def validateVlanToFloodGroupRemove( mode, ipTunnelGroupIntfConfig, vlanList, group ):
   for vlan in vlanList:
      if vlan not in ipTunnelGroupIntfConfig.vlanToFloodGroup:
         mode.addError( "VLAN %d is not mapped." % vlan )
         return
      floodGroup = ipTunnelGroupIntfConfig.vlanToFloodGroup[ vlan ]
      if floodGroup and \
         floodGroup.defaultUnderlayGroup.stringValue != group:
         mode.addError( "VLAN %d is not mapped to group %s." % ( vlan, group ) )
         return
   for vlan in vlanList:
      del ipTunnelGroupIntfConfig.vlanToFloodGroup[ vlan ]
   maybeCleanupIntfConfig( mode )

def convertVlanToFloodGroupSyntax( mode ):
   # New syntax must persist even if Vxlan interface is removed and readded
   vtiConfigDir.vlanFloodGroupRangeSyntax = True

#-------------------------------------------------------------------------------
# Register convertVlanToFloodGroupSyntax via "config convert new-syntax"
#-------------------------------------------------------------------------------
ConfigConvert.registerConfigConvertCallback( convertVlanToFloodGroupSyntax )

class McastVpnFloodGroupCmd( CliCommand.CliCommandClass ):
   syntax = "vxlan vlan [ ACTION ] VLAN_LIST flood group \
   ( MCAST_GROUP_ADDR | MCAST_GROUP_IP6_ADDR )"
   if not Toggles.McastVpnLibToggleLib.toggleMcastVpnUnderlayMulticastV6Enabled():
      syntax = "vxlan vlan [ ACTION ] VLAN_LIST flood group MCAST_GROUP_ADDR"

   noOrDefaultSyntax = "vxlan vlan [ ACTION ] VLAN_LIST flood group ..."
   data = {
      'vxlan': VxlanCli.vxlanNode,
      'vlan': VxlanCli.vlanVxlan1Node,
      'ACTION': CliMatcher.EnumMatcher( {
         'add': 'Add VLAN to flood group mapping',
         'remove': 'Remove VLAN to flood group mapping',
      } ),
      'VLAN_LIST': VxlanCli.vlanMultiMatcher,
      'flood': VxlanCli.matcherFlood,
      'group': groupNodeFlood,
      'MCAST_GROUP_ADDR': addrMatcher,
   }
   if Toggles.McastVpnLibToggleLib.toggleMcastVpnUnderlayMulticastV6Enabled():
      data.update( { 'MCAST_GROUP_IP6_ADDR': addr6Matcher } )

   @staticmethod
   def handler( mode, args ):
      t0( "McastVpnFloodGroupCmd handler args", args )
      if Toggles.McastVpnLibToggleLib.toggleMcastVpnUnderlayMulticastV6Enabled():
         group = args.get( 'MCAST_GROUP_ADDR' ) or args.get( 'MCAST_GROUP_IP6_ADDR' )
      else:
         group = args.get( 'MCAST_GROUP_ADDR' )

      validationError = McastCommonCliLib.validateMulticastAddress(
        group, allowReserved=False )
      if validationError:
         mode.addError( validationError )
         return

      group = getattr( group, 'stringValue', group )
      vlanList = list( args[ 'VLAN_LIST' ].values() )
      intf = mode.intf.name
      ipTunnelGroupIntfConfig = getIpTunnelGroupConfig( mode ).newIntfConfig( intf )
      if 'ACTION' in args:
         if args[ 'ACTION' ] == 'add':
            vlanToFloodGroupAdd( mode, ipTunnelGroupIntfConfig, vlanList, group )
         else:
            validateVlanToFloodGroupRemove( mode, ipTunnelGroupIntfConfig, vlanList,
                                            group )
      else:
         if vtiConfigDir.vlanFloodGroupRangeSyntax:
            for vlan in ipTunnelGroupIntfConfig.vlanToFloodGroup:
               floodGroup = ipTunnelGroupIntfConfig.vlanToFloodGroup[ vlan ]
               if floodGroup and \
                  floodGroup.defaultUnderlayGroup.stringValue == group:
                  mode.addError( ( "VLAN to group mapping already exists for " +
                                   "group %s. Use 'add/remove' keyword to modify " +
                                   "existing map." ) % group )
                  return
         vlanToFloodGroupAdd( mode, ipTunnelGroupIntfConfig, vlanList, group )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      t0( "McastVpnFloodGroupCmd noOrDefaultHandler args", args )
      intf = mode.intf.name
      ipTunnelGroupIntfConfig = getIpTunnelGroupConfig( mode ).intfConfig.get( intf )
      if ipTunnelGroupIntfConfig is None:
         t0( intf, "doesn't exist to cleanup vlanToFloodGroup" )
         return
      vlanList = list( args[ 'VLAN_LIST' ].values() )
      if len( vlanList ) > 1:
         vtiConfigDir.vlanFloodGroupRangeSyntax = True
      for vlanId in vlanList:
         del ipTunnelGroupIntfConfig.vlanToFloodGroup[ vlanId ]
      maybeCleanupIntfConfig( mode )

VxlanCli.VxlanIntfModelet.addCommandClass( McastVpnFloodGroupCmd )

#-------------------------------------------------------------------------------
# config-interface-vxlan: vxlan vlan <id> multicast group <ip-addr|ip6-addr>
#-------------------------------------------------------------------------------
class McastVpnMulticastGroupCmd( CliCommand.CliCommandClass ):
   syntax = "vxlan vlan VLAN_ID multicast group \
   ( MCAST_GROUP_ADDR | MCAST_GROUP_IP6_ADDR )"
   if not Toggles.McastVpnLibToggleLib.toggleMcastVpnUnderlayMulticastV6Enabled():
      syntax = "vxlan vlan VLAN_ID multicast group MCAST_GROUP_ADDR"

   noOrDefaultSyntax = "vxlan vlan VLAN_ID multicast group ..."
   data = {
      'vxlan': VxlanCli.vxlanNode,
      'vlan': VxlanCli.vlanVxlan1Node,
      'VLAN_ID': VlanCli.vlanIdMatcher,
      'multicast': matcherMulticast,
      'group': groupNode,
      'MCAST_GROUP_ADDR': addrMatcher,
   }

   if Toggles.McastVpnLibToggleLib.toggleMcastVpnUnderlayMulticastV6Enabled():
      data.update( { 'MCAST_GROUP_IP6_ADDR': addr6Matcher } )

   @staticmethod
   def handler( mode, args ):
      t0( "McastVpnMulticastGroupCmd handler args", args )
      if Toggles.McastVpnLibToggleLib.toggleMcastVpnUnderlayMulticastV6Enabled():
         mcastGroupAddr = args.get( 'MCAST_GROUP_ADDR' ) or \
            args.get( 'MCAST_GROUP_IP6_ADDR' )
      else:
         mcastGroupAddr = args.get( 'MCAST_GROUP_ADDR' )

      validationError = McastCommonCliLib.validateMulticastAddress(
         mcastGroupAddr, allowReserved=False )
      if validationError:
         mode.addError( validationError )
         return
      vlanId = args[ 'VLAN_ID' ].id
      intf = mode.intf.name
      ipTunnelGroupIntfConfig = getIpTunnelGroupConfig( mode ).newIntfConfig( intf )
      vlanEntry = ipTunnelGroupIntfConfig.vlanToMulticastGroup.newMember( vlanId,
                     'tunnelTypeStatic' )

      mcastGroupAddr = getattr( mcastGroupAddr, 'stringValue', mcastGroupAddr )
      defaultUnderlayGroup = Tac.Value( 'Routing::Multicast::UnderlayAddr',
                                        mcastGroupAddr )
      vlanEntry.tunnelConfigEntry.defaultUnderlayGroup = defaultUnderlayGroup

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      t0( "McastVpnMulticastGroupCmd noOrDefaultHandler args", args )
      intf = mode.intf.name
      ipTunnelGroupIntfConfig = getIpTunnelGroupConfig( mode ).intfConfig.get( intf )
      if ipTunnelGroupIntfConfig is None:
         t0( intf, "doesn't exist to cleanup vlanToMulticastGroup" )
         return
      del ipTunnelGroupIntfConfig.vlanToMulticastGroup[ args[ 'VLAN_ID' ].id ]
      maybeCleanupIntfConfig( mode )

VxlanCli.VxlanIntfModelet.addCommandClass( McastVpnMulticastGroupCmd )

#-------------------------------------------------------------------------------
# config-interface-vxlan: vxlan vrf <name> multicast group <ip-addr>
#-------------------------------------------------------------------------------
class McastVpnVrfGroupCmd( CliCommand.CliCommandClass ):
   syntax = "vxlan VRF multicast group ( MCAST_GROUP_ADDR | MCAST_GROUP_IP6_ADDR )"
   if not Toggles.McastVpnLibToggleLib.toggleMcastVpnUnderlayMulticastV6Enabled():
      syntax = "vxlan VRF multicast group MCAST_GROUP_ADDR"
   noOrDefaultSyntax = "vxlan VRF multicast group ..."
   data = {
      'vxlan': VxlanCli.vxlanNode,
      'VRF': VxlanCli.vrfExprFactory,
      'multicast': CliCommand.Node(
         matcher=matcherMulticast,
         guard=VxlanCli.vxlanMcastRoutingOverUnderlayMcastSupportedGuard ),
      'group': groupNode,
      'MCAST_GROUP_ADDR': addrMatcher,
   }
   if Toggles.McastVpnLibToggleLib.toggleMcastVpnUnderlayMulticastV6Enabled():
      data.update( { 'MCAST_GROUP_IP6_ADDR': addr6Matcher } )

   @staticmethod
   def handler( mode, args ):
      t0( "McastVpnVrfGroupCmd handler args", args )
      if Toggles.McastVpnLibToggleLib.toggleMcastVpnUnderlayMulticastV6Enabled():
         mcastGroupAddr = args.get( 'MCAST_GROUP_ADDR' ) or \
            args.get( 'MCAST_GROUP_IP6_ADDR' )
      else:
         mcastGroupAddr = args.get( 'MCAST_GROUP_ADDR' )

      validationError = McastCommonCliLib.validateMulticastAddress(
         mcastGroupAddr, allowReserved=False )
      if validationError:
         mode.addError( validationError )
         return
      vrfName = args[ 'VRF' ]
      intf = mode.intf.name
      ipTunnelGroupIntfConfig = getIpTunnelGroupConfig( mode ).newIntfConfig( intf )
      vrfEntry = ipTunnelGroupIntfConfig.vrfToMulticastGroup.newMember( vrfName,
                    'tunnelTypeStatic' )

      mcastGroupAddr = getattr( mcastGroupAddr, 'stringValue', mcastGroupAddr )
      defaultUnderlayGroup = Tac.Value( 'Routing::Multicast::UnderlayAddr',
                                        mcastGroupAddr )
      vrfEntry.tunnelConfigEntry.defaultUnderlayGroup = defaultUnderlayGroup

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      t0( "McastVpnVrfGroupCmd noOrDefaultHandler args", args )
      intf = mode.intf.name
      ipTunnelGroupIntfConfig = getIpTunnelGroupConfig( mode ).intfConfig.get( intf )
      if ipTunnelGroupIntfConfig is None:
         t0( intf, "doesn't exist to cleanup vrfToMulticastGroup" )
         return
      vrf = args[ 'VRF' ]
      vrfEntry = ipTunnelGroupIntfConfig.vrfToMulticastGroup.get( vrf )
      if not vrfEntry:
         t0( vrf, " multicast vrf doesn't exist in tunnel group intf config" )
         return
      vrfEntry.tunnelConfigEntry.defaultUnderlayGroup = \
         Tac.Value( "Routing::Multicast::UnderlayAddr" )
      if tunnelConfigEntryDefault( vrfEntry.tunnelConfigEntry ):
         del ipTunnelGroupIntfConfig.vrfToMulticastGroup[ vrf ]
      maybeCleanupIntfConfig( mode )

VxlanCli.VxlanIntfModelet.addCommandClass( McastVpnVrfGroupCmd )

# Returns true if there is no explicit config associated with IpTunnelConfigEntry
def tunnelConfigEntryDefault( tunnelConfigEntry ):
   if tunnelConfigEntry.defaultUnderlayGroup:
      return False
   if tunnelConfigEntry.underlayGroupRange:
      return False
   if tunnelConfigEntry.delayedOverlayToUnderlayGroup:
      return False
   if tunnelConfigEntry.immediateOverlayToUnderlayGroup:
      return False
   return True

matcherEncap = CliMatcher.KeywordMatcher(
   'encap', helpdesc='Underlay multicast group address for encapsulation' )
matcherDelayed = CliMatcher.KeywordMatcher(
   'delayed', helpdesc='Delay encapsulation with specific group for some duration' )
encapRangePrefix = IpAddrMatcher.IpPrefixMatcher(
          'IPv4 multicast group with prefix length between 19 and 32',
          overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO )
# -------------------------------------------------------------------------------
# config-interface-vxlan:
#     vxlan vrf <vrf> multicast group overlay <ipv6/v4Addr> encap <ipAddr> immediate
# -------------------------------------------------------------------------------

class McastVpnVrfGroupOverlayEncapCmd( CliCommand.CliCommandClass ):
   syntax = \
      'vxlan VRF multicast group overlay OVERLAY_ADDR encap ENCAP_ADDR immediate'
   noOrDefaultSyntax = syntax
   data = {
      'vxlan': VxlanCli.vxlanNode,
      'VRF': VxlanCli.vrfExprFactory,
      'multicast': CliCommand.Node(
         matcher=matcherMulticast,
         guard=VxlanCli.vxlanMcastRoutingOverUnderlayMcastSupportedGuard ),
      'group': groupNode,
      'overlay': 'Overlay multicast traffic',
      'OVERLAY_ADDR': IpAddrMatcher.IpAddrMatcher(
               helpdesc='Overlay multicast group address' ),
      'encap': matcherEncap,
      'ENCAP_ADDR': IpAddrMatcher.IpAddrMatcher(
               helpdesc='Underlay multicast group address for encapsulation' ),
      'immediate': 'Immediately encapsulate overlay group traffic with encap group'
   }

   @staticmethod
   def handler( mode, args ):
      t0( 'McastVpnVrfGroupOverlayEncapCmd handler args', args )
      vrf = args[ 'VRF' ]
      overlayAddr = args[ 'OVERLAY_ADDR' ]
      encapAddr = args[ 'ENCAP_ADDR' ]
      if IpAddrMatcher.validateMulticastIpAddr( overlayAddr, allowReserved=False )\
          is not None:
         mode.addError( 'Must provide a valid overlay multicast IP address.' )
         return
      if IpAddrMatcher.validateMulticastIpAddr( encapAddr, allowReserved=False )\
          is not None:
         mode.addError(
            'Must provide a valid multicast IP address for encapsulation.' )
         return
      intf = mode.intf.name
      ipTunnelGroupIntfConfig = getIpTunnelGroupConfig( mode ).newIntfConfig( intf )
      vrfEntry = ipTunnelGroupIntfConfig.vrfToMulticastGroup.newMember(
                    vrf, 'tunnelTypeStatic' )
      tacImmediateColl = vrfEntry.tunnelConfigEntry.immediateOverlayToUnderlayGroup
      tacOverlayAddr = Tac.Value( 'Routing::Multicast::OverlayAddr', overlayAddr )
      tacUnderlayAddr = Tac.Value( 'Routing::Multicast::UnderlayAddr', encapAddr )
      tacImmediateColl[ tacOverlayAddr ] = tacUnderlayAddr

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      t0( 'McastVpnVrfGroupOverlayEncapCmd noOrDefaultHandler args', args )
      intf = mode.intf.name
      ipTunnelGroupIntfConfig = getIpTunnelGroupConfig( mode ).intfConfig.get( intf )
      if ipTunnelGroupIntfConfig is None:
         t0( intf, 'does not exist in tunnel group intf config' )
         return
      vrf = args[ 'VRF' ]
      overlayAddr = args[ 'OVERLAY_ADDR' ]
      encapAddr = args[ 'ENCAP_ADDR' ]
      vrfEntry = ipTunnelGroupIntfConfig.vrfToMulticastGroup.get( vrf )
      if not vrfEntry:
         t0( vrf, ' does not exist in tunnel group intf config' )
         return
      tacOverlayAddr = Tac.Value( 'Routing::Multicast::OverlayAddr', overlayAddr )
      tacUnderlayAddr = Tac.Value( 'Routing::Multicast::UnderlayAddr', encapAddr )
      tacImmediateColl = vrfEntry.tunnelConfigEntry.immediateOverlayToUnderlayGroup
      if tacOverlayAddr not in tacImmediateColl:
         t0( overlayAddr,
             ' does not exist in vrf tunnel group intf immediate collection' )
         return
      existingUnderlayAddr = tacImmediateColl[ tacOverlayAddr ]
      if tacUnderlayAddr != existingUnderlayAddr:
         t0( encapAddr,
             'does not match existing immediate overlay to underlay map' )
         return
      del tacImmediateColl[ tacOverlayAddr ]
      if tunnelConfigEntryDefault( vrfEntry.tunnelConfigEntry ):
         del ipTunnelGroupIntfConfig.vrfToMulticastGroup[ vrf ]
      maybeCleanupIntfConfig( mode )

VxlanCli.VxlanIntfModelet.addCommandClass( McastVpnVrfGroupOverlayEncapCmd )

# -------------------------------------------------------------------------------
# config-interface-vxlan: vxlan vrf <vrf> multicast group encap range <pfx> delayed
# -------------------------------------------------------------------------------

class McastVpnVrfGroupRangeCmd( CliCommand.CliCommandClass ):
   syntax = "vxlan VRF multicast group encap range IP_RANGE delayed"
   noOrDefaultSyntax = syntax
   data = {
      'vxlan': VxlanCli.vxlanNode,
      'VRF': VxlanCli.vrfExprFactory,
      'multicast': CliCommand.Node(
         matcher=matcherMulticast,
         guard=VxlanCli.vxlanMcastRoutingOverUnderlayMcastSupportedGuard ),
      'group': groupNode,
      'encap': matcherEncap,
      'range': 'IPv4 multicast group with prefix length between 19 and 32',
      'IP_RANGE': encapRangePrefix,
      'delayed': matcherDelayed
   }

   @staticmethod
   def handler( mode, args ):
      t0( 'McastVpnVrfGroupRangeCmd handler args', args )
      underlayGroupRange = args[ 'IP_RANGE' ]
      prefixLen = int( underlayGroupRange.split( '/' )[ 1 ] )
      if not 19 <= prefixLen <= 32:
         mode.addError( 'Only prefixes with lengths 19-32 are allowed.' )
         return
      prefix = Arnet.IpGenPrefix( underlayGroupRange )
      if IpAddrMatcher.validateMulticastIpAddr( prefix.v4Addr, allowReserved=False )\
          is not None:
         mode.addError( 'Must provide a valid multicast address range.' )
         return
      intf = mode.intf.name
      ipTunnelGroupIntfConfig = getIpTunnelGroupConfig( mode ).newIntfConfig( intf )
      vrfEntry = ipTunnelGroupIntfConfig.vrfToMulticastGroup.newMember(
                    args[ 'VRF' ], 'tunnelTypeStatic' )
      vrfEntry.tunnelConfigEntry.underlayGroupRange = prefix

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      t0( "McastVpnVrfGroupRangeCmd noOrDefaultHandler args", args )
      intf = mode.intf.name
      ipTunnelGroupIntfConfig = getIpTunnelGroupConfig( mode ).intfConfig.get( intf )
      if ipTunnelGroupIntfConfig is None:
         t0( intf, "doesn't exist in tunnel group intf config" )
         return
      vrf = args[ 'VRF' ]
      underlayGroupRange = args[ 'IP_RANGE' ]
      prefix = Arnet.IpGenPrefix( underlayGroupRange )
      vrfEntry = ipTunnelGroupIntfConfig.vrfToMulticastGroup.get( vrf )
      if not vrfEntry:
         t0( vrf, " multicast range doesn't exist in tunnel group intf config" )
         return
      if vrfEntry.tunnelConfigEntry.underlayGroupRange != prefix:
         t0( underlayGroupRange,
             " multicast range not found in tunnel group intf config" )
         return
      vrfEntry.tunnelConfigEntry.underlayGroupRange = \
         Tac.Value( 'Arnet::IpGenPrefix' )
      if tunnelConfigEntryDefault( vrfEntry.tunnelConfigEntry ):
         del ipTunnelGroupIntfConfig.vrfToMulticastGroup[ vrf ]
      maybeCleanupIntfConfig( mode )

VxlanCli.VxlanIntfModelet.addCommandClass( McastVpnVrfGroupRangeCmd )

protocolNode = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'protocol', helpdesc='Protocol settings' ),
      guard=VxlanCli.vxlanUnderlayMcastSupportedGuard )

multicastModes = {
   'asm': "Any Source Multicast",
   'ssm': "Source Specific Multicast (default)"
}
matcherRouteType = CliMatcher.EnumMatcher( multicastModes )

tacMulticastModeMap = {
   'ssm': UnderlayRouteType.pimssm,
   'asm': UnderlayRouteType.pimasm,
}

def _setModeType( mode, trafficType, modeType=None ):
   intfName = mode.intf.name
   ipTunlGrpCfg = getIpTunnelGroupConfig( mode )
   ipTunnelGroupIntfConfig = ipTunlGrpCfg.intfConfig.get( intfName )
   if not ipTunnelGroupIntfConfig:
      return
   if not modeType:
      modeType = ipTunnelGroupIntfConfig.underlayRouteTypeDefault
   if trafficType == 'multicast':
      ipTunnelGroupIntfConfig.mcastUnderlayRouteType = modeType
   elif trafficType == 'flood':
      ipTunnelGroupIntfConfig.floodUnderlayRouteType = modeType
   maybeCleanupIntfConfig( mode )

#-------------------------------------------------------------------------------
# config-interface-vxlan:  vxlan multicast protocol pim [ asm | ssm ]
#-------------------------------------------------------------------------------
class McastUnderlayType( CliCommand.CliCommandClass ):
   syntax = "vxlan multicast protocol pim ROUTE_TYPE"
   noOrDefaultSyntax = "vxlan multicast protocol pim ..."
   data = {
      'vxlan': VxlanCli.vxlanNode,
      'multicast': multicastVxlanIntfNode,
      'protocol': protocolNode,
      'pim': 'Protocol Independent Multicast',
      'ROUTE_TYPE': CliCommand.Node( matcher=matcherRouteType,
                  guard=VxlanCli.pimAsmVxlanUnderlayIifSwitchGuard )
   }

   @staticmethod
   def handler( mode, args ):
      t0( "McastUnderlayMode handler args", args )
      modeArg = args.get( 'ROUTE_TYPE', 'ssm' )
      modeType = tacMulticastModeMap.get( modeArg )
      getIpTunnelGroupConfig( mode ).newIntfConfig( mode.intf.name )
      _setModeType( mode, 'multicast', modeType )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      _setModeType( mode, 'multicast' )

VxlanCli.VxlanIntfModelet.addCommandClass( McastUnderlayType )

#-------------------------------------------------------------------------------
# config-interface-vxlan:  vxlan flood protocol pim [ asm | ssm ]
#-------------------------------------------------------------------------------
class FloodUnderlayType( CliCommand.CliCommandClass ):
   syntax = "vxlan flood protocol pim ROUTE_TYPE"
   noOrDefaultSyntax = "vxlan flood protocol pim ..."
   data = {
      'vxlan': VxlanCli.vxlanNode,
      'flood': VxlanCli.matcherFlood,
      'protocol': protocolNode,
      'pim': 'Protocol Independent Multicast',
      'ROUTE_TYPE': matcherRouteType,
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      t0( "McastUnderlayMode handler args", args )
      modeArg = args.get( 'ROUTE_TYPE', 'ssm' )
      modeType = tacMulticastModeMap.get( modeArg )
      getIpTunnelGroupConfig( mode ).newIntfConfig( mode.intf.name )
      _setModeType( mode, 'flood', modeType )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      _setModeType( mode, 'flood' )

if Toggles.McastVpnLibToggleLib.toggleMcastVpnTestCliEnabled():
   VxlanCli.VxlanIntfModelet.addCommandClass( FloodUnderlayType )

#-------------------------------------------------------------------------------
# config-interface-vxlan:  vxlan remote vni <VNI> vrf <VRF>
#-------------------------------------------------------------------------------
class StaticRemoteVniToVrfMap( CliCommand.CliCommandClass ):
   syntax = 'vxlan remote vni VNI VRF'
   noOrDefaultSyntax = 'vxlan remote vni VNI...'
   data = {
      'vxlan': VxlanCli.vxlanNode,
      'remote': CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'remote',
                                            helpdesc='Remote VTEP' ),
                                 guard=VxlanCli.isVxlan1InterfaceGuard ),
      'vni': VxlanCli.vniMatcherForConfig,
      'VNI': VxlanCli.vniMatcher,
      'VRF': VxlanCli.vrfExprFactory,
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      vniArg = int( args[ 'VNI' ] )
      vrfArg = args[ 'VRF' ]
      config = getIpTunnelGroupConfig( mode ).newIntfConfig( mode.intf.name )
      config.remoteVniToVrf[ vniArg ] = vrfArg

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vniArg = int( args.get( 'VNI' ) )
      intf = mode.intf.name
      if intf not in getIpTunnelGroupConfig( mode ).intfConfig:
         return
      intfConfig = getIpTunnelGroupConfig( mode ).intfConfig[ intf ]
      del intfConfig.remoteVniToVrf[ vniArg ]
      maybeCleanupIntfConfig( mode )

VxlanCli.VxlanIntfModelet.addCommandClass( StaticRemoteVniToVrfMap )

# -------------------------------------------------------------------------------
# config-interface-vxlan:  vxlan multicast ( ipv4 | ipv6 ) [ disabled ]
# -------------------------------------------------------------------------------
class McastOverlayAfType( CliCommand.CliCommandClass ):
   syntax = "vxlan multicast AF [ disabled ]"
   noOrDefaultSyntax = "vxlan multicast AF ..."
   data = {
      'vxlan': VxlanCli.vxlanNode,
      'multicast': multicastVxlanIntfNode,
      'AF': VxlanMcastIpFamilyExpr,
      'disabled': 'Disable VXLAN encapsulation if overlay multicast',
   }

   @staticmethod
   def handler( mode, args ):
      config = getIpTunnelGroupConfig( mode ).newIntfConfig( mode.intf.name )
      af = args[ 'AF' ]
      enable = 'disabled' not in args
      if af == 'ipv4':
         config.mcastOverlayV4 = enable
         if enable:
            maybeCleanupIntfConfig( mode )
      elif af == 'ipv6':
         config.mcastOverlayV6 = enable
         if not enable:
            maybeCleanupIntfConfig( mode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intf = mode.intf.name
      if intf not in getIpTunnelGroupConfig( mode ).intfConfig:
         return
      af = args[ 'AF' ]
      config = getIpTunnelGroupConfig( mode ).newIntfConfig( intf )
      if af == 'ipv4':
         config.mcastOverlayV4 = True
      elif af == 'ipv6':
         config.mcastOverlayV6 = False
      maybeCleanupIntfConfig( mode )

if Toggles.McastVpnLibToggleLib.toggleMcastVpnOISMV6Enabled():
   VxlanCli.VxlanIntfModelet.addCommandClass( McastOverlayAfType )

# -------------------------------------------------------------------------------
# config-interface-vxlan:  vxlan multicast headend-replication
# -------------------------------------------------------------------------------
herKw = CliMatcher.KeywordMatcher( 'headend-replication',
      helpdesc='Replicate overlay multicast packets at the ingress'
               ' VTEP in the underlay' )
herKwWithGuard = CliCommand.Node( herKw,
      guard=VxlanCli.vxlanMcastUnderlayHerSupportedGuard )
class McastUnderlayHerCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan multicast headend-replication'
   noOrDefaultSyntax = syntax
   data = {
      'vxlan': VxlanCli.vxlanNode,
      'multicast': multicastVxlanIntfNode,
      'headend-replication': herKwWithGuard,
   }

   @staticmethod
   def handler( mode, args ):
      config = getIpTunnelGroupConfig( mode ).newIntfConfig( mode.intf.name )
      config.underlayHerEnabled = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = getIpTunnelGroupConfig( mode ).intfConfig.get( mode.intf.name )
      if config is not None:
         config.underlayHerEnabled = False
         maybeCleanupIntfConfig( mode )

VxlanCli.VxlanIntfModelet.addCommandClass( McastUnderlayHerCmd )

class VxlanIntfMcastVpnIntf( IntfCli.IntfDependentBase ):
   """Cleanup config in response to VTI destruction."""
   def setDefault( self ):
      t0( "Destroying ipTunnelGroupConfig for", self.intf_.name )
      del ipTunnelGroupConfig.intfConfig[ self.intf_.name ]
      del remoteDomainIpTunnelGroupConfig.intfConfig[ self.intf_.name ]

class RemoteDomainMode( McastVpn.RemoteDomainMode, BasicCli.ConfigModeBase ):
   # Attributes required of every Mode class.
   name = 'Domain Remote Mode'

   def __init__( self, parent, session, intf ):
      McastVpn.RemoteDomainMode.__init__( self, parent, session, intf )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

# --------------------------------------------------------------------------------
# [ no | default ] domain remote
# --------------------------------------------------------------------------------
def oismGatewaySupportedGuard( mode, token ):
   cmdSupported = bridgingHwCapabilities.evpnDciGatewayMultihomingSupported
   return None if cmdSupported else CliParser.guardNotThisPlatform

domainMatcher = CliMatcher.KeywordMatcher( 'domain',
      helpdesc='Multicast domain remote commands' )
domainNode = CliCommand.Node( domainMatcher,
      guard=oismGatewaySupportedGuard )

class RemoteDomainModeCmd( CliCommand.CliCommandClass ):
   syntax = 'domain remote'
   noOrDefaultSyntax = syntax
   data = {
      'domain': domainNode,
      'remote': 'remote',
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( RemoteDomainMode, intf=mode.intf )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      maybeCleanupIntfConfig( mode )

# -------------------------------------------------------------------------------
# config-domain-remote: vlan <id> multicast group <ip-addr|ip6-addr>
# -------------------------------------------------------------------------------
class McastVpnMulticastGroupRemoteDomainCmd( CliCommand.CliCommandClass ):
   syntax = McastVpnMulticastGroupCmd.syntax.replace( "vxlan ", "" )
   noOrDefaultSyntax = McastVpnMulticastGroupCmd.noOrDefaultSyntax.replace(
         "vxlan ", "" )
   data = McastVpnMulticastGroupCmd.data.copy()

   handler = McastVpnMulticastGroupCmd.handler
   noOrDefaultHandler = McastVpnMulticastGroupCmd.noOrDefaultHandler

class McastVpnVrfGroupRangeRemoteDomainCmd( CliCommand.CliCommandClass ):
   syntax = McastVpnVrfGroupRangeCmd.syntax.replace( "vxlan ", "" )
   noOrDefaultSyntax = McastVpnVrfGroupRangeCmd.noOrDefaultSyntax.replace(
         "vxlan ", "" )
   data = McastVpnVrfGroupRangeCmd.data.copy()
   handler = McastVpnVrfGroupRangeCmd.handler
   noOrDefaultHandler = McastVpnVrfGroupRangeCmd.noOrDefaultHandler

class McastVpnVrfGroupRemoteDomainCmd( CliCommand.CliCommandClass ):
   syntax = McastVpnVrfGroupCmd.syntax.replace( "vxlan ", "" )
   noOrDefaultSyntax = McastVpnVrfGroupCmd.noOrDefaultSyntax.replace( "vxlan ", "" )
   data = McastVpnVrfGroupCmd.data.copy()

   handler = McastVpnVrfGroupCmd.handler
   noOrDefaultHandler = McastVpnVrfGroupCmd.noOrDefaultHandler

class McastVpnVrfGroupOverlayEncapRemoteDomainCmd( CliCommand.CliCommandClass ):
   syntax = McastVpnVrfGroupOverlayEncapCmd.syntax.replace( "vxlan ", "" )
   noOrDefaultSyntax = McastVpnVrfGroupOverlayEncapCmd.noOrDefaultSyntax.replace(
         "vxlan ", "" )
   data = McastVpnVrfGroupOverlayEncapCmd.data.copy()

   handler = McastVpnVrfGroupOverlayEncapCmd.handler
   noOrDefaultHandler = McastVpnVrfGroupOverlayEncapCmd.noOrDefaultHandler

if Toggles.McastVpnLibToggleLib.toggleOismGatewayEnabled():
   VxlanCli.VxlanIntfModelet.addCommandClass( RemoteDomainModeCmd )
   RemoteDomainMode.addCommandClass( McastVpnMulticastGroupRemoteDomainCmd )
   RemoteDomainMode.addCommandClass( McastVpnVrfGroupRemoteDomainCmd )
   RemoteDomainMode.addCommandClass( McastVpnVrfGroupOverlayEncapRemoteDomainCmd )
   RemoteDomainMode.addCommandClass( McastVpnVrfGroupRangeRemoteDomainCmd )

def Plugin( entityManager ):
   global ipTunnelGroupConfig
   global remoteDomainIpTunnelGroupConfig
   global vtiConfigDir
   global bridgingHwCapabilities
   typeName = 'Routing::Multicast::IpTunnelGroupConfig'
   ipTunnelGroupConfig = ConfigMount.mount( entityManager,
                                            Tac.Type( typeName ).mountPath,
                                            typeName,
                                            "w" )
   remoteDomainIpTunnelGroupConfig = ConfigMount.mount(
                                         entityManager,
                                         Tac.Type( typeName ).remoteDomainMountPath,
                                         typeName,
                                         "w" )

   vtiConfigDir = LazyMount.mount( entityManager,
                                   "interface/config/eth/vxlan",
                                   "Vxlan::VtiConfigDir", "w" )

   # Need read access to hardware capabilities for CLI guards
   bridgingHwCapabilities = LazyMount.mount( entityManager,
                                             "bridging/hwcapabilities",
                                             "Bridging::HwCapabilities", "r" )

   # Registering this cleanup handler is a NOP when toggle is
   # disabled.
   IntfCli.Intf.registerDependentClass( VxlanIntfMcastVpnIntf )
