# Copyright (c) 2006-2008, 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliMatcher
import CliCommand
import Logging
import Tac
import Tracing
import UtmpDump
import WaitForWarmup

t0 = Tracing.trace0
t1 = Tracing.trace1

# The default parameters of doWaitForWarmup are suited as per the need of cli
# wfw command. We use non-default parameters if we need to wait for some very
# specific purpose and the default arguments may not be suitable. For example,
# call from tryWaitForWarmupAfterVlanPortChange.
# Return True if succeeded, else raise error. If however, ignoreTimeout is
# True, then instead of raising, return False on timeout errors.
def doWaitForWarmup( mode, agentList=None, verbose='concise', numIterations=1,
                     agentsToGrab=None, agentsToSkip=None,
                     ignoreTimeout=False, timeout=1000000000.0, warnAfter=1.0,
                     maxDelay=1.0 ):
   # This is a thin wrapper around WaitForWarmup.wait().  The reason this lives
   # in the Cli package instead of in Sysdb is that we can't have Sysdb depend
   # on Cli.

   # if we are standalone, don't wait
   if mode.session_.isStandalone():
      return True

   for _ in range( numIterations ):
      try:
         WaitForWarmup.wait( mode.entityManager,
                             agentList=agentList,
                             agentsToGrab=agentsToGrab,
                             agentsToSkip=agentsToSkip,
                             sleep=not Tac.activityManager.inExecTime.isZero,
                             verbose=verbose, timeout=timeout,
                             maxDelay=maxDelay, warnAfter=warnAfter )
      except Tac.Timeout:
         if not ignoreTimeout:
            raise
         return False
   return True

agentListMatcher = CliMatcher.PatternMatcher( '[^/]+', helpname='WORD',
                                              helpdesc='Agent name' )

class WaitForWarmupCmd( CliCommand.CliCommandClass ):
   syntax = '''wait-for-warmup [ { <agentList> } ] '''
   data = { 'wait-for-warmup': 'Block until agents declare themselves as warm',
            '<agentList>': agentListMatcher }

   @staticmethod
   def handler( mode, args ):
      agents = args.get( '<agentList>' )
      warm = doWaitForWarmup( mode, agentList=agents, ignoreTimeout=True )
      if not warm:
         mode.addError( 'Command timed out' )

BasicCli.EnableMode.addCommandClass( WaitForWarmupCmd )

# When doing a lot of cli configuration ( e.g., some stress test ) we can make the
# Ebra, FocalPoint, IgmpSnooping, etc, agents really busy and starve them from
# writing their heartbeats. For example creating 200,000 vlan ports takes around 30
# seconds for FocalPoint to process. Running the command twice is sufficient for
# ProcMgr to restart the agent since no heartbeats were written. This is undesirable.
# So after making cli config changes, we call waitForWarmup on agents which we think
# might get busy.
# This approach has a few shortcomings, e.g., We wait-for-warmup for agents that
# may not have anything to do with the config change in question. It would be better
# to have a more general, and less intrusive solution to the tac-update-flow-control
# problem. The other problem this can cause is causing delays in returning cli
# prompt, which gets noticeable when you are running commands using scripts, e.g.,
# gotoConfigMode(); for v in vlans: configureVlan( v ).
# Another example is interface range command, which calls the command iteratively
# for all the interfaces.
# For this particular usage( calls from VlanCli, LagCli, IntfCli, etc), we use
# minTimeBetweenWarmups=1 to avoid delaying commands run through interface range
# command which iteratively calls multiple commands. Or for other similar cli actions
# which call multiple commands inside.
# We use numIterations=2 since if we warmup for Ebra before FocalPoint, we may
# return prematurely since Ebra's output which is input for FocalPoint will be
# available when Ebra is done, and so on.
# ignoreErrors=True since we really don't need to report the failure in cli's
# context. The cli config actually did work. Its the agents that are unhappy.

SYS_SYSTEM_WARMUP_FAILED = Logging.LogHandle(
   "SYS_SYSTEM_WARMUP_FAILED",
   severity=Logging.logNotice,
   fmt="System warmup after cli configuration by %s on %s (%s) failed",
   explanation="The system is taking a long time to warm up after the last "
               " configuration request from the Cli. This may mean an unusually "
               "large configuration change was requested from the Cli, which "
               "resulted in prolonged processing by the switching software. ",
   recommendedAction="No action is needed. However avoid loading the switch with "
               "large amount of configuration changes in quick succession. " )

# The list warmupAgentsOnVlanPortChange contains a collection of callback functions
# each of which when called, returns a list of agent name(s). Various agent cli
# plugins add their callback to this list, if they think they know of agent(s) which
# has a lot of work to do when vlan/port config changes.
# Currently Lag, IgmpSnooping, FocalPoint, SandCells, Ebra, etc use this facility.
warmupAgentsOnVlanPortChange = [ ]
warmupAgentsOnVlanPortChange.append( lambda m: [ 'Sysdb' ] )

def tryWaitForWarmupAfterVlanPortChange( mode ):
   session = mode.session
   lastWarmupTime = session.sessionData( 'lastWarmupTime', defaultValue=-1 )
   lastWarmupAttemptTime = session.sessionData( 'lastWarmupAttemptTime',
                                                defaultValue=-1 )
   t0( "in tryWaitForWarmupAfterVlanPortChange", lastWarmupTime,
       lastWarmupAttemptTime )

   # If we warmed up in the recent past, just return.
   if lastWarmupTime > 0 and ( Tac.now() - lastWarmupTime ) < 10.0:
      t0( "Return as last successful warmup within 1 sec" )
      return

   # If we failed in warming up earlier, some agent might be misbehaving. Suspend
   # warming up for an hour, after which we will try again, hoping things may
   # have been fixed. This lets cli not get hosed up again and again.
   if lastWarmupTime < lastWarmupAttemptTime and \
        ( Tac.now() - lastWarmupAttemptTime ) < 3600:
      # Last warmup attempt didn't succeed. Also the last failure happened in
      # the previous hour itself. Ignore warmup, so that cli works even if agents
      # are misbehaving
      t0( "Return as last failed warmup within 3600 sec" )
      return

   # Set lastWarmupAttemptTime before going into wait since user may interrupt
   # control flow with ctrl-C if wait takes long.
   session.sessionDataIs( 'lastWarmupAttemptTime', Tac.now() )
   agentsToGrab = []
   for func in warmupAgentsOnVlanPortChange:
      agentsToGrab.extend( func( mode ) )
   t0( "Verbose wait for 120 secs" )
   # We use warnAfter so that verbose flag doesn't produce too much clutter.
   warmed = doWaitForWarmup( mode, verbose=True, numIterations=2,
                             agentsToGrab=agentsToGrab,
                             ignoreTimeout=True, timeout=120.0,
                             warnAfter=15.0, maxDelay=0.005 )
   if warmed:
      session.sessionDataIs( 'lastWarmupTime', Tac.now() )
   else:
      info = UtmpDump.getUserInfo()
      Logging.log( SYS_SYSTEM_WARMUP_FAILED, info[ 'user' ],
                   info[ 'tty' ], info[ 'ipAddr' ] )

def Plugin( entMan ):
   WaitForWarmup.doMounts( entMan )
