# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import ConfigMount
import Tac
import CliMatcher
import ShowCommand
from CliCommand import Node
from Intf.IntfRange import intfRangeMatcher
import CliPlugin.TechSupportCli
import CliPlugin.XcvrAllStatusDir
from CliPlugin.XcvrModel import ( InterfaceTransceiverBase, InterfacesTransceiver,
                                  InterfaceTransceiver, InterfaceTransceiverDetails,
                                  InterfaceTransceiverDetailThresholds,
                                  SUPPORTED_FORMATS )
from CliPlugin.XcvrConfigCli import ( gridKw, mhzFreqToGhz )
from CliPlugin.XcvrShowApplicationHandler import showXcvrApplicationInterfacesHandler
from CliPlugin.XcvrShowApplicationModel import TransceiverApplicationInterfaces
from CliPlugin.XcvrShowStatusCliHandler import showInterfacesXcvrStatus
from CliPlugin.XcvrShowStatusModel import InterfacesTransceiverStatus
from CliPlugin.XcvrShowTuningHandler import showInterfacesXcvrTuning
from CliPlugin.XcvrShowTuningModel import InterfacesTransceiverTuning
from CliPlugin.XcvrShowDomHandler import ( showInterfacesXcvrDom,
                                           xcvrDomMinPowerThreshold )
from CliPlugin.XcvrShowDomModel import InterfacesTransceiverDom
import CliPlugin.IntfCli
from CliPlugin.IntfCli import ShowIntfCommand, Intf, interfaceKwMatcher
import CliPlugin.EthIntfCli
from CliPlugin.EthIntfCli import xcvrShowKw
import CliPlugin.EthIntfModel
from CliPlugin.XcvrCliLib import getAllIntfsWrapper
from XcvrLib import ( getLineSideChannels,
                      getXcvrSlotName,
                      getXcvrStatus,
                      isCfp2DcoSlotWithNonDcoInserted,
                      negInfToNone )
import CliToken.Idprom
import CliGlobal
import LazyMount
import Tracing
from TypeFuture import TacLazyType

t0 = Tracing.trace0

DomThreshold = TacLazyType( 'Xcvr::DomThreshold' )
XcvrType = TacLazyType( 'Xcvr::XcvrType' )

gv = CliGlobal.CliGlobal( xgc=None, xcvrCliConfigSliceDir=None, entityManager=None,
                          xcvrStatusDir=None, xcvrPerfMonSliceDir=None,
                          lineSystemStatusSliceDir=None )

ethSpeedToStr = {
   'speedUnknown': 'unknown',
   'speed10Mbps': '10Mbps',
   'speed100Mbps': '100Mbps',
   'speed1Gbps': '1Gbps',
   'speed2p5Gbps': '2.5Gbps',
   'speed5Gbps': '5Gbps',
   'speed10Gbps': '10Gbps',
   'speed25Gbps': '25Gbps',
   'speed40Gbps': '40Gbps',
   'speed50Gbps': '50Gbps',
   'speed100Gbps': '100Gbps',
   'speed200Gbps': '200Gbps',
   'speed400Gbps': '400Gbps',
   }

# ----------------------------------------------------------------------------
#
# "show interface [ <interface> ] transceiver [ detail ] [ csv ]"
#
# ----------------------------------------------------------------------------

def is8MediaLane400gPortRenumberedSlot( slotStatus ):
   """
   Returns True if the passed in XcvrNewStatus represents a 400G
   port-renumbered slot with an optical transceiver with 8 media lanes.

   See AID/8413 for port-renumbering.  The SilverthroneQPortRenum Linecard is a SKU
   with port renumbering, for example.

   Parameters
   ----------
   slotStatus : Xcvr::XcvrNewStatus
      An object derived from XcvrNewStatus.  Must be the status associated with
      the slot.

   Returns
   -------
   bool
   """
   intfName = slotStatus.xcvrConfig.intfName
   statusInUse = getXcvrStatus( slotStatus )
   return ( getLineSideChannels( statusInUse ) == 8 and
            0 in intfName and
            2 in intfName and
            4 in intfName and
            6 in intfName and
            intfName[ 0 ].endswith( "/1" ) and
            intfName[ 2 ].endswith( "/2" ) and
            intfName[ 4 ].endswith( "/3" ) and
            intfName[ 6 ].endswith( "/4" ) )

def _showInterfacesXcvr( mode, intf=None,
                         mod=None, outputFmt=None, detailed=False ):
   res = InterfacesTransceiver( _printFmt=outputFmt, _detailed=detailed,
      _domThresholdOverrideEnabled=gv.xgc.domThresholdOverrideEnabled )
   assert outputFmt in SUPPORTED_FORMATS, "Unrecognized DOM output format"
   ( requestedIntfs, requestedIntfNames ) = getAllIntfsWrapper( mode, intf, mod )
   if not requestedIntfs:
      return res

   for slotStatus in gv.xcvrStatusDir.xcvrStatus.values():
      statusInUse = getXcvrStatus( slotStatus )
      if not statusInUse.xcvrMgrCapabilities.domDynamicValues:
         continue

      if not slotStatus.xcvrConfig.intfName:
         # This should never happen.  If there is nothing in our
         # intfName collection, then we can't do anything.
         continue

      # Lane 0 will always map to the primary interface name.
      primaryIntfName = slotStatus.xcvrConfig.intfName.get( 0 )
      if primaryIntfName is None:
         # This should never happen either.  If there isn't a name at index 0
         # (lane 0), then there is something seriously wrong.  Instead of asserting,
         # we'll just be defensive and continue to the next slot.
         continue

      # List of all interface names associated with this transceiver slot
      slotIntfNames = list( slotStatus.xcvrConfig.intfName.values() )

      if isCfp2DcoSlotWithNonDcoInserted( slotStatus ):
         # This is a weird special case.  When a non-DCO is inserted inside
         # CFP2-DCO slot, we pretend like there is only a /1 interface.  For
         # example, a 100GBASE-LR4 inserted into a DCO slot.
         slotIntfNames = [ primaryIntfName ]

      # Mapping from media lane to the interface token we print.  We only insert
      # media lane keys in this dictionary if we are going to print data for those
      # media lanes.
      mediaToPrintedIntfToken = {}

      slotName = getXcvrSlotName( primaryIntfName )  # sub-intf is chopped off
      numMediaLanes = getLineSideChannels( statusInUse )

      if len( slotIntfNames ) == 1:
         # Slots with only 1 interface include:
         # - SFP slots
         # - 40G QSFP slots on Clearlake, Schooner, Cloverdale, etc.
         if primaryIntfName in requestedIntfNames:
            if numMediaLanes == 1:
               # In the case of a single media lane transceiver, then we do not
               # manually append media lane information to the printed name.
               # Instead we use the primary interface name.
               #
               # This is important, for example, in the case of a SFP in an SFP slot.
               # A SFP slot does not have a /1, so we shouldn't append '/1' to the
               # slot name in this case.  Using the primaryIntfName takes care of
               # this.
               #
               # Note: the primaryIntfName may have '/1' at the end of it if this is
               # an SFP in a QSFP adapter.  This is expected.
               mediaToPrintedIntfToken[ 0 ] = primaryIntfName
            else:
               for mediaLaneId in range( numMediaLanes ):
                  # Media lanes are indexed by 0, so add 1 to get our interface token
                  printName = f"{slotName}/{mediaLaneId + 1}"
                  mediaToPrintedIntfToken[ mediaLaneId ] = printName
      elif is8MediaLane400gPortRenumberedSlot( slotStatus ):
         # Oh boy...there is a bit of history here.  In order to preserve the output
         # customer's see today, we are special casing the port-renumbered slots like
         # on SilverthroneQPortRenum.  When an optical transceiver with more than 4
         # media lanes is used on a port-renumdered slot (like a 400GBASE-SR8), then
         # we map the interfaces like this:
         #
         # EthernetL/X/1 -> media lane 0 and 1
         # EthernetL/X/2 -> media lane 2 and 3
         # EthernetL/X/3 -> media lane 4 and 5
         # EthernetL/X/4 -> media lane 6 and 7
         #
         # Therefore when a customer requests just EthernetL/X/3, for example, we
         # print info for media lanes 4 and 5.  This contradicts our normal behavior.
         # Normally, EthernetL/X/3 means the customer is asking for media lane 3's
         # info.

         assert numMediaLanes == 8
         assert numMediaLanes > len( slotIntfNames )

         # Ok, so we need to map each media lane to an interface.  We can hard code
         # this mapping because we know exactly what kind of slot we're dealing with.
         # To get our interface token, we divide our media lane by 2 and add 1.
         # (See mapping in code comment above)
         for mediaLaneId in range( numMediaLanes ):
            intfName = f"{slotName}/{( mediaLaneId // 2 ) + 1}"
            if intfName in requestedIntfNames:
               # Add 1 because our media lanes are indexed starting at 0
               printName = f"{slotName}/{mediaLaneId + 1}"
               mediaToPrintedIntfToken[ mediaLaneId ] = printName

      else:
         # This scenario describes slots with more than 1 Ethernet interface.  This
         # means we can assume the format of "EthernetX/Y" or "EthernetL/X/Y",
         # where L stands for linecard, X stands for slot, and Y stands for sub-intf
         # number.
         for mediaLaneId in range( numMediaLanes ):
            intfToken = f"{slotName}/{mediaLaneId + 1}"
            if intfToken not in slotIntfNames:
               # This means there is not a clean mapping from media lanes
               # to Ethernet interfaces. For example, there are more media lanes
               # than Ethernet interfaces.  Or perhaps some interfaces are missing
               # like on platforms which skip even numbered interfaces.
               tempMediaLaneId = mediaLaneId - 1  # next lowest Ethernet name
               intfToken = f"{slotName}/{tempMediaLaneId + 1}"
               while intfToken not in slotIntfNames:
                  tempMediaLaneId -= 1
                  intfToken = f"{slotName}/{tempMediaLaneId + 1}"

            if intfToken not in requestedIntfNames:
               continue

            # Add 1 because our media lanes are indexed starting at 0
            printName = f"{slotName}/{mediaLaneId + 1}"
            mediaToPrintedIntfToken[ mediaLaneId ] = printName

      # Populate the CLI model. The mediaToPrintedIntfToken contains all the
      # media lanes requested by this command.
      for mediaLaneId, printName in mediaToPrintedIntfToken.items():
         domValid = statusInUse.domInfoValid[ mediaLaneId ]
         res.interfaces[ printName ] =\
            _createInterfaceTransceiver( printName, domValid )
         if domValid:
            intfXcvr = res.interfaces[ printName ]
            _populateXcvrDomInfo( intfXcvr, statusInUse,
                  statusInUse.domInfo[ mediaLaneId ] )
            if detailed:
               intfXcvr.details = _xcvrDomDetails( statusInUse )
               intfXcvr.details.setDomThresholdOverridden(
                     statusInUse.domThresholdOverridden )
   return res

def _createInterfaceTransceiver( intfName: str, domValid: bool ):
   # Creates an empty InterfaceTransceiver model with the slot and channel set
   model = InterfaceTransceiver() if domValid else InterfaceTransceiverBase()
   slot = getXcvrSlotName( intfName )
   channel = intfName[ len( slot ) + 1 : ]
   model.setSlot( slot )
   model.setChannel( "1" if channel == "" else channel )
   return model

def _populateXcvrDomInfo( model, status, domInfo ):
   # The following checks prevent the TX and RX Power from being
   # reported as N/A when the DomInfo is valid.
   domCapabilities = status.domCapabilities

   if ( domInfo.txPower < Tac.Value( "Xcvr::PowerConstants" ).minimumTxPower
        and domCapabilities.txPower ):
      txPower = Tac.Value( "Xcvr::PowerConstants" ).minimumTxPower
   else:
      txPower = domInfo.txPower

   minimumRxPowerConstant = xcvrDomMinPowerThreshold( status, 'rx' )
   if ( domInfo.rxPower < minimumRxPowerConstant and
        domCapabilities.rxPower ):
      rxPower = minimumRxPowerConstant
   else:
      rxPower = domInfo.rxPower

   if ( domInfo.totalRxPower < minimumRxPowerConstant and
        domCapabilities.totalRxPower ):
      totalRxPower = minimumRxPowerConstant
   else:
      totalRxPower = domInfo.totalRxPower

   # Used for active copper where type is optical but txBias is not supported
   if not domCapabilities.txBias:
      txBias = float( "-inf" )
   else:
      txBias = domInfo.txBias

   model.vendorSn = status.vendorInfo.vendorSn.strip()
   model.mediaType = status.mediaTypeString.strip()
   model.narrowBand = status.narrowBand
   model.updateTime = Tac.utcNow() - ( Tac.now() - domInfo.updateTime )
   model.temperature = negInfToNone( domInfo.temperature )
   model.voltage = negInfToNone( domInfo.voltage )
   model.txBias = negInfToNone( txBias )
   model.txPower = negInfToNone( txPower )
   model.rxPower = negInfToNone( rxPower )
   model.totalRxPower = negInfToNone( totalRxPower )
   return model

def _getDetailThresholdModel( domThreshold, overrideEntry, attr ):
   highAlarm = negInfToNone( getattr( domThreshold, attr + "HighAlarm" ) )
   highWarn = negInfToNone( getattr( domThreshold, attr + "HighWarn" ) )
   lowAlarm = negInfToNone( getattr( domThreshold, attr + "LowAlarm" ) )
   lowWarn = negInfToNone( getattr( domThreshold, attr + "LowWarn" ) )

   highAlarmO = getattr( overrideEntry, attr + "HighAlarm", None ) is not None
   highWarnO = getattr( overrideEntry, attr + "HighWarn", None ) is not None
   lowAlarmO = getattr( overrideEntry, attr + "LowAlarm", None ) is not None
   lowWarnO = getattr( overrideEntry, attr + "LowWarn", None ) is not None
   return InterfaceTransceiverDetailThresholds( highAlarm=highAlarm,
                                                highWarn=highWarn,
                                                lowAlarm=lowAlarm,
                                                lowWarn=lowWarn,
                                                highAlarmOverridden=highAlarmO,
                                                highWarnOverridden=highWarnO,
                                                lowAlarmOverridden=lowAlarmO,
                                                lowWarnOverridden=lowWarnO
                                               )

def _xcvrDomDetails( status ):
   model = InterfaceTransceiverDetails()

   if not status.domThresholdValid:
      return model

   domThreshold = status.domThreshold
   overrideEntry = status.domThresholdOverrideEntry
   domCapabilities = status.domCapabilities

   model.temperature = _getDetailThresholdModel( domThreshold,
                                                 overrideEntry, "temp" )
   model.voltage = _getDetailThresholdModel( domThreshold, overrideEntry, "voltage" )
   if domCapabilities.txBias:
      model.txBias = _getDetailThresholdModel( domThreshold, overrideEntry,
                                               "txBias" )
   model.txPower = _getDetailThresholdModel( domThreshold, overrideEntry, "txPower" )
   model.rxPower = _getDetailThresholdModel( domThreshold, overrideEntry, "rxPower" )
   model.totalRxPower = _getDetailThresholdModel( domThreshold, None,
                                                  "totalRxPower" )
   return model

def showInterfacesXcvrDefault( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   outputFmt = args.get( 'csv', 'default' )
   detailed = 'detail' in args
   return _showInterfacesXcvr( mode, intf, mod, outputFmt=outputFmt,
                               detailed=detailed )

class ShowIntfXcvr( ShowIntfCommand ):
   syntax = 'show interfaces transceiver [ csv | detail ]'
   data = dict( transceiver=xcvrShowKw,
                csv='Show interface transceiver in CSV format',
                detail='Show interface transceiver detail' )
   handler = showInterfacesXcvrDefault
   cliModel = InterfacesTransceiver

BasicCli.addShowCommandClass( ShowIntfXcvr )

# ----------------------------------------------------------------------------
#
# "show interface [ <interface> ] transceiver tuning [ detail ]"
#
# ----------------------------------------------------------------------------

class ShowIntfXcvrTuning( ShowIntfCommand ):
   syntax = 'show interfaces transceiver tuning [ detail ]'
   data = dict( transceiver=xcvrShowKw,
                tuning='Show interface transceiver tuning',
                detail='Show interface transceiver tuning detail' )
   handler = showInterfacesXcvrTuning
   cliModel = InterfacesTransceiverTuning

BasicCli.addShowCommandClass( ShowIntfXcvrTuning )

# -------------------------------------------------------------------------------
# The "show interfaces [<name>] transceiver hardware" command, in enable mode.
#   show interfaces [<name>] transceiver hardware
#   show interfaces module <modnum> transceiver hardware
# -------------------------------------------------------------------------------
class ShowIntfXcvrHardware( ShowIntfCommand ):
   syntax = 'show interfaces transceiver hardware'
   data = dict( transceiver=xcvrShowKw,
                hardware='Show interface transceiver hardware' )
   handler = "XcvrShowHardwareHandler.showInterfacesXcvrHardware"
   cliModel = "XcvrShowHardwareModel.InterfacesTransceiverHardwareBase"

BasicCli.addShowCommandClass( ShowIntfXcvrHardware )

# ----------------------------------------------------------------------------
#
# "show interfaces [<name>] transceiver channels"
#
# ----------------------------------------------------------------------------

freqMatcher = CliMatcher.FloatMatcher( 190000, 200000,
                                       helpdesc='frequency (GHz)',
                                       precisionString='%.9g' )

def computeWavelength( frequency ):
   c = 299792458.0 # speed of light (m/s)
   return c / mhzFreqToGhz( frequency )

class ShowIntfXcvrChannels( ShowIntfCommand ):
   syntax = 'show interfaces transceiver channels [ grid-spacing GRID ] ' \
            '[ min-frequency MIN_FREQ ] [ max-frequency MAX_FREQ ]'
   data = { 'transceiver': xcvrShowKw,
            'channels': 'Show interface transceiver supported channels',
            'grid-spacing': gridKw,
            'GRID': CliMatcher.PatternMatcher(
               r'^((100|50|33|25|12\.5|6\.25)\,)*(100|50|33|25|12\.5|6\.25)$',
               helpname='100,50,25,12.5,6.25', helpdesc='grid-spacing (GHz)' ),
            'min-frequency': 'Specify minimum frequency',
            'MIN_FREQ': freqMatcher,
            'max-frequency': 'Specify maximum frequency',
            'MAX_FREQ': freqMatcher
   }
   handler = "XcvrShowChannelsHandler.showInterfacesXcvrChannels"
   cliModel = "XcvrShowChannelsModel.InterfacesTransceiverChannelsBase"
BasicCli.addShowCommandClass( ShowIntfXcvrChannels )

# ----------------------------------------------------------------------------
#
# "show idprom transceiver [ <interface> ] [ extended ]"
#
# ----------------------------------------------------------------------------

class ShowXcvrIdprom( ShowCommand.ShowCliCommandClass ):
   syntax = 'show idprom ( transceiver | interface ) [ INTFS ] [ extended ]'
   data = {
      'idprom': CliToken.Idprom.idpromNode,
      'transceiver': xcvrShowKw,
      'interface': Node( matcher=interfaceKwMatcher, hidden=True ),
      'INTFS': intfRangeMatcher,
      'extended': CliToken.Idprom.extendedNode,
   }
   handler = "XcvrShowIdpromHandler.showInterfaceXcvrIdprom"
   cliModel = "XcvrShowIdpromModel.InterfacesTransceiverIdpromBase"

BasicCli.addShowCommandClass( ShowXcvrIdprom )

# ----------------------------------------------------------------------------
#
# "show interfaces [ <names> ] transceiver eeprom"
#
# ----------------------------------------------------------------------------

class ShowIntfXcvrEeprom( ShowIntfCommand ):
   syntax = 'show interfaces transceiver eeprom'
   data = dict( transceiver=xcvrShowKw,
                eeprom='Show decoded dump of supported EEPROM pages' )
   handler = "XcvrShowEepromHandler.showInterfaceXcvrEeprom"
   cliModel = "XcvrShowEepromModel.InterfacesTransceiverEepromBase"

BasicCli.addShowCommandClass( ShowIntfXcvrEeprom )

# --------------------------------------------------------------------------------
#
# show transceiver status [ interface INTF ]
#
# --------------------------------------------------------------------------------
class TransceiverStatusCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show transceiver status [ interface INTF ]'
   data = {
      'transceiver': xcvrShowKw,
      'status': 'Show transceiver status information',
      'interface': 'Show internal interface state',
      'INTF': Intf.rangeMatcher,
   }
   handler = showInterfacesXcvrStatus
   cliModel = InterfacesTransceiverStatus

BasicCli.addShowCommandClass( TransceiverStatusCmd )

# --------------------------------------------------------------------------------
#
# "show transceiver application [ interface <INTFS> ]"
#
# --------------------------------------------------------------------------------

class TransceiverApplicationCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show transceiver application [ interface INTFS ]'
   data = {
      'transceiver': xcvrShowKw,
      'application': 'Show transceiver active CMIS application status',
      'interface': 'Show transceiver application for a specific interface',
      'INTFS': Intf.rangeMatcher,
   }
   handler = showXcvrApplicationInterfacesHandler
   cliModel = TransceiverApplicationInterfaces

BasicCli.addShowCommandClass( TransceiverApplicationCmd )

# ----------------------------------------------------------------------------
#
# "show interface [ <interface> ] transceiver performance-monitoring [ raw ]
#    [ thresholds ]"
#
# ----------------------------------------------------------------------------

class ShowIntfXcvrPerf( ShowIntfCommand ):
   syntax = (
      'show interfaces transceiver performance-monitoring [ raw ] [ thresholds ]' )
   data = { 'transceiver': xcvrShowKw,
            'performance-monitoring':
            'Show performance-monitoring parameters for the interface',
            'raw': ( 'Show performance-monitoring parameters with index/channel '
                      'information reported by module EEPROM.' ),
            'thresholds': ( 'Show performance-monitoring parameters with '
                             'thresholds for the interface' )
   }
   handler = "XcvrShowPerfMonHandler.showInterfacesXcvrPerformanceMonitoring"
   cliModel = "XcvrShowPerfMonModel.InterfacesTransceiverPerformanceMonitoring"

BasicCli.addShowCommandClass( ShowIntfXcvrPerf )

# ----------------------------------------------------------------------------
#
# "show interfaces [ <interface> ] transceiver dom [ thresholds ]"
#
# ----------------------------------------------------------------------------

class ShowIntfXcvrDom( ShowIntfCommand ):
   syntax = 'show interfaces transceiver dom [ thresholds ]'
   data = dict( transceiver=xcvrShowKw,
                dom=( 'Show transceiver Digital Optical Monitoring (DOM) parameters '
                      'for the interface' ),
                thresholds=( 'Show transceiver Digital Optical Monitoring (DOM) '
                             'parameters with thresholds for the interface' ) )
   handler = showInterfacesXcvrDom
   cliModel = InterfacesTransceiverDom

BasicCli.addShowCommandClass( ShowIntfXcvrDom )

# ------------------------------------------------------
# Intf capabilities plugin
# ------------------------------------------------------
# Return a dictionary indexed by line rate of the tributaries which
# that line rate supports
def getTributaryCapabilitiesDict( tributaryCaps ):
   LineRateEnum = TacLazyType( "Phy::Coherent::LineRate" )
   lineRateLabelMap = {
      LineRateEnum.lineRate100g: '100G',
      LineRateEnum.lineRate200g: '200G',
      LineRateEnum.lineRate300g: '300G',
      LineRateEnum.lineRate400g: '400G' }
   defaultStr = '(default)'
   lineRateToTrib = {}
   for lineRate in tributaryCaps:
      tributaryCapsForLinerate = tributaryCaps[ lineRate ]
      for trib in tributaryCapsForLinerate.tribCapability:
         lineRateStr = lineRateLabelMap[ lineRate ]
         tributaryStr = str( trib )
         if tributaryCapsForLinerate.defaultTributaryId == trib:
            tributaryStr += defaultStr
         if lineRateStr not in lineRateToTrib:
            lineRateToTrib[ lineRateStr ] = []
         lineRateToTrib[ lineRateStr ].append( tributaryStr )
   return lineRateToTrib

def getInterfaceTributaryCapabilities( _, intfName, label, sysCap=False ):
   tribCaps = {}
   # Do not print "Tributary" for system capabilities, as the trib capabilities
   # are really the capabilities of the inserted transceiver.
   if not sysCap:
      xcvrSlotName = getXcvrSlotName( intfName )
      if xcvrSlotName in gv.xcvrStatusDir.xcvrStatus:
         xcvrStatus = gv.xcvrStatusDir.xcvrStatus[ xcvrSlotName ]
         if( hasattr( xcvrStatus, 'tribSelectLaneCapabilities' ) and
             intfName in xcvrStatus.tribSelectLaneCapabilities ):
            tribCaps = getTributaryCapabilitiesDict(
               xcvrStatus.tribSelectLaneCapabilities[ intfName ].
               tribSelectCapabilities )
   return tribCaps

CliPlugin.EthIntfCli.registerInterfaceCapabilities( 'Tributary', None,
   None, getInterfaceTributaryCapabilities )

# ------------------------------------------------------
# "show tech-support" plugin
# Note: CLIs in show tech do not use CliDynamicPlugin;
#       they live completely within CliPlugin
# ------------------------------------------------------
CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2014-02-05 16:54:34',
   cmds=[ 'show interfaces transceiver detail' ],
   summaryCmds=[ 'show interfaces transceiver' ] )

CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2018-06-18 16:58:26',
   cmds=[ 'show interfaces transceiver tuning detail',
          'show transceiver status' ] )

CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2020-09-04 18:00:00',
   cmds=[ 'show interfaces transceiver dom',
          'show interfaces transceiver dom thresholds' ] )

CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2021-11-11 11:00:00',
   cmds=[ 'show transceiver application' ] )

# ------------------------------------------------------
# Plugin method
# ------------------------------------------------------
def Plugin( em ):
   gv.entityManager = em
   gv.xgc = ConfigMount.mount( em, "hardware/xcvr/xgc", "Xcvr::Xgc", "w" )
   gv.xcvrCliConfigSliceDir = ConfigMount.mount( em,
                                                 "hardware/xcvr/cli/config/slice",
                                                 "Tac::Dir", "wi" )
   gv.xcvrStatusDir = CliPlugin.XcvrAllStatusDir.xcvrAllStatusDir( em )
   gv.lineSystemStatusSliceDir = LazyMount.mount( em,
                                               "hardware/line/system/status/slice",
                                               "Tac::Dir", "ri" )
