# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re
import os
import BasicCli
import CliParser
import ConfigMount
import Tracing
import Tac
from CliPlugin.EthIntfCli import xcvrKw, qsfpKw
from CliPlugin import IntfCli, XcvrAllStatusDir
import CliToken.Monitor
import Cell
import EthIntfLib
import LazyMount
import Toggles.XcvrToggleLib
import MultiRangeRule
import Logging
import Url
from XcvrLib import ( getXcvrStatus, validMediaType, isCmisType, isCmisTypeStr,
                      isPrimaryIntf, domThresholdsCsvParser )
from XcvrMediaTypeLib import mediaTypeCliToEnums, tokensToHelp
from CliMode.Layer1Monitor import Layer1MonitorBaseMode
import CliCommand
import CliMatcher
import CliGlobal
from TypeFuture import TacLazyType

TRANSCEIVER_DOM_THRESHOLD_OVERRIDE_WARNING = Logging.LogHandle(
              "TRANSCEIVER_DOM_THRESHOLD_OVERRIDE_WARNING",
              severity=Logging.logWarning,
              fmt="Custom-defined DOM thresholds have been enabled, and "
                  "will be used instead of the manufacturer-defined thresholds.",
              explanation="Custom DOM thresholds have been enabled by the "
                          "transceiver dom-threshold command. These thresholds "
                          "will apply to modules meeting the custom-defined "
                          "conditions. The custom DOM thresholds will now be used "
                          "in place of manufacturer-defined thresholds.",
              recommendedAction="Ensure the custom DOM thresholds are appropriate "
                                "for the matching transceivers." )

TRANSCEIVER_DOM_THRESHOLD_OVERRIDE_FAILED = Logging.LogHandle(
              "TRANSCEIVER_DOM_THRESHOLD_OVERRIDE_FAILED",
              severity=Logging.logError,
              fmt="%s doesn't exist or contains errors.",
              explanation="The transceiver DOM thresholds weren't overridden.",
              recommendedAction="Check the DOM threshold file"
              "to make sure it exists and has the right format." )

__defaultTraceHandle__ = Tracing.Handle( "XcvrCli" )

# Helper functions

gv = CliGlobal.CliGlobal( xgc=None, xcvrCliConfigSliceDir=None, xcvrStatusDir=None,
                          xcvrConfigDir=None )

sfpHighPowerEnable = Toggles.XcvrToggleLib.toggleSfpHighPowerEnabled()
smbusFailureLogEnable = Toggles.XcvrToggleLib.toggleSmbusErrLogsEnabled()

matcherPerformanceMonitoring = CliMatcher.KeywordMatcher( 'performance-monitoring',
      helpdesc='Set performance-monitoring parameters' )
matcherElectrical = CliMatcher.KeywordMatcher( 'electrical',
      helpdesc='Configure transceiver electrical side parameters' )
matcherTransmitter = CliMatcher.KeywordMatcher( 'transmitter',
      helpdesc='Transmitter parameters' )
diagKeyword = CliMatcher.KeywordMatcher( 'diag',
      helpdesc='Configure transceiver in diagnostics mode' )

XcvrType = TacLazyType( 'Xcvr::XcvrType' )
XcvrMediaType = TacLazyType( 'Xcvr::MediaType' )
XcvrElectricalConfigType = TacLazyType( 'Xcvr::XcvrElectricalConfigType' )
LpoType = TacLazyType( 'Xcvr::LpoType' )
XcvrPresence = TacLazyType( 'Xcvr::XcvrPresence' )

def intfToXcvrName( intfName ):
   """
   Helper function used to allow other files containing config commands to call the
   intfToXcvrName method without accessing xcvrConfigDir directly. This is useful
   in case a command needs to run on slots without a module.

   Parameters:
   -----------
   intfName : String
      The interface name from which we wish to obtain an Xcvr name
      ( and from this value, derive module presence )

   Returns:
   --------
   xcvrName : Optional[ String ]
      Returns the xcvr name associated with a given interface if said module
      is inserted/present, None otherwise
   """
   return gv.xcvrConfigDir.intfToXcvrName.get( intfName )

def mediaIntfReject( intfName, mediaType ):
   xcvrName = gv.xcvrConfigDir.intfToXcvrName.get( intfName )
   if not xcvrName:
      # if Fru haven't started, cli will accept all commands.
      return False
   xcvrStatus = getXcvrStatus( gv.xcvrStatusDir.xcvrStatus.get( xcvrName ) )
   if xcvrStatus and xcvrStatus.presence == XcvrPresence.xcvrPresent:
      if not validMediaType( mediaType, xcvrStatus ):
         return True
   return False

def intfCliReject( intf ):
   """
   Return True if the passed-in interface will never have a transceiver
   or if the passed-in interface is not a primary interface.
   Return False otherwise.

   Parameters
   ----------
   intfName : str

   Returns
   -------
   bool
   """
   intfName = intf if isinstance( intf, str ) else intf.name
   xcvrIntfTypes = [ 'Ethernet', 'Management', 'Fabric' ]
   # BUG912825 we should also check intf.internal() for Fabric intfs
   if not any( intfName.startswith( intfType ) for intfType in xcvrIntfTypes ):
      return True

   # Reject the interface if it is not the primary interface
   return not isPrimaryIntf( intfName )

def primaryIntfGuard( mode, token ):
   if intfCliReject( mode.intf ):
      return CliParser.guardNotThisInterface
   return None

def xcvrCmisGuardFn( intf ):
   # Guard on primary interfaces
   if intfCliReject( intf ):
      return CliParser.guardNotThisPlatform

   xcvrName = gv.xcvrConfigDir.intfToXcvrName.get( intf.name )
   if not xcvrName:
      # If Fru hasn't started, cli will accept all commands.
      return None

   # This command will only be visible on CMIS type ports.
   xcvrStatus = getXcvrStatus( gv.xcvrStatusDir.xcvrStatus.get( xcvrName ) )
   if not( xcvrStatus and isCmisType( xcvrStatus.xcvrType ) ):
      return CliParser.guardNotThisPlatform
   return None

def xcvrCmisGuard( mode, token ):
   '''
   CLI guard for CMIS-capable ports.
   '''
   return xcvrCmisGuardFn( mode.intf )

def hostLaneRangeFn( mode, context ):
   '''
   Helper function to be used in DynamicIntegerMatcher to compute the correct range
   of host lanes for a given port.
   '''
   intfName = mode.intf.name
   xcvrName = gv.xcvrConfigDir.intfToXcvrName.get( intfName )
   if not xcvrName:
      # If Fru hasn't started, cli will accept all lanes.
      return ( 1, 8 )
   xcvrStatus = getXcvrStatus( gv.xcvrStatusDir.xcvrStatus.get( xcvrName ) )
   if not xcvrStatus:
      return ( 1, 8 )
   return ( 1, xcvrStatus.capabilities.maxChannels )

nodeTransceiver = CliCommand.guardedKeyword( 'transceiver',
      helpdesc='configure transceiver settings', guard=primaryIntfGuard )
matcherHostLaneNumber = CliMatcher.DynamicIntegerMatcher(
      helpdesc='Host lane number', rangeFn=hostLaneRangeFn )

def getXcvrConfigCliDir( intfName ):
   """
   Gets the XcvrConfigCli object belonging to the passed in interface.

   Uses the globally scoped xcvrCliConfigSliceDir set in this file's
   CLI plugin function.

   Parameters
   ----------
   intfName : str
      Interface to retrive the XcvrConfigCli for
   Returns
   -------
   Xcvr::XcvrConfigCliDir
      The XcvrConfigCliDir associated with the passed in interface
   """
   return getXcvrConfigCliDirHelper( intfName, gv.xcvrCliConfigSliceDir )

def getXcvrConfigCliDirHelper( intfName, sliceDir ):
   """
   Gets the XcvrConfigCli object belonging to the passed in interface.

   Parameters
   ----------
   intfName : str
      Interface to retrieve the XcvrConfigCli for

   sliceDir : Tac::Dir
      Slicified directory containing XcvrConfigCli objects

   Returns
   -------
   Xcvr::XcvrConfigCliDir
      The XcvrConfigCliDir associated with the passed in interface
   """
   # Helper function to get coathanger associated with interface
   isModular = Cell.cellType() == "supervisor"

   # Supervisors having xcvr slots will be present in FixedSystem slice
   if isModular and not Tac.Type( "Arnet::MgmtIntfId" ).isMgmtIntfId( intfName ):
      sliceName = EthIntfLib.sliceName( intfName )
      if not sliceName:
         return None
      xcvrConfigCliDir = sliceDir.get( sliceName )
   else:
      xcvrConfigCliDir = sliceDir[ 'FixedSystem' ]
   return xcvrConfigCliDir

def getXcvrConfigCliForConfigCommand( intfName, configDir, create ):
   """
   This helper function gets the XcvrConfigCli object to be used in config command
   handler functions.

   Parameters
   ----------
   intfName : str
      Interface name to get/create the XcvrConfigCli for

   configDir : Xcvr::XcvrConfigCliDir
      Slicified directory containing XcvrConfigCli entities

   create : boolean
      When true this function will create the XcvrConfigCli object if it does
   not exist.

   Returns
   -------
   Xcvr::XcvrConfigCli (or NoneType)
      Returns a XcvrConfigCli reference if it exists or we just created one,
   otherwise returns NoneType.
   """
   xcvrConfigCli = configDir.xcvrConfigCli.get( intfName )
   if xcvrConfigCli is None:
      if create:
         xcvrConfigCli = configDir.newXcvrConfigCli( intfName )

   return xcvrConfigCli

class XcvrConfigModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      # BUG912825: replace this naive Fabric interface check instead with checking
      # for mode.intf.internal() set to False in the future.
      return ( mode.intf.name.startswith( 'Ethernet' ) or
               mode.intf.name.startswith( 'Fabric' ) )

IntfCli.IntfConfigMode.addModelet( XcvrConfigModelet )

# Support for default interface command (AID905)
class XcvrTuningIntfCleaner( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      del gv.xgc.tuningConfig[ self.intf_.name ]
      del gv.xgc.rxTuningConfig[ self.intf_.name ]
      del gv.xgc.txPowerConfig[ self.intf_.name ]
      del gv.xgc.rxPowerConfig[ self.intf_.name ]
      del gv.xgc.txDisabled[ self.intf_.name ]
      del gv.xgc.tributary[ self.intf_.name ]

class XcvrTxDisabledIntfCleaner( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      del gv.xgc.txDisabled[ self.intf_.name ]

class XcvrTributaryIntfCleaner( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      del gv.xgc.tributary[ self.intf_.name ]

# Support for default interface command (AID905)
# clear the config in xcvrConfigCliDir
class XcvrConfigCliIntfCleaner( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      intfName = self.intf_.name
      xcvrConfigCliDir = getXcvrConfigCliDir( intfName )
      if not xcvrConfigCliDir:
         return
      del xcvrConfigCliDir.xcvrConfigCli[ intfName ]

# Support for default interface command (AID905)
class PerformanceMonitoringIntfCleaner( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      if self.intf_.name in gv.xgc.performanceMonitoringConfig:
         del gv.xgc.performanceMonitoringConfig[ self.intf_.name ]

##### [no|default] "service unsupported-transceiver <licenseeName> <key>" #####

class XcvrModelet( CliParser.SingletonModelet ):
   pass

BasicCli.GlobalConfigMode.addModelet( XcvrModelet )

# --------------------------------------------------------------------------------
# [ no | default ] service unsupported-transceiver LICENSEE AUTH_KEY
# --------------------------------------------------------------------------------
class ServiceUnsupportedTransceiverCmd( CliCommand.CliCommandClass ):
   syntax = 'service unsupported-transceiver LICENSEE AUTH_KEY'
   noOrDefaultSyntax = 'service unsupported-transceiver ...'
   data = {
      'service': 'Configure service parameters',
      'unsupported-transceiver': 'Enable unsupported transceivers, if authorized',
      'LICENSEE': CliMatcher.PatternMatcher( r'\w+', helpname='WORD',
         helpdesc='Licensee name' ),
      'AUTH_KEY': CliCommand.Node(
         matcher=CliMatcher.PatternMatcher( r'[a-fA-F0-9]{1,8}', helpname='HEX_KEY',
            helpdesc='Key to authorize unsupported transceivers' ),
         sensitive=True )
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      gv.xgc.licensee = args[ 'LICENSEE' ]
      gv.xgc.key = int( args[ 'AUTH_KEY' ], 16 )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      gv.xgc.licensee = ''
      gv.xgc.key = 0

XcvrModelet.addCommandClass( ServiceUnsupportedTransceiverCmd )

# --------------------------------------------------------------------------------
# [ no | default ] transceiver qsfp ethernet-compliance bit7-is-psm4
# --------------------------------------------------------------------------------
class QsfpComplianceBit( CliCommand.CliCommandClass ):
   syntax = 'transceiver qsfp ethernet-compliance bit7-is-psm4'
   noOrDefaultSyntax = syntax
   data = {
      'transceiver': xcvrKw,
      'qsfp': qsfpKw,
      'ethernet-compliance': ( 'Override standard meaning of bits in QSFP '
                                'eth-compliance field' ),
      'bit7-is-psm4': 'Bit7 identifies 40GBASE-PSM4 media type',
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      gv.xgc.ethComplianceBit7IsPsm4 = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      gv.xgc.ethComplianceBit7IsPsm4 = False

BasicCli.GlobalConfigMode.addCommandClass( QsfpComplianceBit )

# --------------------------------------------------------------------------------
# [ no | default ] transceiver frequency FREQUENCY
# --------------------------------------------------------------------------------
def mhzFreqToGhz( frequency ):
   return float( frequency ) / 1e3

def ghzFreqToMhz( frequency ):
   return float( frequency ) * 1e3

def getTuningFreq( baseFrequency, channel, grid ):
   return baseFrequency + ( channel - 1 ) * grid

def getTuningChannel( baseFrequency, freq, grid ):
   """
   This helper function is for DZ-T [SFP-10G-DZT] and CFPX [100G-DWDM-E]
   definitions of an optical channel.

   In this context an optical channel is the number of multiples of the grid spacing
   frequency is from the base frequency to the operating frequency.
   I.g Channel 10 of 100GHz spacing, with a base frequency of 191200 GHZ, is at
   192200 GHz.

   This method returns the channel number if valid. Returns None otherwise.

   Parameters
   ----------
   baseFrequency : float
      The center optical frequency. Units are all of the same magnitude.
   freq : float
      The frequency this channel operates.
   grid : float
      The grid spacing.

   Returns
   ----------
      int | None
   """

   # Round down to the nearest channel
   assert freq >= baseFrequency
   if ( freq - baseFrequency ) % grid != 0:
      # Not a multiple of our channel spacing!
      return None
   return int( 1 + ( freq - baseFrequency ) / grid )

def cmdSetXcvrChannel( mode, args ):
   channel = args.get( 'CHANNEL' )
   grid = args.get( 'GRID' )
   frequency = args.get( 'FREQUENCY' )
   displayUnits = 'ghz' in args
   tuningConfigName = mode.intf.name

   if channel or grid:
      if tuningConfigName not in gv.xgc.tuningConfig:
         gv.xgc.newTuningConfig( tuningConfigName )
      tuningConfig = gv.xgc.tuningConfig[ tuningConfigName ]
      tuningConstants = Tac.Value( 'Xcvr::TuningConstants' )
      tuningConfig.grid = int( ghzFreqToMhz( grid ) ) if grid else \
                                         tuningConstants.defaultGrid
      tuningConfig.channel = channel
      tuningConfig.frequency = 0
   elif frequency:
      if tuningConfigName not in gv.xgc.tuningConfig:
         gv.xgc.newTuningConfig( tuningConfigName )
      tuningConfig = gv.xgc.tuningConfig[ tuningConfigName ]
      tuningConfig.grid = 0
      tuningConfig.channel = 0
      tuningConfig.frequency = int( round( ghzFreqToMhz( frequency ) ) )
      tuningConfig.displayUnits = displayUnits
   else:
      if tuningConfigName in gv.xgc.tuningConfig:
         del gv.xgc.tuningConfig[ tuningConfigName ]

gridKw = CliMatcher.KeywordMatcher( 'grid-spacing',
                          helpdesc='Grid-spacing tuning for optical transceiver' )
matcherGhzUnits = CliMatcher.KeywordMatcher( 'ghz', helpdesc='Units in gigahertz' )

# command for configuring the transmitter tuning frequency
class TransceiverFrequencyFrequencyCmd( CliCommand.CliCommandClass ):
   syntax = 'transceiver frequency FREQUENCY [ ghz ]'
   noOrDefaultSyntax = 'transceiver frequency ...'
   data = {
      'transceiver': nodeTransceiver,
      'frequency': 'Frequency tuning for optical transmitter',
      'FREQUENCY': CliMatcher.FloatMatcher( 180000, 200000,
         helpdesc='Frequency (GHz)', precisionString='%.9g' ),
      'ghz': matcherGhzUnits,
   }

   handler = cmdSetXcvrChannel
   noOrDefaultHandler = cmdSetXcvrChannel

XcvrConfigModelet.addCommandClass( TransceiverFrequencyFrequencyCmd )

# --------------------------------------------------------------------------------
# [ no | default ] transceiver channel CHANNEL [ grid-spacing GRID ]
# --------------------------------------------------------------------------------
class TransceiverChannelChannelCmd( CliCommand.CliCommandClass ):
   syntax = 'transceiver channel CHANNEL [ grid-spacing GRID ]'
   noOrDefaultSyntax = 'transceiver channel ...'
   data = {
      'transceiver': xcvrKw,
      'channel': 'Channel tuning for optical transceiver',
      'CHANNEL': CliMatcher.IntegerMatcher( 1, 1000, helpdesc='Channel number' ),
      'grid-spacing': gridKw,
      'GRID': CliMatcher.PatternMatcher(
         pattern='(100|50|33|25)(\\.0{0,2})?|12\\.50?|6\\.25',
         helpdesc='Spacing (GHz)', helpname='<100|50|25|12.5|6.25>' ),
   }

   handler = cmdSetXcvrChannel
   noOrDefaultHandler = handler

XcvrConfigModelet.addCommandClass( TransceiverChannelChannelCmd )

# --------------------------------------------------------------------------------
# transceiver receiver signal-power POWER
# --------------------------------------------------------------------------------
matcherRx = CliMatcher.KeywordMatcher( 'receiver', helpdesc='Receiver parameters' )
matcherSignalPower = CliMatcher.KeywordMatcher( 'signal-power',
      helpdesc='Optical signal power' )

class TransceiverReceiverSignalPowerPowerCmd( CliCommand.CliCommandClass ):
   syntax = 'transceiver receiver signal-power POWER'
   noOrDefaultSyntax = 'transceiver receiver signal-power ...'
   data = {
      'transceiver': xcvrKw,
      'receiver': matcherRx,
      'signal-power': matcherSignalPower,
      'POWER': CliMatcher.FloatMatcher( -30, 10, helpdesc='Power (dBm)',
         precisionString='%.4g' ),
   }

   @staticmethod
   def handler( mode, args ):
      power = args[ 'POWER' ]
      if mode.intf.name not in gv.xgc.rxPowerConfig:
         gv.xgc.newRxPowerConfig( mode.intf.name )

      rxPowerConfig = gv.xgc.rxPowerConfig[ mode.intf.name ]
      rxPowerConfig.signalPower = power

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      del gv.xgc.rxPowerConfig[ mode.intf.name ]

XcvrConfigModelet.addCommandClass( TransceiverReceiverSignalPowerPowerCmd )

# --------------------------------------------------------------------------------
# transceiver receiver frequency FREQUENCY
# --------------------------------------------------------------------------------
def rxFrequencyGuard( mode, token ):
   xcvrName = gv.xcvrConfigDir.intfToXcvrName.get( mode.intf.name )
   if not xcvrName:
      return None
   # If Fru hasn't started, cli will accept all commands.
   xcvrStatus = getXcvrStatus( gv.xcvrStatusDir.xcvrStatus.get( xcvrName ) )
   if ( xcvrStatus.presence != XcvrPresence.xcvrPresent and
        xcvrStatus.xcvrType in [ XcvrType.cfp2 ] ):
      # If no module is inserted, cli will accept all commands.
      return None
   if xcvrStatus.mediaType == XcvrMediaType.xcvr400GDwdmDco:
      # Only the Dp04 DCO module is able to configure rx frequency
      return None
   return "not supported for this transceiver"

matcherRxFrequency = CliMatcher.KeywordMatcher( 'frequency',
                                       'Frequency tuning for optical receiver' )
nodeRxFrequencyGuarded = CliCommand.Node( matcher=matcherRxFrequency,
                                          guard=rxFrequencyGuard )

class TransceiverReceiverFrequencyCmd( CliCommand.CliCommandClass ):
   syntax = 'transceiver receiver frequency_GUARDED FREQUENCY ghz'
   noOrDefaultSyntax = 'transceiver receiver frequency ...'
   data = {
      'transceiver': xcvrKw,
      'receiver': matcherRx,
      'frequency_GUARDED': nodeRxFrequencyGuarded,
      'frequency': matcherRxFrequency,
      'FREQUENCY': CliMatcher.FloatMatcher( 190000, 200000,
         helpdesc='Frequency (GHz)', precisionString='%.9g' ),
      'ghz': matcherGhzUnits,
   }
   # the guard function is dynamic (depends on inserted transceivers)
   # so we need to disable cache
   allowCache = False

   @staticmethod
   def handler( mode, args ):
      frequency = args.get( 'FREQUENCY' )
      if mode.intf.name not in gv.xgc.rxTuningConfig:
         gv.xgc.newRxTuningConfig( mode.intf.name )

      rxTuningConfig = gv.xgc.rxTuningConfig[ mode.intf.name ]
      rxTuningConfig.rxTuningControlledByTx = False
      rxTuningConfig.frequency = int( round( ghzFreqToMhz( frequency ) ) )
      rxTuningConfig.grid = 0
      rxTuningConfig.channel = 0

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      del gv.xgc.rxTuningConfig[ mode.intf.name ]

XcvrConfigModelet.addCommandClass( TransceiverReceiverFrequencyCmd )

# --------------------------------------------------------------------------------
# transceiver receiver frequency fine FINE_FREQUENCY
# --------------------------------------------------------------------------------
matcherRxFrequencyFine = CliMatcher.KeywordMatcher( 'fine',
   helpdesc='Bright fine tuning of frequency for optical receiver' )

class TransceiverReceiverFrequencyFineCmd( CliCommand.CliCommandClass ):
   # NOTE: the noOrDefault syntax conflicts with TransceiverReceiverFrequencyCmd,
   # but is resolved due to '...' having a lower priority. However, this only
   # works if the frequency keywords share the same matcher.
   syntax = 'transceiver receiver frequency_GUARDED fine FINE_FREQUENCY ghz'
   noOrDefaultSyntax = 'transceiver receiver frequency fine ...'
   data = {
      'transceiver': xcvrKw,
      'receiver': matcherRx,
      'frequency_GUARDED': nodeRxFrequencyGuarded,
      'frequency': matcherRxFrequency,
      'fine': matcherRxFrequencyFine,
      'FINE_FREQUENCY': CliMatcher.FloatMatcher( -10.0, 10.0,
         helpdesc='Frequency (GHz)', precisionString='%.9g' ),
      'ghz': matcherGhzUnits,
   }
   # the guard function is dynamic (depends on inserted transceivers)
   # so we need to disable cache
   allowCache = False

   @staticmethod
   def handler( mode, args ):
      fineFrequency = args.get( 'FINE_FREQUENCY' )
      if mode.intf.name not in gv.xgc.rxTuningConfig:
         gv.xgc.newRxTuningConfig( mode.intf.name )

      rxTuningConfig = gv.xgc.rxTuningConfig[ mode.intf.name ]
      rxTuningConfig.fineFrequency = float( round( ghzFreqToMhz( fineFrequency ) ) )
      rxTuningConfig.grid = 0
      rxTuningConfig.channel = 0

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if mode.intf.name in gv.xgc.rxTuningConfig:
         gv.xgc.rxTuningConfig[ mode.intf.name ].fineFrequency = 0.0

XcvrConfigModelet.addCommandClass( TransceiverReceiverFrequencyFineCmd )

# --------------------------------------------------------------------------------
# transceiver transmitter signal-power POWER
# --------------------------------------------------------------------------------
class TransceiverTransmitterSignalPowerPowerCmd( CliCommand.CliCommandClass ):
   syntax = 'transceiver transmitter signal-power POWER'
   noOrDefaultSyntax = 'transceiver transmitter signal-power ...'
   data = {
      'transceiver': xcvrKw,
      'transmitter': matcherTransmitter,
      'signal-power': matcherSignalPower,
      'POWER': CliMatcher.FloatMatcher( -30, 10, helpdesc='Power (dBm)',
         precisionString='%.4g' ),
   }

   @staticmethod
   def handler( mode, args ):
      power = args[ 'POWER' ]
      if mode.intf.name not in gv.xgc.txPowerConfig:
         gv.xgc.newTxPowerConfig( mode.intf.name )

      txPowerConfig = gv.xgc.txPowerConfig[ mode.intf.name ]
      powerConstants = Tac.Value( 'Xcvr::PowerConstants' )
      txPowerConfig.signalPower = power if power is not None else \
                                  powerConstants.defaultTxPower

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if mode.intf.name in gv.xgc.txPowerConfig:
         del gv.xgc.txPowerConfig[ mode.intf.name ]

XcvrConfigModelet.addCommandClass( TransceiverTransmitterSignalPowerPowerCmd )

# --------------------------------------------------------------------------------
# transceiver transmitter disabled
# --------------------------------------------------------------------------------
def txDisabledGuard( mode, token ):
   xcvrName = gv.xcvrConfigDir.intfToXcvrName.get( mode.intf.name )
   if not xcvrName:
      return None
   # If Fru hasn't started, cli will accept all commands.
   xcvrStatus = getXcvrStatus( gv.xcvrStatusDir.xcvrStatus.get( xcvrName ) )
   if ( xcvrStatus.presence != XcvrPresence.xcvrPresent and
        xcvrStatus.xcvrType in [ XcvrType.cfp2, XcvrType.osfp, XcvrType.qsfpDd ] ):
      # If no module is inserted, cli will accept all commands.
      return None
   if xcvrStatus.isCoherent( xcvrStatus.mediaType ):
      # Only the 400G ZR and Dp04 DCO module are able to configure soft TX disable
      return None
   return "not supported for this transceiver"

matcherTxDisabled = CliMatcher.KeywordMatcher( 'disabled',
                                               'Optical transmitter disabled' )
nodeTxDisabledGuarded = CliCommand.Node( matcher=matcherTxDisabled,
                                         guard=txDisabledGuard )

class TransceiverTransmitterDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'transceiver transmitter disabled_GUARDED'
   noOrDefaultSyntax = "transceiver transmitter disabled"
   data = {
      'transceiver': xcvrKw,
      'transmitter': matcherTransmitter,
      'disabled_GUARDED': nodeTxDisabledGuarded,
      'disabled': matcherTxDisabled,
   }
   # the guard function is dynamic (depends on inserted transceivers)
   # so we need to disable cache
   allowCache = False

   @staticmethod
   def handler( mode, args ):
      gv.xgc.txDisabled.add( mode.intf.name )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      del gv.xgc.txDisabled[ mode.intf.name ]

XcvrConfigModelet.addCommandClass( TransceiverTransmitterDisabledCmd )

# --------------------------------------------------------------------------------
# [ no | default ] transceiver power ignore
# --------------------------------------------------------------------------------
def xcvrPowerGuard( mode, token ):
   intfName = mode.intf.name
   # Guard on primary interfaces
   if intfCliReject( mode.intf ):
      return CliParser.guardNotThisPlatform

   xcvrName = gv.xcvrConfigDir.intfToXcvrName.get( intfName )
   if not xcvrName:
      # If Fru hasn't started, cli will accept all commands.
      return None

   # This command will only be visible on CMIS slots and QSFP slots.
   xcvrStatus = gv.xcvrStatusDir.xcvrStatus.get( xcvrName )
   if not( xcvrStatus and ( isCmisType( xcvrStatus.xcvrType ) or
                            xcvrStatus.xcvrType == XcvrType.qsfpPlus or
                            xcvrStatus.xcvrType == XcvrType.sfpPlus ) ):
      return CliParser.guardNotThisPlatform
   return None

matcherPower = CliMatcher.KeywordMatcher(
      'power', helpdesc='Configure transceiver power parameters' )
nodePower = CliCommand.Node( matcher=matcherPower, guard=xcvrPowerGuard )

class TransceiverPowerIgnoreCmd( CliCommand.CliCommandClass ):
   syntax = 'transceiver POWER_GUARDED ignore'
   noOrDefaultSyntax = 'transceiver POWER_UNGUARDED ignore'
   data = {
      'transceiver': xcvrKw,
      'POWER_GUARDED': nodePower,
      'POWER_UNGUARDED': matcherPower,
      'ignore': 'Ignore advertised transceiver power consumption',
   }
   # Guard function is dynamic (depends on inserted transceivers)
   # so we need to disable cache.
   allowCache = False

   @staticmethod
   def handler( mode, args ):
      intfName = mode.intf.name
      xcvrConfigCliDir = getXcvrConfigCliDir( intfName )
      xcvrConfigCli = getXcvrConfigCliForConfigCommand( intfName, xcvrConfigCliDir,
                                                        True )
      xcvrConfigCli.modulePowerIgnore = True
      mode.addWarning( 'You can risk damaging hardware by using transceiver '
                       'modules with high power consumption. We recommend that '
                       'you do this only under direction from '
                       'Arista Networks.' )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfName = mode.intf.name
      xcvrConfigCliDir = getXcvrConfigCliDir( intfName )
      xcvrConfigCli = getXcvrConfigCliForConfigCommand( intfName, xcvrConfigCliDir,
                                                        False )
      if xcvrConfigCli is None:
         return

      xcvrConfigCli.modulePowerIgnore = False
      if xcvrConfigCliDelete( xcvrConfigCli ):
         del xcvrConfigCliDir.xcvrConfigCli[ intfName ]

XcvrConfigModelet.addCommandClass( TransceiverPowerIgnoreCmd )

# --------------------------------------------------------------------------------
# [ no | default ] performance-monitoring period [ SECS | MINS | HOURS | DAYS ]
# --------------------------------------------------------------------------------
class PerformanceMonitoringPeriodCmd( CliCommand.CliCommandClass ):
   syntax = 'performance-monitoring period [ SECS | MINS | HOURS | DAYS ]'
   noOrDefaultSyntax = 'performance-monitoring period ...'
   data = {
      'performance-monitoring': matcherPerformanceMonitoring,
      'period': 'Set performance-monitoring period',
      'SECS': CliMatcher.PatternMatcher( pattern='[0-9]+s', helpdesc='seconds',
         helpname='[0-9]+s' ),
      'MINS': CliMatcher.PatternMatcher( pattern='[0-9]+m', helpdesc='minutes',
         helpname='[0-9]+m' ),
      'HOURS': CliMatcher.PatternMatcher( pattern='[0-9]+h', helpdesc='hours',
         helpname='[0-9]+h' ),
      'DAYS': CliMatcher.PatternMatcher( pattern='[0-9]+d', helpdesc='days',
         helpname='[0-9]+d' ),
   }

   @staticmethod
   def handler( mode, args ):
      monitoringTime = args.get( 'SECS', args.get( 'MINS',
         args.get( 'HOURS', args.get( 'DAYS' ) ) ) )
      no = CliCommand.isNoOrDefaultCmd( args )
      pmPeriod = Tac.Value( 'Xcvr::PerformanceMonitoringPeriodConfig' )
      if monitoringTime is None or no:
         pm = Tac.Value( 'Xcvr::Sff8436PerformanceMonitoring' )
         # pylint: disable-next=consider-using-f-string
         monitoringTime = '%ds' % pm.defaultPMPeriod
      m = re.match( r'([0-9]+)\s*(s|m|h|d)\s*$', monitoringTime )
      unit = m.group( 2 )
      pmPeriod.unit = unit
      if unit is None:
         pmPeriod.valueInSeconds = Tac.endOfTime
      else:
         factor = 1
         if unit != 's':
            factor *= 60
            if unit != 'm':
               factor *= 60
               if unit != 'h':
                  factor *= 24
                  assert unit == 'd', 'wrong time format'
         pmPeriod.valueInSeconds = int( m.group( 1 ) ) * factor
      gv.xgc.performanceMonitoringPeriod = pmPeriod

   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( PerformanceMonitoringPeriodCmd )

# --------------------------------------------------------------------------------
# [ no | default ] performance-monitoring transceiver default
# --------------------------------------------------------------------------------
class PerformanceMonitoringCmd( CliCommand.CliCommandClass ):
   syntax = 'performance-monitoring transceiver default'
   noOrDefaultSyntax = syntax
   data = {
      'performance-monitoring': matcherPerformanceMonitoring,
      'transceiver': xcvrKw,
      'default': 'Performance-monitoring global control',
   }

   @staticmethod
   def handler( mode, args ):
      gv.xgc.performanceMonitoringGlobal = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      gv.xgc.performanceMonitoringGlobal = False

BasicCli.GlobalConfigMode.addCommandClass( PerformanceMonitoringCmd )

# --------------------------------------------------------------------------------
# [ no | default ] performance-monitoring transceiver
# --------------------------------------------------------------------------------
class IntfPerformanceMonitoringCmd( CliCommand.CliCommandClass ):
   syntax = 'performance-monitoring transceiver'
   noOrDefaultSyntax = syntax
   data = {
      'performance-monitoring': matcherPerformanceMonitoring,
      'transceiver': xcvrKw,
   }

   @staticmethod
   def handler( mode, args ):
      # for default, switch to global configuration
      no = CliCommand.isNoCmd( args )
      if CliCommand.isDefaultCmd( args ):
         no = not gv.xgc.performanceMonitoringGlobal

      pmConfigName = mode.intf.name
      if pmConfigName not in gv.xgc.performanceMonitoringConfig:
         gv.xgc.newPerformanceMonitoringConfig( pmConfigName )
      performanceMonitoringConfig = \
            gv.xgc.performanceMonitoringConfig[ pmConfigName ]
      performanceMonitoringConfig.performanceMonitoringEnabled = not no

   noOrDefaultHandler = handler

XcvrConfigModelet.addCommandClass( IntfPerformanceMonitoringCmd )

# --------------------------------------------------------------------------------
# [ no | default ] transceiver 100GE-DWDM2 temperature cooling-target TEMP celsius
# --------------------------------------------------------------------------------
class CoolingTargetDwdm2TemperatureCmd( CliCommand.CliCommandClass ):
   syntax = 'transceiver 100GE-DWDM2 temperature cooling-target TEMP celsius'
   noOrDefaultSyntax = 'transceiver 100GE-DWDM2 temperature cooling-target ...'
   data = {
      'transceiver': xcvrKw,
      '100GE-DWDM2': 'Set 100GE-DWDM2 media type parameters',
      'temperature': 'Set temperature related parameters on the transceiver',
      'cooling-target': 'Set cooling target for the transceiver',
      'TEMP': CliMatcher.IntegerMatcher( 1, 100, helpdesc='Temperature' ),
      'celsius': 'Unit Celsius',
   }

   @staticmethod
   def handler( mode, args ):
      # Sets the target temperature config to 'temperature' Celsius
      mediaType = XcvrMediaType.xcvr100GEDwdm2
      gv.xgc.targetTemperatureForMediaType[ mediaType ] = float( args[ 'TEMP' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      # Resets the target temperature config to default value for Unit Celsius
      mediaType = XcvrMediaType.xcvr100GEDwdm2
      del gv.xgc.targetTemperatureForMediaType[ mediaType ]

BasicCli.GlobalConfigMode.addCommandClass( CoolingTargetDwdm2TemperatureCmd )

# --------------------------------------------------------------------------------
# [ no | default ]
# transceiver media MEDIA_TYPE temperature cooling-target TEMP celsius
# --------------------------------------------------------------------------------
mediaKw = CliMatcher.KeywordMatcher( 'media',
                                     helpdesc='Configure media specific options' )

class CoolingTargetTemperatureCmd( CliCommand.CliCommandClass ):
   syntax = 'transceiver media MEDIA_TYPE temperature cooling-target TEMP celsius'
   noOrDefaultSyntax = 'transceiver media MEDIA_TYPE temperature cooling-target ...'
   data = {
      'transceiver': xcvrKw,
      'media': mediaKw,
      'MEDIA_TYPE': CliMatcher.EnumMatcher( {
         mediaType: helpString.replace( 'media type to be',
                                         'target temperature for' )
         for mediaType, helpString in tokensToHelp.items() } ),
      'temperature': 'Set temperature related parameters on the transceiver',
      'cooling-target': 'Set cooling target for the transceiver',
      'TEMP': CliMatcher.IntegerMatcher( 1, 100, helpdesc='Temperature' ),
      'celsius': 'Unit Celsius',
   }

   @staticmethod
   def handler( mode, args ):
      mediaType = mediaTypeCliToEnums[ args[ 'MEDIA_TYPE' ] ]
      gv.xgc.targetTemperatureForMediaType[ mediaType ] = float( args[ 'TEMP' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mediaType = mediaTypeCliToEnums[ args[ 'MEDIA_TYPE' ] ]
      del gv.xgc.targetTemperatureForMediaType[ mediaType ]

BasicCli.GlobalConfigMode.addCommandClass( CoolingTargetTemperatureCmd )

#### [no|default] transceiver media override <media type> ###

def anyCmisLoopbackConfigured( config ):
   noLoopback = Tac.enumValue( "Xcvr::CmisLoopback::Mode", "NoLoopback" )
   return not all( noLoopback == v for k, v in config.items() )

def xcvrConfigCliDelete( xcvrConfigCli ):
   """
   Parameters
   ----------
   Xcvr::XcvrConfigCli

   Returns
   -------
   boolean
      True if the passed in xcvrConfigCli object is able to be deleted.
      Otherwise, returns False.
   """
   # We need to check every attribute of xcvrConfigCli equals to default one.
   # Otherwise, we cannot delete xcvrConfigCli.
   if xcvrConfigCli.overrideMediaType != "xcvrUnknown" or \
      xcvrConfigCli.overrideTuningParams:
      return False
   if xcvrConfigCli.overrideLpoType != XcvrElectricalConfigType.unset:
      return False
   if xcvrConfigCli.signalBwConfig != LpoType.none:
      return False
   for laneTuningParam in xcvrConfigCli.slotTuningParams.values():
      if laneTuningParam.txInputEqualization.valid or \
         laneTuningParam.txInputAdaptiveEqEnable.valid or \
         laneTuningParam.rxOutputAmplitude.valid or \
         laneTuningParam.rxOutputPreEmphasis.valid or \
         laneTuningParam.rxOutputPostEmphasis.valid or \
         laneTuningParam.txInputEqualization.moduleDefault or \
         laneTuningParam.rxOutputAmplitude.moduleDefault or \
         laneTuningParam.rxOutputPreEmphasis.moduleDefault:
         return False
   if xcvrConfigCli.forceModuleRemoved:
      return False
   if sfpHighPowerEnable and xcvrConfigCli.preventSfpHighPowerEnable:
      return False
   if xcvrConfigCli.forceQsfpManagementMode:
      return False
   if ( xcvrConfigCli.testPatternConfig is not None and
        xcvrConfigCli.testPatternConfig.isActiveConfig() ):
      return False
   if xcvrConfigCli.modulePowerIgnore:
      return False
   if xcvrConfigCli.cmisOverrideApplication:
      return False
   defaultRxAutoSquelchConfig = xcvrConfigCli.rxAutoSquelch.moduleDefault and \
                                not xcvrConfigCli.rxAutoSquelch.override
   if xcvrConfigCli.rxAutoSquelch.valid and not defaultRxAutoSquelchConfig:
      return False
   cmisFecDegradeConfig = xcvrConfigCli.cmisFecDegradeConfig
   if cmisFecDegradeConfig.excessiveDegrade.enabled or \
      cmisFecDegradeConfig.detectedDegrade.enabled:
      return False

   if xcvrConfigCli.cmisLoopbackConfig is not None and (
         anyCmisLoopbackConfigured( xcvrConfigCli.cmisLoopbackConfig.mediaLoopback )
         or anyCmisLoopbackConfigured(
            xcvrConfigCli.cmisLoopbackConfig.hostLoopback ) ):
      return False

   if xcvrConfigCli.legacySRBDCompatibilityConfig:
      return False

   if xcvrConfigCli.reinitRequest:
      return False

   return True

def setMediaType( mode, args ):
   intfName = mode.intf.name
   noOrDefault = CliCommand.isNoOrDefaultCmd( args )
   mediaType = mediaTypeCliToEnums.get( args.get( 'MEDIA_TYPE' ) )

   if mediaType and mediaIntfReject( intfName, mediaType ):
      # pylint: disable-next=consider-using-f-string
      mode.addError( "The overridden media type doesn't match"
                     " the transceiver type of %s. " % intfName )
      return

   xcvrConfigCliDir = getXcvrConfigCliDir( intfName )
   xcvrConfigCli = getXcvrConfigCliForConfigCommand( intfName, xcvrConfigCliDir,
                                                     not noOrDefault )
   if xcvrConfigCli is None:
      assert noOrDefault
      return

   assert xcvrConfigCli
   if mediaType:
      xcvrConfigCli.overrideMediaType = mediaType
   else:
      xcvrConfigCli.overrideMediaType = "xcvrUnknown"
      if xcvrConfigCliDelete( xcvrConfigCli ):
         del xcvrConfigCliDir.xcvrConfigCli[ intfName ]

def setTributaryId( mode, args ):
   intfName = mode.intf.name
   noOrDefault = CliCommand.isNoOrDefaultCmd( args )
   tributaryId = args.get( 'TRIBUTARY_ID' )

   if noOrDefault:
      del gv.xgc.tributary[ intfName ]
   else:
      gv.xgc.tributary[ intfName ] = tributaryId

# --------------------------------------------------------------------------------
# [ no | default ] transceiver media override MEDIA_TYPE
# --------------------------------------------------------------------------------
class TransceiverMediaOverrideCmd( CliCommand.CliCommandClass ):
   syntax = 'transceiver media override MEDIA_TYPE'
   noOrDefaultSyntax = 'transceiver media override ...'
   data = {
      'transceiver': nodeTransceiver,
      'media': mediaKw,
      'override': 'Override the detected media type',
      'MEDIA_TYPE': CliMatcher.EnumMatcher( tokensToHelp ),
   }

   handler = setMediaType
   noOrDefaultHandler = handler

XcvrConfigModelet.addCommandClass( TransceiverMediaOverrideCmd )

# --------------------------------------------------------------------------------
# [ no | default ] transceiver media tributary TRIBUTARY_ID
# --------------------------------------------------------------------------------
class TransceiverMediaTributaryCmd( CliCommand.CliCommandClass ):
   syntax = 'transceiver media tributary TRIBUTARY_ID'
   noOrDefaultSyntax = 'transceiver media tributary ...'
   data = {
      'transceiver': xcvrKw,
      'media': mediaKw,
      'tributary': 'Optical tributary to assign to interface',
      'TRIBUTARY_ID': CliMatcher.IntegerMatcher( 0, 4, helpdesc='Tributary' ),
   }

   handler = setTributaryId
   noOrDefaultHandler = handler

XcvrConfigModelet.addCommandClass( TransceiverMediaTributaryCmd )

##### [no | default] transceiver dom-threshold file <default|filename> #####

def enableDomThresholdOverride( mode, args ):
   fileOrUrl = args.get( 'URL', 'default' )
   if fileOrUrl == 'default':
      domThresholdDataSrc = '/usr/share/Xcvr/AristaStandardDOMThresholds.csv'
   elif os.path.isfile( fileOrUrl.localFilename() ):
      domThresholdDataSrc = fileOrUrl.localFilename()
   else:
      mode.addError( f"{fileOrUrl} doesn't exist." )
      Logging.log( TRANSCEIVER_DOM_THRESHOLD_OVERRIDE_FAILED,
                   fileOrUrl.localFilename() )
      return

   parseErrors = domThresholdsCsvParser( gv.xgc, domThresholdDataSrc )

   if not parseErrors:
      gv.xgc.domThresholdOverrideEnabled = True
      gv.xgc.domThresholdOverrideSrc = str( fileOrUrl )
      if fileOrUrl != 'default':
         mode.addWarning(
            "Custom-defined DOM thresholds have been enabled, "
            "and will be used instead of" )
         mode.addWarning( "the manufacturer-defined thresholds." )
         mode.addWarning(
            "Ensure the custom DOM thresholds are appropriate "
            "for the matching transceivers." )
         Logging.log( TRANSCEIVER_DOM_THRESHOLD_OVERRIDE_WARNING )
   else:
      for err in parseErrors:
         mode.addError( f"File: {fileOrUrl}, Row: {err.row}, Col: {err.col}\n"
                        f"Parse Error: {err.errMsg}" )
      Logging.log( TRANSCEIVER_DOM_THRESHOLD_OVERRIDE_FAILED,
                   fileOrUrl.localFilename() )

def disableDomThresholdOverride( mode, args ):
   gv.xgc.domThresholdOverrideEnabled = False

class DomThresholdFileCmd( CliCommand.CliCommandClass ):
   syntax = "transceiver dom-threshold file default | URL | ANYFILE"
   noOrDefaultSyntax = "transceiver dom-threshold file ..."
   data = {
      'transceiver': xcvrKw,
      'dom-threshold': 'Configure optics transceiver DOM thresholds',
      'file': 'Override DOM thresholds with values from specified file',
      'default': 'Default Arista-standardized thresholds',
      'URL': Url.UrlMatcher( lambda fs: fs.scheme in [ 'flash:' ] and
                                fs.supportsRead(),
                                'File location', acceptSimpleFile=False ),
      'ANYFILE': CliCommand.Node(
         matcher=Url.UrlMatcher( lambda fs: ( fs.scheme in [ 'file:' ] and
                                              fs.supportsRead() ),
                                 'File location', acceptSimpleFile=False ),
         alias='URL',
         hidden=True )
      }

   handler = enableDomThresholdOverride
   noOrDefaultHandler = disableDomThresholdOverride

BasicCli.GlobalConfigMode.addCommandClass( DomThresholdFileCmd )

# --------------------------------------------------------------------------------
# [ no | default ] transceiver electrical lane LANE_LIST
#                       [ { ( tx-input-equalization TX_INPUT_EQUALIZATION ) |
#                           ( rx-output-amplitude RX_OUTPUT_AMPLITUDE ) |
#                           ( rx-output-pre-emphasis RX_OUTPUT_PREEMPHASIS ) |
#                           ( rx-output-post-emphasis RX_OUTPUT_POSTEMPHASIS ) } ]
# --------------------------------------------------------------------------------

def copyCliLaneTuningParam( cliLaneTuningParam=None ):
   laneTuningParam = Tac.Value( "Xcvr::LaneTuningParameter" )
   laneTuningParam.txInputEqualization = Tac.Value( "Xcvr::TuningValue",
      cliLaneTuningParam.txInputEqualization.valid,
      cliLaneTuningParam.txInputEqualization.val,
      cliLaneTuningParam.txInputEqualization.moduleDefault )
   laneTuningParam.rxOutputAmplitude = Tac.Value( "Xcvr::TuningValue",
      cliLaneTuningParam.rxOutputAmplitude.valid,
      cliLaneTuningParam.rxOutputAmplitude.val,
      cliLaneTuningParam.rxOutputAmplitude.moduleDefault )
   laneTuningParam.rxOutputPreEmphasis = Tac.Value( "Xcvr::TuningValue",
      cliLaneTuningParam.rxOutputPreEmphasis.valid,
      cliLaneTuningParam.rxOutputPreEmphasis.val,
      cliLaneTuningParam.rxOutputPreEmphasis.moduleDefault )
   laneTuningParam.rxOutputPostEmphasis = Tac.Value( "Xcvr::TuningValue",
      cliLaneTuningParam.rxOutputPostEmphasis.valid,
      cliLaneTuningParam.rxOutputPostEmphasis.val,
      cliLaneTuningParam.rxOutputPostEmphasis.moduleDefault )
   laneTuningParam.txInputAdaptiveEqEnable = Tac.Value( "Xcvr::TuningValue",
      cliLaneTuningParam.txInputAdaptiveEqEnable.valid,
      cliLaneTuningParam.txInputAdaptiveEqEnable.val,
      cliLaneTuningParam.txInputAdaptiveEqEnable.moduleDefault )

   return laneTuningParam

# gyorgym NOTE: BUG504031
# This function is never called.
def moduleDefaultKeywordGuard( mode, token ):
   intfName = mode.intf.name
   xcvrName = gv.xcvrConfigDir.intfToXcvrName.get( intfName )
   if not xcvrName:
      return None

   xcvrStatus = getXcvrStatus( gv.xcvrStatusDir.xcvrStatus.get( xcvrName ) )
   if not xcvrStatus:
      return None

   if xcvrStatus.xcvrType != 'qsfpPlus':
      # Guard module default token for non-qsfp ports
      return CliParser.guardNotThisPlatform

   return None

def autoTxInputEqKeywordGuard( mode, token ):
   intfName = mode.intf.name
   xcvrName = gv.xcvrConfigDir.intfToXcvrName.get( intfName )
   if not xcvrName:
      return None

   xcvrStatus = getXcvrStatus( gv.xcvrStatusDir.xcvrStatus.get( xcvrName ) )
   if not xcvrStatus:
      return None

   if not isCmisTypeStr( xcvrStatus.xcvrType ):
      # Guard this token for non-osfp and non-qsfpDd ports
      return CliParser.guardNotThisPlatform

   cmisTuningSupp = xcvrStatus.cmisTuningParamCapabilities.tuningParamImplemented
   autoImpl = cmisTuningSupp.txInputAdaptiveEqualizationImpl

   xcvrConfigCliDir = getXcvrConfigCliDir( intfName )
   xcvrConfigCli = xcvrConfigCliDir.xcvrConfigCli.get( intfName )
   overrideEn = xcvrConfigCli and xcvrConfigCli.overrideTuningParams

   if( xcvrStatus.presence == XcvrPresence.xcvrPresent and
       not ( autoImpl or overrideEn ) ):
      # When a transceiver is inserted, require it to advertise 'auto' support, or
      # the overrideTuning cli to be enabled
      return CliParser.guardNotThisPlatform

   return None

def valueTxInputEqKeywordGuard( mode, token ):
   intfName = mode.intf.name
   xcvrName = gv.xcvrConfigDir.intfToXcvrName.get( intfName )
   if not xcvrName:
      return None

   xcvrStatus = getXcvrStatus( gv.xcvrStatusDir.xcvrStatus.get( xcvrName ) )
   if not xcvrStatus:
      return None

   cmisEqImpl = False
   if isCmisTypeStr( xcvrStatus.xcvrType ):
      cmisTuningSupp = xcvrStatus.cmisTuningParamCapabilities.tuningParamImplemented
      cmisEqImpl = cmisTuningSupp.txInputFixedEqualizationImpl

   xcvrConfigCliDir = getXcvrConfigCliDir( intfName )
   xcvrConfigCli = xcvrConfigCliDir.xcvrConfigCli.get( intfName )
   overrideEn = xcvrConfigCli and xcvrConfigCli.overrideTuningParams

   if( isCmisTypeStr( xcvrStatus.xcvrType ) and
       xcvrStatus.presence == XcvrPresence.xcvrPresent and
       not ( cmisEqImpl or overrideEn ) ):
      # When a CMIS transceiver is inserted, require it to advertise
      # txInputFixedEqualization support, or the overrideTuning cli to be enabled
      return CliParser.guardNotThisPlatform

   return None

def sfpKeywordGuard( mode, token ):
   intfName = mode.intf.name
   xcvrName = gv.xcvrConfigDir.intfToXcvrName.get( intfName )
   if not xcvrName:
      return None
   xcvrStatus = getXcvrStatus( gv.xcvrStatusDir.xcvrStatus.get( xcvrName ) )
   if not xcvrStatus:
      return None

   if xcvrStatus.xcvrType != 'sfpPlus':
      # Guard sfp token for non-sfp ports
      return CliParser.guardNotThisPlatform

   return None

def cmisTuningParamGuardHelper( token, xcvrStatus, overrideEn ):
   if token == 'tx-input-equalization':
      paramSupportedStr = 'txInputFixedEqualizationImpl'
      paramSupportedStrAuto = 'txInputAdaptiveEqualizationImpl'
   elif token == 'rx-output-amplitude':
      paramSupportedStr = 'rxOutputAmplitudeImpl'
   elif token == 'rx-output-pre-emphasis':
      paramSupportedStr = 'rxOutputPreEmphasisImpl'
   elif token == 'rx-output-post-emphasis':
      paramSupportedStr = 'rxOutputPostEmphasisImpl'
   else:
      # Guard unsupported tuning param
      return CliParser.guardNotThisPlatform

   tuningParamImpl = getattr(
      xcvrStatus.cmisTuningParamCapabilities.tuningParamImplemented,
      paramSupportedStr )
   if token == 'tx-input-equalization' and not tuningParamImpl:
      # For tx-input-equalization, we check the fixed equalization first. If it is
      # not supported, we need to check the auto equalization. If both are not
      # supported, token 'tx-input-equalization' will be guarded.
      tuningParamImpl = getattr(
      xcvrStatus.cmisTuningParamCapabilities.tuningParamImplemented,
      paramSupportedStrAuto )

   if xcvrStatus.presence == XcvrPresence.xcvrPresent and (
         not xcvrStatus.writableEeprom() or not ( tuningParamImpl or overrideEn ) ):
      # Guard this command when the osfp/qsfpDd xcvr is present but it isn't optical
      # or it doesn't support tuning the param without overriding to enable tuning
      return CliParser.guardNotThisPlatform

   return None

def nonCmisTuningParamGuardHelper( token, xcvrStatus, overrideEn ):
   qsfp28Id = Tac.enumValue( "Xcvr::Sff8024::IdentifierCode", "idenQsfp28" )

   if token == 'tx-input-equalization':
      paramSupportedStr = 'txInputEqualizationSupported'
   elif token == 'rx-output-amplitude':
      paramSupportedStr = 'rxAmplitudeSupported'
   elif token == 'rx-output-pre-emphasis':
      paramSupportedStr = 'rxPreEmphasisSupported'
   else:
      # Guard unsupported tuning param
      return CliParser.guardNotThisPlatform

   tuningParamSupp = getattr( xcvrStatus.xcvrTuningParamCapabilities,
                              paramSupportedStr )
   if xcvrStatus.xcvrType != 'qsfpPlus':
      # Guard this command for non-qsfp ports
      return CliParser.guardNotThisPlatform
   elif xcvrStatus.presence == XcvrPresence.xcvrPresent and \
        ( xcvrStatus.sff8436IdEepromContents.identifier != qsfp28Id or
          not xcvrStatus.optical or not ( tuningParamSupp or overrideEn ) ):
      # Guard this command when the qsfp xcvr is present but not qsfp28 or it
      # doesn't support tuning the param without overriding to enable tuning
      return CliParser.guardNotThisPlatform

   return None

def tuningParamGuard( mode, token ):
   # Accept only tunable parameters
   intfName = mode.intf.name
   xcvrName = gv.xcvrConfigDir.intfToXcvrName.get( intfName )
   if not xcvrName:
      return None

   xcvrStatus = getXcvrStatus( gv.xcvrStatusDir.xcvrStatus.get( xcvrName ) )
   if not xcvrStatus:
      return None

   xcvrConfigCliDir = getXcvrConfigCliDir( intfName )
   xcvrConfigCli = xcvrConfigCliDir.xcvrConfigCli.get( intfName )
   overrideEn = xcvrConfigCli and xcvrConfigCli.overrideTuningParams

   if isCmisTypeStr( xcvrStatus.xcvrType ):
      return cmisTuningParamGuardHelper( token, xcvrStatus, overrideEn )
   else:
      # qsfp check happens inside this function
      return nonCmisTuningParamGuardHelper( token, xcvrStatus, overrideEn )

def setTuningParam( mode, args ):
   intfName = mode.intf.name
   xcvrConfigCliDir = getXcvrConfigCliDir( intfName )
   xcvrConfigCli = getXcvrConfigCliForConfigCommand( intfName, xcvrConfigCliDir,
                                                     True )

   for laneId in args[ 'LANE_LIST' ]:
      # laneId starts from 1
      if xcvrConfigCli.slotTuningParams.get( laneId - 1 ):
         # copy the last cli tuning config
         laneTuningParam = copyCliLaneTuningParam(
            xcvrConfigCli.slotTuningParams.get( laneId - 1 ) )
      else:
         laneTuningParam = Tac.Value( "Xcvr::LaneTuningParameter" )

      for tuningParam, tuningValue in args[ 'tuningParams' ]:
         if tuningValue == 'module-default':
            laneTuningValue = Tac.Value( "Xcvr::TuningValue", False, 0, True )
         elif tuningValue == 'auto':
            laneTuningValue = Tac.Value( "Xcvr::TuningValue", True, 1, False )
         else:
            laneTuningValue = Tac.Value( "Xcvr::TuningValue", True, tuningValue,
                                         False )

         if tuningParam == 'txInputEqualization':
            if tuningValue == 'auto':
               laneTuningParam.txInputEqualization = \
                  Tac.Value( "Xcvr::TuningValue", False, 0, False )
               laneTuningParam.txInputAdaptiveEqEnable = laneTuningValue
            elif tuningValue == 'module-default':
               laneTuningParam.txInputEqualization = laneTuningValue
               laneTuningParam.txInputAdaptiveEqEnable = \
                  Tac.Value( "Xcvr::TuningValue", False, 0, False )
            else:
               laneTuningParam.txInputEqualization = laneTuningValue
               # txInputEqualization and txInputAdaptiveEqEnable are mutually
               # exclusive. To make txInputEqualization configured, we need
               # to set txInputAdaptiveEqEnable to 0 and make it valid so as to
               # disable txInputAdaptiveEq
               laneTuningParam.txInputAdaptiveEqEnable = \
                  Tac.Value( "Xcvr::TuningValue", True, 0, False )
         elif tuningParam == 'rxOutputAmplitude':
            laneTuningParam.rxOutputAmplitude = laneTuningValue
         elif tuningParam == 'rxOutputPreEmphasis':
            laneTuningParam.rxOutputPreEmphasis = laneTuningValue
         elif tuningParam == 'rxOutputPostEmphasis':
            laneTuningParam.rxOutputPostEmphasis = laneTuningValue

      xcvrConfigCli.slotTuningParams[ laneId - 1 ] = laneTuningParam

def noTuningParam( mode, args ):
   intfName = mode.intf.name
   xcvrConfigCliDir = getXcvrConfigCliDir( intfName )
   xcvrConfigCli = getXcvrConfigCliForConfigCommand( intfName, xcvrConfigCliDir,
                                                     False )
   if xcvrConfigCli is None:
      # no xcvrConfigCli on this interface
      return

   if 'LANE_LIST' not in args:
      for laneId in xcvrConfigCli.slotTuningParams.keys():
         # clear all the tuning params for all the lanes
         del xcvrConfigCli.slotTuningParams[ laneId ]
   else:
      tuningParams = args[ 'tuningParams' ]
      for laneId in args[ 'LANE_LIST' ]:
         # laneId starts from 1
         if not xcvrConfigCli.slotTuningParams.get( laneId - 1 ):
            # no tuning param on this lane, skip unnecessary default command
            continue

         if not tuningParams:
            # clear all tuning params on this lane
            del xcvrConfigCli.slotTuningParams[ laneId - 1 ]
            continue

         laneTuningParam = copyCliLaneTuningParam(
            xcvrConfigCli.slotTuningParams.get( laneId - 1 ) )
         for tuningParam, _ in tuningParams:
            laneTuningValue = Tac.Value( "Xcvr::TuningValue", False, 0, False )
            if tuningParam == 'txInputEqualization':
               laneTuningParam.txInputEqualization = laneTuningValue
               laneTuningParam.txInputAdaptiveEqEnable = laneTuningValue
            elif tuningParam == 'rxOutputAmplitude':
               laneTuningParam.rxOutputAmplitude = laneTuningValue
            elif tuningParam == 'rxOutputPreEmphasis':
               laneTuningParam.rxOutputPreEmphasis = laneTuningValue
            elif tuningParam == 'rxOutputPostEmphasis':
               laneTuningParam.rxOutputPostEmphasis = laneTuningValue

         xcvrConfigCli.slotTuningParams[ laneId - 1 ] = laneTuningParam

   if xcvrConfigCliDelete( xcvrConfigCliDir.xcvrConfigCli[ intfName ] ):
      del xcvrConfigCliDir.xcvrConfigCli[ intfName ]

matcherModuleDefault = CliMatcher.KeywordMatcher( 'module-default',
      helpdesc='Configure tuning parameter to module\'s default' )

class TxInputEqualizationExpr( CliCommand.CliExpression ):
   expression = 'TX_MODULE_DEFAULT | auto | TX_INPUT_EQUALIZATION'
   data = {
      'TX_MODULE_DEFAULT': CliCommand.Node(
         matcher=matcherModuleDefault,
         alias='TX_INPUT_EQUALIZATION' ),
      'auto': CliCommand.guardedKeyword( 'auto',
         helpdesc='enable adaptive tx input equalization',
         guard=autoTxInputEqKeywordGuard,
         alias='TX_INPUT_EQUALIZATION' ),
      'TX_INPUT_EQUALIZATION': CliCommand.Node(
         matcher=CliMatcher.IntegerMatcher( 0, 12, helpdesc='Tuning value' ),
         guard=valueTxInputEqKeywordGuard ),
   }

class RxOutputAmplitudeExpr( CliCommand.CliExpression ):
   expression = 'AMPLITUDE_MODULE_DEFAULT | RX_OUTPUT_AMPLITUDE'
   data = {
      'AMPLITUDE_MODULE_DEFAULT': CliCommand.Node(
         matcher=matcherModuleDefault,
         alias='RX_OUTPUT_AMPLITUDE' ),
      'RX_OUTPUT_AMPLITUDE': CliMatcher.IntegerMatcher( 0, 3,
         helpdesc='Tuning value' ),
   }

class RxOutputPreEmphasisExpr( CliCommand.CliExpression ):
   expression = 'EMPHASIS_MODULE_DEFAULT | RX_OUTPUT_PREEMPHASIS'
   data = {
      'EMPHASIS_MODULE_DEFAULT': CliCommand.Node( matcher=matcherModuleDefault,
         alias='RX_OUTPUT_PREEMPHASIS' ),
      'RX_OUTPUT_PREEMPHASIS': CliMatcher.IntegerMatcher( 0, 7,
         helpdesc='Tuning value' ),
   }

class TuningParamCmd( CliCommand.CliCommandClass ):
   syntax = ( 'transceiver electrical lane LANE_LIST '
                     '[ { ( tx-input-equalization TX_INPUT_EQUALIZATION ) | '
                         '( rx-output-amplitude RX_OUTPUT_AMPLITUDE ) | '
                         '( rx-output-pre-emphasis RX_OUTPUT_PREEMPHASIS ) | '
                         '( rx-output-post-emphasis RX_OUTPUT_POSTEMPHASIS ) } ]' )
   noOrDefaultSyntax = ( 'transceiver electrical [ lane LANE_LIST '
               '[ { ( tx-input-equalization [ TX_INPUT_EQUALIZATION ] ) | '
                   '( rx-output-amplitude [ RX_OUTPUT_AMPLITUDE ] ) | '
                   '( rx-output-pre-emphasis [ RX_OUTPUT_PREEMPHASIS ] ) | '
                   '( rx-output-post-emphasis [ RX_OUTPUT_POSTEMPHASIS ] ) } ] ]' )
   data = {
      'transceiver': nodeTransceiver,
      'electrical': matcherElectrical,
      'lane': 'Transceiver lane id(s)',
      'LANE_LIST': MultiRangeRule.MultiRangeMatcher(
         rangeFn=lambda: ( 1, 8 ),
         noSingletons=False,
         value=lambda mode, genericRangeList: list( genericRangeList.values() ),
         helpdesc='lane number(s)' ),
      'tx-input-equalization': CliCommand.guardedKeyword( 'tx-input-equalization',
         helpdesc='configure tx input equalization tuning',
         guard=tuningParamGuard, maxMatches=1 ),
      'TX_INPUT_EQUALIZATION': TxInputEqualizationExpr,
      'rx-output-amplitude': CliCommand.guardedKeyword( 'rx-output-amplitude',
         helpdesc='configure rx output amplitude tuning',
         guard=tuningParamGuard, maxMatches=1 ),
      'RX_OUTPUT_AMPLITUDE': RxOutputAmplitudeExpr,
      'rx-output-pre-emphasis': CliCommand.guardedKeyword( 'rx-output-pre-emphasis',
         helpdesc='configure rx output pre-emphasis tuning',
         guard=tuningParamGuard, maxMatches=1 ),
      'RX_OUTPUT_PREEMPHASIS': RxOutputPreEmphasisExpr,
      'rx-output-post-emphasis': CliCommand.guardedKeyword(
         'rx-output-post-emphasis',
         helpdesc='configure rx output post-emphasis tuning',
         guard=tuningParamGuard, maxMatches=1 ),
      'RX_OUTPUT_POSTEMPHASIS': CliMatcher.IntegerMatcher( 0, 7,
         helpdesc='Tuning value' ),
   }
   # Parameter keyword's guard functions are dynamic (depends on the state of
   # the inserted transceiver) so we need to disable cache
   allowCache = False

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'tuningParams' not in args:
         args[ 'tuningParams' ] = []

      if 'tx-input-equalization' in args:
         args[ 'tuningParams' ].append( ( 'txInputEqualization',
            args.get( 'TX_INPUT_EQUALIZATION', [ None ] )[ 0 ] ) )
         del args[ 'tx-input-equalization' ]
      if 'rx-output-amplitude' in args:
         args[ 'tuningParams' ].append( ( 'rxOutputAmplitude',
            args.get( 'RX_OUTPUT_AMPLITUDE', [ None ] )[ 0 ] ) )
         del args[ 'rx-output-amplitude' ]
      if 'rx-output-pre-emphasis' in args:
         args[ 'tuningParams' ].append( ( 'rxOutputPreEmphasis',
            args.get( 'RX_OUTPUT_PREEMPHASIS', [ None ] )[ 0 ] ) )
         del args[ 'rx-output-pre-emphasis' ]
      if 'rx-output-post-emphasis' in args:
         args[ 'tuningParams' ].append( ( 'rxOutputPostEmphasis',
            args.get( 'RX_OUTPUT_POSTEMPHASIS', [ None ] )[ 0 ] ) )
         del args[ 'rx-output-post-emphasis' ]

   handler = setTuningParam
   noOrDefaultHandler = noTuningParam

XcvrConfigModelet.addCommandClass( TuningParamCmd )

# --------------------------------------------------------------------------------
# [ no | default ] transceiver electrical tuning override
# --------------------------------------------------------------------------------
class TransceiverElectricalTuningOverrideCmd( CliCommand.CliCommandClass ):
   syntax = 'transceiver electrical tuning override'
   noOrDefaultSyntax = syntax
   data = {
      'transceiver': nodeTransceiver,
      'electrical': matcherElectrical,
      'tuning': 'Configure transceiver tuning capabilities',
      'override': ( 'Ignore transceiver EEPROM tuning capabilities and treat all as '
                    'tunable' ),
   }

   @staticmethod
   def handler( mode, args ):
      noOrDefault = CliCommand.isNoOrDefaultCmd( args )
      intfName = mode.intf.name
      xcvrConfigCliDir = getXcvrConfigCliDir( intfName )
      # Pass in 'not noOrDefault' for 'create', meaning we will create a
      # xcvrConfigCli object only when the command is NOT the no/default variant.
      xcvrConfigCli = getXcvrConfigCliForConfigCommand( intfName, xcvrConfigCliDir,
                                                        not noOrDefault )
      if xcvrConfigCli is None:
         assert noOrDefault
         return

      assert xcvrConfigCli
      if noOrDefault:
         xcvrConfigCli.overrideTuningParams = False
         if xcvrConfigCliDelete( xcvrConfigCli ):
            del xcvrConfigCliDir.xcvrConfigCli[ intfName ]
      else:
         xcvrConfigCli.overrideTuningParams = True

   noOrDefaultHandler = handler

XcvrConfigModelet.addCommandClass( TransceiverElectricalTuningOverrideCmd )

# --------------------------------------------------------------------------------
# [ no | default ] transceiver electrical tuning file URL
# --------------------------------------------------------------------------------
class TransceiverElectricalTuningFileCmd( CliCommand.CliCommandClass ):
   syntax = 'transceiver electrical tuning file URL'
   noOrDefaultSyntax = 'transceiver electrical tuning file ...'
   data = {
      'transceiver': xcvrKw,
      'electrical': matcherElectrical,
      'tuning': 'Configure transceiver tuning capabilities',
      'file': 'Apply transceiver tuning values from file',
      'URL': Url.UrlMatcher( lambda fs: fs.scheme in [ 'flash:' ] and
                                fs.supportsRead(),
                                'File location', acceptSimpleFile=False ),
   }

   @staticmethod
   def handler( mode, args ):
      fileUrl = args[ 'URL' ]
      if os.path.isfile( fileUrl.localFilename() ):
         gv.xgc.overrideTuningExceptionSrc = fileUrl.localFilename()
         gv.xgc.overrideTuningExceptionGenerationId += 1
      else:
         mode.addError( f"File {fileUrl} does not exist." )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      gv.xgc.overrideTuningExceptionSrc = ""
      gv.xgc.overrideTuningExceptionGenerationId += 1

if Toggles.XcvrToggleLib.toggleDefaultTuningExceptionEnabled():
   BasicCli.GlobalConfigMode.addCommandClass( TransceiverElectricalTuningFileCmd )

# --------------------------------------------------------------------------------
# [ no | default ] transceiver application override
#                     ( ( ID [lanes start START [end END]] ) | 100gbase-srbd )
# --------------------------------------------------------------------------------
matcherApplication = CliMatcher.KeywordMatcher(
   'application', helpdesc='Configure CMIS transceiver applications' )
nodeApplication = CliCommand.Node( matcher=matcherApplication, guard=xcvrCmisGuard )
matcherOverride = CliMatcher.KeywordMatcher(
   'override', 'Override the active CMIS application' )
srbdLegacyKeyword = '100gbase-srbd'
matcherLegacySrbd = CliMatcher.KeywordMatcher(
   srbdLegacyKeyword, 'Configure legacy mode for 100GBASE-SRBD interoperability' )

def _handleGenericApplicationOverrideCmd( xcvrConfigCli, args ):
   # Since start and end lanes are optional arguments let's initialize them in
   # the beginning to cover all lanes.
   # Note that the START and END arguments for startLane and endLane start
   # counting at 1. We will encode a missing 'start' token as startLane 0.
   startLane = 0
   # We assume our application may cover as much as 8 lanes, unless we specify
   # an end lane. A numLanes of 0 is equivalent to the command missing the "end"
   # tokens
   numLanes = 0
   apSel = args[ 'ID' ]
   if 'START' in args:
      startLane = args[ 'START' ]
      if 'END' in args:
         endLane = args[ 'END' ]
         # If an endLane is specified, then we need to update our application's
         # laneCount
         numLanes = endLane - startLane + 1

   overrideApplication = Tac.Value( "Xcvr::CmisOverrideApplication" )

   overrideApplication.apSel = apSel
   overrideApplication.laneCount = numLanes

   xcvrConfigCli.cmisOverrideApplication[ startLane ] = overrideApplication

class TransceiverApplicationOverrideCmd( CliCommand.CliCommandClass ):
   syntax = ( 'transceiver GUARD_APPLICATION override ( ( ID [ lanes start '
              'START [ end END ] ] ) | 100gbase-srbd )' )
   noOrDefaultSyntax = 'transceiver UNGUARD_APPLICATION override [ ID ] ...'

   # According to the CMIS spec there can be up to 15 advertised applications
   _cmisMaxAppNumber = 15
   data = {
      'transceiver': xcvrKw,
      'GUARD_APPLICATION': nodeApplication,
      'UNGUARD_APPLICATION': matcherApplication,
      'override': matcherOverride,
      'ID': CliMatcher.IntegerMatcher( 0, _cmisMaxAppNumber,
                                        helpdesc='application number' ),
      'lanes': 'Specify which host lanes the override should by applied to',
      'start': 'Application start host lane',
      'START': matcherHostLaneNumber,
      'end': 'Application end host lane',
      'END': matcherHostLaneNumber,
      srbdLegacyKeyword: matcherLegacySrbd,
   }

   @staticmethod
   def handler( mode, args ):
      intfName = mode.intf.name
      xcvrConfigCliDir = getXcvrConfigCliDir( intfName )
      xcvrConfigCli = getXcvrConfigCliForConfigCommand( intfName, xcvrConfigCliDir,
                                                        True )
      if srbdLegacyKeyword in args:
         xcvrConfigCli.legacySRBDCompatibilityConfig = True
         xcvrConfigCli.cmisOverrideApplication.clear()
      else:
         _handleGenericApplicationOverrideCmd( xcvrConfigCli, args )
         xcvrConfigCli.legacySRBDCompatibilityConfig = False

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfName = mode.intf.name
      xcvrConfigCliDir = getXcvrConfigCliDir( intfName )
      xcvrConfigCli = getXcvrConfigCliForConfigCommand( intfName, xcvrConfigCliDir,
                                                        False )
      if xcvrConfigCli is None:
         # xcvrConfigCli doesn't exist => we are done
         return

      apSel = args.get( 'ID' )
      if apSel:
         # Remove only the application overrides matching this ApSel
         for lane in list( xcvrConfigCli.cmisOverrideApplication ):
            if xcvrConfigCli.cmisOverrideApplication[ lane ].apSel == apSel:
               del xcvrConfigCli.cmisOverrideApplication[ lane ]
      else:
         # Remove all application overrides
         xcvrConfigCli.legacySRBDCompatibilityConfig = False
         xcvrConfigCli.cmisOverrideApplication.clear()

      if xcvrConfigCliDelete( xcvrConfigCli ):
         del xcvrConfigCliDir.xcvrConfigCli[ intfName ]

XcvrConfigModelet.addCommandClass( TransceiverApplicationOverrideCmd )

####
# To prevent sfp enable high power:
# transceiver sfp power low
# Default mode - enable high power:
# no|default transceiver sfp power [low]
####
sfpMatcher = CliMatcher.KeywordMatcher(
             'sfp', helpdesc='SFP transceiver' )

class configSfpPowerCmd( CliCommand.CliCommandClass ):
   syntax = "transceiver sfp power low"
   noOrDefaultSyntax = "transceiver sfp power [low]"

   data = {
         'transceiver': xcvrKw,
         'sfp': CliCommand.Node( matcher=sfpMatcher, guard=sfpKeywordGuard ),
         'power': 'Configure transceiver power settings',
         'low': 'Do not permit module to draw more than 1 Watt'
         }

   @staticmethod
   def handler( mode, args ):
      intfName = mode.intf.name
      xcvrConfigCliDir = getXcvrConfigCliDir( intfName )
      xcvrConfigCli = getXcvrConfigCliForConfigCommand( intfName, xcvrConfigCliDir,
                                                        True )
      xcvrConfigCli.preventSfpHighPowerEnable = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfName = mode.intf.name
      xcvrConfigCliDir = getXcvrConfigCliDir( intfName )
      xcvrConfigCli = getXcvrConfigCliForConfigCommand( intfName, xcvrConfigCliDir,
                                                        False )

      if xcvrConfigCli is None:
         # no xcvrConfigCli on this interface
         return

      xcvrConfigCli.preventSfpHighPowerEnable = False
      if xcvrConfigCliDelete( xcvrConfigCli ):
         del xcvrConfigCliDir.xcvrConfigCli[ intfName ]

if sfpHighPowerEnable:
   XcvrConfigModelet.addCommandClass( configSfpPowerCmd )

# Layer1Monitor mode
class Layer1MonitorMode( Layer1MonitorBaseMode, BasicCli.ConfigModeBase ):
   name = "Layer 1 monitor configuration"

   def __init__( self, parent, session ):
      Layer1MonitorBaseMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

# --------------------------------------------------------------------------------
# monitor layer1
# --------------------------------------------------------------------------------
class MonitorLayer1Cmd( CliCommand.CliCommandClass ):
   syntax = 'monitor layer1'
   noOrDefaultSyntax = syntax
   data = {
      'monitor': CliToken.Monitor.monitorMatcher,
      'layer1': 'Layer 1 monitor configuration',
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( Layer1MonitorMode )
      mode.session_.gotoChildMode( childMode )

   noOrDefaultHandler = Layer1MonitorMode.clear

BasicCli.GlobalConfigMode.addCommandClass( MonitorLayer1Cmd )

# --------------------------------------------------------------------------------
# [ no | default ] logging transceiver
# --------------------------------------------------------------------------------

matcherLogging = CliMatcher.KeywordMatcher( "logging",
                                            helpdesc="Configure logging" )
matcherLoggingTransceiver = CliMatcher.KeywordMatcher(
      "transceiver", helpdesc="Configure transceiver monitoring logging" )

class LoggingTransceiverCmd( CliCommand.CliCommandClass ):
   syntax = 'logging transceiver'
   noOrDefaultSyntax = syntax
   data = {
      'logging': matcherLogging,
      'transceiver': matcherLoggingTransceiver,
   }

   @staticmethod
   def handler( mode, args ):
      gv.xgc.domThresholdLogging = True
      if smbusFailureLogEnable:
         gv.xgc.smbusFailureLogging = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      gv.xgc.domThresholdLogging = False
      if smbusFailureLogEnable:
         gv.xgc.smbusFailureLogging = False

Layer1MonitorMode.addCommandClass( LoggingTransceiverCmd )

# ------------------------------------------------------
# Plugin method
# ------------------------------------------------------

def Plugin( em ):

   gv.xgc = ConfigMount.mount( em, "hardware/xcvr/xgc", "Xcvr::Xgc", "w" )
   gv.xcvrCliConfigSliceDir = ConfigMount.mount( em,
                                                 "hardware/xcvr/cli/config/slice",
                                                 "Tac::Dir", "wi" )
   gv.xcvrStatusDir = XcvrAllStatusDir.xcvrAllStatusDir( em )
   gv.xcvrConfigDir = LazyMount.mount( em, "hardware/xcvr/config/all",
                                    "Xcvr::AllConfigDir", "r" )
   IntfCli.Intf.registerDependentClass( XcvrTuningIntfCleaner )
   IntfCli.Intf.registerDependentClass( XcvrTxDisabledIntfCleaner )
   IntfCli.Intf.registerDependentClass( XcvrTributaryIntfCleaner )
   IntfCli.Intf.registerDependentClass( XcvrConfigCliIntfCleaner )
   IntfCli.Intf.registerDependentClass( PerformanceMonitoringIntfCleaner )
