# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
#
# This file contains the CLI command definition for the
# '[no|default] transceiver diag management' command.

import CliCommand
import CliMatcher
import CliParser
from CliPlugin.XcvrConfigCli import XcvrConfigModelet
from CliPlugin.XcvrConfigCli import nodeTransceiver
from CliPlugin.XcvrConfigCli import diagKeyword
import LazyMount
import Tac
import CliGlobal

gv = CliGlobal.CliGlobal( xcvrConfigDir=None )

# --------------------------------------------------------------------------------
# [ no | default ] transceiver diag management
# --------------------------------------------------------------------------------
managementKeyword = CliMatcher.KeywordMatcher( 'management',
      helpdesc='Configure transceiver software management mode' )

def capabilityGuard( mode, token ):
   """
   Guard function for the 'transceiver diag management' command.

   The command is guarded by capability information which comes from a
   platform's FDL.  This means only ports with this special capability can
   run this command.
   """
   xcvrName = gv.xcvrConfigDir.intfToXcvrName.get( mode.intf.name )
   if not xcvrName:
      return CliParser.guardNotThisPlatform

   xcvrConfig = gv.xcvrConfigDir.xcvrConfig.get( xcvrName )
   capable = ( isinstance( xcvrConfig, Tac.Type( "Xcvr::QsfpConfig" ) ) and
               xcvrConfig.qsfpSlotCapabilities.whisperPortCapable )
   if capable:
      return None
   else:
      return CliParser.guardNotThisPlatform

class TransceiverDiagManagementCmd( CliCommand.CliCommandClass ):
   syntax = 'transceiver diag management'
   noOrDefaultSyntax = syntax
   data = {
      'transceiver': nodeTransceiver,
      'diag': diagKeyword,
      'management': CliCommand.Node( matcher=managementKeyword,
                                      guard=capabilityGuard ),
   }

   handler = "XcvrManagementConfigCliHandler.TransceiverDiagManagementCmdHandler"
   # [ no | default ] uses the same handler function
   noOrDefaultHandler = handler

XcvrConfigModelet.addCommandClass( TransceiverDiagManagementCmd )

# -----------------------------------------------------------------------------------
# Plugin
# -----------------------------------------------------------------------------------
xcvrConfigDir = None

def Plugin( em ):
   gv.xcvrConfigDir = LazyMount.mount( em, "hardware/xcvr/config/all",
                                           "Xcvr::AllConfigDir", "r" )
