#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliGlobal
import CliParser
import LazyMount
import Toggles.XcvrToggleLib

from CliPlugin import EthIntfCli

gv = CliGlobal.CliGlobal( xgc=None, inventoryDir=None, entMibStatus=None )

# --------------------------------------------------------------------------------
# [ no | default ] transceiver power-cycle on-boot
# --------------------------------------------------------------------------------

def powerCycleOnBootCmdGuard( mode, token ):
   model = gv.entMibStatus.root.modelName
   if "modularSystem" in gv.inventoryDir and model.startswith( "DCS-78" ):
      return None
   return CliParser.guardNotThisPlatform

powerCycleOnBootKw = CliCommand.guardedKeyword(
   "power-cycle",
   helpdesc="Perform a power cycle operation",
   guard=powerCycleOnBootCmdGuard
)

class TransceiverPowerCycleOnBoot( CliCommand.CliCommandClass ):
   syntax = "transceiver power-cycle on-boot"
   noOrDefaultSyntax = syntax

   data = {
      "transceiver": EthIntfCli.xcvrKw,
      "power-cycle": powerCycleOnBootKw,
      "on-boot": "Enable linecard power cycling for faulty transceivers"
   }

   @staticmethod
   def handler( mode, args ):
      gv.xgc.recoveryEnabled = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      gv.xgc.recoveryEnabled = False

if Toggles.XcvrToggleLib.toggleXcvrRecoveryEnabled():
   BasicCli.GlobalConfigMode.addCommandClass( TransceiverPowerCycleOnBoot )

def Plugin( entityManager ):

   gv.inventoryDir = LazyMount.mount( entityManager,
                                      "hardware/inventory",
                                      "Tac::Dir",
                                      "ri" )

   gv.entMibStatus = LazyMount.mount( entityManager,
                                      "hardware/entmib",
                                      "EntityMib::Status",
                                      "r" )

   gv.xgc = LazyMount.mount( entityManager,
                             "hardware/xcvr/xgc",
                             "Xcvr::Xgc",
                             "w" )
