#!/usr/bin/env python3
# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliGlobal
import CliParser
import LazyMount
import Tac

from CliPlugin import EthIntfCli, IntfStatusMatcher

gv = CliGlobal.CliGlobal( inventoryDir=None, hwEcbDirConfig=None,
                          hwIslConfig=None, entMibStatus=None )

slotMatcher = IntfStatusMatcher.DefaultIntfMatcher(
   "slot",
   # All three types of linecards support up to 16-slot chassis
   # Max port supported is 48 on Clearwater2
   # Note we only vaguely check the range of linecard number and port number
   # Detailed checks are done in the handler
   IntfStatusMatcher.DefaultIntfNumberMatcher(
      [ ( 3, 19 ), ( 1, 48 ) ],
      minParts=2, maxParts=2 ),
   helpdesc="Hardware transceiver slot"
)

def powerCycleEnabled( powerCycleMethod ) -> bool:
   powerCycleMethodEnum = Tac.Type( "Hardware::PowerController::PowerCycleMethod" )
   return powerCycleMethod != powerCycleMethodEnum.powerCycleNotSupported

def hasIslRiser( slot: int ) -> bool:
   # check if there is a riser card supports power cycle on this linecard slot
   return any( ( powerController.name.startswith( f"Linecard{slot}" ) and
                 powerCycleEnabled( powerController.powerCycleMethod ) )
               for powerController in gv.hwIslConfig.controller.values() )

def hasEcbRiser( slot: int ) -> bool:
   # check if there is a riser ecb supports power cycle on this linecard slot
   lcName = "Linecard" + str( slot )
   if lcName not in gv.hwEcbDirConfig.card.keys():
      return False

   return any( ( "Ports" in ecb.name and powerCycleEnabled( ecb.powerCycleMethod ) )
               for ecb in gv.hwEcbDirConfig.card[ lcName ].ecb.values() )

def hasRiser( modelName: str, slot: int ) -> bool:
   # checker function for modelName of the linecard
   # specify true if:
   # 1) the linecard is within DenaliModular family ( 7800.* )
   # 2) there are risers on the linecard
   return ( modelName.startswith( "7800" ) and
            ( hasIslRiser( slot ) or hasEcbRiser( slot ) ) )

def powerCycleCmdGuard( mode, token ):
   modularSystemInv = gv.inventoryDir.entity.get( "modularSystem" )
   if modularSystemInv is None:
      return CliParser.guardNotThisPlatform

   # the riser power cycle command is only available if there is any DenaliModular
   # linecards with risers inserted on the switch
   if ( any( hasRiser( card.modelName, card.slot )
             for card in modularSystemInv.card.values() ) ):
      return None

   return CliParser.guardNotThisPlatform

powerCycleKw = CliCommand.guardedKeyword(
   "power-cycle",
   helpdesc="Perform a power cycle operation",
   guard=powerCycleCmdGuard
)

class TransceiverPowerCycle( CliCommand.CliCommandClass ):
   syntax = "transceiver power-cycle SLOT"

   data = {
      "transceiver": EthIntfCli.xcvrKw,
      "power-cycle": powerCycleKw,
      "SLOT": slotMatcher
   }
   handler = "XcvrRiserPowerCycleCliHandler.xcvrPowerCycleCliHandler"

BasicCli.EnableMode.addCommandClass( TransceiverPowerCycle )

def Plugin( entityManager ):

   gv.inventoryDir = LazyMount.mount( entityManager,
                                      "hardware/inventory",
                                      "Tac::Dir",
                                      "ri" )

   gv.hwEcbDirConfig = LazyMount.mount( entityManager,
                                        "hardware/ecb/config",
                                        "Hardware::Ecb::EcbDirConfig",
                                        "r" )

   gv.hwIslConfig = LazyMount.mount( entityManager,
                                     "hardware/powercontroller/config/isl6812x",
                                     "Hardware::Isl6812X::Config",
                                     "r" )

   gv.entMibStatus = LazyMount.mount( entityManager,
                                      "hardware/entmib",
                                      "EntityMib::Status",
                                      "r" )
