# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliPlugin.XcvrShowApplicationModel import (
                     TransceiverApplications,
                     TransceiverApplicationHostLanes,
                     TransceiverApplicationInterfaces,
                     TransceiverApplicationStatus )
import CliPlugin.XcvrAllStatusDir
from CliPlugin.XcvrCliLib import getAllIntfsWrapper
from XcvrLib import ( getXcvrSlotName,
                      getXcvrStatus,
                      isCmisTransceiver )
import CliGlobal
from TypeFuture import TacLazyType

XcvrMediaType = TacLazyType( 'Xcvr::MediaType' )
Cmis400GSRBDApplications = TacLazyType( 'Xcvr::Cmis400GSRBDApplications' )
XcvrTypesApi = TacLazyType( 'XcvrTypes::XcvrTypesApi' )

gv = CliGlobal.CliGlobal( xcvrStatusDir=None )

# --------------------------------------------------------------------------------
#
# "show transceiver application [ interface <INTFS> ]"
#
# --------------------------------------------------------------------------------

def _detectUndefinedLegacySrbdMediaType( mediaType, apSelCode ):
   '''
   400GBASE-SR4.2 transceivers have a legacy 100GBASE-SRBD application (apSel 3)
   that do not have a defined media interface id (0x0 in eeprom). This function will
   return the correct media type string if both the media type matches and the
   configured application id matches this application.
   '''
   if ( mediaType == XcvrMediaType.xcvr400GBaseSr4_2 and
        apSelCode == Cmis400GSRBDApplications.legacySRBD ):
      return XcvrTypesApi.mediaTypeStringHelper( XcvrMediaType.xcvr100GBaseSrbd, 0 )
   return None

def _populateXcvrApplications( portName ):
   '''
   Helper function for populating the model associated with the given portName.
   It will use the information found in xcvrStatus.applicationStatus which is
   only populated for CMIS transceivers.

   Parameters
   ----------
   portName : str
      Represents the name of the port for which we want to generate a Cli model.
      E.g. EthernetX on fixed systems, EthernetX/Y on modular systems.

   Return
   ------
   TransceiverApplications object
   '''
   xcvrStatus = getXcvrStatus( gv.xcvrStatusDir.xcvrStatus.get( portName ) )
   # The information that this command is concerned with is only available on CMIS
   # transceivers.
   if not xcvrStatus or not isCmisTransceiver( xcvrStatus ):
      return None

   applicationStatus = xcvrStatus.applicationStatus

   if not applicationStatus:
      return None

   model = TransceiverApplications()
   moduleType = xcvrStatus.eepromContents.moduleType
   vendorType = xcvrStatus.eepromContents.vendorType

   advertisedApplications = xcvrStatus.eepromContents.application

   # Max number of host lanes
   numLanes = xcvrStatus.capabilities.maxChannels

   model.mixedAppsCombinationRejected = \
         applicationStatus.mixedAppsCombinationRejected

   for lane in range( numLanes ):
      laneApSel = applicationStatus.activeApplication.get( lane )
      if laneApSel is None:
         continue
      apSelCode = laneApSel.apSel

      # startLane and subsequent hostLanes are indexed starting at 1 within
      # this model
      startLane = laneApSel.startLane + 1
      statusCode = applicationStatus.configErrorCodes.get( lane )
      if apSelCode not in model.applications:
         # Create a new model for this application if we didn't see it already
         model.applications[ apSelCode ] = TransceiverApplicationStatus()
         model.applications[ apSelCode ].applicationCode = apSelCode

         hostInterface = mediaInterface = 'unknown'
         if apSelCode in advertisedApplications:
            osfpApp = advertisedApplications[ apSelCode ]
            hostInterface = osfpApp.hostElectricalEntry().appName
            if 'Unknown' in hostInterface:
               hostInterface = f'0x{osfpApp.hostElectricalInterface:X}'
            mediaInterface = osfpApp.moduleMediaEntry( moduleType,
                                                       vendorType ).appName
            if 'Unknown' in mediaInterface:
               mediaInterface = f'0x{osfpApp.moduleMediaInterface:X}'
            elif 'Undefined' in mediaInterface:
               mediaInterface = _detectUndefinedLegacySrbdMediaType(
                  xcvrStatus.mediaType, apSelCode ) or mediaInterface

         model.applications[ apSelCode ].hostInterface = hostInterface
         model.applications[ apSelCode ].mediaInterface = mediaInterface

      if startLane not in model.applications[ apSelCode ].hostLanes:
         # Create a new model for a new set of host lanes
         model.applications[ apSelCode ].hostLanes[ startLane ] = \
               TransceiverApplicationHostLanes()

      if statusCode not in model.applications[ apSelCode ].statusCodes:
         # Create a new model for a new set of config error codes
         model.applications[ apSelCode ].statusCodes[ statusCode ] = \
               TransceiverApplicationHostLanes()

      # model lanes are indexed starting at 1
      modelLane = lane + 1
      model.applications[ apSelCode ].hostLanes[ startLane ].append( modelLane )
      model.applications[ apSelCode ].statusCodes[ statusCode ].append( modelLane )

   return model

def _showXcvrApplicationInterfaces( mode, intf, mod ):
   '''
   Helper function used to build the full Cli model for this command.

   Parameters
   ----------
   mode : Cli mode
   intf : an Intf or a MultiRangeRule.IntfList
      This represents the targeted intf(s)
   mod : int
      Represents the module number of this interface. Output is restricted to
      physical interfaces found on this module if not None. Here module is
      referring to Linecard. See IntfCli.py for more details.
   '''
   model = TransceiverApplicationInterfaces()

   ( intfs, intfNames ) = getAllIntfsWrapper( mode, intf, mod )
   if not intfs:
      return model

   # This information is descriptive of the whole transceiver, so we will only
   # create one entry in our model per port.
   for intfName in intfNames:
      portName = getXcvrSlotName( intfName )
      if portName not in model.ports:
         xcvrAppModel = _populateXcvrApplications( portName )
         if not xcvrAppModel:
            continue
         model.ports[ portName ] = xcvrAppModel

   return model

def showXcvrApplicationInterfacesHandler( mode, args ):
   intf = args.get( 'INTFS' )
   mod = args.get( 'MOD' )
   return _showXcvrApplicationInterfaces( mode, intf, mod )

# ------------------------------------------------------
# Plugin method
# ------------------------------------------------------
def Plugin( em ):
   gv.xcvrStatusDir = CliPlugin.XcvrAllStatusDir.xcvrAllStatusDir( em )
