# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
from CliModel import Dict, Model, Str, Int, List, Bool
import Tac

# --------------------------------------------------------------------------------
#
# Models for "show transceiver application"
#
# --------------------------------------------------------------------------------

class TransceiverApplicationHostLanes( Model ):
   # list of lanes indexed starting at 1
   lanes = List( valueType=int, help='Host lanes (indexed starting at 1)' )

   def getFirstLane( self ):
      return self.lanes[ 0 ]

   def append( self, lane ):
      self.lanes.append( lane )

   def getRenderedAttr( self ):
      return ', '.join( str( lane ) for lane in self.lanes )

class TransceiverApplicationStatus( Model ):
   applicationCode = Int( help='Indicates the active application code' )
   hostInterface = Str( help=( 'Indicates the host interface of the active '
                               'application' ) )
   mediaInterface = Str( help=( 'Indicates the media interface of the active '
                                'application' ) )
   hostLanes = Dict( keyType=int, valueType=TransceiverApplicationHostLanes,
                     help=( 'Mapping between application start lane and all '
                            'the lanes under the given application' ) )
   statusCodes = Dict(
         keyType=str, valueType=TransceiverApplicationHostLanes,
         help=( 'Mapping between configuration error code and list of lanes that '
                'report the given code' ) )

   def _decodeConfigErrorCode( self, configCode ):
      '''
      This helper will decode the given config error code according to
      Table 8-64 (CMIS 4.0) and Table 8-81 (CMIS 5.0).
      '''
      configCodeType = Tac.Type( 'Xcvr::CmisConfigurationErrorCode' )
      # Standard codes
      if configCode == configCodeType.configCodeNoStatus:
         return 'pending', None
      if configCode == configCodeType.configCodeAccepted:
         return 'accepted', None
      if configCode == configCodeType.configCodeRejectedUnknown:
         return 'rejected', 'unknown'
      if configCode == configCodeType.configCodeRejectedInvalid:
         return 'rejected', 'invalid code'
      if configCode == configCodeType.configCodeRejectedCombo:
         return 'rejected', 'invalid set of lanes'
      if configCode == configCodeType.configCodeRejectedSI:
         return 'rejected', 'invalid SI'
      if configCode == configCodeType.configCodeRejectedInUse:
         return 'rejected', 'in use'
      if configCode == configCodeType.configCodeRejectedIncomplete:
         return 'rejected', 'incomplete lane info'
      if configCode == configCodeType.configCodeRejectedFailed:
         return 'rejected', 'failed'
      if configCode == configCodeType.configCodeRejectedInProgress:
         return 'rejected', 'in progress'
      # Custom codes
      if configCode == configCodeType.configCodeRejectedCustom0xD:
         return 'rejected', '0xD'
      if configCode == configCodeType.configCodeRejectedCustom0xE:
         return 'rejected', '0xE'
      if configCode == configCodeType.configCodeRejectedCustom0xF:
         return 'rejected', '0xF'
      # Anything else appears as "reserved" in CMIS 4.0
      return None, None

   def getLowestLane( self ):
      # Get the lowest starting lane
      lane = min( self.hostLanes )
      if not self.applicationCode:
         # Application 0 doesn't really have a proper startLane, so we will need
         # to grab the first lane from our list
         lane = self.hostLanes[ lane ].getFirstLane()
      return lane

   def render( self ):
      if self.applicationCode:
         # pylint: disable-next=consider-using-f-string
         print( 'Application {}: host: {}, media: {}'.format( self.applicationCode,
                                                              self.hostInterface,
                                                              self.mediaInterface ) )
      else:
         print( 'Application 0: inactive' )

      for hostLanesList in sorted(
            self.hostLanes.values(), key=lambda lanes: lanes.getFirstLane() ):
         hostLanesStr = hostLanesList.getRenderedAttr()
         print( 'Host lanes: ' + hostLanesStr )

      for statusCode in sorted(
            self.statusCodes,
            key=lambda code: self.statusCodes[ code ].getFirstLane() ):
         hostLanesList = self.statusCodes[ statusCode ]
         status, errorReason = self._decodeConfigErrorCode( statusCode )
         hostLanesStr = hostLanesList.getRenderedAttr()
         if not status:
            # If statusCode was not a code that we know of, then we will skip
            continue
         if not errorReason:
            # If we didn't get an error reason, then the configuration was
            # successful.
            print( f'Config {status} on host lanes: {hostLanesStr}' )
         else:
            # pylint: disable-next=consider-using-f-string
            print( 'Config {} on host lanes: {} (reason: {})'.format(
               status, hostLanesStr, errorReason ) )

class TransceiverApplications( Model ):
   applications = Dict( keyType=int, valueType=TransceiverApplicationStatus,
                        help=( 'Mapping between application IDs and the'
                               ' information associated with each application' ) )
   mixedAppsCombinationRejected = Bool( help=( 'Combination of applications was'
                                               ' accepted' ) )

   def hasApplications( self ):
      """
      Model contains application statuses
      """
      # In certain cases (e.g. Blanco bypass module) we can have an empty model
      # with no programmed applications.
      return bool( self.applications )

   def render( self ):
      for application in sorted( self.applications.values(),
                                 key=lambda app: app.getLowestLane() ):
         application.render()
      if self.mixedAppsCombinationRejected:
         print( "Unsupported application combination(s) requested. A valid set"
                " of supported applications was applied instead." )

class TransceiverApplicationInterfaces( Model ):
   ports = Dict( keyType=str, valueType=TransceiverApplications,
                 help='Mapping between port name and transceiver applications' )

   def render( self ):
      if not self.ports:
         return

      for port in Arnet.sortIntf( self.ports ):
         if not self.ports[ port ].hasApplications():
            continue
         print( port )
         self.ports[ port ].render()
         print()
