# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
from CliModel import Bool
from CliModel import Dict
from CliModel import OrderedDict
from CliModel import Float
from CliModel import Model
from CliModel import Str
from CliModel import Int
from CliModel import List
from CliModel import Submodel
from collections import OrderedDict as ordDict
from collections import namedtuple
from XcvrLib import ( getXcvrStatus,
                      isCmisTransceiver,
                      isOsfpToQsfpSwizzler,
                      isQsfpDdToQsfpSwizzler,
                      isQsfpCmisToQsfpAdapter,
                      isQsfpToSfpSwizzler )
import Ark
import Tac
from TypeFuture import TacLazyType
from time import ctime
import codecs
import Toggles.XcvrToggleLib

CmisModuleState = TacLazyType( "Xcvr::CmisModuleState" )
DataPathState = TacLazyType( "Xcvr::DataPathState" )
LpoType = TacLazyType( 'Xcvr::LpoType' )
MediaType = TacLazyType( "Xcvr::MediaType" )
TributaryId = TacLazyType( 'Xcvr::TributaryId' )
XcvrPresence = TacLazyType( "Xcvr::XcvrPresence" )

lpoTypeToStr = {
   LpoType.none: 'none',
   LpoType.lowBw: 'low',
   LpoType.highBw: 'high'
}

xcvrPresenceToStr = {
   XcvrPresence.xcvrPresenceUnknown: 'unknown',
   XcvrPresence.xcvrNotPresent: 'not present',
   XcvrPresence.xcvrPresent: 'present'
}

moduleStateToStr = {
   CmisModuleState.moduleUnknown: 'unknown',
   CmisModuleState.moduleLowPwr: 'low power',
   CmisModuleState.modulePwrUp: 'power up',
   CmisModuleState.moduleReady: 'ready',
   CmisModuleState.modulePwrDn: 'power down',
   CmisModuleState.moduleFault: 'fault'
}

dataPathStateToStr = {
   DataPathState.dataPathUnknown: 'unknown',
   DataPathState.dataPathDeactivated: 'deactivated',
   DataPathState.dataPathInit: 'init',
   DataPathState.dataPathDeinit: 'deinit',
   DataPathState.dataPathActivated: 'activated',
   DataPathState.dataPathTxTurnOn: 'TX turn on',
   DataPathState.dataPathTxTurnOff: 'TX turn off',
   DataPathState.dataPathInitialized: 'initialized'
}

# --------------------------------------------------------------------------------
#
# Models for "show transceiver status [ interfaces <interface> ]"
#
# --------------------------------------------------------------------------------

AttrDesc = namedtuple( 'AttrDesc', 'attr header renderedTrue renderedFalse' )

def fmtGeneric( attr, attrName, fmt="  %-29.29s %-20.20s %7.7s       %s" ):
   if not attr:
      return
   renderableAttrs = attr.getRenderedAttr()
   values = ( attrName,
              renderableAttrs.get( "state", '' ),
              renderableAttrs.get( "changes", '' ),
              renderableAttrs.get( "lastChange", '' ) )
   if any( values ):
      print( fmt % values )

def fmtDescriptive( attrDesc, attrHeader="",
                    fmt="  %-29.29s %-20.20s %7.7s       %s" ):
   '''
   Helper function used to print a formatted line in the case when we would
   want to use a more descriptive token for the state, instead of true/false.

   Parameters
   ----------
   attrDesc : AttrDesc object
      Namedtuple object holding the attribute that must be rendered, the line
      header, and the tokens used to replace the boolean state in the output
   '''
   assert isinstance( attrDesc, AttrDesc )
   if not attrDesc or not attrDesc.attr:
      return
   renderableAttrs = attrDesc.attr.getRenderedAttr()
   renderedState = attrDesc.renderedTrue \
                   if "true" in renderableAttrs.get( "state", '' ) \
                   else attrDesc.renderedFalse
   header = attrHeader if attrHeader else attrDesc.header
   values = ( header,
              renderedState,
              renderableAttrs.get( "changes", '' ),
              renderableAttrs.get( "lastChange", '' ) )
   if any( values ):
      print( fmt % values )

def fmtFecDegradeStatus( headers,
                         adminAttrs,
                         alarmAttrs,
                         headerFmt="  %-29.29s",
                         statusFmt="    %-27.27s %-20.20s %7.7s       %s" ):
   '''
   Helper function used to print formatted output for Coherent FEC Degrade
   parameters ( detected or excessive )
   Includes headers (parameter name ), admin status and alarm state

   Parameters
   ----------
   headers : list [ str ]
      A list of headers for degrade parameters

   adminAttrs : list[ AttrDesc ]
      A list of AttrDesc objects for degrade parameters admin status

   alarmAttrs : list[ AttrDesc ]
      A list of AttrDesc objects for degrade parameters alarm state

   headerFmt : str
         Formatted string for common degrade parameter

   statusFmt : str
         Formatted string for admin status and alarm state
   '''
   for header, adminAttr, alarmAttr in \
       zip( headers, adminAttrs, alarmAttrs ):

      for attrDesc in ( adminAttr, alarmAttr ):
         assert isinstance( attrDesc, AttrDesc )

      print( headerFmt % header )
      fmtDescriptive( adminAttr, fmt=statusFmt )
      # Print alarm state only if feature is enabled
      if adminAttr.attr.state:
         fmtDescriptive( alarmAttr, fmt=statusFmt )

class InterfacesTransceiverStatusValueFormat( Model ):
   changes = Int( help="Changes since system boot", optional=True )
   lastChange = Float( help="Last update time in UTC", optional=True )

   def _getRenderedAttr( self ):
      return { "changes": str( self.changes if self.changes is not None else '' ),
               "lastChange": ( Ark.timestampToStr( self.lastChange ) if
                                self.lastChange is not None else '' ) }

   def getRenderedAttr( self ):
      return self._getRenderedAttr()

class InterfacesTransceiverStatusValueFormatInt(
      InterfacesTransceiverStatusValueFormat ):
   state = Int( help="Current value" )

   def getRenderedAttr( self ):
      renderDict = self._getRenderedAttr()
      renderDict[ "state" ] = str( self.state )
      return renderDict

class InterfacesTransceiverStatusValueFormatTrib(
      InterfacesTransceiverStatusValueFormat ):
   state = Int( help="Current value" )

   def getRenderedAttr( self ):
      renderDict = self._getRenderedAttr()
      noTrib = TributaryId.tributaryIdUnknown
      if self.state == noTrib:
         renderDict[ "state" ] = "none"
      else:
         renderDict[ "state" ] = str( self.state )
      return renderDict

class InterfacesTransceiverStatusValueFormatHex(
      InterfacesTransceiverStatusValueFormat ):
   state = Int( help="Current value" )

   def getRenderedAttr( self ):
      renderDict = self._getRenderedAttr()
      renderDict[ "state" ] = hex( self.state )
      return renderDict

class InterfacesTransceiverStatusValueFormatStr(
      InterfacesTransceiverStatusValueFormat ):
   state = Str( help="Current value" )

   def getRenderedAttr( self ):
      renderDict = self._getRenderedAttr()
      renderDict[ "state" ] = self.state
      return renderDict

class InterfacesTransceiverStatusValueFormatBool(
      InterfacesTransceiverStatusValueFormat ):
   state = Bool( help="Current value" )

   def getRenderedAttr( self ):
      renderDict = self._getRenderedAttr()
      renderDict[ "state" ] = str( self.state ).lower()
      return renderDict

class InterfacesTransceiverStatusValueFormatFloat(
      InterfacesTransceiverStatusValueFormat ):
   state = Float( help="Current value" )

   def getRenderedAttr( self ):
      renderDict = self._getRenderedAttr()
      renderDict[ "state" ] = str( self.state )
      return renderDict

class InterfacesTransceiverStatusValueFormatScientificFloat(
      InterfacesTransceiverStatusValueFormat ):
   state = Float( help="Current value" )

   def getRenderedAttr( self ):
      renderDict = self._getRenderedAttr()
      renderDict[ "state" ] = f"{self.state:.2e}"
      return renderDict

class InterfacesTransceiverStatuValuesSwizzlerList(
      InterfacesTransceiverStatusValueFormat ):
   state = List( valueType=str, help="List of adapters in use" )

   def getRenderedAttr( self ):
      listValues = ", ".join( self.state or [ 'none' ] )
      renderDict = self._getRenderedAttr()
      renderDict[ "state" ] = listValues
      return renderDict

class InterfacesTransceiverStatusValueFormatModuleStateDec(
      InterfacesTransceiverStatusValueFormat ):

   init = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                    help='Module state initialization' )
   lowPwr = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                      help='Module state low power' )
   hiPwrUp = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                       help='Module state high power up' )
   txOff = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                     help='Module state TX off' )
   txTurnOn = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                        help='Module state TX turn on' )
   ready = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                     help='Module state ready' )
   fault = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                     help='Module state fault' )
   txTurnOff = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                         help='Module state TX turn off' )
   hiPwrDown = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                         help='Module state high power down' )

   def renderModel( self, fmtState="    %-40.40s %-20.20s %7.7s       %s" ):
      moduleStateAttrs = [
         ( self.init, "Initialization" ),
         ( self.lowPwr, "Low power" ),
         ( self.hiPwrUp, "High power up" ),
         ( self.txOff, "TX off" ),
         ( self.txTurnOn, "TX turn on" ),
         ( self.ready, "Ready" ),
         ( self.fault, "Fault" ),
         ( self.txTurnOff, "TX turn off" ),
         ( self.hiPwrDown, "High power down" ),
      ]
      print( "  Module state" )
      for attr in moduleStateAttrs:
         fmtGeneric( *attr, fmt=fmtState )

class InterfacesTransceiverStatusValueFormatTxTurnUpStateDec(
      InterfacesTransceiverStatusValueFormat ):

   txInit = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                      help='TX turn up state path initialization' )
   txDataPathLock = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='TX turn up state data path lock' )
   txLasReadyOff = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='TX turn up state TX laser ready off' )
   txLaserReady = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='TX turn up state TX laser ready' )
   txModulatorConverge = \
      Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                help='TX turn up state TX modulator converge' )
   txOutPwrAdj = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='TX turn up state TX output power adjustment' )

   def renderModel( self, fmtState="    %-40.40s %-20.20s %7.7s       %s" ):
      txTurnUpStateAttrs = [
         ( self.txInit, "Path initialization" ),
         ( self.txDataPathLock, "Data path lock" ),
         ( self.txLasReadyOff, "Laser ready off" ),
         ( self.txLaserReady, "Laser ready" ),
         ( self.txModulatorConverge, "Modulator converge" ),
         ( self.txOutPwrAdj, "Output power adjustment" ) ]
      print( "  TX turn-up state" )
      for attr in txTurnUpStateAttrs:
         fmtGeneric( *attr, fmt=fmtState )

class InterfacesTransceiverStatusValueFormatRxTurnUpStateDec(
      InterfacesTransceiverStatusValueFormat ):

   rxInit = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                      help='RX turn up state path initialization' )
   rxLoLaserReady = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                              help='RX turn up state RX LO laser ready' )
   rxWaitForInput = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                              help='RX turn up state wait for signal input' )
   adcOutput = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                         help='RX turn up state ADC output' )
   dispersionLock = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                              help='RX turn up state dispersion lock' )
   rxDemodLock = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                           help='RX turn up state demodulator lock' )

   def renderModel( self, fmtState="    %-40.40s %-20.20s %7.7s       %s" ):
      rxTurnUpStateAttrs = [
         ( self.rxInit, "Path initialization" ),
         ( self.rxLoLaserReady, "LO laser ready" ),
         ( self.rxWaitForInput, "Wait for signal input" ),
         ( self.adcOutput, "ADC output" ),
         ( self.dispersionLock, "Dispersion lock" ),
         ( self.rxDemodLock, "Demodulator lock" ),
      ]
      print( "  RX turn-up state" )
      for attr in rxTurnUpStateAttrs:
         fmtGeneric( *attr, fmt=fmtState )

class InterfacesTransceiverStatusValueFormatModuleGeneralStatus(
      InterfacesTransceiverStatusValueFormat ):

   hiPowerOn = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='Module general status high power on' )
   pmIntervalDone = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='Module general status PM interval done',
      optional=True )
   outOfAlgn = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='Module general status out of alignment' )
   rxNetworkLOL = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='Module general status RX network loss of lock' )
   rxLOS = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='Module general status RX loss of signal' )
   txHostLOL = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='Module general status TX host loss of lock' )
   txLOF = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='Module general status TX loss of signal functionality' )
   hwInterlock = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='Module general status hw interlock' )

   def renderModel( self, fmtStatus="    %-40.40s %-20.20s %7.7s       %s" ):
      modGeneralStatusAttrs = [
         ( self.hiPowerOn, "High power on" ),
         ( self.outOfAlgn, "Host lane alignment" ),
         ( self.rxNetworkLOL, "RX network LOL" ),
         ( self.rxLOS, "RX LOS" ),
         ( self.txHostLOL, "TX host LOL" ),
         ( self.txLOF, "TX LOSF" ),
         ( self.hwInterlock, "HW interlock" ) ]
      print( "  Module general status" )
      for attr in modGeneralStatusAttrs:
         fmtGeneric( *attr, fmt=fmtStatus )

class InterfacesTransceiverStatusValueFormatModuleFaultStatus(
      InterfacesTransceiverStatusValueFormat ):

   checkSum = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='Module fault status checksum' )
   powerSupplyFault = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='Module fault status power supply fault' )
   pldFlashInitFault = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='Module fault status pld flash init fault' )
   modSpecFault = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='Module fault status module specific hw fault' )
   modOverTempFault = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='Module fault status module over temperature fault' )
   powerSupplyFault = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='Module fault status power supply fault' )

   def renderModel( self, fmtFault="    %-40.40s %-20.20s %7.7s       %s" ):
      modFaultStatusAttrs = [
         ( self.checkSum, "CFP table checksum" ),
         ( self.powerSupplyFault, "Power supply fault" ),
         ( self.pldFlashInitFault, "PLD flash init fault" ),
         ( self.modSpecFault, "Module specific HW fault" ),
         ( self.modOverTempFault, "Module over temp fault" ), ]
      print( "  Module fault status" )
      for attr in modFaultStatusAttrs:
         fmtGeneric( *attr, fmt=fmtFault )

class InterfacesTransceiverStatusValueFormatNetworkLaneVendorFAWS(
      InterfacesTransceiverStatusValueFormat ):

   oaPumpBiasHighAlarm = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='OA pump bias high alarm' )
   oaPumpBiasHighWarn = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='OA pump bias high warn' )
   oaPumpBiasLowAlarm = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='OA pump bias low alarm' )
   oaPumpBiasLowWarn = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='OA pump bias low warn' )
   rxPhaseCtrlLoopHighAlarm = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX phase control loop high alarm' )
   rxPhaseCtrlLoopHighWarn = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX phase control loop high warn' )
   rxPhaseCtrlLoopLowAlarm = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX phase control loop low alarm' )
   rxPhaseCtrlLoopLowWarn = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX phase control loop low warn' )
   txModBiasVOAHighAlarm = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='TX mod bias VOA control loop high alarm' )
   txModBiasVOAHighWarn = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='TX mod bias VOA control loop high warn' )
   rxTunedChannelPwrHighAlarm = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX tuned channel power high alarm' )
   rxTunedChannelPwrHighWarn = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX tuned channel power high warn' )
   rxTunedChannelPwrLowAlarm = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX tuned channel power low alarm' )
   rxTunedChannelPwrLowWarn = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX tuned channel power low warn' )
   modBiasConvergeFault = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='Modulator bias converge fault' )
   adcOutputLow = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='ADC output low' )

   def renderModel( self, fmtFaws="    %-40.40s %-20.20s %7.7s       %s" ):
      networkLaneVendorFAWSAttrs = [
         ( self.oaPumpBiasHighAlarm, "OA pump bias high alarm" ),
         ( self.oaPumpBiasHighWarn, "OA pump bias high warn" ),
         ( self.oaPumpBiasLowAlarm, "OA pump bias low alarm" ),
         ( self.oaPumpBiasLowWarn, "OA pump bias low warn" ),
         ( self.rxPhaseCtrlLoopHighAlarm, "RX phase control loop high alarm" ),
         ( self.rxPhaseCtrlLoopHighWarn, "RX phase control loop high warn" ),
         ( self.rxPhaseCtrlLoopLowAlarm, "RX phase control loop low alarm" ),
         ( self.rxPhaseCtrlLoopLowWarn, "RX phase control loop low warn" ),
         ( self.txModBiasVOAHighAlarm,
           "TX mod bias VOA ctrl loop high alarm" ),
         ( self.txModBiasVOAHighWarn,
           "TX mod bias VOA ctrl loop high warn" ),
         ( self.rxTunedChannelPwrHighAlarm,
           "RX tuned channel power high alarm" ),
         ( self.rxTunedChannelPwrHighWarn,
           "RX tuned channel power high warn" ),
         ( self.rxTunedChannelPwrLowAlarm,
           "RX tuned channel power low alarm" ),
         ( self.rxTunedChannelPwrLowWarn,
           "RX tuned channel power low warn" ),
         ( self.modBiasConvergeFault, "Modulator bias converge fault" ),
         ( self.adcOutputLow, "ADC output low" ),
      ]
      print( "  Network lane vendor specific faults/status" )
      for attr in networkLaneVendorFAWSAttrs:
         fmtGeneric( *attr, fmt=fmtFaws )

class InterfacesTransceiverStatusValueFormatNetworkLaneFaultStatus(
      InterfacesTransceiverStatusValueFormat ):

   rxTECFault = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX TEC fault' )
   rxFIFOFault = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX FIFO fault' )
   rxLOL = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX LOL' )
   rxLOS = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX LOS' )
   txLOL = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='TX LOL' )
   txLOSF = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='TX LOSF' )
   wavelengthUnlockFault = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='Wavelength unlock fault' )
   tecFault = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='TEC fault' )

   def renderModel( self, fmtAttrs="    %-40.40s %-20.20s %7.7s       %s" ):
      networkLaneFaultStatusAttrs = [
         ( self.rxTECFault, "RX TEC fault" ),
         ( self.rxFIFOFault, "RX FIFO fault" ),
         ( self.rxLOL, "RX LOL" ),
         ( self.rxLOS, "RX LOS" ),
         ( self.txLOL, "TX LOL" ),
         ( self.txLOSF, "TX LOSF" ),
         ( self.wavelengthUnlockFault, "Wavelength unlock fault" ),
         ( self.tecFault, "TEC fault" ),
      ]
      print( "  Network lane fault/status" )
      for attr in networkLaneFaultStatusAttrs:
         fmtGeneric( *attr, fmt=fmtAttrs )

class InterfacesTransceiverStatusValueFormatLaneAlignment(
      InterfacesTransceiverStatusValueFormat ):

   netLaneTxAlignment = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Network lane TX alignment' )
   modemLockFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='RX alignment modem lock fault' )
   modemSyncDetectFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='RX alignment modem sync detect fault' )

   def renderModel( self, fmtTx="  %-42.42s %-20.20s %7.7s       %s",
                    fmtRx="    %-40.40s %-20.20s %7.7s       %s" ):
      fmtGeneric( self.netLaneTxAlignment, "Network TX alignment",
                  fmt=fmtTx )
      print( "  Network RX alignment" )
      for attr in [ ( self.modemLockFault, "Modem lock fault" ),
                    ( self.modemSyncDetectFault,
                      "Modem sync detect fault" ), ]:
         fmtGeneric( *attr, fmt=fmtRx )

class InterfacesTransceiverStatusValueFormatNetworkLaneRxOTNStatus(
      InterfacesTransceiverStatusValueFormat ):

   fastPMRxSignalDegradeAlarm = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Fast PM RX signal degrade alarm' )
   fastPMRxSignalFailAlarm = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Fast PM RX signal fail alarm' )
   fastPMTxSignalDegradeAlarm = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Fast PM TX signal degrade alarm' )
   fastPMTxSignalFailAlarm = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Fast PM TX signal fail alarm' )
   ingressFDDAlarm = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Ingress FDD alarm' )
   ingressFEDAlarm = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Ingress FED alarm' )
   oduLCKMaintSignal = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='ODU LCK maintenance signal' )
   oduAISMaintSignal = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='ODU AIS maintenance signal' )
   sectionMonitoringBackDefect = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Section monitoring back defect' )
   outOfMultiframe = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Out of multiframe' )
   lossOfMultiframe = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Loss of multiframe' )
   outOfFrame = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Out of frame' )
   lossOfFrame = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Loss of frame' )

   def renderModel( self, fmtRxOTN="    %-40.40s %-20.20s %7.7s       %s" ):
      networkRxOTNStatusAttrs = [
         ( self.fastPMRxSignalDegradeAlarm, "Fast PM RX signal degrade alarm" ),
         ( self.fastPMRxSignalFailAlarm, "Fast PM RX signal fail alarm" ),
         ( self.fastPMTxSignalDegradeAlarm, "Fast PM TX signal degrade alarm" ),
         ( self.fastPMTxSignalFailAlarm, "Fast PM TX signal fail alarm" ),
         ( self.ingressFDDAlarm, "Ingress FDD alarm" ),
         ( self.ingressFEDAlarm, "Ingress FED alarm" ),
         ( self.oduLCKMaintSignal, "ODU LCK maintenance signal" ),
         ( self.oduAISMaintSignal, "ODU AIS maintenance signal" ),
         ( self.sectionMonitoringBackDefect, "Section monitoring back defect" ),
         ( self.outOfMultiframe, "Out of multiframe" ),
         ( self.lossOfMultiframe, "Loss of multiframe" ),
         ( self.outOfFrame, "Out of frame" ),
         ( self.lossOfFrame, "Loss of frame" ),
      ]
      print( "  Network lane RX OTN status" )
      for attr in networkRxOTNStatusAttrs:
         fmtGeneric( *attr, fmt=fmtRxOTN )

class InterfacesTransceiverStatusValueFormatTxModulatorBiasVOA(
      InterfacesTransceiverStatusValueFormat ):

   loop1Alarm = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Net mod bias VOA loop 1 alarm' )
   loop1Warn = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Net mod bias VOA loop 1 warn' )
   loop2Alarm = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Net mod bias VOA loop 2 alarm' )
   loop2Warn = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Net mod bias VOA loop 2 warn' )
   loop3Alarm = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Net mod bias VOA loop 3 alarm' )
   loop3Warn = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Net mod bias VOA loop 3 warn' )
   loop4Alarm = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Net mod bias VOA loop 4 alarm' )
   loop4Warn = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Net mod bias VOA loop 4 warn' )
   loop5Alarm = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Net mod bias VOA loop 5 alarm' )
   loop5Warn = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Net mod bias VOA loop 5 warn' )
   loop6Alarm = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Net mod bias VOA loop 6 alarm' )
   loop6Warn = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Net mod bias VOA loop 6 warn' )
   loop7Alarm = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Net mod bias VOA loop 7 alarm' )
   loop7Warn = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Net mod bias VOA loop 7 warn' )
   loop8Alarm = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Net mod bias VOA loop 8 alarm' )
   loop8Warn = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Net mod bias VOA loop 8 warn' )

   def renderModel( self, fmtAttrs="    %-40.40s %-20.20s %7.7s       %s" ):
      networkModBiasVOAAttrs = [
         ( self.loop1Alarm, "Loop 1 alarm" ),
         ( self.loop1Warn, "Loop 1 warn" ),
         ( self.loop2Alarm, "Loop 2 alarm" ),
         ( self.loop2Warn, "Loop 2 warn" ),
         ( self.loop3Alarm, "Loop 3 alarm" ),
         ( self.loop3Warn, "Loop 3 warn" ),
         ( self.loop4Alarm, "Loop 4 alarm" ),
         ( self.loop4Warn, "Loop 4 warn" ),
         ( self.loop5Alarm, "Loop 5 alarm" ),
         ( self.loop5Warn, "Loop 5 warn" ),
         ( self.loop6Alarm, "Loop 6 alarm" ),
         ( self.loop6Warn, "Loop 6 warn" ),
         ( self.loop7Alarm, "Loop 7 alarm" ),
         ( self.loop7Warn, "Loop 7 warn" ),
         ( self.loop8Alarm, "Loop 8 alarm" ),
         ( self.loop8Warn, "Loop 8 warn" ),
      ]
      print( "  Network TX modulator bias VOA AWS" )
      for attr in networkModBiasVOAAttrs:
         fmtGeneric( *attr, fmt=fmtAttrs )

class InterfacesTransceiverStatusValueFormatVendorSpecificFaultStatus(
      InterfacesTransceiverStatusValueFormat ):

   bootEepromCRCFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Boot EEPROM CRC fault' )
   devInitFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Device initialization fault' )
   ctrlProcImgABBootFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Controller proc image AB boot fault' )
   fwHwMismatchFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='FW HW mismatch fault' )
   powerSupplyRWFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Power supply RW fault' )
   cryptoDevRWFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Crypto device RW fault' )
   cpldDevRWFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='CPLD device RW fault' )
   viMonitorDevRWFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='VI monitor device RW fault' )
   dacRefClockRWFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='DAC reference clock RW fault' )
   ctrlProcIntCommFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Controller proc internal comm fault' )
   modemProcIntCommFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Modem proc internal comm fault' )
   opticsProcIntCommFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='Optics proc internal comm fault' )
   txITLAFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='TX ITLA fault' )
   rxITLAFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='RX ITLA fault' )
   adcCalibrationFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='ADC calibration fault' )
   dacCalibrationFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='DAC calibration fault' )
   picEepromRWFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='PIC EEPROM RW fault' )
   picEepromCRCFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='PIC EEPROM CRC fault' )
   apexRWFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='APEX RW fault' )
   picXTiaRWFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='PIC X TIA RW fault' )
   picYTiaRWFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='PIC Y TIA RW fault' )
   picDriverRWFault = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatStr,
       help='PIC driver RW fault' )

   def renderModel( self, fmtAttrs="    %-40.40s %-20.20s %7.7s       %s" ):
      vendorSpecificFaultStatusAttrs = [
         ( self.bootEepromCRCFault, "Boot EEPROM CRC fault" ),
         ( self.devInitFault, "Device initialization fault" ),
         ( self.ctrlProcImgABBootFault, "Controller proc img AB boot fault" ),
         ( self.fwHwMismatchFault, "FW HW mismatch fault" ),
         ( self.powerSupplyRWFault, "Power supply RW fault" ),
         ( self.cryptoDevRWFault, "Crypto device RW fault" ),
         ( self.cpldDevRWFault, "CPLD device RW fault" ),
         ( self.viMonitorDevRWFault, "VI monitor device RW fault" ),
         ( self.dacRefClockRWFault, "DAC ref clock RW fault" ),
         ( self.ctrlProcIntCommFault, "Controller proc internal comm fault" ),
         ( self.modemProcIntCommFault, "Modem proc internal comm fault" ),
         ( self.opticsProcIntCommFault, "Optics proc internal comm fault" ),
         ( self.txITLAFault, "TX ITLA fault" ),
         ( self.rxITLAFault, "RX ITLA fault" ),
         ( self.adcCalibrationFault, "ADC calibration fault" ),
         ( self.dacCalibrationFault, "DAC calibration fault" ),
         ( self.picEepromRWFault, "PIC EEPROM RW fault" ),
         ( self.picEepromCRCFault, "PIC EEPROM CRC fault" ),
         ( self.apexRWFault, "APEX RW fault" ),
         ( self.picXTiaRWFault, "PIC X TIA RW fault" ),
         ( self.picYTiaRWFault, "PIC Y TIA RW fault" ),
         ( self.picDriverRWFault, "PIC driver RW fault" ),
      ]
      print( "  Vendor specific fault status" )
      for attr in vendorSpecificFaultStatusAttrs:
         fmtGeneric( *attr, fmt=fmtAttrs )

class InterfacesTransceiverStatusExtCfp2Dco( Model ):
   _dualLaserModulePresent = Bool( help="The dual laser DP04 DCO module requires "
                                        "custom CLI status formatting when "
                                        "inserted" )

   def dualLaserModulePresentIs( self, present=False ):
      self._dualLaserModulePresent = present

   def dualLaserModulePresent( self ):
      return self._dualLaserModulePresent

   controllerState = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                               help="Controller state" )
   controllerSubState = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help="Controller substate" )
   moduleState = Submodel( valueType=InterfacesTransceiverStatusValueFormatHex,
                           help="Module state",
                           optional=True )
   txTurnUpState = Submodel( valueType=InterfacesTransceiverStatusValueFormatHex,
                             help="TX turn-up state",
                             optional=True )
   rxTurnUpState = Submodel( valueType=InterfacesTransceiverStatusValueFormatHex,
                             help="RX turn-up state",
                             optional=True )
   moduleGeneral = Submodel( valueType=InterfacesTransceiverStatusValueFormatHex,
                             help="Module general status",
                             optional=True )
   moduleFaults = Submodel( valueType=InterfacesTransceiverStatusValueFormatHex,
                            help="Module fault status",
                            optional=True )
   vendorFaults = Submodel( valueType=InterfacesTransceiverStatusValueFormatHex,
                            help="Vendor specific faults/status",
                            optional=True )
   networkVendorFaults = Submodel(
         valueType=InterfacesTransceiverStatusValueFormatHex,
         help="Network vendor-specific faults/status",
         optional=True )
   laserFaultCode = Submodel( valueType=InterfacesTransceiverStatusValueFormatHex,
                              help="Laser fault code" )
   sopFastTracking = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                               help="State of Polarization (SOP) fast tracking",
                               optional=True )

   # The following attributes currently apply only to the DP04 module
   moduleStateDec = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatModuleStateDec,
      help="Module state deciphered for cfp2Dco Dp04 module",
      optional=True )
   txTurnUpStateDec = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatTxTurnUpStateDec,
      help="TX turn up state deciphered for cfp2Dco Dp04 module",
      optional=True )
   rxTurnUpStateDec = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatRxTurnUpStateDec,
      help="RX turn up state deciphered for cfp2Dco Dp04 module",
      optional=True )
   moduleGeneralStatus = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatModuleGeneralStatus,
      help="Module general status for cfp2Dco Dp04 module",
      optional=True )
   moduleFaultStatus = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatModuleFaultStatus,
      help="Module fault status for cfp2Dco Dp04 module",
      optional=True )
   txModBiasHighAlarm = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='TX modulator bias high alarm',
      optional=True )
   txModBiasHighWarn = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='TX modulator bias high warn',
      optional=True )
   txModBiasLowAlarm = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='TX modulator bias low alarm',
      optional=True )
   txModBiasLowWarn = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='TX modulator bias low warn',
      optional=True )
   txLaserPowerHighAlarm = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='TX laser power high alarm',
      optional=True )
   txLaserPowerHighWarn = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='TX laser power high warn',
      optional=True )
   txLaserPowerLowAlarm = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='TX laser power low alarm',
      optional=True )
   txLaserPowerLowWarn = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='TX laser power low warn',
      optional=True )
   rxLaserPowerHighAlarm = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX laser power high alarm',
      optional=True )
   rxLaserPowerHighWarn = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX laser power high warn',
      optional=True )
   rxLaserPowerLowAlarm = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX laser power low alarm',
      optional=True )
   rxLaserPowerLowWarn = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX laser power low warn',
      optional=True )
   txLaserTempHighAlarm = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='TX laser temperature high alarm',
      optional=True )
   txLaserTempHighWarn = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='TX laser temperature high warn',
      optional=True )
   txLaserTempLowAlarm = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='TX laser temperature low alarm',
      optional=True )
   txLaserTempLowWarn = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='TX laser temperature low warn',
      optional=True )
   rxLaserTempHighAlarm = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX laser temperature high alarm',
      optional=True )
   rxLaserTempHighWarn = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX laser temperature high warn',
      optional=True )
   rxLaserTempLowAlarm = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX laser temperature low alarm',
      optional=True )
   rxLaserTempLowWarn = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX laser temperature low warn',
      optional=True )
   rxPowerHighAlarm = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX total power high alarm',
      optional=True )
   rxPowerHighWarn = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX total power high warn',
      optional=True )
   rxPowerLowAlarm = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX total power low alarm',
      optional=True )
   rxPowerLowWarn = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='RX total power low warn',
      optional=True )
   netLaneVendorSpecificFAWS = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatNetworkLaneVendorFAWS,
      help='Network lane vendor specific fault/status',
      optional=True )
   netLaneFaultStatus = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatNetworkLaneFaultStatus,
      help='Network lane fault/status',
      optional=True )
   laneAlignment = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatLaneAlignment,
      help='TX/RX alignment',
      optional=True )
   networkLaneRxOTNStatus = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatNetworkLaneRxOTNStatus,
      help='Network lane RX OTN status',
      optional=True )
   networkTxModulatorBiasVoaAWS = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatTxModulatorBiasVOA,
      help='Network TX modulator bias VOA AWS',
      optional=True )
   adcFaultCode = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatHex,
       help='ADC fault code',
       optional=True )
   dacFaultCode = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatHex,
       help='DAC fault code',
       optional=True )
   modulatorConvergenceIter = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatInt,
       help='Modulator convergence iteration count',
       optional=True )
   modulatorConvergenceStep = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatInt,
       help='Modulator convergence step',
       optional=True )
   modulatorBiasFaultCode = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatHex,
       help='Modulator bias fault code',
       optional=True )
   vendorSpecificFaultStatus = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatVendorSpecificFaultStatus,
      help='Vendor specific fault status',
      optional=True )
   rxLaserFaultCode = Submodel(
       valueType=InterfacesTransceiverStatusValueFormatHex,
       help='RX laser fault code',
       optional=True )

   def renderExt( self ):
      attrs = [ ( self.controllerState, "Controller state" ),
                ( self.controllerSubState, "Controller substate" ) ]
      if not self._dualLaserModulePresent:
         attrs.extend( [ ( self.moduleState, "Module state" ),
                         ( self.txTurnUpState, "TX turn-up state" ),
                         ( self.rxTurnUpState, "RX turn-up state" ),
                         ( self.moduleGeneral, "Module general status" ),
                         ( self.moduleFaults, "Module fault status" ),
                         ( self.vendorFaults, "Vendor specific faults/status" ),
                         ( self.networkVendorFaults, "Network vendor-specific "
                                                     "faults/status" ),
                         ( self.laserFaultCode, "Laser fault code" ),
                         ( self.sopFastTracking, "SOP fast tracking" ) ] )
         for attr in attrs:
            fmtGeneric( *attr )
      else:
         fmt = "  %-42.42s %-20.20s %7.7s       %s"
         for attr in attrs:
            fmtGeneric( *attr, fmt=fmt )
         self.moduleStateDec.renderModel()
         self.txTurnUpStateDec.renderModel()
         self.rxTurnUpStateDec.renderModel()
         self.moduleGeneralStatus.renderModel()
         self.moduleFaultStatus.renderModel()
         alarmStatusAttrs = [
            ( self.txModBiasHighAlarm, "TX modulator bias high alarm" ),
            ( self.txModBiasHighWarn, "TX modulator bias high warn" ),
            ( self.txModBiasLowAlarm, "TX modulator bias low alarm" ),
            ( self.txModBiasLowWarn, "TX modulator bias low warn" ),
            ( self.txLaserPowerHighAlarm, "TX laser power high alarm" ),
            ( self.txLaserPowerHighWarn, "TX laser power high warn" ),
            ( self.txLaserPowerLowAlarm, "TX laser power low alarm" ),
            ( self.txLaserPowerLowWarn, "TX laser power low warn" ),
            ( self.rxLaserPowerHighAlarm, "RX laser power high alarm" ),
            ( self.rxLaserPowerHighWarn, "RX laser power high warn" ),
            ( self.rxLaserPowerLowAlarm, "RX laser power low alarm" ),
            ( self.rxLaserPowerLowWarn, "RX laser power low warn" ),
            ( self.txLaserTempHighAlarm, "TX laser temperature high alarm" ),
            ( self.txLaserTempHighWarn, "TX laser temperature high warn" ),
            ( self.txLaserTempLowAlarm, "TX laser temperature low alarm" ),
            ( self.txLaserTempLowWarn, "TX laser temperature low warn" ),
            ( self.rxLaserTempHighAlarm, "RX laser temperature high alarm" ),
            ( self.rxLaserTempHighWarn, "RX laser temperature high warn" ),
            ( self.rxLaserTempLowAlarm, "RX laser temperature low alarm" ),
            ( self.rxLaserTempLowWarn, "RX laser temperature low warn" ),
            ( self.rxPowerHighAlarm, "RX total power high alarm" ),
            ( self.rxPowerHighWarn, "RX total power high warn" ),
            ( self.rxPowerLowAlarm, "RX total power low alarm" ),
            ( self.rxPowerLowWarn, "RX total power low warn" ),
         ]
         for attr in alarmStatusAttrs:
            fmtGeneric( *attr, fmt=fmt )
         self.netLaneVendorSpecificFAWS.renderModel()
         self.netLaneFaultStatus.renderModel()
         self.laneAlignment.renderModel()
         self.networkLaneRxOTNStatus.renderModel()
         self.networkTxModulatorBiasVoaAWS.renderModel()
         for attr in [
            ( self.adcFaultCode, "ADC fault code" ),
            ( self.dacFaultCode, "DAC fault code" ),
            ( self.modulatorConvergenceIter, "Modulator convergence iteration" ),
            ( self.modulatorConvergenceStep, "Modulator convergence step" ),
            ( self.modulatorBiasFaultCode, "Modulator bias fault code" ) ]:
            fmtGeneric( *attr, fmt=fmt )
         self.vendorSpecificFaultStatus.renderModel()
         fmtGeneric( self.laserFaultCode, "TX laser fault code", fmt=fmt )
         fmtGeneric( self.rxLaserFaultCode, "RX laser fault code", fmt=fmt )
         fmtGeneric( self.sopFastTracking, "SOP fast tracking", fmt=fmt )

class InterfacesTransceiverStatusIntfValueFormatIngressAlarmStatus(
      InterfacesTransceiverStatusValueFormat ):

   rxLocalFault = Submodel( valueType=InterfacesTransceiverStatusValueFormatBool,
                            help='RX local fault' )
   rxRemoteFault = Submodel( valueType=InterfacesTransceiverStatusValueFormatBool,
                             help='RX remote fault' )
   blockLock = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                         help='PCS block lock' )
   alignmentMarkerLock = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='PCS alignment marker lock' )
   bipErrorsDetected = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='PCS BIP errors detected' )
   erroredBlocksDetected = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatStr,
      help='PCS errored blocks detected' )
   loa = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                   help='PCS loss of alignment' )
   los = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                   help='PCS loss of signal',
                   optional=True )

   def renderModel( self, fmt="    %-40.40s %-20.20s %7.7s       %s" ):
      print( "  Client ingress alarm status" )
      ingressAlarmStatusAttrs = [
         ( self.rxLocalFault, 'RX local fault' ),
         ( self.rxRemoteFault, 'RX remote fault' ),
         ( self.blockLock, 'PCS block lock' ),
         ( self.alignmentMarkerLock, 'PCS alignment marker lock' ),
         ( self.bipErrorsDetected, 'PCS BIP errors detected' ),
         ( self.erroredBlocksDetected, 'PCS errored blocks detected' ),
         ( self.loa, 'PCS LOA' ),
         ( self.los, 'PCS LOS' ) ]
      for attr in ingressAlarmStatusAttrs:
         fmtGeneric( *attr, fmt=fmt )

class InterfacesTransceiverStatusIntfValueFormatEgressAlarmStatus(
      InterfacesTransceiverStatusIntfValueFormatIngressAlarmStatus ):

   highBer = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                       help='PCS high BER' )

   def renderModel( self, fmt="    %-40.40s %-20.20s %7.7s       %s" ):
      print( "  Client egress alarm status" )
      egressAlarmStatusAttrs = [
         ( self.rxLocalFault, 'RX local fault' ),
         ( self.rxRemoteFault, 'RX remote fault' ),
         ( self.blockLock, 'PCS block lock' ),
         ( self.highBer, 'PCS high BER' ),
         ( self.alignmentMarkerLock, 'PCS alignment marker lock' ),
         ( self.bipErrorsDetected, 'PCS BIP errors detected' ),
         ( self.erroredBlocksDetected, 'PCS errored blocks detected' ),
         ( self.loa, 'PCS LOA' ) ]
      for attr in egressAlarmStatusAttrs:
         fmtGeneric( *attr, fmt=fmt )

class InterfacesTransceiverStatusExtCfp2DcoIntf( Model ):
   pcsAlarmStatus = Submodel( valueType=InterfacesTransceiverStatusValueFormatHex,
                              help="Client physical coding sublayer alarm status",
                              optional=True )
   _dualLaserModulePresent = Bool( help="The dual laser DP04 DCO module requires "
                                        "custom CLI status formatting when "
                                        "inserted" )
   # The following only apply to DP04 dual laser modules
   laneCount = Submodel( valueType=InterfacesTransceiverStatusValueFormatInt,
                         help='Client speed PMD lane count',
                         optional=True )
   speed = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                     help='Client operational speed',
                     optional=True )
   fec = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                   help='Client forward error correction type',
                   optional=True )
   rsFecCodewordSize = Submodel( valueType=InterfacesTransceiverStatusValueFormatInt,
                                 help='Client Reed-Solomon FEC codeword size',
                                 optional=True )
   fecUncorrectedBlocks = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatInt,
      help='Client FEC uncorrected blocks',
      optional=True )
   preFecBer = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                         help='Client pre-FEC bit error rate',
                         optional=True )
   ingressAlarmStatus = Submodel(
      valueType=InterfacesTransceiverStatusIntfValueFormatIngressAlarmStatus,
      help='Client ingress alarms/status',
      optional=True )
   egressAlarmStatus = Submodel(
      valueType=InterfacesTransceiverStatusIntfValueFormatEgressAlarmStatus,
      help='Client egress alarms/status',
      optional=True )
   tributary = Submodel( valueType=InterfacesTransceiverStatusValueFormatTrib,
                         help='Optical tributary',
                         optional=True )

   def dualLaserModulePresentIs( self, present=False ):
      self._dualLaserModulePresent = present

   def dualLaserModulePresent( self ):
      return self._dualLaserModulePresent

   def renderExt( self, intfFmt="  %-42.42s %-20.20s %7.7s       %s" ):
      if self.dualLaserModulePresent():
         cfp2DcoIntfAttrs = [
            ( self.laneCount, 'Lane count' ),
            ( self.fec, 'Forward error correction' ),
            ( self.rsFecCodewordSize, 'Reed-Solomon codeword size' ),
            ( self.fecUncorrectedBlocks, 'FEC uncorrected blocks' ),
            ( self.preFecBer, 'Pre-FEC bit error rate' ) ]
         for attr in cfp2DcoIntfAttrs:
            fmtGeneric( *attr, fmt=intfFmt )
         self.ingressAlarmStatus.renderModel()
         self.egressAlarmStatus.renderModel()
      else:
         fmtGeneric( self.pcsAlarmStatus, "Client PCS alarm status" )

def renderChAttr( channelsDict, printChLabel=False ):
   if len( channelsDict ) > 1:
      # Follows the format of:
      #   Attribute name
      #     Channel 1
      #     Channel 2
      #     Channel ...
      #     Channel n
      attrs = [ ( "RX LOS", "rxLos", "no signal", "ok" ),
                ( "TX fault", "txFault", "fault", "ok" ),
                ( "RX CDR LOL", "rxCdrLol", "no lock", "ok" ),
                ( "TX power high alarm", "txPowerHiAlarm", "alarm", "ok" ),
                ( "TX power high warn", "txPowerHiWarn", "warn", "ok" ),
                ( "TX power low alarm", "txPowerLoAlarm", "alarm", "ok" ),
                ( "TX power low warn", "txPowerLoWarn", "warn", "ok" ),
                ( "TX bias high alarm", "txBiasHiAlarm", "alarm", "ok" ),
                ( "TX bias high warn", "txBiasHiWarn", "warn", "ok" ),
                ( "TX bias low alarm", "txBiasLoAlarm", "alarm", "ok" ),
                ( "TX bias low warn", "txBiasLoWarn", "warn", "ok" ),
                ( "RX power high alarm", "rxPowerHiAlarm", "alarm", "ok" ),
                ( "RX power high warn", "rxPowerHiWarn", "warn", "ok" ),
                ( "RX power low alarm", "rxPowerLoAlarm", "alarm", "ok" ),
                ( "RX power low warn", "rxPowerLoWarn", "warn", "ok" ),
                ( "TX output signal status", "txOutputSignalStats",
                  "ok", "not ok" ) ]

      for attrName, attrValue, renderedTrue, renderedFalse in attrs:
         if any( getattr( i, attrValue ) for i in channelsDict.values() ):
            print( f"  {attrName}" )
            for ch in sorted( channelsDict ):
               attrDesc = AttrDesc( getattr( channelsDict[ ch ], attrValue ),
                                    attrName, renderedTrue, renderedFalse )
               fmtDescriptive( attrDesc, attrHeader=f"Channel {ch}",
                               fmt="    %-27.27s %-20.20s %7.7s       %s" )

      # Need multilane handler for txOutputSignalStatus
      lane = 1
      if lane in channelsDict:
         if channelsDict[ lane ].coherentAlarms is not None:
            channelsDict[ lane ].coherentAlarms.renderModel()
         if channelsDict[ lane ].frequencyTuningAlarms is not None:
            channelsDict[ lane ].frequencyTuningAlarms.renderModel()
   elif len( channelsDict ) == 1:
      ch, chModel = list( channelsDict.items() )[ 0 ]
      if printChLabel:
         print( f"Channel {ch}" )
      chModel.renderModel()

def renderEnhancedDomAttrs( hostLanesDict ):
   enhancedDomAttrs = ordDict( [ ( "Pre-FEC bit error rate",
                                   "preFecBERCurrHost" ),
                                 ( "Post-FEC errored frames ratio",
                                   "errFramesCurHost" ) ] )
   for attrName, attrValue in enhancedDomAttrs.items():
      applicableLanes = sum( getattr( i, attrValue, None ) is not None
                             for i in hostLanesDict.values() )
      if applicableLanes > 1:
         print( f"  {attrName}" )
         for lane in sorted( hostLanesDict ):
            fmtGeneric( getattr( hostLanesDict[ lane ], attrValue ),
                        f"Host lane {lane}",
                        fmt="    %-27.27s %-20.20s %7.7s       %s" )
      elif applicableLanes == 1:
         for lane in hostLanesDict:
            fmtGeneric( getattr( hostLanesDict[ lane ], attrValue ),
                        attrName )

def renderHostLaneAttr( hostLanesDict, printHostLaneLabel=False ):
   if len( hostLanesDict ) > 1:
      # Follows the format of:
      #   Attribute name
      #     Host lane 1
      #     Host lane 2
      #     Host lane ...
      #     Host lane n
      attrs = [ ( "TX LOS", "txLos", "no signal", "ok" ),
                ( "TX CDR LOL", "txCdrLol", "no lock", "ok" ),
                ( "TX adaptive input EQ fault", "txAdaptiveInputEqFault", "fault",
                  "ok" ),
                ( "RX output signal status",
            "rxOutputSignalStats", "ok", "not ok" ),
              ]
      for attrName, attrValue, renderedTrue, renderedFalse in attrs:
         if any( getattr( i, attrValue ) for i in hostLanesDict.values() ):
            print( f"  {attrName}" )
            for lane in sorted( hostLanesDict ):
               attrDesc = AttrDesc( getattr( hostLanesDict[ lane ], attrValue ),
                                    attrName, renderedTrue, renderedFalse )
               fmtDescriptive( attrDesc, attrHeader=f"Host lane {lane}",
                               fmt="    %-27.27s %-20.20s %7.7s       %s" )
   elif len( hostLanesDict ) == 1:
      lane, laneModel = list( hostLanesDict.items() )[ 0 ]
      if printHostLaneLabel:
         print( f"Host lane {lane}" )
      laneModel.renderModel()

class InterfacesTransceiverStatusFrequencyTuningChannelSpecific( Model ):
   tuningInProgress = Submodel( valueType=InterfacesTransceiverStatusValueFormatBool,
                                help="Frequency tuning in progress", optional=True )
   tuningNotAccepted = Submodel(
         valueType=InterfacesTransceiverStatusValueFormatBool,
         help="Frequency tuning busy", optional=True )
   invalidChannel = Submodel(
         valueType=InterfacesTransceiverStatusValueFormatBool,
         help="Frequency tuning invalid channel ( configured channel is outside the \
               range for the selected grid spacing )",
         optional=True )
   tuningComplete = Submodel( valueType=InterfacesTransceiverStatusValueFormatBool,
                              help="Frequency tuning completed",
                              optional=True )

   def renderModel( self ):
      attrs = [ AttrDesc( self.tuningInProgress, 'Freq tuning in progress',
                          'in progress', 'idle' ),
                AttrDesc( self.tuningNotAccepted, 'Freq tuning busy', 'alarm',
                          'ok' ),
                AttrDesc( self.invalidChannel, 'Freq tuning invalid channel',
                          'alarm', 'ok' ),
                AttrDesc( self.tuningComplete, 'Freq tuning completed', 'yes',
                          'no' )
              ]

      for attrDesc in attrs:
         fmtDescriptive( attrDesc )

class InterfacesTransceiverStatusCoherentChannelSpecific( Model ):
   txLossOfAlignment = Submodel(
         valueType=InterfacesTransceiverStatusValueFormatBool,
         help="Transmit loss of alignment", optional=True )
   txOutOfAlignment = Submodel( valueType=InterfacesTransceiverStatusValueFormatBool,
                                help="Transmit out of alignment", optional=True )
   txCmuLossOfLock = Submodel( valueType=InterfacesTransceiverStatusValueFormatBool,
                        help="Transmit clock monitor unit clock loss of lock",
                        optional=True )
   txRefClkLossOfLock = Submodel(
         valueType=InterfacesTransceiverStatusValueFormatBool,
         help="Transmit reference clock loss of lock",
         optional=True )
   txDeskewLossOfLock = Submodel(
         valueType=InterfacesTransceiverStatusValueFormatBool,
         help="Transmit deskew clock loss of lock",
         optional=True )
   txFifoErr = Submodel( valueType=InterfacesTransceiverStatusValueFormatBool,
                           help="Transmit FIFO error", optional=True )
   rxDemodulationLossOfLock = Submodel(
                          valueType=InterfacesTransceiverStatusValueFormatBool,
                          help="Receive demodulator loss of lock",
                          optional=True )
   rxChrDispCompensationLossOfLock = \
         Submodel( valueType=InterfacesTransceiverStatusValueFormatBool,
                   help="Receive chromatic dispersion compensation loss of lock",
                   optional=True )
   rxLossOfAlignment = Submodel(
         valueType=InterfacesTransceiverStatusValueFormatBool,
         help="Receive loss of alignment", optional=True )
   rxOutOfAlignment = Submodel( valueType=InterfacesTransceiverStatusValueFormatBool,
                                help="Receive out of alignment", optional=True )
   rxDeskewLossOfLock = Submodel(
         valueType=InterfacesTransceiverStatusValueFormatBool,
         help="Receive deskew loss of lock", optional=True )
   rxFifoErr = Submodel( valueType=InterfacesTransceiverStatusValueFormatBool,
                           help="Receive FIFO error", optional=True )
   fecExcessiveDegrade = Submodel(
         valueType=InterfacesTransceiverStatusValueFormatBool,
         help="Receive FEC excessive degrade ( indicates that BER exceeded the \
               excessive degrade threshold )",
         optional=True )
   fecExcessiveDegradeEnabled = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatBool,
      help="Receive FEC excessive degrade monitoring enabled",
      optional=True )
   fecDetectedDegrade = Submodel(
         valueType=InterfacesTransceiverStatusValueFormatBool,
         help="Receive FEC detect degrade ( indicates that BER exceeded the \
               detected degrade threshold )",
               optional=True )
   fecDetectedDegradeEnabled = Submodel(
      valueType=InterfacesTransceiverStatusValueFormatBool,
      help="Receive FEC detected degrade monitoring enabled",
      optional=True )

   def renderModel( self ):
      attrs = [ AttrDesc( self.txLossOfAlignment, 'TX loss of alignment',
                          'unaligned', 'ok' ),
                AttrDesc( self.txOutOfAlignment, 'TX out of alignment',
                          'unaligned', 'ok' ),
                AttrDesc( self.txCmuLossOfLock, 'TX clock monitor unit LOL',
                          'no lock', 'ok' ),
                AttrDesc( self.txRefClkLossOfLock, 'TX reference clock LOL',
                          'no lock', 'ok' ),
                AttrDesc( self.txDeskewLossOfLock, 'TX deskew LOL', 'no lock',
                          'ok' ),
                AttrDesc( self.txFifoErr, 'TX FIFO error', 'error', 'ok' ),
                AttrDesc( self.rxDemodulationLossOfLock, 'RX demodulator LOL',
                          'no lock', 'ok' ),
                AttrDesc( self.rxChrDispCompensationLossOfLock,
                          'RX CD compensation LOL', 'no lock', 'ok' ),
                AttrDesc( self.rxLossOfAlignment, 'RX loss of alignment',
                          'unaligned', 'ok' ),
                AttrDesc( self.rxOutOfAlignment, 'RX out of alignment',
                          'unaligned', 'ok' ),
                AttrDesc( self.rxDeskewLossOfLock, 'RX deskew LOL', 'no lock',
                          'ok' ),
                AttrDesc( self.rxFifoErr, 'RX FIFO error', 'error', 'ok' ),
              ]
      for attrDesc in attrs:
         fmtDescriptive( attrDesc )

      # Print Coherent FEC Degrade attributes
      self.renderCoherentDegradeModel()

   def renderCoherentDegradeModel( self ):
      # Print C-CMIS FEC Degrade attributes
      coherentFecDegradeEnabled = \
         Toggles.XcvrToggleLib.toggleCoherentFecDegradeEnabled()
      if coherentFecDegradeEnabled:
         fecDegradeHeaders = [ 'RX FEC detected degrade',
                               'RX FEC excessive degrade' ]
         fecDegradeAdminAttrs = [
            AttrDesc( self.fecDetectedDegradeEnabled, 'Monitoring',
                      'enabled', 'disabled' ),
            AttrDesc( self.fecExcessiveDegradeEnabled, 'Monitoring',
                      'enabled', 'disabled' ),
        ]
         fecDegradeAlarmAttrs = [
            AttrDesc( self.fecDetectedDegrade, 'Alarm state',
                      'alarm', 'ok' ),
            AttrDesc( self.fecExcessiveDegrade, 'Alarm state',
                      'alarm', 'ok' ),
         ]

         fmtFecDegradeStatus( fecDegradeHeaders,
                              fecDegradeAdminAttrs,
                              fecDegradeAlarmAttrs )
      else:
         fecDegradeAttrs = [
            AttrDesc( self.fecExcessiveDegrade, 'RX FEC excessive degrade',
                      'alarm', 'ok' ),
            AttrDesc( self.fecDetectedDegrade, 'RX FEC detected degrade',
                      'alarm', 'ok' ),
         ]
         for attrDesc in fecDegradeAttrs:
            fmtDescriptive( attrDesc )

class InterfacesTransceiverStatusChannelSpecific( Model ):
   rxLos = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                     help="Receive loss of signal", optional=True )
   txFault = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                       help="Transmit fault", optional=True )
   rxCdrLol = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                        help="Receive clock and data recovery loss of lock",
                        optional=True )
   txPowerHiAlarm = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                              help="Transmit high power alarm", optional=True )
   txPowerLoAlarm = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                              help="Transmit low power alarm", optional=True )
   txPowerHiWarn = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                             help="Transmit high power warning", optional=True )
   txPowerLoWarn = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                             help="Transmit low power warning", optional=True )
   txBiasHiAlarm = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                             help="Transmit high bias alarm", optional=True )
   txBiasLoAlarm = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                             help="Transmit low bias alarm", optional=True )
   txBiasHiWarn = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                            help="Transmit high bias warning", optional=True )
   txBiasLoWarn = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                            help="Transmit low bias warning", optional=True )
   rxPowerHiAlarm = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                              help="Receive high power alarm", optional=True )
   rxPowerLoAlarm = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                              help="Receive low power alarm", optional=True )
   rxPowerHiWarn = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                             help="Receive high power warning", optional=True )
   rxPowerLoWarn = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                             help="Receive low power warning", optional=True )
   txOutputSignalStats = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                             help="Transmit output signal status",
                                   optional=True )
   coherentAlarms = Submodel(
      valueType=InterfacesTransceiverStatusCoherentChannelSpecific,
      help="Coherent media channel specific alarms",
      optional=True )
   frequencyTuningAlarms = Submodel(
         valueType=InterfacesTransceiverStatusFrequencyTuningChannelSpecific,
         help="Frequency tuning media channel specific CMIS alarms",
         optional=True )

   def renderModel( self ):
      attrs = [ AttrDesc( self.rxLos, "RX LOS", 'no signal', 'ok' ),
                AttrDesc( self.txFault, "TX fault", 'fault', 'ok' ),
                AttrDesc( self.rxCdrLol, "RX CDR LOL", 'no lock', 'ok' ),
                AttrDesc( self.txPowerHiAlarm, "TX power high alarm", 'alarm',
                          'ok' ),
                AttrDesc( self.txPowerHiWarn, "TX power high warn", 'warn', 'ok' ),
                AttrDesc( self.txPowerLoAlarm,
            "TX power low alarm", 'alarm', 'ok' ),
                AttrDesc( self.txPowerLoWarn, "TX power low warn", 'warn', 'ok' ),
                AttrDesc( self.txBiasHiAlarm, "TX bias high alarm", 'alarm', 'ok' ),
                AttrDesc( self.txBiasHiWarn, "TX bias high warn", 'warn', 'ok' ),
                AttrDesc( self.txBiasLoAlarm, "TX bias low alarm", 'alarm', 'ok' ),
                AttrDesc( self.txBiasLoWarn, "TX bias low warn", 'warn', 'ok' ),
                AttrDesc( self.rxPowerHiAlarm, "RX power high alarm", 'alarm',
                          'ok' ),
                AttrDesc( self.rxPowerHiWarn, "RX power high warn", 'warn', 'ok' ),
                AttrDesc( self.rxPowerLoAlarm,
            "RX power low alarm", 'alarm', 'ok' ),
                AttrDesc( self.rxPowerLoWarn, "RX power low warn", 'warn', 'ok' ),
                AttrDesc( self.txOutputSignalStats, "TX output signal status",
                          'ok', 'not ok' ),
              ]
      for attrDesc in attrs:
         fmtDescriptive( attrDesc )

      if self.coherentAlarms is not None:
         self.coherentAlarms.renderModel()

      if self.frequencyTuningAlarms is not None:
         self.frequencyTuningAlarms.renderModel()

class InterfacesTransceiverStatusHostLaneSpecific( Model ):

   preFecBERCurrHost = Submodel(
         valueType=InterfacesTransceiverStatusValueFormatScientificFloat,
         help="Current value of host-side pre-FEC bit error rate", optional=True )
   errFramesCurHost = Submodel(
         valueType=InterfacesTransceiverStatusValueFormatScientificFloat,
         help="Current value of host-side post-FEC errored frames ratio",
         optional=True )

   txLos = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                     help="Transmit loss of signal", optional=True )
   txCdrLol = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                        help="Transmit clock and data recovery loss of lock",
                        optional=True )
   txAdaptiveInputEqFault = Submodel(
         valueType=InterfacesTransceiverStatusValueFormat,
         help="Transmit adaptive input equalization fault", optional=True )
   rxOutputSignalStats = Submodel(
      valueType=InterfacesTransceiverStatusValueFormat,
      help="Receive output signal status", optional=True )

   def renderModel( self ):
      fmtGeneric( self.preFecBERCurrHost, "Pre-FEC bit error rate" )
      fmtGeneric( self.errFramesCurHost, "Post-FEC errored frames ratio" )
      attrs = [ AttrDesc( self.txLos, "TX LOS", 'no signal', 'ok' ),
                AttrDesc( self.txCdrLol, "TX CDR LOL", 'no lock', 'ok' ),
                AttrDesc( self.txAdaptiveInputEqFault, "TX adaptive input EQ fault",
                          'fault', 'ok' ),
                AttrDesc( self.rxOutputSignalStats, "RX output signal status",
                            'ok', 'not ok' )
              ]
      for attrDesc in attrs:
         fmtDescriptive( attrDesc )

class InterfacesTransceiverStatusInterfaceSpecific( Model ):
   interface = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                         help="Interface name", optional=True )
   operSpeed = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                         help="Operational speed of the interface", optional=True )
   channels = Dict( valueType=InterfacesTransceiverStatusChannelSpecific,
                    keyType=int, help="Channel specific attributes indexed by "
                                      "channel number", optional=True )
   cfp2Dco = Submodel( valueType=InterfacesTransceiverStatusExtCfp2DcoIntf,
                       help="CFP2-DCO specific attributes", optional=True )
   hostLanes = Dict( valueType=InterfacesTransceiverStatusHostLaneSpecific,
                    keyType=int, help="Host lane specific attributes indexed by "
                                      "lane number", optional=True )

   def renderModel( self ):
      # For CMIS coherent modules ( e.g. ZR ) we want to display attributes in a
      # different order. Because these module only have one media lane, the channel
      # specific attributes are identical for all interfaces. So we chose to display
      # them before the interface label.
      if self.hasCoherentAttributes():
         # If we have a ZR, all that is left to display is the
         # host lane attributes.
         self._renderCmisCoherentInterfaceAttrs()
         return

      # If this is any other module, display: interface label, speed, channel
      # attributes, and host lane attributes
      print( self.interface.state )
      # pylint: disable-msg=protected-access
      if self.cfp2Dco and self.cfp2Dco._dualLaserModulePresent:
         fmt = "  %-42.42s %-20.20s %7.7s       %s"
         fmtGeneric( self.operSpeed, "Operational speed", fmt=fmt )
         fmtGeneric( self.cfp2Dco.tributary, "Tributary", fmt=fmt )
      else:
         fmtGeneric( self.operSpeed, "Operational speed" )

      # Print CMIS4.0 VDM Host parameters
      if self.hasEnhancedDomHostAttributes( self.hostLanes ):
         renderEnhancedDomAttrs( self.hostLanes )

      renderChAttr( self.channels )
      renderHostLaneAttr( self.hostLanes )
      if self.cfp2Dco:
         self.cfp2Dco.renderExt()

   def _renderCmisCoherentInterfaceAttrs( self ):
      '''
      The only interface specific lines that we want to display for CMIS coherent
      modules are:
      - the interface label
      - the operational speed
      - all the host lane attributes
      '''
      print( self.interface.state )
      fmtGeneric( self.operSpeed, "Operational speed" )
      renderEnhancedDomAttrs( self.hostLanes )
      renderHostLaneAttr( self.hostLanes )

   def hasCoherentAttributes( self ):
      # Whether we are in 4x100G or 1x400G ZR application, there will only be one
      # media lane.
      mediaLane = 1
      return ( mediaLane in self.channels and
               ( self.channels[ mediaLane ].coherentAlarms is not None or
                 self.channels[ mediaLane ].frequencyTuningAlarms is not None ) )

   def hasEnhancedDomHostAttributes( self, hostLanesDict ):
      enhancedDomHostParams = [ "preFecBERCurrHost",
                                "errFramesCurHost" ]
      for param in enhancedDomHostParams:
         if any( getattr( i, param, None ) is not None for i in
                   hostLanesDict.values() ):
            return True
      return False

class InterfacesTransceiverStatusValues( Model ):
   mediaType = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                           help="Transceiver media type" )
   serialNumber = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                            help="Transceiver serial number" )
   if Toggles.XcvrToggleLib.toggleXcvrLpoEnabled():
      bandwidth = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                        help="Signal bandwidth of linear pluggable optics",
                        optional=True )
   presence = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                        help="Indication of transceiver presence", optional=True )
   eepromReadTimeouts = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                                  help="Number of Eeprom read timeouts",
                                  optional=True )
   badEepromChecksums = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                                  help="Number of bad Eeprom checksums",
                                  optional=True )
   spuriousDet = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                               help="Number of spurious detections of the "
                                    "transceiver", optional=True )
   domControlFail = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                              help="Number of DOM control/status failures",
                              optional=True )
   resets = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                      help="Number of transceiver resets", optional=True )
   interrupts = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                          help="Number of transceiver interrupts", optional=True )
   smbusFailures = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                             help="Number of transceiver system managements bus "
                                  "failures", optional=True )
   mdioFailures = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                            help="Number of transceiver management data "
                                 "input/output failures", optional=True )
   interfaces = Dict( valueType=InterfacesTransceiverStatusInterfaceSpecific,
                      keyType=str, help="Interface specific attributes indexed "
                                        "by interface name", optional=True )
   channels = Dict( valueType=InterfacesTransceiverStatusChannelSpecific,
                    keyType=int, help="Channel specific attributes indexed by "
                                      "channel number", optional=True )
   cfp2Dco = Submodel( valueType=InterfacesTransceiverStatusExtCfp2Dco,
                       help="CFP2-DCO specific attributes", optional=True )
   adapters = Submodel( valueType=InterfacesTransceiverStatuValuesSwizzlerList,
                        help="Form factor adapter types in use", optional=True )
   dataPathFirmwareFault = Submodel(
                           valueType=InterfacesTransceiverStatusValueFormat,
                           help="Data path firmware fault", optional=True )
   moduleFirmwareFault = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                                   help="Module firmware fault", optional=True )
   tempHiAlarm = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                           help="High temperature alarm", optional=True )
   tempLoAlarm = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                           help="Low temperature alarm", optional=True )
   tempHiWarn = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                          help="High temperature warn", optional=True )
   tempLoWarn = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                          help="Low temperature warn", optional=True )
   voltageHiAlarm = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                              help="High voltage alarm", optional=True )
   voltageLoAlarm = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                              help="Low voltage alarm", optional=True )
   voltageHiWarn = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                             help="High voltage alarm", optional=True )
   voltageLoWarn = Submodel( valueType=InterfacesTransceiverStatusValueFormat,
                             help="Low voltage alarm", optional=True )
   moduleState = Submodel( valueType=InterfacesTransceiverStatusValueFormatStr,
                           help="Module state", optional=True )
   dataPathState = Dict( valueType=InterfacesTransceiverStatusValueFormatStr,
                         keyType=int, help="Data path state indexed by "
                                           "host lane number", optional=True )
   hostLanes = Dict( valueType=InterfacesTransceiverStatusHostLaneSpecific,
                    keyType=int, help="Host lane specific attributes indexed by "
                                      "lane number", optional=True )

   def setMediaType( self, xcvrStatus ):
      """
      Parameters
      ----------
      xcvrStatus : Xcvr::XcvrNewStatus
         The XcvrNewStatus belonging to the inserted transceiver
      """
      self.mediaType = InterfacesTransceiverStatusValueFormatStr(
         state=xcvrStatus.mediaTypeString,
         changes=xcvrStatus.presenceChanges,
         lastChange=xcvrStatus.lastPresenceChange )

   def setSerialNumber( self, xcvrStatus ):
      """
      Parameters
      ----------
      xcvrStatus : Xcvr::XcvrNewStatus
         The XcvrNewStatus belonging to the inserted transceiver
      """
      sn = xcvrStatus.vendorInfo.vendorSn
      self.serialNumber = InterfacesTransceiverStatusValueFormatStr(
         state=( "" if not sn else
                 codecs.decode( sn.strip().encode( 'unicode_escape' ), 'UTF-8' ) ) )

   def setBandwidth( self, xcvrStatus ):
      """
      Parameters
      ----------
      xcvrStatus : Xcvr::XcvrNewStatus
         The XcvrNewStatus belonging to the inserted transceiver
      """
      signalBwConfig = None
      if xcvrStatus and xcvrStatus.isLpo:
         signalBwConfig = lpoTypeToStr[ xcvrStatus.lpoType ]
      self.bandwidth = InterfacesTransceiverStatusValueFormatStr(
         state=signalBwConfig )

   def setPresence( self, xcvrStatus ):
      """
      Parameters
      ----------
      xcvrStatus : Xcvr::XcvrNewStatus
         The XcvrNewStatus belonging to the inserted transceiver
      """
      self.presence = InterfacesTransceiverStatusValueFormatStr(
         state=xcvrPresenceToStr[ xcvrStatus.presence ] )

   def setBadEepromChecksums( self, xcvrStatus ):
      """
      Parameters
      ----------
      xcvrStatus : Xcvr::XcvrNewStatus
         The XcvrNewStatus belonging to the inserted transceiver
      """
      self.badEepromChecksums = InterfacesTransceiverStatusValueFormat(
         changes=xcvrStatus.xcvrStats.badEepromChecksumEvents,
         lastChange=xcvrStatus.xcvrStats.lastBadEepromChecksumEvent )

   def setResets( self, xcvrStatus ):
      """
      Parameters
      ----------
      xcvrStatus : Xcvr::XcvrNewStatus
         The XcvrNewStatus belonging to the inserted transceiver
      """
      self.resets = InterfacesTransceiverStatusValueFormat(
         changes=xcvrStatus.xcvrStats.resetCount,
         lastChange=xcvrStatus.xcvrStats.lastReset )

   def setInterrupts( self, xcvrStatus ):
      """
      Parameters
      ----------
      xcvrStatus : Xcvr::XcvrNewStatus
         The XcvrNewStatus belonging to the inserted transceiver
      """
      self.interrupts = InterfacesTransceiverStatusValueFormat(
         changes=xcvrStatus.xcvrStats.interruptCount,
         lastChange=xcvrStatus.xcvrStats.lastInterrupt )

   def setSmbusFailures( self, xcvrStatus ):
      """
      Parameters
      ----------
      xcvrStatus : Xcvr::XcvrNewStatus
         The XcvrNewStatus belonging to the inserted transceiver
      """
      if xcvrStatus.smbusFailCountValid:
         self.smbusFailures = InterfacesTransceiverStatusValueFormat(
            changes=xcvrStatus.xcvrStats.smbusFailCount,
            lastChange=xcvrStatus.xcvrStats.lastSmbusFail )

   def setAdapters( self, xcvrStatus ):
      """
      Parameters
      ----------
      xcvrStatus : Xcvr::XcvrNewStatus
         The XcvrNewStatus belonging to the slot.  Otherwise known as the
         base XcvrStatus or parent XcvrStatus.
      """
      swizzlers = []
      curXcvrStatus = xcvrStatus
      if isOsfpToQsfpSwizzler( curXcvrStatus ):
         swizzlers.append( 'oqa' )
         curXcvrStatus = curXcvrStatus.qsfpStatus
      if( isQsfpDdToQsfpSwizzler( curXcvrStatus ) or
          isQsfpCmisToQsfpAdapter( curXcvrStatus ) ):
         curXcvrStatus = curXcvrStatus.qsfpStatus
      if isQsfpToSfpSwizzler( curXcvrStatus ):
         if getXcvrStatus( xcvrStatus ).qsa28:
            swizzlers.append( 'qsa28' )
         else:
            swizzlers.append( 'qsa10' )
      self.adapters = InterfacesTransceiverStatuValuesSwizzlerList(
         state=swizzlers )

   def setDataPathFirmwareFault( self, xcvrStatus ):
      """
      Parameters
      ----------
      xcvrStatus : Xcvr::XcvrNewStatus
         The XcvrNewStatus belonging to the inserted transceiver
      """
      assert isCmisTransceiver( xcvrStatus )
      cmisModuleStats = xcvrStatus.cmisModuleStatsEnt
      if cmisModuleStats is None:
         return
      self.dataPathFirmwareFault = InterfacesTransceiverStatusValueFormatBool(
         state=bool( cmisModuleStats.dataPathFirmwareFault.current ),
         changes=cmisModuleStats.dataPathFirmwareFault.changes,
         lastChange=cmisModuleStats.dataPathFirmwareFault.lastChange )

   def setModuleFirmwareFault( self, xcvrStatus ):
      """
      Parameters
      ----------
      xcvrStatus : Xcvr::XcvrNewStatus
         The XcvrNewStatus belonging to the inserted transceiver
      """
      assert isCmisTransceiver( xcvrStatus )
      cmisModuleStats = xcvrStatus.cmisModuleStatsEnt
      if cmisModuleStats is None:
         return
      self.moduleFirmwareFault = InterfacesTransceiverStatusValueFormatBool(
         state=bool( cmisModuleStats.moduleFirmwareFault.current ),
         changes=cmisModuleStats.moduleFirmwareFault.changes,
         lastChange=cmisModuleStats.moduleFirmwareFault.lastChange )

   def setModuleState( self, xcvrStatus ):
      """
      Parameters
      ----------
      xcvrStatus : Xcvr::XcvrNewStatus
         The XcvrNewStatus belonging to the inserted transceiver
      """
      assert isCmisTransceiver( xcvrStatus )
      cmisModuleStats = xcvrStatus.cmisModuleStatsEnt
      if cmisModuleStats is None:
         return
      self.moduleState = InterfacesTransceiverStatusValueFormatStr(
         state=moduleStateToStr[ xcvrStatus.moduleState ],
         changes=cmisModuleStats.moduleStateChanged.changes,
         lastChange=cmisModuleStats.moduleStateChanged.lastChange )

   def setLaneDataPathState( self, laneId, xcvrStatus ):
      """
      Parameters
      ----------
      laneId : int
         Integer representing lane number (indexed by 0)

      xcvrStatus : Xcvr::XcvrNewStatus
      """
      laneStats = xcvrStatus.laneStats.get( laneId )
      if laneStats is None:
         return
      dataPathStateChange = laneStats.dataPathStateChange
      self.setLaneDataPathStateFromRawData( laneId,
                                            xcvrStatus.dataPathState[ laneId ],
                                            dataPathStateChange.changes,
                                            dataPathStateChange.lastChange )

   def setLaneDataPathStateFromRawData( self, laneId, dataPathEnum,
                                        changes, lastChange ):
      """
      Parameters
      ----------
      laneId : int
         Integer representing lane number (indexed by 0)

      dataPathEnum : Xcvr::DataPathState

      changes : int
         Integer representing number of state changes

      lastChange : Tac::Seconds
         Last time state was updated
      """
      # Note: the model indexes lanes starting at 1
      self.dataPathState[ laneId + 1 ] = InterfacesTransceiverStatusValueFormatStr(
         state=dataPathStateToStr[ dataPathEnum ],
         changes=changes, lastChange=lastChange )

   def alarmWarnValueFactory( self, xcvrStatusEnt, attrName: str ) \
         -> InterfacesTransceiverStatusValueFormatBool:
      condition = getattr( xcvrStatusEnt, attrName )
      return InterfacesTransceiverStatusValueFormatBool(
         state=bool( condition.current ),
         changes=condition.changes,
         lastChange=condition.lastChange )

   def setAlarmsAndWarns( self, xcvrStatus, attrPrefix: str ) -> None:
      xcvrStatsEnt = xcvrStatus.xcvrStatsEnt
      if not xcvrStatsEnt:
         return

      postfixes = [ "HiAlarm", "LoAlarm", "HiWarn", "LoWarn" ]
      for postfix in postfixes:
         attr = attrPrefix + postfix
         setattr( self, attr, self.alarmWarnValueFactory( xcvrStatsEnt, attr ) )

   def setVoltageAlarmAndWarn( self, xcvrStatus ):
      """
      Parameters
      ----------
      xcvrStatus : Xcvr::XcvrNewStatus
         The XcvrNewStatus belonging to the inserted transceiver
      """
      self.setAlarmsAndWarns( xcvrStatus, "voltage" )

   def setTemperatureAlarmAndWarn( self, xcvrStatus ):
      """
      Parameters
      ----------
      xcvrStatus : Xcvr::XcvrNewStatus
         The XcvrNewStatus belonging to the inserted transceiver
      """
      self.setAlarmsAndWarns( xcvrStatus, "temp" )

   def renderModel( self ):
      if Toggles.XcvrToggleLib.toggleXcvrLpoEnabled():
         attrs = [ ( self.mediaType, "Transceiver" ),
                ( self.serialNumber, "Transceiver SN" ),
                ( self.bandwidth, "LPO signal bandwidth" ),
                ( self.presence, "Presence" ),
                ( self.adapters, "Adapters" ),
                ( self.eepromReadTimeouts, "Xcvr EEPROM read timeout" ),
                ( self.badEepromChecksums, "Bad EEPROM checksums" ),
                ( self.spuriousDet, "Spurious xcvr detection" ),
                ( self.domControlFail, "DOM control/status fail" ),
                ( self.resets, "Resets" ),
                ( self.interrupts, "Interrupts" ),
                ( self.smbusFailures, "Smbus failures" ),
                ( self.mdioFailures, "Mdio failures" ),
              ]
      else:
         attrs = [ ( self.mediaType, "Transceiver" ),
                  ( self.serialNumber, "Transceiver SN" ),
                  ( self.presence, "Presence" ),
                  ( self.adapters, "Adapters" ),
                  ( self.eepromReadTimeouts, "Xcvr EEPROM read timeout" ),
                  ( self.badEepromChecksums, "Bad EEPROM checksums" ),
                  ( self.spuriousDet, "Spurious xcvr detection" ),
                  ( self.domControlFail, "DOM control/status fail" ),
                  ( self.resets, "Resets" ),
                  ( self.interrupts, "Interrupts" ),
                  ( self.smbusFailures, "Smbus failures" ),
                  ( self.mdioFailures, "Mdio failures" ),
               ]
      attrDescs = [ AttrDesc( self.dataPathFirmwareFault,
                              "Data path firmware fault", 'fault', 'ok' ),
                    AttrDesc( self.moduleFirmwareFault, "Module firmware fault",
                              'fault', 'ok' ),
                    AttrDesc( self.tempHiAlarm, "Temperature high alarm",
                              'alarm', 'ok' ),
                    AttrDesc( self.tempHiWarn, "Temperature high warn",
                              'warn', 'ok' ),
                    AttrDesc( self.tempLoAlarm, "Temperature low alarm",
                              'alarm', 'ok' ),
                    AttrDesc( self.tempLoWarn, "Temperature low warn", 'warn',
                              'ok' ),
                    AttrDesc( self.voltageHiAlarm, "Voltage high alarm", 'alarm',
                              'ok' ),
                    AttrDesc( self.voltageHiWarn, "Voltage high warn", 'warn',
                              'ok' ),
                    AttrDesc( self.voltageLoAlarm, "Voltage low alarm", 'alarm',
                              'ok' ),
                    AttrDesc( self.voltageLoWarn, "Voltage low warn", 'warn',
                              'ok' ),
                  ]

      dualLaserFmt = "  %-42.42s %-20.20s %7.7s       %s"
      defaultFmt = "  %-29.29s %-20.20s %7.7s       %s"
      fmt = defaultFmt
      # pylint: disable-msg=protected-access
      if self.cfp2Dco and self.cfp2Dco._dualLaserModulePresent:
         fmt = dualLaserFmt
      for attr in attrs:
         fmtGeneric( *attr, fmt=fmt )
      for attrDesc in attrDescs:
         fmtDescriptive( attrDesc, fmt=fmt )
      fmtGeneric( self.moduleState, "Module state", fmt=fmt )

      if self.cfp2Dco:
         self.cfp2Dco.renderExt()

      for( hostLane,
           dathPathStateModel ) in sorted( self.dataPathState.items() ):
         dataPathStateAttrName = f"Data path {hostLane} state"
         fmtGeneric( dathPathStateModel, dataPathStateAttrName )

      intfNames = Arnet.sortIntf( self.interfaces )
      # Check if this module is CMIS coherent ( e.g. ZR ). And if so, we want to
      # display the channel attributes before any of the interface specific ones.
      # This is because a ZR only has one media lane ( channel ). So all interfaces
      # would otherwise duplicate the same information.
      for intf in intfNames:
         if self.interfaces[ intf ].hasCoherentAttributes():
            renderChAttr( self.interfaces[ intf ].channels )
            # Since we duplicate the same media channel data for all interfaces,
            # we only want to display everything once
            break
      for intf in intfNames:
         self.interfaces[ intf ].renderModel()

      # pylint: disable-msg=protected-access
      if self.cfp2Dco and self.cfp2Dco.dualLaserModulePresent():
         # Channel parameters are not populated by dual laser dco module
         return
      renderChAttr( self.channels, printChLabel=True )
      renderHostLaneAttr( self.hostLanes, printHostLaneLabel=True )

class InterfacesTransceiverStatus( Model ):
   ports = OrderedDict( keyType=str, valueType=InterfacesTransceiverStatusValues,
                 help='Mapping between port name and transceiver status' )

   def render( self ):
      if not self.ports:
         return

      print( f"Current System Time: { str( ctime( Tac.utcNow() ) ) }" )
      fmt = "%-31.31s %-20.20s %-13.13s %11.11s"
      for port in self.ports:
         if ( self.ports[ port ].cfp2Dco and
              self.ports[ port ].cfp2Dco.dualLaserModulePresent() ):
            fmt = "  %-42.42s %-20.20s %7.7s       %s"
         print( fmt % ( "", "Current State", "Changes", "Last Change" ) )
         print( fmt % ( "", "-------------", "-------", "-----------" ) )

         print( port )
         self.ports[ port ].renderModel()
         print()
