# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
from CliPlugin.XcvrShowTuningModel import (
      InterfacesTransceiverTuning, InterfaceTransceiverTuning,
      InterfaceTransceiverTuningParameter, InterfaceTransceiverTuningDetail,
      TUNING_TYPE_TO_PARAM )
import CliPlugin.XcvrAllStatusDir
import CliPlugin.IntfCli
import CliPlugin.EthIntfCli
import CliPlugin.EthIntfModel
from CliPlugin.XcvrConfigCli import getXcvrConfigCliDir
from CliPlugin.XcvrCliLib import getAllIntfsWrapper
from XcvrLib import ( getAndFilterPrintableIntfNameByLane,
                      getXcvrStatus, isCmisType )
import Tac
import Toggles
from TypeFuture import TacLazyType

XcvrMediaType = TacLazyType( 'Xcvr::MediaType' )
XcvrType = TacLazyType( 'Xcvr::XcvrType' )
TuningValue = Tac.Type( "Xcvr::TuningValue" )

xcvrStatusDir_ = None

# ----------------------------------------------------------------------------
#
# "show interface [ <interface> ] transceiver tuning [ detail ]"
#
# ----------------------------------------------------------------------------
autoStr = 'auto'

def _tuningExceptionSupported( xcvrType: str ) -> bool:
   return Toggles.XcvrToggleLib.toggleDefaultTuningExceptionEnabled() and \
          ( xcvrType == XcvrType.qsfpPlus or isCmisType( xcvrType ) )

def _getTuningSupported( status ):
   if isCmisType( status.typeInUse ):
      tuningCap = status.cmisTuningParamCapabilities.tuningParamImplemented
      txAdaEquSupp = tuningCap.txInputAdaptiveEqualizationImpl
      txEquSupp = tuningCap.txInputFixedEqualizationImpl
      rxPostEmpSupp = tuningCap.rxOutputPostEmphasisImpl
      rxPreEmpSupp = tuningCap.rxOutputPreEmphasisImpl
      rxAmpSupp = tuningCap.rxOutputAmplitudeImpl
   else:
      tuningCap = status.xcvrTuningParamCapabilities
      txAdaEquSupp = False
      txEquSupp = tuningCap.txInputEqualizationSupported
      rxPostEmpSupp = False
      rxPreEmpSupp = tuningCap.rxPreEmphasisSupported
      rxAmpSupp = tuningCap.rxAmplitudeSupported
   return txAdaEquSupp, txEquSupp, rxPostEmpSupp, rxPreEmpSupp, rxAmpSupp

def _getTuningOverride( status ):
   primaryLaneId = 0
   assert status.xcvrConfig.intfName
   primaryIntfName = status.xcvrConfig.intfName[ primaryLaneId ]
   xcvrConfigCliDir = getXcvrConfigCliDir( primaryIntfName )
   xcvrConfigCli = xcvrConfigCliDir.xcvrConfigCli.get( primaryIntfName )
   overrideEn = xcvrConfigCli and xcvrConfigCli.overrideTuningParams
   return overrideEn

def _showInterfacesXcvrTuning( mode, intf=None, mod=None ):
   ( intfs, intfNames ) = getAllIntfsWrapper( mode, intf, mod )
   model = InterfacesTransceiverTuning()
   model.detailIs( False )
   if not intfs:
      return model
   xcvrStatus = xcvrStatusDir_.xcvrStatus

   for xcvrName in Arnet.sortIntf( xcvrStatus ):
      status = getXcvrStatus( xcvrStatus[ xcvrName ] )

      if status.presence != "xcvrPresent":
         # Skip transceivers that aren't present
         continue

      txAdaEquSupp, txEquSupp, rxPostEmpSupp, rxPreEmpSupp, rxAmpSupp = \
         _getTuningSupported( status )
      overrideEn = _getTuningOverride( status )

      if not any( [ txAdaEquSupp, txEquSupp, rxPostEmpSupp, rxPreEmpSupp, rxAmpSupp,
                    overrideEn ] ):
         continue

      maxChannels = status.capabilities.maxChannels
      for laneId in range( maxChannels ):
         name = getAndFilterPrintableIntfNameByLane( laneId, status, maxChannels,
                                                     intfNames )
         if not name:
            continue

         # Create and populate a new submodel for this interface's tunings
         submodel = InterfaceTransceiverTuning()

         # Populate the slot and lane attributes
         submodel.slot = xcvrName
         submodel.lane = str( laneId + 1 )

         # Create the tuning parameter submodels
         submodel.rxOutputAmplitude = InterfaceTransceiverTuningParameter()
         submodel.rxOutputPreEmphasis = InterfaceTransceiverTuningParameter()
         submodel.rxOutputPostEmphasis = InterfaceTransceiverTuningParameter()
         submodel.txInputEqualization = InterfaceTransceiverTuningParameter()

         # Initialize supported flags to false
         submodel.rxOutputAmplitude.supported = False
         submodel.rxOutputPreEmphasis.supported = False
         submodel.rxOutputPostEmphasis.supported = False
         submodel.txInputEqualization.supported = False

         # Generate the output string for each tuning parameter
         if status.typeInUse == XcvrType.cfp2:
            submodel.rxOutputAmplitude.operational = None
         else:
            submodel.rxOutputAmplitude.operational = \
                  str( status.rxOutputAmplitude[ laneId ] )
            submodel.rxOutputAmplitude.supported = rxAmpSupp

         submodel.rxOutputPreEmphasis.operational = \
               str( status.rxOutputPreEmphasis[ laneId ] )
         submodel.rxOutputPreEmphasis.supported = rxPreEmpSupp

         if isCmisType( status.typeInUse ):
            submodel.rxOutputPostEmphasis.operational = \
                  str( status.rxOutputPostEmphasis[ laneId ] )
            submodel.rxOutputPostEmphasis.supported = rxPostEmpSupp
         else:
            submodel.rxOutputPostEmphasis.operational = None
         if ( isCmisType( status.typeInUse )
               and status.txInputAdaptiveEqEnable[ laneId ] ):
            submodel.txInputEqualization.operational = autoStr
            submodel.txInputEqualization.supported = txAdaEquSupp
         else:
            submodel.txInputEqualization.operational = \
                  str( status.txInputEqualization[ laneId ] )
            submodel.txInputEqualization.supported = txEquSupp

         # put the submodel into the dictionary for this interface
         model.interfaces[ name ] = submodel
   return model
# ----------------------------------------------------------------------------
#
# "show interface [ <interface> ] transceiver tuning detail"
#
# ----------------------------------------------------------------------------

def _getCmisTxInputEq( txAdaEquSupp, txEquSupp, status, xcvrConfigCli, laneId ):
   # Indicates whether the TX input equalization parameter is supported
   txInputEquSupported = False
   overrideEn = xcvrConfigCli and xcvrConfigCli.overrideTuningParams

   # Module Default Value
   moduleDefExists = laneId in status.defaultCmisTuningParamValue
   if moduleDefExists:
      if status.defaultCmisTuningParamValue[ laneId ].txInputAdaptiveEqEnable:
         moduleStr = autoStr
         txInputEquSupported = txAdaEquSupp
      else:
         moduleStr = str( status.defaultCmisTuningParamValue[
                             laneId ].txInputEqualization )
         txInputEquSupported = txEquSupp
   else:
      moduleStr = None

   slotDefExists = laneId in status.xcvrConfig.fdlTuningParam
   configExists = laneId in xcvrConfigCli.slotTuningParams\
                  if xcvrConfigCli else False
   configFixEq = xcvrConfigCli.slotTuningParams[ laneId ].txInputEqualization \
                 if configExists else None
   configAdaEq = xcvrConfigCli.slotTuningParams[ laneId ].txInputAdaptiveEqEnable \
                 if configExists else None

   exceptionAdaEq = None
   exceptionFixEq = None

   if ( _tuningExceptionSupported( status.typeInUse ) and
        status.isUserException ):
      exceptionAdaEq = status.txInputAdaptiveEqualizationException.get( laneId )
      exceptionFixEq = status.txInputEqualizationException.get( laneId )

   # Configure Value
   if txAdaEquSupp or txEquSupp or overrideEn:
      if ( txAdaEquSupp or overrideEn ) and configExists and configAdaEq.valid and \
         configAdaEq.val:
         configuredStr = autoStr
      elif ( txEquSupp or overrideEn ) and configExists and configFixEq.valid:
         configuredStr = str( configFixEq.val )
      elif exceptionAdaEq and exceptionAdaEq.moduleDefault:
         configuredStr = 'module default+'
      elif exceptionAdaEq and exceptionAdaEq.valid and exceptionAdaEq.val != 0:
         configuredStr = f"{autoStr}+"
      elif exceptionFixEq and exceptionFixEq.moduleDefault:
         configuredStr = 'module default+'
      elif exceptionFixEq and exceptionFixEq.valid:
         configuredStr = f"{exceptionFixEq.val}+"
      elif slotDefExists and ( status.xcvrConfig.fdlTuningParam[
                                  laneId ].txInputAdaptiveEqEnable.valid or
                               status.xcvrConfig.fdlTuningParam[
                                  laneId ].txInputEqualization.valid ):
         configuredStr = 'slot default'
      else:
         configuredStr = 'module default'
   else:
      configuredStr = 'module default'

   # Operational Value
   operationalAdaEqEnable = status.txInputAdaptiveEqEnable[ laneId ]
   operationalFixEq = status.txInputEqualization[ laneId ]
   if operationalAdaEqEnable:
      operationalStr = autoStr
   else:
      operationalStr = str( operationalFixEq )

   return moduleStr, configuredStr, operationalStr, txInputEquSupported

def _getTuningParam( tuningType, tuningParamSupp, status, xcvrConfigCli, laneId ):
   if tuningType == 'Rx Output Amplitude':
      defaultParamStr = 'defaultRxOutputAmplitude'
      paramStr = 'rxOutputAmplitude'
      exceptionStr = 'rxOutputAmplitudeException'
   elif tuningType == 'Rx Output Pre-emphasis':
      defaultParamStr = 'defaultRxOutputPreEmphasis'
      paramStr = 'rxOutputPreEmphasis'
      exceptionStr = 'rxOutputPreEmphasisException'
   elif tuningType == 'Rx Output Post-emphasis':
      defaultParamStr = ''
      paramStr = 'rxOutputPostEmphasis'
      exceptionStr = 'rxOutputPostEmphasisException'
   else:
      defaultParamStr = 'defaultTxInputEqualization'
      paramStr = 'txInputEqualization'
      exceptionStr = 'txInputEqualizationException'

   if status.typeInUse == XcvrType.cfp2 and paramStr == "rxOutputAmplitude":
      return None, None, None

   if not isCmisType( status.typeInUse ) and paramStr == "rxOutputPostEmphasis":
      return None, None, None

   overrideEn = xcvrConfigCli and xcvrConfigCli.overrideTuningParams

   moduleStr = None
   if isCmisType( status.typeInUse ):
      slotDefExists = laneId in status.xcvrConfig.fdlTuningParam
      slotDefault = getattr( status.xcvrConfig.fdlTuningParam[ laneId ],
                             paramStr ) if slotDefExists else None
      moduleDefExists = laneId in status.defaultCmisTuningParamValue
      if moduleDefExists:
         moduleStr = str( getattr( status.defaultCmisTuningParamValue[ laneId ],
                                   paramStr ) )
   else:
      slotDefExists = status.typeInUse == XcvrType.qsfpPlus and\
                      laneId in getattr( status.xcvrConfig, paramStr )
      slotDefault = getattr( status.xcvrConfig, paramStr ).get( laneId ) \
                    if slotDefExists else None
      moduleDefExists = status.typeInUse == XcvrType.qsfpPlus and\
                        laneId in getattr( status, defaultParamStr )
      if moduleDefExists:
         moduleStr = str( getattr( status, defaultParamStr )[ laneId ] )

   configExists = laneId in xcvrConfigCli.slotTuningParams\
                  if xcvrConfigCli else False
   configVal = getattr( xcvrConfigCli.slotTuningParams[ laneId ], paramStr ) \
               if configExists else None

   # exceptionVal is a Xcvr::TuningValue that we will get from the tuning exception
   # attributes in its xcvrStatus.
   exceptionVal = TuningValue( False, 0, False )
   if ( _tuningExceptionSupported( status.typeInUse ) and
        status.isUserException ):
      exceptionVal = getattr( status, exceptionStr ).get(
                        laneId,
                        TuningValue( False, 0, False ) )

   if tuningParamSupp or overrideEn:
      if configVal and configVal.valid:
         # CLI-configured value
         configuredStr = str( configVal.val )
      elif configVal and configVal.moduleDefault:
         # CLI-configured module-default
         configuredStr = "module default"
      elif exceptionVal and exceptionVal.valid:
         configuredStr = f"{exceptionVal.val}+"
      elif exceptionVal and exceptionVal.moduleDefault:
         configuredStr = "module default+"
      elif slotDefault and slotDefault.valid:
         # CLI-default results in slot default
         configuredStr = "slot default"
      else:
         # Fall back to module default when no slot default exists
         configuredStr = "module default"
   else:
      configuredStr = "module default"

   operationalStr = str( getattr( status, paramStr )[ laneId ] )
   return moduleStr, configuredStr, operationalStr

def _showInterfacesXcvrTuningDetail( mode, intf=None, mod=None ):
   ( intfs, intfNames ) = getAllIntfsWrapper( mode, intf, mod )
   model = InterfacesTransceiverTuning()
   model.detailIs( True )
   if not intfs:
      return model
   xcvrStatus = xcvrStatusDir_.xcvrStatus

   # Check if at least one xcvr support at least one tuning param or override its
   # tuning capabilities
   tuningSupported = False
   for xcvrName in Arnet.sortIntf( xcvrStatus ):
      status = getXcvrStatus( xcvrStatus[ xcvrName ] )
      if status.presence != "xcvrPresent":
         # Skip transceivers that aren't present
         continue

      # Nothing shows up if all tuning params on all interfaces are not supported.
      txAdaEquSupp, txEquSupp, rxPostEmpSupp, rxPreEmpSupp, rxAmpSupp = \
         _getTuningSupported( status )
      overrideEn = _getTuningOverride( status )

      if any( [ txAdaEquSupp, txEquSupp, rxPostEmpSupp, rxPreEmpSupp,
                rxAmpSupp, overrideEn ] ):
         tuningSupported = True
         break

   # None of the xcvrs support tuning
   if not tuningSupported:
      return model

   # At least one xcvr supports at least one tuning param
   for tuningType, paramStr in TUNING_TYPE_TO_PARAM.items():
      for xcvrName in Arnet.sortIntf( xcvrStatus ):
         status = getXcvrStatus( xcvrStatus[ xcvrName ] )
         if status.presence != "xcvrPresent":
            # Skip transceivers that aren't present
            continue

         assert status.xcvrConfig.intfName
         primaryLaneId = 0
         primaryIntfName = status.xcvrConfig.intfName[ primaryLaneId ]
         xcvrConfigCliDir = getXcvrConfigCliDir( primaryIntfName )
         xcvrConfigCli = xcvrConfigCliDir.xcvrConfigCli.get( primaryIntfName )
         overrideEn = xcvrConfigCli and xcvrConfigCli.overrideTuningParams

         txAdaEquSupp, txEquSupp, rxPostEmpSupp, rxPreEmpSupp, rxAmpSupp = \
            _getTuningSupported( status )

         if not any( [ txAdaEquSupp, txEquSupp, rxPostEmpSupp, rxPreEmpSupp,
                       rxAmpSupp, overrideEn ] ):
            continue

         if tuningType == 'Rx Output Amplitude':
            tuningParamSupp = rxAmpSupp
         elif tuningType == 'Rx Output Pre-emphasis':
            tuningParamSupp = rxPreEmpSupp
         elif tuningType == 'Rx Output Post-emphasis':
            tuningParamSupp = rxPostEmpSupp
         elif tuningType == 'Tx Input Equalization':
            tuningParamSupp = txEquSupp
         else:
            tuningParamSupp = False

         maxChannels = status.capabilities.maxChannels
         for laneId in range( maxChannels ):
            tuningParam = InterfaceTransceiverTuningParameter()
            tuningParam.detail = InterfaceTransceiverTuningDetail()
            name = getAndFilterPrintableIntfNameByLane( laneId, status, maxChannels,
                                                        intfNames )
            if not name:
               continue

            # Make sure there is an entry in the interface dictionary
            if name not in model.interfaces:
               interface = InterfaceTransceiverTuning()
               interface.slot = status.name
               interface.lane = str( laneId + 1 )
               model.interfaces[ name ] = interface

            if( isCmisType( status.typeInUse ) and
                tuningType == 'Tx Input Equalization' ):
               moduleStr, configuredStr, operationalStr, supported =\
                     _getCmisTxInputEq( txAdaEquSupp,
                  txEquSupp, status, xcvrConfigCli, laneId )
               tuningParam.supported = supported
            else:
               moduleStr, configuredStr, operationalStr = _getTuningParam(
                  tuningType, tuningParamSupp, status, xcvrConfigCli, laneId )
               tuningParam.supported = tuningParamSupp
            tuningParam.detail.moduleDefault = moduleStr
            tuningParam.detail.configured = configuredStr
            tuningParam.operational = operationalStr

            # Add the module name and the details submodel to the model
            interface = model.interfaces[ name ]
            setattr( interface, paramStr, tuningParam )
   return model

def showInterfacesXcvrTuning( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   if 'detail' in args:
      return _showInterfacesXcvrTuningDetail( mode, intf, mod )
   else:
      return _showInterfacesXcvrTuning( mode, intf, mod )

# ------------------------------------------------------
# Plugin method
# ------------------------------------------------------
def Plugin( em ):
   global xcvrStatusDir_
   xcvrStatusDir_ = CliPlugin.XcvrAllStatusDir.xcvrAllStatusDir( em )
