# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
from CliModel import ( Bool, Dict, Model, Str, Submodel )
from IntfModels import Interface
import TableOutput
from collections import OrderedDict

# Ordered dictionary between tuning type string and its param string
TUNING_TYPE_TO_PARAM = OrderedDict( [
   ( "Rx Output Amplitude", "rxOutputAmplitude" ),
   ( "Rx Output Pre-emphasis", "rxOutputPreEmphasis" ),
   ( "Rx Output Post-emphasis", "rxOutputPostEmphasis" ),
   ( "Tx Input Equalization", "txInputEqualization" ) ] )

# --------------------------------------------------------------------------------
#
# Models for "show interfaces transceiver tuning [detail]"
#
# --------------------------------------------------------------------------------

class InterfaceTransceiverTuningDetail( Model ):
   moduleDefault = Str( optional=True,
         help="The default tuning value of the module." )
   configured = Str( optional=True,
         help="The user configured tuning value." )

class InterfaceTransceiverTuningParameter( Model ):
   detail = Submodel( optional=True,
         help="Detail output of module default and configured values",
         valueType=InterfaceTransceiverTuningDetail )
   operational = Str( optional=True,
         help="The tuning value currently programmed into the module" )
   supported = Bool( help="This tuning parameter is supported" )

   def renderDetail( self, table, slot, lane ):
      # Fill in n/a for missing string values and add a "*" to unsupported parameters
      moduleDefaultStr = self.detail.moduleDefault
      if not moduleDefaultStr:
         moduleDefaultStr = "n/a"
      elif not self.supported:
         moduleDefaultStr += "*"

      configuredStr = self.detail.configured if self.detail.configured else "n/a"

      operationalStr = self.operational
      if not operationalStr:
         operationalStr = "n/a"
      elif not self.supported:
         operationalStr += "*"

      table.newRow( slot, lane, moduleDefaultStr, configuredStr, operationalStr )

class InterfaceTransceiverTuning( Model ):
   slot = Str( help="Transceiver slot number" )
   lane = Str( help="Transceiver host electrical lane number" )
   rxOutputAmplitude = Submodel( valueType=InterfaceTransceiverTuningParameter,
         help="RX Output Amplitude" )
   rxOutputPreEmphasis = Submodel( valueType=InterfaceTransceiverTuningParameter,
         help="RX Output Pre-emphasis" )
   rxOutputPostEmphasis = Submodel( valueType=InterfaceTransceiverTuningParameter,
         help="RX Output Post-emphasis" )
   txInputEqualization = Submodel( valueType=InterfaceTransceiverTuningParameter,
         help="TX Input Equalization" )

   def renderBasic( self, table ):
      rowOutput = [ self.slot, self.lane ]

      # Fill in n/a for missing string values and add a "*" to unsupported parameters
      for _, parameterName in TUNING_TYPE_TO_PARAM.items():
         param = getattr( self, parameterName )
         paramStr = param.operational
         if not paramStr:
            paramStr = "n/a"
         elif not param.supported:
            paramStr += "*"
         rowOutput.append( paramStr )

      table.newRow( *rowOutput )

class InterfacesTransceiverTuning( Model ):
   interfaces = Dict( optional=True, keyType=Interface,
         valueType=InterfaceTransceiverTuning,
         help="Mapping between interface name and tuning info" )

   # This is set to true for detailed format, otherwise false
   _detail = Bool( help="Indicates that the tuning detail data is present." )

   def detailIs( self, val ):
      self._detail = val

   def _printStarHelp( self, detailed=False ):
      print( "* = Unsupported by module, value may be inaccurate" )
      if detailed:
         print( "+ = From tuning file" )

   def _createTuningTable( self ):
      headers = ( "Slot", "Lane", "Amplitude", "Pre-emphasis", "Post-emphasis",
                  "Equalization" )
      formats = ( TableOutput.Format( justify="left", minWidth=13 ),
                  TableOutput.Format( justify="right", minWidth=4 ),
                  TableOutput.Format( justify="right", minWidth=9 ),
                  TableOutput.Format( justify="right", minWidth=12 ),
                  TableOutput.Format( justify="right", minWidth=13 ),
                  TableOutput.Format( justify="right", minWidth=12 ), )

      for fmt in formats:
         fmt.noPadLeftIs( True )
         fmt.padLimitIs( True )

      table = TableOutput.createTable( headers )
      table.formatColumns( *formats )
      return table

   def _createTuningParamTable( self ):
      headers = ( "Slot", "Lane", "Module Default", "Configured", "Operational" )
      formats = ( TableOutput.Format( justify="left", minWidth=13 ),
                  TableOutput.Format( justify="right", minWidth=4 ),
                  TableOutput.Format( justify="right", minWidth=14 ),
                  TableOutput.Format( justify="right", minWidth=14 ),
                  TableOutput.Format( justify="right", minWidth=11 ), )
      for fmt in formats:
         fmt.noPadLeftIs( True )
         fmt.padLimitIs( True )

      table = TableOutput.createTable( headers )
      table.formatColumns( *formats )
      return table

   def renderNoDetail( self ):
      table = self._createTuningTable()
      for intf in Arnet.sortIntf( self.interfaces ):
         self.interfaces[ intf ].renderBasic( table )
      print( table.output(), end="" )

   def renderDetail( self ):
      # Prints table for each tuning param
      for printName, attrName in TUNING_TYPE_TO_PARAM.items():
         table = self._createTuningParamTable()
         for i in Arnet.sortIntf( self.interfaces ):
            intf = self.interfaces[ i ]
            getattr( intf, attrName ).renderDetail( table, intf.slot, intf.lane )
         print( printName )
         print( table.output(), end="" )

   def render( self ):
      if not self.interfaces:
         return
      self._printStarHelp( self._detail )
      if not self._detail:
         self.renderNoDetail()
      else:
         self.renderDetail()
