# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import BasicCli
import LazyMount
import CliParser
import CliMatcher
import CliCommand
import ShowCommand
import Toggles.XcvrToggleLib
import CliPlugin.XcvrAllStatusDir
from CliPlugin.IntfCli import Intf
from CliPlugin.EthIntfCli import xcvrKw, xcvrShowKw
from CliPlugin.XcvrConfigCli import ( XcvrConfigModelet, matcherRx )
from XcvrLib import getXcvrStatus, isPrimaryIntf
from TypeFuture import TacLazyType
import CliGlobal

gv = CliGlobal.CliGlobal( xcvrStatusDir=None, xcvrConfigDir=None )

XcvrType = TacLazyType( 'Xcvr::XcvrType' )

rxAutoSquelchEnabled = Toggles.XcvrToggleLib.toggleRxAutoSquelchEnabled()

# --------------------------------------------------------------------------------
# [ no | default ] transceiver receiver squelch auto [ disabled ]
# --------------------------------------------------------------------------------

def rxSquelchGuard( mode, token ):
   xcvrName = gv.xcvrConfigDir.intfToXcvrName.get( mode.intf.name )
   if not xcvrName:
      return None

   xcvrStatus = getXcvrStatus( gv.xcvrStatusDir.xcvrStatus.get( xcvrName ) )
   if xcvrStatus.typeInUse not in [ XcvrType.osfp, XcvrType.qsfpDd ]:
      # Guard this token for non-OSFP and non-QSFP-DD ports
      return CliParser.guardNotThisXcvrSlotType

   if not isPrimaryIntf( mode.intf.name ):
      # Only primary interfaces can configure Rx auto squelch
      return CliParser.guardNotThisInterface

   # Don't use guard to check for inserted xcvr or xcvr features

   return None

matcherRxSquelch = CliMatcher.KeywordMatcher( 'squelch',
      helpdesc='Transceiver RX squelch settings' )
nodeRxSquelchGuarded = CliCommand.Node( matcher=matcherRxSquelch,
                                          guard=rxSquelchGuard )
matcherRxAuto = CliMatcher.KeywordMatcher( 'auto',
      helpdesc='Transceiver RX auto squelching' )
matcherRxAutoDisabled = CliMatcher.KeywordMatcher( 'disabled',
      helpdesc='Disable transceiver RX auto squelching' )

class TransceiverReceiverSquelchAutoCmd( CliCommand.CliCommandClass ):
   syntax = 'transceiver receiver squelch_GUARDED auto [ disabled ]'
   noOrDefaultSyntax = 'transceiver receiver squelch auto...'
   data = {
      'transceiver': xcvrKw,
      'receiver': matcherRx,
      'squelch_GUARDED': nodeRxSquelchGuarded,
      'squelch': matcherRxSquelch,
      'auto': matcherRxAuto,
      'disabled': matcherRxAutoDisabled
   }
   # the guard function is dynamic (depends on inserted transceivers)
   # so we need to disable cache
   allowCache = False

   handler = (
      "XcvrSquelchCliHandler.transceiverRxSquelchAutoCmdHandler" )
   noOrDefaultHandler = (
      "XcvrSquelchCliHandler.transceiverRxSquelchAutoCmdHandler" )

if rxAutoSquelchEnabled:
   XcvrConfigModelet.addCommandClass( TransceiverReceiverSquelchAutoCmd )

# --------------------------------------------------------------------------------
# show transceiver squelch [ interface INTF ]
# --------------------------------------------------------------------------------

class TransceiverSquelchCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show transceiver squelch [ interface INTFS ]'
   data = {
      'transceiver': xcvrShowKw,
      'squelch': 'Show transceiver squelch information',
      'interface': 'Show transceiver squelch for a specific interface',
      'INTFS': Intf.rangeMatcher,
   }
   handler = "XcvrSquelchCliHandler.showXcvrSquelchInterfaces"
   cliModel = "XcvrSquelchModel.TransceiverSquelchSlots"

if rxAutoSquelchEnabled:
   BasicCli.addShowCommandClass( TransceiverSquelchCmd )

# ------------------------------------------------------
# Plugin method
# ------------------------------------------------------
def Plugin( em ):
   gv.xcvrStatusDir = CliPlugin.XcvrAllStatusDir.xcvrAllStatusDir( em )
   gv.xcvrConfigDir = LazyMount.mount( em, "hardware/xcvr/config/all",
                                    "Xcvr::AllConfigDir", "r" )
