# Copyright (c) 2006-2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import itertools
from collections import namedtuple

from CliExtensions import CliHook
from CliSaveBlock import SaveBlockModelRenderer
from CliSaveBlock import sanitizedString
from CliSaveBlock import ( ShowRunningConfigOptions,
                           ShowRunningConfigRenderOptions )
from CliSaveMode import Mode, GlobalConfigMode # pylint: disable=unused-import
import CliSession as CS
import ConfigMount
import Plugins
import Tac
import Tracing

PICKLE_PROTO = 2

th = Tracing.defaultTraceHandle()
trace = th.trace0
traceTopSort = th.trace1
traceDetail = th.trace2

configCounter_ = Tac.singleton( "Tac::ConfigCounter" )

runningConfigCacheEnabledHook = CliHook()

# Maps pathPrefix to a dict, which maps typename to a list of functions to be called
# to save each entity of type typename in the subtree rooted at pathPrefix.
saveHandlers_ = {}
# for secure CliSave
secureMonitorSaveHandlers_ = {}
# for command-tag CliSave
commandTagSaveHandlers_ = {}

# header hooks (only 2 entries for now)
headerHooks_ = [ None ] * 2

# Type to store handler related information.
TypeHandlerInfo = namedtuple( 'TypeHandlerInfo',
                              'pathPrefix attrName requireMounts hasDefaultConfig' )

RootSaveBlock = namedtuple( 'RootSaveBlock', [ 'saveBlock', 'configCounter' ] )

def saver( typename, pathPrefix, attrName='', requireMounts=(),
      secureMonitor=False, commandTagSupported=False, hasDefaultConfig=False ):
   """A decorator that marks a function as being a CliSave function
   for entities of type 'typename' in the subtree rooted at the entity
   whose name (relative to the Sysdb root) is 'pathPrefix'.  If
   'attrName' is specified, only entities within the subtrees rooted
   at the collection attribute with this name are considered.
   'attrName' must be an instantiating collection attribute containing
   Entity pointers.

   A typical use might be like this:

     @CliSave.saver( 'Ip::VrfConfig', 'ip/config' )
     def saveVrfConfig( entity, root ):
        ...

   or like this:

      @CliSave.saver( 'Interface::EthIntfConfig', 'interface/config/ethernet',
                      attrName='ethIntfConfig' )
      def saveEthIntfConfig( entity, root ):
         ...

   The root argument passed to the saver is an instance of
   GlobalConfigMode.

   If 'requireMounts' is specified then CliSave supplies the named entities
   to the saver via the optional dict argument also named 'requireMounts'.
   The saver get the entities via this dict, not via the sysdbRoot argument.

   This is useful when the saver needs information from other entities in Sysdb
   other that those under the primary path. For e.g. focalpoint saver requires
   'hardware/focalpoint/config' to check the platform type before saving focalpoint
   commands.

   If 'secureMonitor' is True, this is also used for generating secure-monitor config
   that does not show up in normal running-config.

   If 'commandTagSupported' is True, this is used for generating command-tag related
   config that also shows up in normal running-config.
   """
   for path in itertools.chain( [ pathPrefix ], requireMounts ):
      assert path, 'Path cannot be empty.'
      assert not path.startswith( '/' ), 'Path prefixes must be relative.'

   def decorator( func ):
      # Register func as a handler for entities of type typename in the subtree
      # rooted at pathPrefix.
      typeHandlerInfo = TypeHandlerInfo( pathPrefix, attrName,
                                         requireMounts, hasDefaultConfig )
      handlers = [ saveHandlers_ ]
      if secureMonitor:
         handlers.append( secureMonitorSaveHandlers_ )
      if commandTagSupported:
         handlers.append( commandTagSaveHandlers_ )

      for h in handlers:
         typeList = h.setdefault( typeHandlerInfo, {} ).setdefault( typename, [] )
         typeList.append( func )
      return func
   return decorator

_pluginsLoaded = False
_pluginsWanted = True

def pluginsWantedIs( w ):
   '''Pass True if you want me to load all plugins, or False if you plan to
   manually load them or otherwise provide the CliSave handlers yourself.'''
   global _pluginsWanted
   if w == _pluginsWanted:
      return
   if not w and _pluginsLoaded:
      raise Exception( "Too late, I already loaded plugins" )
   _pluginsWanted = w

def maybeLoadCliSavePlugins( entityManager ):
   # Load the CliSave plugins the first time this function is called.
   global _pluginsLoaded
   if not _pluginsLoaded and _pluginsWanted:
      # CliSavePlugins or IntfRangePlugins can call ConfigMount.mount()
      Plugins.loadPlugins( 'CliSavePlugin', context=entityManager )
      _pluginsLoaded = True

def addHeaderHook( headerHook, priority ):
   '''Add a "header hook", which gets called at the very beginning to write
   a header at the start of running-config.  Eos uses this to write a header
   with information about the version of EOS that wrote the running-config
   (see Eos/CliSavePlugin/EosRunningConfigHeader.py).

   priority is 0-based, with lower value being called earlier. If priority is -1,
   it is appended to the end of the list (only used for testing).
   '''
   assert callable( headerHook )
   if priority < 0:
      headerHooks_.append( headerHook )
   else:
      assert priority < len( headerHooks_ )
      assert headerHooks_[ priority ] is None, \
         f"conflicts detected for priority {priority}"
      headerHooks_[ priority ] = headerHook

def registerSyntheticRequireMountGenerator( requireMount, func ):
   """ CLI Save modules use this to register a function
   for synthesizing a require mount object. These are
   necessary in a config session for objects normally
   produced by reactors in other agents. The func
   argument is called with three arguments:
   func( sysdbRoot, sessionRoot, pathHasSessionPrefix ).
   The last is a predicate on an entity path name.
   If it is true, use the session entity (if any),
   not the sysdb entity. """
   syntheticRequireMountGenerators_[ requireMount ] = func

syntheticRequireMountGenerators_ = {}

class EntityRoot:
   ''' Provides a transparent access to entities in Sysdb. '''

   def __init__( self, entityManager ):
      self.em_ = entityManager
      self.sysdbRoot_ = entityManager.root()

   def __getitem__( self, path ):
      if self.em_.localEntityExists( path ):
         return self.em_.getLocalEntity( path )
      if path in self.sysdbRoot_.entity:
         return self.sysdbRoot_.entity[ path ]

      raise KeyError( path )

   def fullName( self, path ):
      if self.em_.localEntityExists( path ):
         return ''
      elif path in self.sysdbRoot_.entity:
         return self.sysdbRoot_.fullName
      return None

   def root( self ):
      return self.sysdbRoot_

class ConfigGeneratorBase:
   def __init__( self, entityManager, sessionRoot, sStatus=None,
         cleanConfig=False ):
      maybeLoadCliSavePlugins( entityManager )
      self.entityManager_ = entityManager
      self.entityRoot_ = EntityRoot( entityManager )
      self.sessionRoot_ = sessionRoot
      self.sStatus_ = sStatus
      self.cleanConfig_ = cleanConfig

   def getRequireMountsDict( self, requireMounts ):
      raise NotImplementedError

   def getEntity( self, prefixPath ):
      raise NotImplementedError

   def getPathTrie( self ):
      return None

   @property
   def configRootFilter( self ):
      raise NotImplementedError

   def getRootSaveBlockAndCounter( self, options, pathTrie=None ):
      handlers = saveHandlers_
      if options.secureMonitor:
         handlers = secureMonitorSaveHandlers_
      elif options.commandTag:
         handlers = commandTagSaveHandlers_

      root = GlobalConfigMode()
      with Tac.ActivityLockHolder():
         configCounter = configCounter_.counterInternal
         for typeHandlerInfo, typeHandlers in sorted( handlers.items() ):
            path = typeHandlerInfo.pathPrefix
            if ( pathTrie and
                 not typeHandlerInfo.hasDefaultConfig and
                 not pathTrie.hasPrefix( path ) and
                 not any( pathTrie.hasPrefix( p )
                          for p in typeHandlerInfo.requireMounts ) ):
               continue

            if ( ( self.configRootFilter is not None and
                   self.configRootFilter.rootTrie.hasPrefix( path ) )
                 != options.showFilteredRoot ):
               continue

            requireMountsDict = self.getRequireMountsDict(
               typeHandlerInfo.requireMounts )
            e = self.getEntity( path )
            traceDetail( 'saving tree of', e )
            self._saveTree( e, root, typeHandlers, typeHandlerInfo.attrName,
                            requireMountsDict, options )

      param = SaveParam( self, options )
      return RootSaveBlock( root.getRootSaveBlock( param ), configCounter )

   def getRootSaveBlock( self, options, pathTrie=None ):
      return self.getRootSaveBlockAndCounter( options, pathTrie=pathTrie ).saveBlock

   def saveConfig( self, stream, options, renderOptions ):
      # Given a stream and options it will render the config to the stream
      rootSaveBlock = self.getRootSaveBlock( options )
      headers = self.generateHeaders( renderOptions )
      SaveBlockModelRenderer.render( stream, renderOptions, headers, rootSaveBlock )

   def generateHeadersAndSaveBlock( self, options, renderOptions,
                                    theirConfigGenerator ):
      # generate the save blocks for myConfig
      myHeaders = self.generateHeaders( renderOptions )
      myRootSaveBlock = self.getRootSaveBlock( options )

      # generate the save blocks for theirConfig
      theirHeaders = theirConfigGenerator.generateHeaders( renderOptions )
      theirRootSaveBlock = theirConfigGenerator.getRootSaveBlock( options )

      return myHeaders, myRootSaveBlock, theirHeaders, theirRootSaveBlock

   def diffConfig( self, stream, options, renderOptions, theirConfigGenerator ):
      myHeaders, myRootSaveBlock, theirHeaders, theirRootSaveBlock = \
         self.generateHeadersAndSaveBlock( options, renderOptions,
                                           theirConfigGenerator )
      # display the diff
      SaveBlockModelRenderer.renderDiff( stream, renderOptions,
                                         theirHeaders, myHeaders,
                                         theirRootSaveBlock, myRootSaveBlock )

   def diffConfigCliCommands( self, stream, options, renderOptions,
                              theirConfigGenerator ):
      myHeaders, myRootSaveBlock, theirHeaders, theirRootSaveBlock = \
         self.generateHeadersAndSaveBlock( options, renderOptions,
                                           theirConfigGenerator )
      # display the cli commands of the diff
      SaveBlockModelRenderer.renderDiffCliCommands( stream, renderOptions,
                                          theirHeaders, myHeaders,
                                          theirRootSaveBlock, myRootSaveBlock )

   def generateHeaders( self, options ):
      headers = []
      if options.showHeader:
         # If the user is requesting headers iterate through all of the header
         # hooks and add the header. The header is only applicable for the top
         # level and doesn't really figure into modes
         for h in headerHooks_:
            if not h:
               continue
            # pylint: disable=not-callable
            header = h( self.entityRoot_, options ).strip()
            if header:
               headers.append( header )
      return headers

   def _saveTree( self, entity, root, typeHandlers, attrName, requireMountsDict,
         options ):
      """
      Saves the state of the instantiating subtree rooted at 'entity' by calling
      the handler functions in 'typeHandlers[ type ]' for each entity in the tree.
      We do a preorder traversal of the tree.
      """

      walk = Tac.newInstance( "Cli::CliSaveWalk" )
      if attrName:
         # EntityWalk with an attribute filter will not search down an
         # attribute unless it's an instantiating Entity attribute.  We
         # should not be looking a non-instantiating attribute
         # instantiating anyway -- we should go to the instantiating
         # collection directly and walk it.
         a = entity.tacType.attr( attrName )
         assert a.instantiating and a.memberType.isEntity
         walk.attribute = attrName

      for k in typeHandlers:
         walk.handledType[ k ] = True

      walk.root = entity

      for e in walk.match.values():
         handlers = typeHandlers.get( e.tacType.fullTypeName )
         for func in handlers:
            traceDetail( 'Calling handler function', func, 'on entity', e )
            func( e, root, requireMountsDict, options )

      traceDetail( 'Done saving', entity )

   def _getSessionEntity( self, path ):
      """ Given an entity path, return the entity appropriate to
      the session state. """
      assert path
      sessionRoot = self._getEntitySessionRoot( path )
      if sessionRoot:
         return sessionRoot.entity[ path ]
      return self.entityRoot_[ path ]

   def _getEntitySessionRoot( self, path ):
      # return the session root for path, include rollbackConfig if available.
      #
      # None if the entity is not inside a session.
      if self.cleanConfig_:
         if path in CS.sessionStatus.cleanConfigStatus.cleanConfigRoot:
            return self.sessionRoot_
      elif self.sStatus_:
         if ( self.sStatus_.localPrefix and
              self.sStatus_.localPrefix.hasPrefix( path ) ):
            return self.sessionRoot_
         if ( self.sStatus_.rollbackConfig and
              CS.configRoot( self.entityManager_ ).rootTrie.hasPrefix( path ) ):
            # the session has been rolled back, so all paths under config root
            # should belong here
            return self.sStatus_.rollbackConfig
      return None

   def _pathHasSessionPrefix( self, path ):
      """ Consult the session trie to see if it has
      a prefix for the given path. The trie may be null,
      implying no prefixes. Special-case if clean config,
      using the presence of the path in cleanConfigRoot
      as a predicate. """
      return self._getEntitySessionRoot( path ) is not None

class RunningConfigGenerator( ConfigGeneratorBase ):

   cachedRootSaveBlock = None # RootSaveBlock
   # only cache the following which are heavily used
   cacheOpNormal = ShowRunningConfigOptions()

   cacheHits = 0


   def __init__( self, entityManager ):
      sessionRoot = EntityRoot( entityManager )
      super().__init__( entityManager, sessionRoot )
      self.configRootFilter_ = CS.activeConfigRootFilter( entityManager )

   def getRequireMountsDict( self, requireMounts ):
      return RequireMountsDict( requireMounts, self.entityManager_,
            self.entityRoot_ )

   def getEntity( self, prefixPath ):
      return self.entityRoot_[ prefixPath ]

   @property
   def configRootFilter( self ):
      return self.configRootFilter_

   def _getCacheOptions( self, options ):
      # some options don't matter for caching, just set them to default
      change = options.intfFilter
      if change:
         return options._replace( intfFilter=None )
      return options

   def getRootSaveBlockAndCounter( self, options, pathTrie=None ):
      # no caching if filtering is enabled
      cacheEnabled = ( not pathTrie and
                       not self.configRootFilter and
                       runningConfigCacheEnabledHook.all() )
      if cacheEnabled:
         cacheOptions = self._getCacheOptions( options )
         if cacheOptions == RunningConfigGenerator.cacheOpNormal:
            cached = RunningConfigGenerator.cachedRootSaveBlock
            if cached and cached.configCounter == configCounter_.counterInternal:
               RunningConfigGenerator.cacheHits += 1
               return cached

      # get root save block from super class method
      r = ConfigGeneratorBase.getRootSaveBlockAndCounter( self, options,
                                                          pathTrie=pathTrie )
      # save cache only when config counter is non-zero (cohab tests will always have
      # it as zero)
      if ( cacheEnabled and
           r.configCounter and
           options == RunningConfigGenerator.cacheOpNormal ):
         RunningConfigGenerator.cachedRootSaveBlock = r
      return r

class SessionConfigGenerator( ConfigGeneratorBase ):
   def __init__( self, entityManager, sessionName, sessionRoot, cleanConfig=False ):
      trace( 'Saving session-config of', sessionName, 'rooted at', sessionRoot )
      if cleanConfig:
         sStatus = None
      else:
         sStatus = CS.sessionStatus.pendingChangeStatusDir.status.get( sessionName )

      super().__init__( entityManager, sessionRoot,
                        sStatus=sStatus, cleanConfig=cleanConfig )

      # Generate the mapping for the registered synthetic requireMount generators.
      traceDetail( 'generating synthetic requireMount map' )
      self.syntheticRequireMountMap_ = self._generateSyntheticRequireMountMap()

   def getRequireMountsDict( self, requireMounts ):
      return SessionRequireMountsDict( requireMounts, self.entityManager_,
            self.entityRoot_, self, self.sessionRoot_,
            self.syntheticRequireMountMap_ )

   def getEntity( self, prefixPath ):
      return self._getSessionEntity( prefixPath )

   def getPathTrie( self ):
      # we do not support rollbackConfig with merge-on-commit
      assert self.sStatus_.rollbackConfig is None
      return self.sStatus_.localPrefix

   @property
   def configRootFilter( self ):
      return None

   def _generateSyntheticRequireMountMap( self ):
      """ This is called by saveSessionConfig,
          given to instances of SessionRequireMountsDict.
      """
      synthMap = {}
      for requireMount, func in syntheticRequireMountGenerators_.items():
         synthMap[ requireMount ] = func( self.entityRoot_, self.sessionRoot_,
               self._pathHasSessionPrefix )
      return synthMap

def saveRunningConfig( entityManager, stream, **kwargs ):
   """Saves the running-config to the file-like object stream."""
   with ConfigMount.ConfigMountDisabler( disable=True ):
      _saveConfigInternal( entityManager, stream, None, 'running-config', False,
                           **kwargs )

def saveSessionConfig( entityManager, stream, sessionName, cleanConfig=False,
                       **kwargs ):
   """Saves the config of the session sessionName to the file-like object stream."""
   _saveConfigInternal( entityManager, stream, sessionName, 'session-config',
         cleanConfig, **kwargs )

def saveAncestorSessionConfig( entityManager, stream, sessionName, **kwargs ):
   """Saves the ancestor config of the session to the file-like object stream."""
   _saveConfigInternal( entityManager, stream, sessionName, 'ancestor-config',
         False, **kwargs )

def _getGenerationAndRenderOptions( kwargs ):
   # separate options for generation and rendering
   renderKwargs = {}
   for kw in ( 'showSanitized', 'showJson', 'showHeader' ):
      if kw in kwargs:
         renderKwargs[ kw ] = kwargs.pop( kw )

   options = ShowRunningConfigOptions( **kwargs )
   renderOptions = ShowRunningConfigRenderOptions( **renderKwargs )

   # some sanity check on flags
   if options.saveAll:
      assert options.saveCleanConfig
   else:
      assert not options.saveAllDetail

   return options, renderOptions

def _saveConfigInternal( entityManager, stream, sessionName, generatorType,
                         cleanConfig, **kwargs ):
   if generatorType != 'running-config':
      # these are only supported for running config
      assert 'secureMonitor' not in kwargs, 'not supported for config session'
      assert 'showFilteredRoot' not in kwargs, 'not supported for config session'
      assert 'commandTag' not in kwargs, 'not supported for config session'

   options, renderOptions = _getGenerationAndRenderOptions( kwargs )

   configGenerator = _getGenerator( entityManager, sessionName, generatorType,
                                    cleanConfig )
   if configGenerator is None:
      return
   configGenerator.saveConfig( stream, options, renderOptions )

def saveSessionDiffs( entityManager, stream, sessionName, theirSource, mySource,
                      diffType='', mySessionName='', **kwargs ):
   """A function generating the diff between mySource and theirSource, which
   can be 'running-config', 'session-config', 'merged-config', 'ancestor-config'.
   The sessionName is used to generate config for mySource or theirSource based
   on the given type above.
   A special case when comparing 2 sessions: sessionName is the session of
   theirSource while mySessionName is the session of mySource.
   The 'cli' diffType can be set to generate diff in cli format.
   """
   assert 'secureMonitor' not in kwargs, 'not supported for config session'
   assert 'showFilteredRoot' not in kwargs, 'not supported for config session'
   assert 'commandTag' not in kwargs, 'not supported for config session'
   assert theirSource != 'merged-config', 'merged-config not a supported source'
   if mySessionName != '':
      assert mySource == 'session-config' and theirSource == 'session-config'
   else:
      mySessionName = sessionName
   if mySource == 'merged-config':
      assert diffType != 'cli'
      # merged config is a bit special so call that function instead of this one
      saveMergedSessionConfig( entityManager, stream, sessionName, diff=True,
                               localRootsOnly=True, **kwargs )
      return

   options, renderOptions = _getGenerationAndRenderOptions( kwargs )

   theirConfigGenerator = _getGenerator( entityManager, sessionName, theirSource )
   myConfigGenerator = _getGenerator( entityManager, mySessionName, mySource )

   if theirConfigGenerator is None or myConfigGenerator is None:
      return

   if diffType == 'cli':
      myConfigGenerator.diffConfigCliCommands( stream, options, renderOptions,
                                                theirConfigGenerator )
   else:
      myConfigGenerator.diffConfig( stream, options, renderOptions,
                                    theirConfigGenerator )

def getSessionSaveBlock( entityManager, sessionName, **kwargs ):
   # when we get our save blocks we don't want our mode ranges to be compressed
   # as our diffs will be incorrect
   kwargs[ 'expandMergeRange' ] = False
   options, _ = _getGenerationAndRenderOptions( kwargs )

   # session-config generator
   sessionConfigGenerator = _getGenerator( entityManager, sessionName,
         'session-config' )
   return sessionConfigGenerator.getRootSaveBlock( options, None )

def getConfigSaveBlockCommon( entityManager, sessionName, generatorType,
                              cleanConfig, **kwargs ):
   # when we get our save blocks we don't want our mode ranges to be compressed
   # as our diffs will be incorrect
   kwargs[ 'expandMergeRange' ] = False
   options, _ = _getGenerationAndRenderOptions( kwargs )

   configGenerator = _getGenerator( entityManager, sessionName, generatorType,
                                    cleanConfig=cleanConfig )
   return configGenerator.getRootSaveBlock( options, None )

def getCleanConfigSaveBlock( entityManager, **kwargs ):
   return getConfigSaveBlockCommon( entityManager, 'clean', 'session-config',
                                    True, **kwargs )

def getRunningConfigSaveBlock( entityManager, **kwargs ):
   return getConfigSaveBlockCommon( entityManager, None, 'running-config', False,
                                    **kwargs )

def saveMergedSessionConfig( entityManager, stream, sessionName, diff=False,
                             localRootsOnly=False, **kwargs ):
   # when we get our save blocks we don't want our mode ranges to be compressed
   # as our diffs will be incorrect
   kwargs[ 'expandMergeRange' ] = False
   options, renderOptions = _getGenerationAndRenderOptions( kwargs )

   # session-config generator
   sessionConfigGenerator = _getGenerator( entityManager, sessionName,
         'session-config' )
   trie = sessionConfigGenerator.getPathTrie() if localRootsOnly else None
   sessionSaveBlock = sessionConfigGenerator.getRootSaveBlock( options, trie )

   # ancestor-config generator
   ancestorConfigGenerator = _getGenerator( entityManager, sessionName,
                                            'ancestor-config' )
   ancestorSaveBlock = ancestorConfigGenerator.getRootSaveBlock( options, trie )

   # running-config generator
   runningConfigGenerator = _getGenerator( entityManager, None, 'running-config' )
   rootSaveBlock = runningConfigGenerator.getRootSaveBlock( options, trie )

   # now they we have all of the save blocks generated, create the merged one
   mergedSaveBlock = sessionSaveBlock.getMergedModel( ancestorSaveBlock,
                                                      rootSaveBlock )

   # We should use the session header, because that's what would take effect
   # we also don't support merging headers
   myHeaders = sessionConfigGenerator.generateHeaders( renderOptions )
   if diff:
      # compare the session header with the running-config header
      theirHeaders = runningConfigGenerator.generateHeaders( renderOptions )
      SaveBlockModelRenderer.renderDiff( stream, renderOptions,
                                         theirHeaders, myHeaders,
                                         rootSaveBlock, mergedSaveBlock )
   else:
      SaveBlockModelRenderer.render( stream, renderOptions, myHeaders,
                                     mergedSaveBlock )

   # return if there are any differences between the merged save blocks and the
   # session saved blocks
   return mergedSaveBlock.hasChanges( sessionSaveBlock )

def _getGenerator( entityManager, sessionName, generatorType, cleanConfig=False ):
   if generatorType == 'running-config':
      return RunningConfigGenerator( entityManager )
   elif generatorType == 'session-config':
      sessionRoot = _getSessionRoot( entityManager, sessionName, cleanConfig )
      if sessionRoot is None:
         return None
      return SessionConfigGenerator( entityManager, sessionName, sessionRoot,
            cleanConfig )
   elif generatorType == 'ancestor-config':
      if cleanConfig:
         ancestorRoot = None
      else:
         ancestorRoot = _getAncestorRoot( entityManager, sessionName )
      if ancestorRoot is None:
         return None
      return SessionConfigGenerator( entityManager, sessionName, ancestorRoot )
   else:
      assert False, f'Unknown generator type {generatorType}'

   return None

def _getAncestorRoot( em, sessionName ):
   ancestorRoot = em.root()[ 'session' ][ 'ancestorDir' ].get( sessionName )
   if ancestorRoot is None:
      raise CS.SessionAlreadyCompletedError( sessionName )
   return ancestorRoot

def _getSessionRoot( em, sessionName, cleanConfig ):
   CS.waitForSessionStatusInitialized( em )
   sStatus = CS.sessionStatus
   assert ( sStatus and sStatus.cleanConfigStatus and
         sStatus.cleanConfigStatus.cleanConfigComplete ), ( 'Config sessions '
               'should be up by now!!!' )

   if cleanConfig:
      return sStatus.cleanConfig

   sessionRoot = em.root()[ 'session' ][ 'sessionDir' ].get( sessionName )
   if sessionRoot is None:
      raise CS.SessionAlreadyCompletedError( sessionName )
   return sessionRoot

def sanitizedOutput( options, originalString ):
   if options and options.showSanitized:
      return sanitizedString
   else:
      return originalString

class RequireMountsDict:
   """ Maps decorator-specified requireMounts paths
   to the corresponding entities. An instance is passed
   to savers as the optional kwarg 'requireMounts'. """

   def __init__( self, requireMounts, entityManager, entityRoot ):
      self.requireMounts_ = requireMounts
      self.entityManager_ = entityManager
      self.entityRoot_ = entityRoot

   def getValue( self, requireMount ):
      # check if the path exists in the entityManager local coll
      if self.entityManager_.localEntityExists( requireMount ):
         return self.entityManager_.getLocalEntity( requireMount )

      return self.entityRoot_[ requireMount ]

   def __getitem__( self, requireMount ):
      if requireMount not in self.requireMounts_:
         raise KeyError( "attempt to access undeclared requireMount " +
                         requireMount )

      return self.getValue( requireMount )

   def __contains__( self, key ):
      return key in self.requireMounts_

   def get( self, requireMount, defaultValue=None ):
      try:
         return self.__getitem__( requireMount )
      except KeyError:
         return defaultValue

   def getRootPathname( self, entity ):
      rootPath = self.entityRoot_.root().fullName
      assert entity.fullName.startswith( rootPath )
      return rootPath

   def getEntityManager( self ):
      return self.entityManager_

   def getRoot( self ):
      return self.entityRoot_

class SessionRequireMountsDict( RequireMountsDict ):
   """ Maps decorator-specified requireMounts paths
   to the corresponding entities, taking into account
   the session configuration. An instance is passed
   to savers as the optional kwarg 'requireMounts'. """
   def __init__( self, requireMounts,
                 entityManager, entityRoot,
                 configGenerator,
                 sessionRoot,
                 syntheticRequireMountMap ):
      self.sessionRoot_ = sessionRoot
      self.syntheticRequireMountMap_ = syntheticRequireMountMap
      self.configGenerator_ = configGenerator
      RequireMountsDict.__init__( self, requireMounts, entityManager, entityRoot )

   def getValue( self, requireMount ):
      # Check for a synthetic requireMount mapping.
      if requireMount in self.syntheticRequireMountMap_:
         return self.syntheticRequireMountMap_[ requireMount ]

      # check if the path exists in the entityManager local coll
      if self.entityManager_.localEntityExists( requireMount ):
         return self.entityManager_.getLocalEntity( requireMount )

      return self.configGenerator_.getEntity( requireMount )

   def getRootPathname( self, entity ):
      # There is only one session root unlike 'Sysdb' root
      return self.sessionRoot_.fullName

class SaveParam:
   __slots__ = ( 'configGenerator_', 'options', 'comments' )

   def __init__( self, configGenerator, options ):
      self.configGenerator_ = configGenerator

      # store the options
      self.options = options

      self.comments = dict( self.getSessionEntity( 'cli/config' ).comment )

   def getSessionEntity( self, prefixPath ):
      return self.configGenerator_.getEntity( prefixPath )

# Add a blank command sequence so we can enable relative dependencies.
# For example, all low-priority sequences should use
# after=[ 'config.priority' ], and all high-priority sequences should
# use before=[ 'config.priority' ].
# By default, command sequences are not ordered wrt config.priority.
GlobalConfigMode.addCommandSequence( 'config.priority' )

def hasComments( commentKey, requireMounts ):
   return commentKey in requireMounts[ 'cli/config' ].comment

def escapeFormatString( string ):
   # escape { or } so they are not interpreted as format strings
   return string.replace( '{', '{{' ).replace( '}', '}}' )
