# Copyright (c) 2006-2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from functools import total_ordering
import os

import CliSaveBlock
import Tac
import Tracing

th = Tracing.defaultTraceHandle()
trace = th.trace0
traceTopSort = th.trace1
traceDetail = th.trace2

class CommandSequenceGenerator:
   __slots__ = ( 'name_', 'useInsertionOrder_' )

   def __init__( self, name, useInsertionOrder=False ):
      self.name_ = name
      self.useInsertionOrder_ = useInsertionOrder

@total_ordering
class Mode:
   """A container for the SaveBlocks that are to be output in a instance of a
   particular mode.  Plugins should create a subclass of this class for each CLI mode
   they define (that is, for each subclass of CliParser.Mode they define)."""

   __slots__ = ( 'param_', 'saveBlocks_', 'saveBlockMap_' )

   # all save blocks are sorted by sort key; if absent, the classname is used
   sortKey = ""
   # The following can be set to True if we want to merge modes with identical
   # commands into one range block. In addition it has to do the following:
   # 1. implement a static method of enterRangeCmd() that takes a list of modes
   #    and returns a command that enters the range.
   # 2. overload the canMergeRange() function to return True when conditions
   #    are met.
   mergeRange = False

   def __init__( self, param ):
      self.param_ = param

      with Tac.ActivityLockHolder():
         if not hasattr( self, 'saveBlockGenerators_' ):
            # pylint: disable=protected-access
            self.__class__._sortSaveBlockGenerators()
            # pylint: enable=protected-access

         # Instantiate all the SaveBlocks for this Mode instance.
         self.saveBlocks_ = []
         self.saveBlockMap_ = {}
         for generator in self.saveBlockGenerators_:
            if isinstance( generator, CommandSequenceGenerator ):
               saveBlock = CliSaveBlock.CommandSequence( generator.name_,
                     generator.useInsertionOrder_ )
               self.saveBlockMap_[ generator.name_ ] = saveBlock
            else:
               saveBlock = CliSaveBlock.ModeCollection( generator )
               self.saveBlockMap_[ generator ] = saveBlock
            self.saveBlocks_.append( saveBlock )

   def skipIfEmpty( self ):
      # This indicates that we should not print the mode in running-config if it is
      # empty (and no comments) even if someone called getOrCreateModeInstance().
      return False

   def hideInactive( self, param ):
      """This can be overridden if certain modes could be hidden at run time."""
      return False

   def hideUnconnected( self, param ):
      """This can be overridden if certain modes could be hidden at run time."""
      return False

   @classmethod
   def modeSortKey( cls ):
      if cls.sortKey:
         return cls.sortKey
      else:
         return cls.__name__

   @classmethod
   def addCommandSequence( cls, name, before=None, after=None,
         useInsertionOrder=False ):
      """Registers a command sequence name for instances of this Mode subclass.
      'before' and 'after' are lists of command sequence names or other Mode subclass
      objects.  'before' represents those save blocks that this command sequence must
      come before.  'after' represents those save blocks that this command sequence
      must come after.  Therefore:

       MyParentMode.addCommandSequence( 'seq1' )
       MyParentMode.addCommandSequence( 'seq2' )
       MyParentMode.addCommandSequence( 'seq3', before=[ 'seq1' ], after=[ 'seq2' ] )

      will cause the save blocks to be output in the order: seq2, seq3, seq1.

      useInsertionOrder: Whether the commands within the command sequence use
         insertion order to be displayed. BUG876816 for some more details
      """
      assert isinstance( name, str )
      generator = CommandSequenceGenerator( name, useInsertionOrder )
      cls._addSaveBlockGeneratorRecord( generator, before, after )

   @classmethod
   def addChildMode( cls, childModeClass, before=None, after=None ):
      """Registers a Mode subclass as being a child mode of this Mode subclass.
      'before' and 'after' are lists of command sequence names or other Mode subclass
      objects.  'before' represents those save blocks that the child mode must come
      before.  'after' represents those save blocks that the child mode must come
      after.  Therefore:

       MyParentMode.addCommandSequence( 'seq1' )
       MyParentMode.addCommandSequence( 'seq2' )
       MyParentMode.addChildMode( MyChildMode, before=[ 'seq1' ], after=[ 'seq2' ] )

      will cause the save blocks to be output in the order: seq2, MyChildMode,
      seq1."""
      assert issubclass( childModeClass, Mode )
      cls._addSaveBlockGeneratorRecord( childModeClass, before, after )

   @classmethod
   def _addSaveBlockGeneratorRecord( cls, generator, before, after ):
      """Adds a 'save block generator record' to this Mode subclass.  A 'save block
      generator' is either the name of a command sequence (a string) or a child mode
      class object.  A 'save block generator record' contains the generator, plus
      information about its dependencies on other save blocks."""
      if before is None:
         before = []
      if after is None:
         after = []
      if 'DEBUG_CLISAVE' in os.environ:
         # Get information about the caller of this function's caller.
         import traceback # pylint: disable=import-outside-toplevel
         ( filename, lineNo, _, _ ) = traceback.extract_stack()[ -3 ]
         debug = ( filename, lineNo )
      else:
         debug = None
      record = ( generator, before, after, debug )
      try:
         cls.saveBlockGeneratorRecords_.append( record )
      except AttributeError:
         cls.saveBlockGeneratorRecords_ = [ record ]

   @classmethod
   def _sortSaveBlockGenerators( cls ):
      """Sorts the save block generators for this mode class according to their
      dependencies.  This function is called the first time that an instance of this
      Mode subclass is created."""
      try:
         generatorRecords = cls.saveBlockGeneratorRecords_
      except AttributeError:
         # There are no child mode classes.
         generatorRecords = []

      generatorsMap = {}
      for ( generator, before, after, debug ) in generatorRecords:
         if isinstance( generator, CommandSequenceGenerator ):
            generatorsMap[ generator.name_ ] = generator
         else:
            generatorsMap[ generator ] = generator

      generators = list( generatorsMap.keys() )

      partialOrder = []
      for ( generator, before, after, debug ) in generatorRecords:
         generatorRecord = ( generator.name_
               if isinstance( generator, CommandSequenceGenerator ) else generator )
         partialOrder.extend( [ ( generatorRecord, b ) for b in before ] )
         partialOrder.extend( [ ( a, generatorRecord ) for a in after ] )

         # Check that none of the SaveBlocks have been given a dependency on a
         # non-existent SaveBlock.
         for b in before + after:
            if not b in generatorsMap:
               location = "an unknown location " \
                   "(run with DEBUG_CLISAVE=1 for more debug info)"
               try:
                  location = "%s:%d" % debug
               except ValueError:
                  pass
               except TypeError:
                  pass
               msg = "At %s\nyou called %s.addChildMode() or " \
                     "%s.addCommandSequence(), " \
                     "passing %r as an entry in the 'before' or 'after' lists.\n" \
                     "However, %r is not a valid CLI save block for mode %s.\n" \
                     "Perhaps you passed the mode name rather than the mode class " \
                     "object?" % \
                     ( location, cls.__name__, cls.__name__,
                       b, b, cls.__name__ )
               raise Exception( msg )

      # Sort the generators into a deterministic order; the
      # topological sort is not fully constraining.
      def _generatorKey( g ):
         if isinstance( g, str ):
            return g
         else:
            return g.modeSortKey()

      generators.sort( key=_generatorKey )
      traceTopSort( 'Sorting save blocks %s for class %s' %
              ( generators, cls.__name__ ) )
      traceTopSort( 'Using partial order %s' % partialOrder )
      generators = Tac.topologicalSort( generators, partialOrder )
      traceTopSort( 'Sorted as %s' % generators )
      assert not hasattr( cls, 'saveBlockGenerators_' )
      cls.saveBlockGenerators_ = [ generatorsMap[ generator ]
            for generator in generators ]

   def comments( self, param ):
      # pylint: disable=no-member
      commentKey = self.commentKey()
      if not commentKey:
         return None
      return param.comments.get( commentKey )

   def emptyCmds( self, param ):
      for b in self.saveBlocks_:
         if not b.empty( param ):
            return False
      return not self.comments( param )

   def empty( self, param ):
      """We are empty if no commands have been added to this Mode
      instance, and the enterCmd() is empty."""
      return ( ( self.skipIfEmpty() or not self.enterCmd() ) and
               self.emptyCmds( param ) )

   def expandMode( self, param ):
      """A mode might need to be 'expanded' aka something like
      interface profiles where if the option is given the commands
      are expanded"""
      pass # pylint: disable=unnecessary-pass

   def canMergeRange( self ):
      """Whether this mode instance can participate in range merge.
      For example, VLAN 1 is not supposed to merge with other VLAN
      instances, partly due to its different defaults."""
      return False

   @classmethod
   def useInsertionOrder( cls ):
      ''' Normally modes within a collection won't change order since they are
      normally sorted by alphabetically. However some mode collections have
      modes that are sorted instead by something external (like a priority)
      that isn't part of the command. For diffs and merging to be applied
      properly this need to overridden to True
      '''
      return False

   def instanceKey( self ):
      """Used by range merge. If you override you may have to override
      `useInsertionOrder` as well to avoid BUG876816"""
      return self.param_

   def content( self, param ):
      """Returns comments + saveBlocks as a tuple"""
      blocks = [ self.comments( param ) ]
      for i in self.saveBlocks_:
         blocks.append( i.content( param ) )
      return tuple( blocks )

   def modeSeparator( self ):
      """For printing modes inside a mode collection, whether a separator is needed
      between two modes."""
      return True

   def __hash__( self ):
      return hash( self.instanceKey() )

   def __eq__( self, other ):
      if not isinstance( other, Mode ):
         return False
      ik1 = self.instanceKey()
      ik2 = other.instanceKey()
      return ik1 == ik2

   def __lt__( self, other ):
      ik1 = self.instanceKey()
      ik2 = other.instanceKey()
      return ik1 < ik2

   def enterCmd( self ):
      """Subclasses should either override this method to return the
      command to be used to enter this CLI mode instance, or inherit a
      CliMode to implement this method. Otherwise, NotImplementedError
      is raised."""
      raise NotImplementedError

   def revertCmd( self ):
      """Subclasses can override this method to return the command
      to be used to revert the CLI mode instance.  By default,
      the 'default <enterCmd> is used."""
      return 'default ' + self.enterCmd()

   def __getitem__( self, key ):
      """Returns the SaveBlock for a particular key.  The key may be either the name
      of a command sequence (a string) or a child mode class object."""
      return self.saveBlockMap_[ key ]

class GlobalConfigMode( Mode ):
   def __init__( self ):
      # Instances of GlobalConfigMode do not have parameters.
      Mode.__init__( self, None )

   def enterCmd( self ):
      raise NotImplementedError

   def getRootSaveBlock( self, param ):
      saveBlockModel = CliSaveBlock.ModeEntryModel( None, None, True )
      for b in self.saveBlocks_:
         if b.empty( param ):
            continue
         saveBlockModel.addSaveBlockModel( b.generateSaveBlockModel( param ) )
      return saveBlockModel
