#!/usr/bin/env python3
# Copyright (c) 2008, 2009, 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import CliSave, Tracing
from CliSavePlugin.IntfCliSave import IntfConfigMode
from CliMode.Acl import BaseAclMode, CpMode
from AclCliLib import sortedSequenceNumbers, getRuleValue
from AclLib import AclPermitResponseType

__defaultTraceHandle__ = Tracing.Handle( 'AclCliSave' )

#-------------------------------------------------------------------------------
# Object used for saving commands in "config-acl-foo" mode.
#-------------------------------------------------------------------------------
class IpAclConfigMode( BaseAclMode, CliSave.Mode ):
   def __init__( self, param ):
      BaseAclMode.__init__( self, 'ip', False, param )
      CliSave.Mode.__init__( self, param )

class IpAclConfigStdMode( BaseAclMode, CliSave.Mode ):
   def __init__( self, param ):
      BaseAclMode.__init__( self, 'ip', True, param )
      CliSave.Mode.__init__( self, param )

class Ip6AclConfigMode( BaseAclMode, CliSave.Mode ):
   def __init__( self, param ):
      BaseAclMode.__init__( self, 'ipv6', False, param )
      CliSave.Mode.__init__( self, param )

class Ip6AclConfigStdMode( BaseAclMode, CliSave.Mode ):
   def __init__( self, param ):
      BaseAclMode.__init__( self, 'ipv6', True, param )
      CliSave.Mode.__init__( self, param )

class MacAclConfigMode( BaseAclMode, CliSave.Mode ):
   def __init__( self, param ):
      BaseAclMode.__init__( self, 'mac', False, param )
      CliSave.Mode.__init__( self, param )

class CpConfigMode( CpMode, CliSave.Mode ):
   def __init__( self, param ):
      CpMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

# interface mode goes first
#
# This way if one ACL is applied on many interfaces, at startup time
# it won't cause hardware to apply the same ACL multiple times.
CliSave.GlobalConfigMode.addChildMode( IpAclConfigMode,
                                       after=[ IntfConfigMode ] )
IpAclConfigMode.addCommandSequence( 'Acl.ipacl' )

CliSave.GlobalConfigMode.addChildMode( IpAclConfigStdMode,
                                       after=[ IntfConfigMode ] )
IpAclConfigStdMode.addCommandSequence( 'Acl.ipaclstd' )

CliSave.GlobalConfigMode.addChildMode( Ip6AclConfigMode,
                                       after=[ IntfConfigMode ] )
Ip6AclConfigMode.addCommandSequence( 'Acl.ip6acl' )

CliSave.GlobalConfigMode.addChildMode( Ip6AclConfigStdMode,
                                       after=[ IntfConfigMode ] )
Ip6AclConfigStdMode.addCommandSequence( 'Acl.ip6aclstd' )

CliSave.GlobalConfigMode.addChildMode( MacAclConfigMode,
                                       after=[ IntfConfigMode ] )
MacAclConfigMode.addCommandSequence( 'Acl.macacl' )

CliSave.GlobalConfigMode.addChildMode( CpConfigMode,
                                       after=[ IpAclConfigMode,
                                               IpAclConfigStdMode,
                                               Ip6AclConfigMode,
                                               Ip6AclConfigStdMode,
                                               MacAclConfigMode ] )
CpConfigMode.addCommandSequence( 'Acl.cp' )

CliSave.GlobalConfigMode.addCommandSequence( 'Acl.udfAlias',
                                             before=[ IpAclConfigMode,
                                                      IpAclConfigStdMode,
                                                      MacAclConfigMode ],
                                             after=[ IntfConfigMode ] )

CliSave.GlobalConfigMode.addCommandSequence( 'Acl.convertSymbols',
                                             before=[ IntfConfigMode ] )

def saveStatisticsIfEnabled( entity, cmds, saveAll ):
   if entity.currCfg.countersEnabled:
      cmds.addCommand( "counters per-entry" )
   elif saveAll:
      cmds.addCommand( "no counters per-entry" )

def saveFragmentsIfEnabled( entity, cmds, saveAll ):
   if not entity.currCfg.fragmentsEnabled:
      cmds.addCommand( "no fragment-rules" )
   elif saveAll:
      cmds.addCommand( "fragment-rules" )

@CliSave.saver( 'Acl::AclConfig', 'acl/config/cli',
                requireMounts=( 'acl/paramconfig', ),
                secureMonitor=True )
def saveIpAclConfig( entity, root, requireMounts, options ):
   if options.intfFilter:
      # quickly bypass ACL generation if we only look at interfaces.
      return

   paramConfig = requireMounts[ 'acl/paramconfig' ]
   convertSymbols = paramConfig.convertSymbols

   if ( entity.readonly or entity.type != 'ip' or entity.standard or
        entity.dynamic or entity.secureMonitor != options.secureMonitor ):
      return
   mode = root[ IpAclConfigMode ].getOrCreateModeInstance( entity.name )
   cmds = mode[ 'Acl.ipacl' ]
   if entity.currCfg is None:
      return
   saveStatisticsIfEnabled( entity, cmds, options.saveAll )
   saveFragmentsIfEnabled( entity, cmds, options.saveAll )
   for seq in sortedSequenceNumbers( entity.currCfg ):
      rule = getRuleValue( entity.currCfg, seq, 'ip', convert=convertSymbols )
      if options.showNoSeqNum:
         cmd = "%s" % ( rule )
      else:
         cmd = "%d %s" % ( seq, rule )
      cmds.addCommand( cmd )
   if entity.permitResponse == AclPermitResponseType.natTraffic:
      cmd = "permit response traffic nat"
      cmds.addCommand( cmd )

@CliSave.saver( 'Acl::AclConfig', 'acl/config/cli',
                requireMounts=( 'acl/paramconfig', ),
                secureMonitor=True )
def saveStdIpAclConfig( entity, root, requireMounts, options ):
   if options.intfFilter:
      # quickly bypass ACL generation if we only look at interfaces.
      return

   paramConfig = requireMounts[ 'acl/paramconfig' ]
   convertSymbols = paramConfig.convertSymbols

   if ( entity.readonly or entity.type != 'ip' or not entity.standard or
        entity.dynamic or entity.secureMonitor != options.secureMonitor ):
      return
   mode = root[ IpAclConfigStdMode ].getOrCreateModeInstance( entity.name )
   cmds = mode[ 'Acl.ipaclstd' ]
   if entity.currCfg is None:
      return
   saveStatisticsIfEnabled( entity, cmds, options.saveAll )
   saveFragmentsIfEnabled( entity, cmds, options.saveAll )
   for seq in sortedSequenceNumbers( entity.currCfg ):
      rule = getRuleValue( entity.currCfg, seq, 'ip', True, convertSymbols )
      if options.showNoSeqNum:
         cmd = "%s" % ( rule )
      else:
         cmd = "%d %s" % ( seq, rule )
      cmds.addCommand( cmd )

@CliSave.saver( 'Acl::AclConfig', 'acl/config/cli',
                requireMounts=( 'acl/paramconfig', ),
                secureMonitor=True )
def saveIp6AclConfig( entity, root, requireMounts, options ):
   if options.intfFilter:
      # quickly bypass ACL generation if we only look at interfaces.
      return

   paramConfig = requireMounts[ 'acl/paramconfig' ]
   convertSymbols = paramConfig.convertSymbols

   if ( entity.readonly or entity.type != 'ipv6' or entity.standard or
        entity.dynamic or entity.secureMonitor != options.secureMonitor ):
      return
   mode = root[ Ip6AclConfigMode ].getOrCreateModeInstance( entity.name )
   cmds = mode[ 'Acl.ip6acl' ]
   if entity.currCfg is None:
      return
   saveStatisticsIfEnabled( entity, cmds, options.saveAll )
   saveFragmentsIfEnabled( entity, cmds, options.saveAll )
   for seq in sortedSequenceNumbers( entity.currCfg ):
      rule = getRuleValue( entity.currCfg, seq, 'ipv6', convert=convertSymbols )
      if options.showNoSeqNum:
         cmd = "%s" % ( rule )
      else:
         cmd = "%d %s" % ( seq, rule )
      cmds.addCommand( cmd )

@CliSave.saver( 'Acl::AclConfig', 'acl/config/cli',
                requireMounts=( 'acl/paramconfig', ),
                secureMonitor=True )
def saveStdIp6AclConfig( entity, root, requireMounts, options ):
   if options.intfFilter:
      # quickly bypass ACL generation if we only look at interfaces.
      return

   paramConfig = requireMounts[ 'acl/paramconfig' ]
   convertSymbols = paramConfig.convertSymbols

   if ( entity.readonly or entity.type != 'ipv6' or not entity.standard or
        entity.dynamic or entity.secureMonitor != options.secureMonitor ):
      return
   mode = root[ Ip6AclConfigStdMode ].getOrCreateModeInstance( entity.name )
   cmds = mode[ 'Acl.ip6aclstd' ]
   if entity.currCfg is None:
      return
   saveStatisticsIfEnabled( entity, cmds, options.saveAll )
   saveFragmentsIfEnabled( entity, cmds, options.saveAll )
   for seq in sortedSequenceNumbers( entity.currCfg ):
      rule = getRuleValue( entity.currCfg, seq, 'ipv6', True, convertSymbols )
      if options.showNoSeqNum:
         cmd = "%s" % ( rule )
      else:
         cmd = "%d %s" % ( seq, rule )
      cmds.addCommand( cmd )

@CliSave.saver( 'Acl::AclConfig', 'acl/config/cli', secureMonitor=True )
def saveMacAclConfig( entity, root, requireMounts, options ):
   if options.intfFilter:
      # quickly bypass ACL generation if we only look at interfaces.
      return

   if ( entity.readonly or entity.type != 'mac' or
        entity.dynamic or entity.secureMonitor != options.secureMonitor ):
      return
   mode = root[ MacAclConfigMode ].getOrCreateModeInstance( entity.name )
   cmds = mode[ 'Acl.macacl' ]
   if entity.currCfg is None:
      return
   saveStatisticsIfEnabled( entity, cmds, options.saveAll )
   for seq in sortedSequenceNumbers( entity.currCfg ):
      proto = getRuleValue( entity.currCfg, seq, 'mac' )
      if type( proto ) is int: # pylint: disable=unidiomatic-typecheck
         if options.showNoSeqNum:
            cmd = "0x%x" % ( proto )
         else:
            cmd = "%d 0x%x" % ( seq, proto )
      else:
         if options.showNoSeqNum:
            cmd = "%s" % ( proto )
         else:
            cmd = "%d %s" % ( seq, proto )
      cmds.addCommand( cmd )

@CliSave.saver( 'Acl::ParamConfig', 'acl/paramconfig' )
def saveAutoConversionConfig( entity, root, requireMounts, options ):
   cmd = root[ 'Acl.convertSymbols' ]
   if entity.convertSymbols == False: # pylint: disable=singleton-comparison
      cmd.addCommand( 'no service configuration access-list symbols' )

@CliSave.saver( 'Acl::UdfAlias', 'acl/udfalias' )
def saveUdfAlias( entity, root, requireMounts, options ):
   if options.intfFilter:
      # quickly bypass ACL generation if we only look at interfaces.
      return

   def _createAliasCmd( alias, aliasType ):
      cmd = aliasType
      cmd += ' access-list payload alias'
      cmd += ' %s' % alias.name
      if alias.headerType == 'l4Header':
         cmd += ' header layer-4 ' + alias.headerStart
      elif alias.headerStart == 'start':
         cmd += ' header start'
      payload = alias.payload
      cmd += ' offset %d pattern 0x%08x mask 0x%08x' % (
                      payload.offset, payload.pattern, 0xFFFFFFFF ^ payload.mask )
      return cmd

   cmds = root[ 'Acl.udfAlias' ]
   for alias in entity.ipAlias.values():
      cmd = _createAliasCmd( alias, 'ip' )
      cmds.addCommand( cmd )

   for alias in entity.macAlias.values():
      cmd = _createAliasCmd( alias, 'mac' )
      cmds.addCommand( cmd )
