# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import operator
import Tac
import CliExtensions
import CliSave
from AleCounters import tunnelTypeToStr
from AleFlexCounterTypes import (
   counterFeatureMap,
   counterFeatureConfigCommand,
   unitsCountModeToTokenMap,
)
from CliMode.AleCounters import (
   MonitorCounterMode,
   MonitorCounterEthernetMode,
   MonitorQueueCounterMode,
   RouteCountersModeBase,
   RouteCountersVrfModeBase,
   TunnelCountersMode,
   TunnelCountersTransmitMode,
)
from CliSavePlugin.RouterGeneralCliSave import RouterGeneralCliSaveMode
from TypeFuture import TacLazyType

CliConfigConstants = TacLazyType( 'Ale::Counters::CliConfigConstants' )
TunnelCountersPriority = TacLazyType( "Tunnel::TunnelCountersPriority" )

CliSave.GlobalConfigMode.addCommandSequence( 'AleCounters' )

@CliSave.saver( 'Ale::Counters::InternalDrop::CliConfig',
                'hardware/counter/internalDrop/config/cli' )
def saveInternalDropCliConfig( entity, root, requireMounts, options ):
   syslogEventThresholds = sorted( entity.syslogEventThreshold.keys() )
   for counterName in syslogEventThresholds:
      root[ 'AleCounters' ].addCommand( 'hardware counter drop log %s %d'
            % ( counterName, entity.syslogEventThreshold[ counterName ] ) )

def flexCounterSaveCounterCliConfig( entity, root, requireMounts, options,
                                     fcFeatureStatusDir=None ):
   def addCommand( prefix, featureId, suffix="" ):
      if featureId == 'Route':
         return

      command = prefix + counterFeatureConfigCommand( featureId ) + suffix
      root[ "AleCounters" ].addCommand( command )

   if fcFeatureStatusDir is None:
      return

   defaultFeatAttr = Tac.Value( 'FlexCounters::FeatureAttributeState' )
   for featureId, featureStatus in sorted( fcFeatureStatusDir.entityPtr.items(),
                                           key=lambda p: p[ 0 ] ):
      featureInfo = counterFeatureMap.get( featureId )
      if featureInfo is None or not featureInfo.autogenCliCfg:
         continue

      featureConfig = entity.feature.get( featureId )
      featureAttrConfig = entity.featureAttribute.get( featureId, defaultFeatAttr )
      if featureConfig is not None:
         if (
               # Feature is explicitly enabled.
               ( not featureStatus.defaultState and featureConfig == "enabled" ) or
               # Feature at default state, but all/allDetail requested.
               ( featureStatus.defaultState and
                 ( options.saveAll or options.saveAllDetail ) ) or
               # Feature at default state, but has a non-default featureAttribute.
               ( featureStatus.defaultState and
                 featureAttrConfig != defaultFeatAttr )
         ):
            featAttrSuffix = ""
            unitsTypeToken = unitsCountModeToTokenMap[ featureAttrConfig.countMode ]
            if unitsTypeToken is not None:
               featAttrSuffix += " units " + unitsTypeToken
            addCommand( "", featureId, featAttrSuffix )
      else:
         if featureStatus.defaultState or options.saveAllDetail:
            addCommand( "no ", featureId )

# Hook for platform featureStatusDir.
# The hook extension must be a callable such that:
# - it accepts the following arguments: `sysdbRoot`;
# - it returns the featureStatusDir.
# No more than 1 extension shall be registered at the same time.
counterFeatureStatusDirHook = CliExtensions.CliHook()

def getFeatureStatusDir():
   extensions = counterFeatureStatusDirHook.extensions()
   if not extensions:
      return None
   if os.environ.get( "ALECOUNTERS_CLIHOOK_STRICT" ):
      assert len( extensions ) <= 1, "More than 1 hook registered"
   # Take the first extension.
   extension = next( iter( extensions ) )
   return extension( None )

@CliSave.saver( 'Ale::FlexCounter::FeatureConfigDir',
                'flexCounter/featureConfigDir/cliAgent' )
def saveCounterCliConfig( entity, root, requireMounts, options ):
   flexCounterSaveCounterCliConfig(
      entity, root, requireMounts, options,
      fcFeatureStatusDir=getFeatureStatusDir() )

def flexCounterSaveRouteCounterCliConfig( entity, root, requireMounts, options ):
   def addCommand( routeType, vrfName, prefix ):
      if prefix is None:
         return

      command = None

      if vrfName == 'default':
         command = f"hardware counter feature route {routeType} {prefix}"
      else:
         command = ( f"hardware counter feature route {routeType} "
                     f"vrf {vrfName} {prefix}" )

      root[ "AleCounters" ].addCommand( command )

   for vrfNameAndPrefixObj in entity.route:
      vrfName = vrfNameAndPrefixObj.vrfName
      prefixString = vrfNameAndPrefixObj.ipGenPrefix.stringValue
      routeType = vrfNameAndPrefixObj.ipGenPrefix.af

      addCommand( routeType, vrfName, prefixString )

# -----------------------------------------------------------------
# CliSave Plugin for monitor counter mode
# -----------------------------------------------------------------
class MonitorCounterConfigMode( MonitorCounterMode, CliSave.Mode ):
   def __init__( self, param ):
      MonitorCounterMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

#-----------------------------------------------------------------
# CliSave Plugin for ethernet interfaces mode
#-----------------------------------------------------------------
class MonitorCounterEthernetConfigMode( MonitorCounterEthernetMode, CliSave.Mode ):
   def __init__( self, param ):
      MonitorCounterEthernetMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

#-----------------------------------------------------------------
# CliSave Plugin for queue counters mode
#-----------------------------------------------------------------
class MonitorQueueCounterConfigMode( MonitorQueueCounterMode, CliSave.Mode ):
   def __init__( self, param ):
      MonitorQueueCounterMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

# -----------------------------------------------------------------
# CliSave Plugin for route counter mode
# -----------------------------------------------------------------
class RouteCountersConfigMode( RouteCountersModeBase, CliSave.Mode ):
   def __init__( self, param ):
      RouteCountersModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

# -----------------------------------------------------------------
# CliSave Plugin for route counter vrf mode
# -----------------------------------------------------------------
class RouteCountersVrfConfigMode( RouteCountersVrfModeBase, CliSave.Mode ):
   def __init__( self, param ):
      RouteCountersVrfModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

# -----------------------------------------------------------------
# CliSave Plugin for tunnel counters mode
# -----------------------------------------------------------------
class TunnelCountersConfigMode( TunnelCountersMode, CliSave.Mode ):
   def __init__( self, param ):
      TunnelCountersMode.__init__( self )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

#-----------------------------------------------------------------
# CliSave Plugin for tunnel counters transmit mode
#-----------------------------------------------------------------
class TunnelCountersTransmitConfigMode( TunnelCountersTransmitMode, CliSave.Mode ):
   def __init__( self, param ):
      TunnelCountersTransmitMode.__init__( self )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

RouterGeneralCliSaveMode.addChildMode( RouteCountersConfigMode )
RouteCountersConfigMode.addCommandSequence( 'RouteCounters.config' )
RouteCountersConfigMode.addChildMode( RouteCountersVrfConfigMode )
RouteCountersVrfConfigMode.addCommandSequence( 'RouteCounters.vrf.config' )

CliSave.GlobalConfigMode.addChildMode( MonitorCounterConfigMode )
CliSave.GlobalConfigMode.addChildMode( TunnelCountersConfigMode )

TunnelCountersConfigMode.addCommandSequence( 'AleCounters.tunnelCounters' )
TunnelCountersConfigMode.addChildMode( TunnelCountersTransmitConfigMode )
TunnelCountersTransmitConfigMode.addCommandSequence(
   'AleCounters.transmitTunnelCounters' )

MonitorCounterConfigMode.addChildMode( MonitorCounterEthernetConfigMode )
MonitorCounterEthernetConfigMode.addCommandSequence( 'AleCounters.intfconfig' )
MonitorCounterConfigMode.addChildMode( MonitorQueueCounterConfigMode )
MonitorQueueCounterConfigMode.addCommandSequence( 'AleCounters.queueconfig' )

def savablePollPeriod( config ):
   # Don't save if config.periodPoll in not in range. The only way to get to
   # this state is through Acons for debug purposes, and saving it breaks ASU
   return ( config.periodPoll >= CliConfigConstants.minPeriodPoll and
            config.periodPoll <= CliConfigConstants.maxPeriodPoll and
            config.periodPoll != config.defaultPeriodPoll )

@CliSave.saver( 'Ale::Counters::CliConfig', 'counter/global/intfconfig' )
def saveIntfGlobalConfig( config, root, requireMounts, options ):
   monMode = root[ MonitorCounterConfigMode ].getSingletonInstance()
   mode = monMode[ MonitorCounterEthernetConfigMode ].getSingletonInstance()

   cmds = mode[ 'AleCounters.intfconfig' ]

   # normal and fast polling are separate config, so only add non-defaults
   if savablePollPeriod( config ):
      cmds.addCommand( 'update interval %.1f' % config.periodPoll )
   if config.periodFastPoll != config.defaultPeriodFastPoll:
      cmds.addCommand( 'update fast-interval %d' % config.periodFastPoll )

@CliSave.saver( 'Ale::Counters::CliConfig', 'counter/global/queueconfig' )
def saveQueueGlobalConfig( config, root, requireMounts, options ):
   # only create the mode if there is anything to save
   monMode = root[ MonitorCounterConfigMode ].getSingletonInstance()
   mode = monMode[ MonitorQueueCounterConfigMode ].getSingletonInstance()

   cmds = mode[ 'AleCounters.queueconfig' ]
   if savablePollPeriod( config ):
      cmds.addCommand( 'update interval %.1f' % config.periodPoll )

@CliSave.saver( 'Ale::PrefixCounterCliConfig', 'hardware/counter/prefix/config' )
def saveRouteCountersConfig( config, root, requireMounts, options ):
   generalMode = root[ RouterGeneralCliSaveMode ].getSingletonInstance()
   routeCountersMode = generalMode[ RouteCountersConfigMode ].getSingletonInstance()
   cmds = routeCountersMode[ 'RouteCounters.config' ]
   if config.primaryBackupDedicated:
      cmds.addCommand( 'primary backup dedicated' )
   for vrfName in config.vrfPrefixLens:
      vrfMode = routeCountersMode[ RouteCountersVrfConfigMode ].\
            getOrCreateModeInstance( vrfName )
      cmds = vrfMode[ 'RouteCounters.vrf.config' ]
      if config.vrfPrefixLens[ vrfName ].prefixLen:
         cmd = ( 'ipv4 prefix-length ' +
                 ' '.join( str( length ) for length
                           in config.vrfPrefixLens[ vrfName ].prefixLen ) )
         cmds.addCommand( cmd )

@CliSave.saver( 'Ale::TunnelCountersPriorityTable',
                'hardware/counter/tunnel/tunnelTypePriority' )
def saveTunnelTypePriorityConfig( config, root, requireMounts, options ):
   tCountersMode = root[ TunnelCountersConfigMode ].getSingletonInstance()
   mode = tCountersMode[ TunnelCountersTransmitConfigMode ].getSingletonInstance()

   if not config.allTunnelCountersEnabled:
      mode[ "AleCounters.transmitTunnelCounters" ].addCommand( "selective" )
   cmd = "source-protocol "
   tunnelTypeByPrio = sorted( config.tunnelTypePriority.items(),
                              key=operator.itemgetter( 1 ) )
   for tunnelType, priority in tunnelTypeByPrio:
      tunnelTypeStr = tunnelTypeToStr[ tunnelType ]
      tunnelTypeStr += " priority " + str( priority )
      mode[ "AleCounters.transmitTunnelCounters" ].addCommand( cmd + tunnelTypeStr )

@CliSave.saver( 'Ale::Counters::TunnelCountersCliConfig',
                'hardware/counter/tunnel/cliConfig' )
def saveTunnelCountersCliConfig( config, root, requireMounts, options ):
   loadInterval = config.tunnelCountersLoadInterval

   if loadInterval.val != loadInterval.defaultVal or options.saveAll:
      tunnelCountersMode = root[ TunnelCountersConfigMode ].getSingletonInstance()
      cmds = tunnelCountersMode[ 'AleCounters.tunnelCounters' ]
      cmd = 'rate period decay %d seconds' % loadInterval.val
      cmds.addCommand( cmd )
