#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliMode.AleCpuPolicy import (
   CpuTrafficPolicyEnforcementConfigMode,
   CpuTrafficPolicyEnforcementModeBase,
)
from CliMode.TrafficPolicy import TrafficPoliciesConfigMode
from CliSavePlugin.IntfCliSave import IntfConfigMode
from CliSavePlugin.TrafficPolicyCliSave import (
   TrafficPoliciesSaveMode, TrafficPoliciesVrfSaveMode,
   allFieldSetSaveModes,
)
import CliSave
import Tac

tacMatchOption = Tac.Type( 'PolicyMap::ClassMapMatchOption' )
matchIpAccessGroup = tacMatchOption.matchIpAccessGroup
matchIpv6AccessGroup = tacMatchOption.matchIpv6AccessGroup

class CpuPolicyEnforcementSaveMode( CpuTrafficPolicyEnforcementModeBase,
                                    CliSave.Mode ):
   def __init__( self, param ):
      CpuTrafficPolicyEnforcementModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

TrafficPoliciesSaveMode.addChildMode( CpuPolicyEnforcementSaveMode,
                                      before=allFieldSetSaveModes() )

TrafficPoliciesSaveMode.addChildMode( TrafficPoliciesVrfSaveMode,
                                      before=allFieldSetSaveModes() )

TrafficPoliciesSaveMode.addCommandSequence( 'CpuPolicyGlobalConfig' )

CpuPolicyEnforcementSaveMode.addCommandSequence( 'TrafficPolicy.CpuPolicies' )

class AleCpuPolicySaver:
   def __init__( self, entity, root, requireMounts, options ):
      self.entity = entity
      self.root = root
      self.requireMounts = requireMounts
      self.options = options
      self.trafficPolicy = None
      self.currPolicyName = None
      self.matchRule = None

   def saveCpuEnforcementConfig( self, policyName, vrfListStr,
                                 includeManagement, policiesMode ):
      param = ( policyName, vrfListStr, CpuTrafficPolicyEnforcementConfigMode, )
      cpuEnforcementMode = (
         policiesMode[ CpuPolicyEnforcementSaveMode ].
            getOrCreateModeInstance( param ) )
      cmds = cpuEnforcementMode[ 'TrafficPolicy.CpuPolicies' ]
      if includeManagement:
         cmds.addCommand( 'enforcement management' )

   def savePerVrfCpuPolicyConfig( self, policyName, vrfName,
                                  includeManagement, policiesMode ):
      param = ( vrfName, TrafficPoliciesConfigMode, 'traffic-policy', self.entity,
                None )

      vrfMode = (
         policiesMode[ TrafficPoliciesVrfSaveMode ].
         getOrCreateModeInstance( param )
      )
      cmds = vrfMode[ 'TrafficPolicy.TrafficPoliciesVrf' ]
      cmd = f'cpu traffic-policy {policyName} fallback traffic-policy none'
      cmds.addCommand( cmd )
      if includeManagement:
         indent = 3 * " "
         cmds.addCommand( f'{indent}enforcement management' )

   # save policy map vrf settings
   def saveCpuVrfConfigAll( self, cpuVrfConfig, policiesMode ):
      # print out pmaps in sorted order
      vrfPmapList = sorted( cpuVrfConfig.trafficPolicies.items(),
                            key=lambda x:x[ 0 ] )
      for vrfName, policyConfig in vrfPmapList:
         includeManagement = policyConfig.includeManagement
         if vrfName == 'all':
            self.saveCpuEnforcementConfig( policyConfig.policyName, vrfName,
                                          includeManagement,
                                          policiesMode )
         else:
            self.savePerVrfCpuPolicyConfig( policyConfig.policyName, vrfName,
                                          includeManagement,
                                          policiesMode )

   def save( self ):
      cpuVrfConfig = self.entity
      trafficPoliciesMode = \
         self.root[ TrafficPoliciesSaveMode ].getSingletonInstance()
      self.saveCpuVrfConfigAll( cpuVrfConfig, trafficPoliciesMode )

# Save all traffic-policy
@CliSave.saver( 'PolicyMap::VrfConfig', 'trafficPolicies/cpu/vrf' )
def saveConfig( entity, root, requireMounts, options ):
   cliDumper = AleCpuPolicySaver( entity, root, requireMounts, options )
   cliDumper.save()

INTF_CPU_POLICY_CMD_SEQ = 'TrafficPolicies.IntfCpuPolicy'
IntfConfigMode.addCommandSequence( INTF_CPU_POLICY_CMD_SEQ )

@CliSave.saver( "PolicyMap::IntfConfig", "trafficPolicies/cpu/intf" )
def saveIntfCpuPolicyConfig( entity, root, requireMounts, options ):
   if entity is None:
      return
   for intf, policyName in entity.intf.items():
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intf )
      cmds = mode[ INTF_CPU_POLICY_CMD_SEQ ]
      cmds.addCommand( f'cpu traffic-policy {policyName} '
                       'fallback traffic-policy vrf' )

@CliSave.saver( "TrafficPolicy::CpuPolicyGlobalConfig",
                "trafficPolicies/param/config/cpu" )
def saveCpuPolicyGlobalConfig( entity, root, requireMounts, options ):
   if entity is None:
      return
   if entity.installPermitFragment is False:
      fragCmd = "cpu traffic-policy fragment implicit-permit disabled"
      trafficPoliciesMode = \
            root[ TrafficPoliciesSaveMode ].getOrCreateModeInstance( None )
      cmds = trafficPoliciesMode[ 'CpuPolicyGlobalConfig' ]
      cmds.addCommand( fragCmd )
   if entity.enforcementIpTtlExpired:
      enforcementIpTtlExpiredCmd = "cpu traffic-policy enforcement ip ttl expired"
      trafficPoliciesMode = \
            root[ TrafficPoliciesSaveMode ].getOrCreateModeInstance( None )
      cmds = trafficPoliciesMode[ 'CpuPolicyGlobalConfig' ]
      cmds.addCommand( enforcementIpTtlExpiredCmd )
