# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import CliSave
from CliSavePlugin.IntfCliSave import IntfConfigMode
from IpLibConsts import DEFAULT_VRF
from RoutingIntfUtils import allIpIntfNames
from Toggles.ArpToggleLib import toggleForcedPeriodicArpRefreshEnabled

IntfConfigMode.addCommandSequence( 'Arp.config',
      after=[ 'Ira.ipIntf' ] )
CliSave.GlobalConfigMode.addCommandSequence( 'Ira.arp', after=[ 'Ira.routes' ] )
CliSave.GlobalConfigMode.addCommandSequence( 'Ira.staticNeighbor6',
      after=[ IntfConfigMode ] )
tristateBool = Tac.Type( "Ip::TristateBool" )
defaultEthernetTimeout = Tac.Type( 'Arp::ArpIntfConfig' ).defaultEthernetTimeout

@CliSave.saver( 'Arp::Config', 'arp/config',
                requireMounts = ( 'interface/config/all', 'interface/status/all' ) )
def saveArpConfig( entity, root, requireMounts, options ):
   if entity.restoreOnReboot:
      root[ 'Ira.arp' ].addCommand( "arp cache persistent" )

   if entity.ip6RestoreOnReboot:
      root[ 'Ira.staticNeighbor6' ].addCommand( 
         "ipv6 neighbor cache persistent" )

   if entity.ip4PersistanceEnabled:
      root[ 'Ira.arp' ].addCommand( 
            f"arp persistent refresh-delay {entity.refreshDelay}" )

   if entity.ip6PersistanceEnabled:
      root[ 'Ira.staticNeighbor6' ].addCommand( 
            f"ipv6 neighbor persistent refresh-delay {entity.refreshDelay}" )

   if entity.proxyMaxDelay != entity.proxyMaxDelayDefault:
      root[ 'Ira.arp' ].addCommand(
         f"arp proxy max-delay {entity.proxyMaxDelay}" )
   elif options.saveAll:
      root[ 'Ira.arp' ].addCommand(
         f"arp proxy max-delay {entity.proxyMaxDelayDefault}" )

   if entity.globalMonitorMac:
      root[ 'Ira.arp' ].addCommand( "arp monitor mac-address" )
   elif options.saveAll:
      root[ 'Ira.arp' ].addCommand( "no arp monitor mac-address" )
   
   if entity.globalArpTimeout != defaultEthernetTimeout:
      root[ 'Ira.arp' ].addCommand( 
            f"arp aging timeout default {entity.globalArpTimeout}" )
   elif options.saveAll:
      root[ 'Ira.arp' ].addCommand( "default arp aging timeout default" )

   if options.saveAllDetail:
      cfgIntfNames = allIpIntfNames( requireMounts, includeEligible=True )
   elif options.saveAll:
      cfgIntfNames = set( allIpIntfNames( requireMounts ) )
      cfgIntfNames.update( entity.arpIntfConfig )
   else:
      cfgIntfNames = entity.arpIntfConfig
   for intfName in sorted( cfgIntfNames ):
      intfConfig = entity.arpIntfConfig.get( intfName )
      if not intfConfig:
         if options.saveAll:
            intfConfig = Tac.newInstance( 'Arp::ArpIntfConfig', intfName )
         else:
            continue
      saveArpIntfConfig( intfConfig, root, options )

def saveArpIntfConfig( entity, root, options ):
   mode = root[ IntfConfigMode ].getOrCreateModeInstance( entity.name )
   cmds = mode[ 'Arp.config' ]
   if entity.timeoutV4:
      cmds.addCommand( f"arp aging timeout {entity.timeoutV4}" )
   elif options.saveAll:
      cmds.addCommand( "default arp aging timeout" )
   if entity.arpCacheCapacity is not None:
      cmds.addCommand(
         f"arp cache dynamic capacity {entity.arpCacheCapacity.capacity}"
      )
   elif options.saveAll:
      cmds.addCommand( "default arp cache dynamic capacity" )
   if entity.timeoutV6:
      cmds.addCommand( f"ipv6 nd cache expire {entity.timeoutV6}" )
   elif options.saveAll:
      cmds.addCommand( "default ipv6 nd cache expire" )
   if entity.nbrCacheCapacity is not None:
      excluded = (
         ' link-local excluded'
         if entity.nbrCacheCapacity.linkLocalExcluded else ''
      )
      cmds.addCommand(
         f"ipv6 nd cache dynamic capacity {entity.nbrCacheCapacity.capacity}"
         f"{excluded}"
      )
   elif options.saveAll:
      cmds.addCommand( "default ipv6 nd cache dynamic capacity" )
   if entity.monitorMac == tristateBool.isTrue:
      cmds.addCommand( "arp monitor mac-address" )
   elif entity.monitorMac == tristateBool.isFalse:
      cmds.addCommand( "no arp monitor mac-address" )
   elif options.saveAll and entity.name.startswith( 'Vlan' ):
      cmds.addCommand( "default arp monitor mac-address" )
   if entity.checkProxyNdEnabled():
      if entity.proxyNdPrefixConnected:
         cmds.addCommand( "ipv6 nd proxy prefix connected" )
      for prefix in sorted( entity.proxyNdPrefix ):
         cmds.addCommand( f"ipv6 nd proxy prefix {prefix}" )
   elif options.saveAll and entity.name.startswith( "Vlan" ):
      cmds.addCommand( "no ipv6 nd proxy" )
   if entity.proxyNdDad:
      cmds.addCommand( "ipv6 nd dad proxy" )
   elif options.saveAll:
      cmds.addCommand( "no ipv6 nd dad proxy" )
   if toggleForcedPeriodicArpRefreshEnabled():
      if entity.forceArpRequestOnTimeout:
         cmds.addCommand( "arp aging refresh always" )
      elif options.saveAll:
         cmds.addCommand( "no arp aging refresh always" )
      if entity.forceNsOnTimeout:
         cmds.addCommand( "ipv6 nd cache refresh always" )
      elif options.saveAll:
         cmds.addCommand( "no ipv6 nd cache refresh always" )

@CliSave.saver( 'Arp::InputConfigVrf', 'arp/input/config/cli',
                attrName = 'vrf' )
def saveVrfArpIntfConfig( entity, root, requireMounts, options ):
   saveStaticArpEntries( entity, root, options, entity.name )
   saveStaticIp6NeighEntries( entity, root, options, entity.name )

def saveStaticArpEntries( entity, root, options, vrfName ):
   arpCmd = 'arp'
   if vrfName != DEFAULT_VRF:
      arpCmd += f' vrf {vrfName}'

   for key in sorted( entity.ipv4 ):
      staticArp = entity.ipv4[ key ]
      cmd = f"{arpCmd} {staticArp.addr.stringValue} {staticArp.ethAddr} arpa"
      root[ 'Ira.arp' ].addCommand( cmd )

def saveStaticIp6NeighEntries( entity, root, options, vrfName ):
   snCmd = 'ipv6 neighbor'
   if vrfName != DEFAULT_VRF:
      snCmd += f' vrf {vrfName}'

   for sn in sorted( entity.ipv6 ):
      neighbor = entity.ipv6[ sn ]
      cmd = (
         f'{snCmd} {neighbor.addr.stringValue} {neighbor.intfId} '
         f'{neighbor.ethAddr}'
      )
      root[ 'Ira.staticNeighbor6' ].addCommand( cmd )

