#!/usr/bin/env python3
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import CliSave
import MultiRangeRule
import Arnet
from CliSavePlugin.RoutingBgpCliSave import RouterBgpBaseConfigMode
from CliMode.BgpMacVrfConfigMode import (
   BgpMacVrfMode,
   BgpMacVrfVpwsPwMode,
)

from BgpLib import routeTargetToPrint
from ArnetLib import formatRd
from RouteMapLib import isAsdotConfigured, bgpFormatAsn
from Toggles.ArBgpToggleLib import ( toggleVpwsRouteTargetExportRcfEnabled,
                                     toggleEvpnUmrEnabled )
from TypeFuture import TacLazyType

import Tracing

th = Tracing.Handle( 'BgpMacVrfConfigCliSave' )
t0 = th.trace0

ControlWordOverride = TacLazyType( "Pseudowire::Hardware::ControlWordOverride" )
EvpnEtid = TacLazyType( "Evpn::Etid" )
FxcMode = TacLazyType( "Pseudowire::Fxc::Mode" )

class BgpMacVrfConfigSaveMode( BgpMacVrfMode, CliSave.Mode ):
   def __init__( self, params ):
      BgpMacVrfMode.__init__( self, *params )
      CliSave.Mode.__init__( self, params[ 0 ] )
      self.saveOrder_ = 1
      if self.isVniMacVrf:
         self.saveOrder_ = 2
      elif self.isVpws:
         self.saveOrder_ = 3
      elif self.isBundle:
         self.saveOrder_ = 4

   def instanceKey( self ):
      '''Used for sorting Mode instances.'''
      return ( self.saveOrder_, Arnet.intfNameKey( self.macVrfId ) )

RouterBgpBaseConfigMode.addChildMode( BgpMacVrfConfigSaveMode )
BgpMacVrfConfigSaveMode.addCommandSequence( 'Bgp.macvrf.config' )

class BgpMacVrfVpwsPwSaveMode( BgpMacVrfVpwsPwMode, CliSave.Mode ):
   def __init__( self, pwName ):
      BgpMacVrfVpwsPwMode.__init__( self, pwName )
      CliSave.Mode.__init__( self, pwName )

BgpMacVrfConfigSaveMode.addChildMode( BgpMacVrfVpwsPwSaveMode )
BgpMacVrfVpwsPwSaveMode.addCommandSequence( 'Bgp.macvrf.pw.config' )

def routeTargetToPrintCached( routeTargetCache, rt, asdotConfigured ):
   rtStr = routeTargetCache.get( rt )
   if rtStr is None:
      rtStr = routeTargetToPrint( rt, asdotConfigured )
      routeTargetCache[ rt ] = rtStr
   return rtStr

def saveRouteTarget( routeTargetCache, cmds, macVrfConfig,
                     asdotConfigured, options ):
   # route-target < import | export | import export | both >
   #     [ evpn domain < remote | all > ] RT
   #
   # EVPN VPWS MAC-VRFs share this helper method but they use a slightly
   # different route-target syntax:
   # - extra "evpn" token for local domain RTs, this should be supported
   #   by regular MAC-VRFs too in the future, see bug/951376
   # - no "both" (only uses "import export")
   isVpwsMacVrf = macVrfConfig.isVpwsMacVrf()
   evpn = ' evpn' if isVpwsMacVrf else ''
   for rt in sorted( macVrfConfig.importRtList ):
      # Check importExportAll, importAll, importExport, both
      if rt not in macVrfConfig.routeTargetCli:
         direction = 'import' + evpn
      else:
         cliFlags = macVrfConfig.routeTargetCli[ rt ]
         if cliFlags.importExportAll:
            direction = 'import export evpn domain all'
         elif cliFlags.importAll:
            direction = 'import evpn domain all'
         elif cliFlags.importExport:
            direction = 'import export' + evpn
         elif cliFlags.both:
            assert not isVpwsMacVrf
            direction = 'both'
         else:
            direction = 'import' + evpn
      cmds.addCommand(
         'route-target {} {}'.format( direction,
            routeTargetToPrintCached( routeTargetCache, rt, asdotConfigured ) ) )
   for rt in sorted( macVrfConfig.importRemoteDomainRtList ):
      # Check importExportRemote
      if rt not in macVrfConfig.routeTargetCli:
         alreadyPrinted = False
         direction = 'import evpn domain remote'
      else:
         cliFlags = macVrfConfig.routeTargetCli[ rt ]
         alreadyPrinted = cliFlags.importExportAll or cliFlags.importAll
         if cliFlags.importExportRemote:
            direction = 'import export evpn domain remote'
         else:
            direction = 'import evpn domain remote'
      if not alreadyPrinted:
         cmds.addCommand(
            'route-target {} {}'.format( direction,
               routeTargetToPrintCached( routeTargetCache, rt, asdotConfigured ) ) )
   for rt in sorted( macVrfConfig.exportRtList ):
      # Check exportAll
      if rt not in macVrfConfig.routeTargetCli:
         alreadyPrinted = False
         direction = 'export' + evpn
      else:
         cliFlags = macVrfConfig.routeTargetCli[ rt ]
         alreadyPrinted = ( cliFlags.importExportAll or cliFlags.both or
                            cliFlags.importExport )
         if cliFlags.exportAll:
            direction = 'export evpn domain all'
         else:
            direction = 'export' + evpn
      if not alreadyPrinted:
         cmds.addCommand(
            'route-target {} {}'.format( direction,
              routeTargetToPrintCached( routeTargetCache, rt, asdotConfigured ) ) )
   for rt in sorted( macVrfConfig.exportRemoteDomainRtList ):
      direction = 'export evpn domain remote'
      if rt not in macVrfConfig.routeTargetCli:
         alreadyPrinted = False
      else:
         cliFlags = macVrfConfig.routeTargetCli[ rt ]
         alreadyPrinted = ( cliFlags.importExportAll or cliFlags.exportAll or
                            cliFlags.importExportRemote )
      if not alreadyPrinted:
         cmds.addCommand(
            'route-target {} {}'.format( direction,
               routeTargetToPrintCached( routeTargetCache, rt, asdotConfigured ) ) )

   if isVpwsMacVrf:
      # add route-target export rcf
      if toggleVpwsRouteTargetExportRcfEnabled():
         cmd = 'route-target export evpn rcf'
         if macVrfConfig.exportRtRcf:
            cmds.addCommand( cmd + ' ' + macVrfConfig.exportRtRcf + '()' )
         elif options.saveAll:
            cmds.addCommand( 'no ' + cmd )
   else:
      # add route-target export auto
      if macVrfConfig.autoExportRt:
         if macVrfConfig.autoExportRtAsn > 0:
            cmds.addCommand( 'route-target export auto %s' % (
                             bgpFormatAsn( macVrfConfig.autoExportRtAsn,
                                           asdotConfigured ) ) )
         else:
            cmds.addCommand( 'route-target export auto' )

      # add route-target import auto < asn >
      for asn in sorted( macVrfConfig.autoImportRtList ):
         cmds.addCommand( 'route-target import auto %s' % (
                          bgpFormatAsn( asn, asdotConfigured ) ) )

class VpwsMacVrfConfigSaver:
   def __init__( self, macVrfConfig, macVrfMode, bgpConfig, asdotConfigured ):
      self.macVrfConfig = macVrfConfig
      self.macVrfMode = macVrfMode
      self.bgpConfig = bgpConfig
      self.asdotConfigured = asdotConfigured

   def saveMplsControlWord( self, cmds, options ):
      """mpls control-word"""
      if self.macVrfConfig.mplsControlWord:
         cmds.addCommand( 'mpls control-word' )
      elif options.saveAll:
         cmds.addCommand( 'no mpls control-word' )

   def saveFlowLabel( self, cmds, options ):
      """label flow"""
      if self.macVrfConfig.flowLabel:
         cmds.addCommand( 'label flow' )
      elif options.saveAll:
         cmds.addCommand( 'no label flow' )

   def saveMtu( self, cmds, options ):
      """mtu ( ignore | MTU )"""
      if self.macVrfConfig.pseudowireMtu != 0:
         cmds.addCommand( f'mtu {self.macVrfConfig.pseudowireMtu}' )
      elif options.saveAll:
         cmds.addCommand( 'mtu ignore' )

   def saveFxcSignaling( self, cmds, options ):
      """flexible-cross-connect"""
      t0( 'flexible-cross-connect: macVrfConfig.fxcMode', self.macVrfConfig.fxcMode )
      if self.macVrfConfig.fxcMode == FxcMode.defaultFxc:
         t0( "flexible-cross-connect: saving as defaultFxc" )
         cmds.addCommand( 'flexible-cross-connect' )
      elif options.saveAll:
         t0( "flexible-cross-connect: saving as notFxc for saveAll" )
         cmds.addCommand( 'no flexible-cross-connect' )
      else:
         t0( "flexible-cross-connect: not saving" )

   def savePwEvpnVpwsId( self, cmds, options, pwConfig ):
      """evpn vpws id local LOCAL remote REMOTE"""
      if not ( pwConfig.vpwsIdLocal == EvpnEtid.max and
               pwConfig.vpwsIdRemote == EvpnEtid.max ):
         cmds.addCommand( 'evpn vpws id local {} remote {}'.format(
            pwConfig.vpwsIdLocal, pwConfig.vpwsIdRemote ) )
      elif options.saveAll:
         cmds.addCommand( 'no evpn vpws id' )

   @staticmethod
   def _saveOneOverride( cmds, options, direction, value ):
      status = {
         ControlWordOverride.cwDisabled: 'disabled',
         ControlWordOverride.cwAlways: 'always',
      }.get( value )

      cmd = 'mpls control-word data-plane {direction}'
      if status:
         cmds.addCommand( cmd.format( direction=direction ) + " " + status )
      elif options.saveAll:
         cmds.addCommand( 'no ' + cmd.format( direction=direction ) )

   def savePwControlWordDataPlaneOverride( self, cmds, options, pwConfig ):
      """mpls control-word data-plane (receive|transmit) (always|disabled)"""
      override = pwConfig.controlWordOverride
      self._saveOneOverride( cmds, options, 'receive', override.rxOverride )
      self._saveOneOverride( cmds, options, 'transmit', override.txOverride )

   def savePwColor( self, cmds, options, pwConfig ):
      """color INTEGER"""
      if pwConfig.color is not None:
         cmds.addCommand( f'color {pwConfig.color}' )
      elif options.saveAll:
         cmds.addCommand( 'no color' )

   def savePw( self, options, pwName ):
      """pseudowire NAME"""
      pwConfig = self.macVrfConfig.pseudowireConfig.get( pwName )
      if not pwConfig:
         return
      pwMode = self.macVrfMode[ BgpMacVrfVpwsPwSaveMode ].getOrCreateModeInstance(
            pwName )
      cmds = pwMode[ 'Bgp.macvrf.pw.config' ]
      self.savePwEvpnVpwsId( cmds, options, pwConfig )
      self.savePwControlWordDataPlaneOverride( cmds, options, pwConfig )
      self.savePwColor( cmds, options, pwConfig )

   def save( self, cmds, options ):
      self.saveMplsControlWord( cmds, options )
      self.saveFlowLabel( cmds, options )
      self.saveMtu( cmds, options )
      self.saveFxcSignaling( cmds, options )
      for pwName in sorted( self.macVrfConfig.pseudowireConfig ):
         self.savePw( options, pwName )

def getRedistributeCmdCached( redistributeCache, macVrfConfig ):
   cmds = redistributeCache.get( macVrfConfig.redistribute )
   if cmds is not None:
      return cmds

   cmds = []
   if macVrfConfig.isRedistributeAll():
      cmds.append( 'redistribute all' )
   else:
      if macVrfConfig.isInRedistribute( 'redistributeLearned' ):
         cmds.append( 'redistribute learned' )
      if macVrfConfig.isInRedistribute( 'redistributeStatic' ):
         cmds.append( 'redistribute static' )
      if macVrfConfig.isInRedistribute( 'redistributeDot1x' ):
         cmds.append( 'redistribute dot1x' )
      if macVrfConfig.isInRedistribute( 'redistributeLinkLocal' ):
         cmds.append( 'redistribute link-local ipv6' )
      if macVrfConfig.isInRedistribute( 'redistributeIgmp' ):
         cmds.append( 'redistribute igmp' )
      if macVrfConfig.isInRedistribute( 'redistributeMld' ):
         cmds.append( 'redistribute mld' )
      if macVrfConfig.isInRedistribute( 'redistributeSysMac' ):
         cmd = 'redistribute router-mac system'
         if macVrfConfig.isInRedistribute( 'redistributeSysMacGateway' ):
            cmd += ' default-gateway'
         cmds.append( cmd )
      if not macVrfConfig.isInRedistribute( 'redistributeHostRoute' ):
         cmds.append( 'no redistribute host-route' )
      if not macVrfConfig.isInRedistribute( 'redistributeRouterMac' ):
         cmds.append( 'no redistribute router-mac' )
      if macVrfConfig.isInRedistribute( 'redistributeRemoteLearned' ):
         cmds.append( 'redistribute learned remote' )
      if macVrfConfig.isInRedistribute( 'redistributeRouterMacPrimary' ):
         cmds.append( 'redistribute router-mac next-hop vtep primary' )
      if macVrfConfig.isInRedistribute( 'redistributeRouterMacVirtualIp' ):
         cmd = 'redistribute router-mac virtual-ip'
         if macVrfConfig.isInRedistribute( 'redistributeRouterMacVirtualIpPrimary' ):
            cmd += ' next-hop vtep primary'
         cmds.append( cmd )
      if macVrfConfig.isInRedistribute( 'redistributeSysMacPrimaryIp' ):
         cmds.append( 'redistribute router-mac system ip' )
      if macVrfConfig.isInRedistribute( 'redistributeMrouter' ):
         cmds.append( 'redistribute multicast-router' )

   redistributeCache[ macVrfConfig.redistribute ] = cmds
   return cmds

def saveMacVrfConfig( macVrfConfig, parentMode, bgpConfig, asdotConfigured,
                      options, redistributeCache ):
   routeTargetCache = {}
   isVlanMacVrf = macVrfConfig.isVlanMacVrf()
   isVniMacVrf = macVrfConfig.isVniMacVrf()
   isVpwsMacVrf = macVrfConfig.isVpwsMacVrf()
   mode = parentMode[ BgpMacVrfConfigSaveMode ].getOrCreateModeInstance(
      ( macVrfConfig.name, macVrfConfig.isBundle, isVniMacVrf, isVpwsMacVrf ) )
   cmds = mode[ 'Bgp.macvrf.config' ]

   rd = macVrfConfig.rd
   remoteRd = macVrfConfig.remoteRd
   if macVrfConfig.rdAll:
      cmds.addCommand( 'rd evpn domain all ' + formatRd( rd, asdotConfigured ) )
   else:
      if rd and rd != 'INVALID':
         cmds.addCommand( 'rd ' + formatRd( rd, asdotConfigured ) )
      if macVrfConfig.autoRd:
         cmds.addCommand( 'rd auto' )
      if remoteRd and remoteRd != 'INVALID':
         cmds.addCommand( 'rd evpn domain remote ' +
                          formatRd( remoteRd, asdotConfigured ) )

   saveRouteTarget( routeTargetCache, cmds, macVrfConfig, asdotConfigured, options )

   # redistribute < learned [remote] | static | dot1x |
   #                link-local ipv6 |
   #                router-mac [system [ip | default-gateway] |
   #                            (next-hop vtep primary) |
   #                            virtual-ip [ next-hop vtep primary ] ] ) |
   #                host-route | igmp |  multicast-router | all >

   for cmd in getRedistributeCmdCached( redistributeCache, macVrfConfig ):
      cmds.addCommand( cmd )

   if isVlanMacVrf:
      if options.saveAll:
         if macVrfConfig.isInRedistribute( 'redistributeRouterMac' ):
            cmds.addCommand( 'redistribute router-mac' )
         if macVrfConfig.isInRedistribute( 'redistributeHostRoute' ):
            cmds.addCommand( 'redistribute host-route' )

   if macVrfConfig.isVniMacVrf():
      if macVrfConfig.redistributeVcs:
         cmds.addCommand( 'redistribute service vxlan' )

   if macVrfConfig.macAliasDefaultGateway:
      cmds.addCommand( 'router mac-address alias default-gateway' )

   if macVrfConfig.isBundle:
      # vlan-aware-bundle configuration
      if isVniMacVrf:
         cmdStr = 'vni'
      else:
         cmdStr = 'vlan'
      if macVrfConfig.brIdToEtId:
         defaultEtidKeys = []
         nonDefaultEtidKeys = []
         for bdId in sorted( macVrfConfig.brIdToEtId ):
            etid = macVrfConfig.brIdToEtId.get( bdId )
            if etid == 0:
               if macVrfConfig.isVniMacVrf():
                  defaultEtidKeys.append( bdId.vni )
               else:
                  defaultEtidKeys.append( bdId.vlanId )
            else:
               nonDefaultEtidKeys.append( bdId )
         rangeStr = MultiRangeRule.multiRangeToCanonicalString( defaultEtidKeys )
         if rangeStr:
            cmds.addCommand( '%s ' % ( cmdStr ) + rangeStr )
         for bdId in nonDefaultEtidKeys:
            etid = macVrfConfig.brIdToEtId[ bdId ]
            cmds.addCommand( f'{cmdStr} {bdId.vlanId} etid {etid}' )
   elif isVpwsMacVrf:
      # VPWS
      vpws = VpwsMacVrfConfigSaver( macVrfConfig, mode, bgpConfig, asdotConfigured )
      vpws.save( cmds, options )

   # maximum-routes configuration
   if macVrfConfig.maxRoutes != 0:
      cmds.addCommand( 'maximum-routes %d' % macVrfConfig.maxRoutes )
   elif options.saveAll:
      if isVlanMacVrf or isVniMacVrf:
         cmds.addCommand( 'maximum-routes 0' )

   if macVrfConfig.preferenceMode.isSet:
      cmds.addCommand(
         'designated-forwarder election preference rule ' +
         ( 'low' if macVrfConfig.preferenceMode.value == 'preferenceLow' else
           'high' ) )
   elif options.saveAll:
      if macVrfConfig.isVlanMacVrf() or macVrfConfig.isVpwsMacVrf():
         cmds.addCommand( 'designated-forwarder election preference rule high' )

   if toggleEvpnUmrEnabled() and isVlanMacVrf:
      if macVrfConfig.originateUmr:
         cmds.addCommand( 'unknown-mac-route originate' )
      elif options.saveAll:
         cmds.addCommand( 'no unknown-mac-route originate' )

@CliSave.saver( 'Routing::Bgp::MacVrfConfigDir', 'routing/bgp/macvrf',
                requireMounts = ( 'routing/bgp/config',
                                  'routing/bgp/asn/config' ) )
def saveBgpMacVrfConfig( macVrfConfigDir, root, requireMounts, options ):
   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   if bgpConfig.asNumber == 0:
      return
   
   asdotConfigured = isAsdotConfigured( asnConfig )
   parentMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
         bgpConfig.asNumber, asdotConfigured ) )
   redistributeCache = {}
   for vrfConfig in macVrfConfigDir.config.values():
      saveMacVrfConfig( vrfConfig, parentMode, bgpConfig, asdotConfigured, options,
                        redistributeCache )
