# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac, CliSave
from CliSavePlugin.RoutingBgpCliSave import RouterBgpBaseConfigMode
from CliSavePlugin.RoutingBgpCliSave import RouterBgpVrfConfigMode
from IpLibConsts import DEFAULT_VRF
from RouteMapLib import isAsdotConfigured

RouterBgpBaseConfigMode.addCommandSequence( 'ArBgp.trace',
                                            before=[ 'Bgp.config' ] )
RouterBgpVrfConfigMode.addCommandSequence( 'ArBgp.vrf.trace',
                                           before=[ 'Bgp.vrf.config' ] )

def saveBgpTraceFacility( traceFacility, options ):
   cmds = []
   saveAll = options.saveAll

   if traceFacility.peerEntryForAll.isSet or saveAll:
      if traceFacility.peerEntryForAll.enabled:
         cmds.append( 'bgp trace neighbor all' )
      else:
         cmds.append( 'no bgp trace neighbor all' )

   for addr in traceFacility.peerEntry:
      # pylint: disable-next=consider-using-f-string
      cmds.append( 'bgp trace neighbor %s' % str( addr ) )

   if traceFacility.routeKeyEntryForAll.isSet or saveAll:
      if traceFacility.routeKeyEntryForAll.enabled:
         cmds.append( 'bgp trace route-key all' )
      else:
         cmds.append( 'no bgp trace route-key all' )

   for afiSafi, routeKeyEntryForAfiSafi in \
         traceFacility.routeKeyEntryForAfiSafi.items():
      if routeKeyEntryForAfiSafi.isSet:
         if afiSafi.afi == 'afiIpv4' and \
               afiSafi.safi == 'safiUnicast':
            afiSafiToken = 'ipv4 unicast'
         elif afiSafi.afi == 'afiIpv6' and \
               afiSafi.safi == 'safiUnicast':
            afiSafiToken = 'ipv6 unicast'
         else:
            continue
         if routeKeyEntryForAfiSafi.enabled:
            # pylint: disable-next=consider-using-f-string
            cmds.append( 'bgp trace route-key %s' % afiSafiToken )
         else:
            # pylint: disable-next=consider-using-f-string
            cmds.append( 'no bgp trace route-key %s' % afiSafiToken )

   for prefix in traceFacility.routeKeyEntryForPrefix:
      # pylint: disable-next=consider-using-f-string
      cmds.append( 'bgp trace route-key %s' % str( prefix ) )

   return cmds

@CliSave.saver( 'Routing::Bgp::VrfTraceFacilityDir', 'routing/arbgp/trace',
                requireMounts = ( 'routing/bgp/config',
                                  'routing/bgp/vrf/config',
                                  'routing/bgp/asn/config' ) )
def saveBgpVrfTraceFacilityDir( vrfTraceFacilityDir, root, requireMounts, options ):
   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   vrfConfigDir = requireMounts[ 'routing/bgp/vrf/config' ]
   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   if bgpConfig.asNumber == 0 or \
         DEFAULT_VRF not in vrfTraceFacilityDir.traceFacility:
      return

   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
         bgpConfig.asNumber, isAsdotConfigured( asnConfig ), ) )

   traceFacility = vrfTraceFacilityDir.traceFacility.get( DEFAULT_VRF )
   if traceFacility is not None:
      cmds = saveBgpTraceFacility( traceFacility, options )
      for cmd in cmds:
         bgpMode[ 'ArBgp.trace' ].addCommand( cmd )

   for vrfName in vrfConfigDir.vrfConfig:
      bgpVrfMode = bgpMode[ RouterBgpVrfConfigMode
            ].getOrCreateModeInstance( vrfName )
      traceFacility = vrfTraceFacilityDir.traceFacility.get( vrfName )
      if traceFacility is not None:
         cmds = saveBgpTraceFacility( traceFacility, options )
         for cmd in cmds:
            bgpVrfMode[ 'ArBgp.vrf.trace' ].addCommand( cmd )
