# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import CliSave
import Tac
from CliMode.Bmp import RoutingBmpStationMode
from CliSavePlugin import RoutingBgpCliSave
from CliSavePlugin.RoutingBgpCliSave import (
   getExplicitNoCmd,
   RouterBgpBaseConfigMode,
   RouterBgpVrfConfigMode,
)
from CliSavePlugin.Security import mgmtSecurityConfigPath
from IpLibConsts import DEFAULT_VRF
import ReversibleSecretCli
from RouteMapLib import isAsdotConfigured
import Toggles.BmpToggleLib
from Toggles.BmpToggleLib import toggleBmpAdjRibinVrfFilteringEnabled

ConnectionMode = Tac.Type( 'Routing::Bmp::ConnectionMode' )
TimestampMode = Tac.Type( 'Routing::Bmp::TimestampMode' )
adjRibOutExportToggle = Toggles.BmpToggleLib.toggleBmpAdjRibOutExportEnabled()
addPathToggle = True

#------------------------------------------------------------------------------------
# monitoring station mode
#------------------------------------------------------------------------------------
class RouterBmpStationConfigMode( RoutingBmpStationMode, CliSave.Mode ):
   def __init__( self, param ):
      RoutingBmpStationMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

RouterBgpBaseConfigMode.addChildMode( RouterBmpStationConfigMode )
RouterBmpStationConfigMode.addCommandSequence( 'Bgp.bmp.station' )

def saveBmpStationConfig( stationConfig, root, options, stationName, parentMode,
                          securityConfig ):

   mode = parentMode[ RouterBmpStationConfigMode ].getOrCreateModeInstance(
         stationName )
   cmds = mode[ 'Bgp.bmp.station' ]

   def isConfiguredAttribute( attrStr ):
      attr = getattr( stationConfig, attrStr )
      attrDefault = getattr( stationConfig, attrStr + 'Default' )
      return attr != attrDefault

   def addSimpleSaveCommand( attrStr, cmd, trail=None, add=True, integer=False ):
      attr = getattr( stationConfig, attrStr )
      attrDefault = getattr( stationConfig, attrStr + 'Default' )
      present = False
      saveCmd = ''

      if isConfiguredAttribute( attrStr ):
         saveCmd = '%s' % ( cmd )
         if attrStr == 'shutdown':
            shutdownMsgAttr = stationConfig.shutdownMsg
            shutdownMsgAttrDefault = stationConfig.shutdownMsgDefault
            if shutdownMsgAttr != shutdownMsgAttrDefault:
               saveCmd += ' reason %s' % shutdownMsgAttr
         if attr == attrDefault:
            saveCmd = 'no ' + saveCmd
         elif not isinstance( attr, bool ):
            if integer:
               attr = int( attr )
            saveCmd = saveCmd + ' ' + str( attr )
            present = True
            if trail:
               saveCmd = saveCmd + ' ' + trail
      elif options.saveAll:
         saveCmd = 'no %s' % ( cmd )
         present = False
      if add and saveCmd:
         cmds.addCommand( saveCmd )
      return saveCmd, present

   addSimpleSaveCommand( 'description', 'description' )
   addSimpleSaveCommand( 'shutdown', 'shutdown' )
   addSimpleSaveCommand( 'sourceIntf', 'update-source' )
   addSimpleSaveCommand( 'statistics', 'statistics' )
   if isConfiguredAttribute( 'authenticationKey' ):
      cmd = ReversibleSecretCli.getCliSaveCommand( 'authentication-key {}',
                                                   securityConfig,
                                                   stationConfig.authenticationKey,
                                                   uniqueKey=stationName + '_passwd',
                                                   algorithm='DES' )
      cmds.addCommand( cmd )
   elif options.saveAll:
      cmds.addCommand( 'no authentication-key' )

   #------------------------------------------------------------------------------
   # CliSave for "[no|default] connection address <ip|ipv6|hostname> [vrf <VRF>]"
   # command in "monitoring station" mode.
   #------------------------------------------------------------------------------

   addrCmd = 'connection address'
   ( vrfCmd, vrfPresent ) = addSimpleSaveCommand( 'vrfName', 'vrf', add=False )
   if isConfiguredAttribute( 'remoteHost' ):
      if vrfPresent:
         ( addrCmd, _ ) = addSimpleSaveCommand( 'remoteHost', addrCmd,
                                                add=False )
         cmds.addCommand( f'{addrCmd} {vrfCmd}' )
      else:
         ( _, _ ) = addSimpleSaveCommand( 'remoteHost', addrCmd, add=True )
   else:
      if options.saveAll:
         cmds.addCommand( 'no %s' % addrCmd )

   #-------------------------------------------------------------------------------
   # CliSave for command
   # "[no|default] connection mode {active port <N> [reconnect-timer <N>]} |
   #                               {passive} "
   # in "monitoring station" mode.
   #-------------------------------------------------------------------------------
   modeCmd = 'connection mode'
   if stationConfig.connectionMode == ConnectionMode.passive:
      cmds.addCommand( '%s passive' % modeCmd )
   elif stationConfig.connectionMode == ConnectionMode.active \
         and isConfiguredAttribute( 'remotePort' ):
      remotePort = stationConfig.remotePort
      if isConfiguredAttribute( 'reconnectTime' ):
         reconnectTime = stationConfig.reconnectTime
         cmds.addCommand( '%s active port %s reconnect-timer %s' %
                          ( modeCmd, remotePort, int( reconnectTime ) ) )
      else:
         cmds.addCommand( f'{modeCmd} active port {remotePort}' )
   elif options.saveAll:
      cmds.addCommand( 'no connection mode' )

   keepAlive = stationConfig.keepaliveOption

   idleTime = keepAlive.idleTime 
   probeInterval = keepAlive.probeInterval
   probeCount = keepAlive.probeCount 

   if keepAlive.enabled:
      cmds.addCommand( 'connection keepalive %s %s %s' %
            ( idleTime, probeInterval, probeCount ) )
   elif options.saveAll:
      cmds.addCommand( 'no connection keepalive' )

   if addPathToggle:
      cmd = "export-policy received routes additional-paths"
      if stationConfig.addPathIdConfigured:
         cmds.addCommand( cmd )
      elif options.saveAll:
         cmd = "no " + cmd
         cmds.addCommand( cmd )

   stationRibInExportPolicy = stationConfig.ribInExportPolicyStationConfig
   cmd = "export-policy received routes"

   if stationRibInExportPolicy.isSet:
      if stationRibInExportPolicy.prePolicyExport:
         cmd += " pre-policy"
      if stationRibInExportPolicy.postPolicyExport:
         cmd += " post-policy"
      if not stationRibInExportPolicy.postPolicyExport \
            and not stationRibInExportPolicy.prePolicyExport:
         cmd = "no " + cmd
      cmds.addCommand( cmd )
   elif options.saveAll:
      cmd = "default " + cmd
      cmds.addCommand( cmd )

   if adjRibOutExportToggle:
      stationRibOutExportPolicy = stationConfig.ribOutExportPolicyStationConfig
      cmd = "export-policy advertised routes"
      if stationRibOutExportPolicy.isSet:
         if stationRibOutExportPolicy.postPolicyExport:
            cmd += " post-policy"
         else:
            cmd = "no " + cmd
         cmds.addCommand( cmd )
      elif options.saveAll:
         cmd = "default " + cmd
         cmds.addCommand( cmd )

   if Toggles.BmpToggleLib.toggleBmpLocRibEnabled():
      cmd = 'export-policy bgp rib bestpaths'
      noCmd = getExplicitNoCmd( cmd, nedMode=True )
      if stationConfig.locRibExportPolicyStationConfig.isSet:
         cmds.addCommand( cmd if
            stationConfig.locRibExportPolicyStationConfig.locRibPolicyExport
            else noCmd )
      elif options.saveAll:
         cmds.addCommand( 'no ' + cmd )

RouterBgpBaseConfigMode.addCommandSequence( 'Bgp.config.bmp' )

@CliSave.saver( 'Routing::Bmp::BmpConfig', 'routing/bmp/config',
                requireMounts=( 'routing/bgp/config', 'routing/bgp/asn/config',
                                'routing/bgp/vrf/config', mgmtSecurityConfigPath ) )
def saveBmpConfig( bmpConfig, root, requireMounts, options ):
   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   vrfConfigDir = requireMounts[ 'routing/bgp/vrf/config' ]

   securityConfig = requireMounts[ mgmtSecurityConfigPath ]
   if bgpConfig.asNumber == 0:
      return

   parentMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
         bgpConfig.asNumber, isAsdotConfigured( asnConfig ), ) )

   for stationName in bmpConfig.bmpStation:
      bmpStationConfig = bmpConfig.bmpStation.get( stationName )
      if not bmpStationConfig:
         continue
      saveBmpStationConfig( bmpStationConfig, root, options, stationName,
                            parentMode, securityConfig )

   cmds = parentMode[ 'Bgp.config.bmp' ]
   # BUG157364: Add non-default VRF support
   vrfName = DEFAULT_VRF
   port = bmpConfig.localPort.get( vrfName )
   if port:
      cmds.addCommand( 'monitoring port %s' % port )
   elif options.saveAll:
      cmds.addCommand( 'no monitoring port' )

   prePolicyExport = bmpConfig.ribInExportPolicyConfig.prePolicyExport
   postPolicyExport = bmpConfig.ribInExportPolicyConfig.postPolicyExport
   ribOutExport = bmpConfig.ribOutExportPolicyConfig.postPolicyExport
   locRibPolicyExport = bmpConfig.locRibExportPolicyConfig.locRibPolicyExport

   if not prePolicyExport:
      cmds.addCommand( 'no monitoring received routes pre-policy' )
   elif options.saveAll:
      cmds.addCommand( 'monitoring received routes pre-policy' )

   if not postPolicyExport:
      cmds.addCommand( 'no monitoring received routes post-policy' )
   elif options.saveAll:
      cmds.addCommand( 'monitoring received routes post-policy' )

   if adjRibOutExportToggle:
      if ribOutExport:
         cmds.addCommand( 'monitoring advertised routes post-policy' )
      elif options.saveAll:
         cmds.addCommand( 'no monitoring advertised routes post-policy' )

   afiSafi = Tac.Value( 'Routing::Bgp::AfiSafi', 'afiIpv4', 'safiUnicast' )
   if not bmpConfig.afiSafiExport[ afiSafi ]:
      cmds.addCommand( 'monitoring received routes address-family '
                       'ipv4 unicast disable' )
   elif options.saveAll:
      cmds.addCommand( 'monitoring received routes address-family ipv4 unicast' )

   afiSafi = Tac.Value( 'Routing::Bgp::AfiSafi', 'afiIpv6', 'safiUnicast' )
   if not bmpConfig.afiSafiExport[ afiSafi ]:
      cmds.addCommand( 'monitoring received routes address-family '
                       'ipv6 unicast disable' )
   elif options.saveAll:
      cmds.addCommand( 'monitoring received routes address-family ipv6 unicast' )

   afiSafi = Tac.Value( 'Routing::Bgp::AfiSafi', 'afiIpv6', 'safiMplsLabels' )
   if afiSafi in bmpConfig.afiSafiExport and bmpConfig.afiSafiExport[ afiSafi ]:
      cmd = [ 'monitoring received routes address-family ipv6 labeled-unicast' ]
      if bmpConfig.exportSixPe:
         cmd.append( '6pe' )
      if bmpConfig.exportIpv6LuTunnel:
         cmd.append( 'tunnel' )
      cmds.addCommand( ' '.join( cmd ) )
   elif options.saveAll:
      cmds.addCommand( 'no monitoring received routes address-family'
                       ' ipv6 labeled-unicast' )
   afiSafi = Tac.Value( 'Routing::Bgp::AfiSafi', 'afiIpv4', 'safiMplsVpn' )
   if afiSafi in bmpConfig.afiSafiExport and bmpConfig.afiSafiExport[ afiSafi ]:
      cmds.addCommand( 'monitoring received routes address-family '
                       'vpn-ipv4' )
   elif options.saveAll:
      cmds.addCommand( 'no monitoring received routes address-family vpn-ipv4' )

   afiSafi = Tac.Value( 'Routing::Bgp::AfiSafi', 'afiIpv6', 'safiMplsVpn' )
   if afiSafi in bmpConfig.afiSafiExport and bmpConfig.afiSafiExport[ afiSafi ]:
      cmds.addCommand( 'monitoring received routes address-family '
                       'vpn-ipv6' )
   elif options.saveAll:
      cmds.addCommand( 'no monitoring received routes address-family vpn-ipv6' )

   if Toggles.BmpToggleLib.toggleBmpLocRibEnabled():
      cmd = 'monitoring bgp rib bestpaths'
      noCmd = getExplicitNoCmd( cmd, nedMode=True )
      defaultCmd = 'no ' + cmd
      if bmpConfig.locRibExportPolicyConfig.isSet:
         cmds.addCommand( cmd if locRibPolicyExport else noCmd )
      elif options.saveAll:
         cmds.addCommand( defaultCmd )

   if Toggles.BmpToggleLib.toggleBmpFlowspecSafiEnabled():
      afiSafi = Tac.Value( 'Routing::Bgp::AfiSafi', 'afiIpv4', 'safiFlowspec' )
      if afiSafi in bmpConfig.afiSafiExport and bmpConfig.afiSafiExport[ afiSafi ]:
         cmds.addCommand( 'monitoring received routes address-family '
                          'flow-spec ipv4' )
      elif options.saveAll:
         cmds.addCommand( 
               'no monitoring received routes address-family flow-spec ipv4' )

      afiSafi = Tac.Value( 'Routing::Bgp::AfiSafi', 'afiIpv6', 'safiFlowspec' )
      if afiSafi in bmpConfig.afiSafiExport and bmpConfig.afiSafiExport[ afiSafi ]:
         cmds.addCommand( 'monitoring received routes address-family '
                          'flow-spec ipv6' )
      elif options.saveAll:
         cmds.addCommand( 
               'no monitoring received routes address-family flow-spec ipv6' )

   timestamp = bmpConfig.timestampMode
   if timestamp == TimestampMode.none:
      cmds.addCommand( 'monitoring timestamp none' )
   elif options.saveAll:
      cmds.addCommand( 'monitoring timestamp send-time' )

   if bmpConfig.dscp != bmpConfig.dscpDefault:
      cmds.addCommand( 'monitoring qos dscp %d' % bmpConfig.dscp )
   elif options.saveAll:
      cmds.addCommand( 'no monitoring qos dscp' )

   if bmpConfig.statsInterval != bmpConfig.statsIntervalDefault:
      cmds.addCommand( 'monitoring statistics interval %d seconds' %
            bmpConfig.statsInterval )
   elif options.saveAll:
      cmds.addCommand( 'monitoring statistics interval %d seconds' %
            bmpConfig.statsIntervalDefault )

   if bmpConfig.bmpActivate:
      cmds.addCommand( 'bgp monitoring' )
   elif options.saveAll:
      cmds.addCommand( 'no bgp monitoring' )

   if Toggles.BmpToggleLib.toggleBmpLocRibEnabled():
      saveBmpLocRibMonitoring( bmpConfig, bgpConfig, asnConfig,
                              vrfConfigDir, root, options )

   if toggleBmpAdjRibinVrfFilteringEnabled():
      saveBmpAdjRibinVrfFilter( bmpConfig, bgpConfig, asnConfig,
                                vrfConfigDir, root, options )

# Bgp CliSave for "neighbor * monitoring"
def saveNeighborMonitoring( peer, peerConfig, saveAll ):
   cmd = 'neighbor %s monitoring' % peer
   if peerConfig.bmpActivatePresent:
      if peerConfig.bmpActivate == True: # pylint: disable=singleton-comparison
         return cmd
      else:
         return 'no ' + cmd
   elif saveAll and not peerConfig.isPeerGroupPeer:
      return 'default ' + cmd
   return None

RoutingBgpCliSave.neighborMonitoringHook.addExtension( saveNeighborMonitoring )

def saveBmpLocRibMonitoring( bmpConfig, bgpConfig, asnConfig,
                             vrfConfigDir, root, options ):
   cmd = 'bgp rib bestpaths monitoring'
   noCmd = getExplicitNoCmd( cmd, nedMode=True )
   # 'no' meaning 'default', even though the config is "enabled" by default.
   #
   # > The "no" commands just set the config back to its default rather
   # > than negating it.
   #
   # ref: Adam Sweeney
   # https://groups/cli-review/c/9wiIqtiydLY/m/tl06ThWRBgAJ
   defaultCmd = 'no ' + cmd

   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
      bgpConfig.asNumber, isAsdotConfigured( asnConfig ) ) )

   if bmpConfig.locRibMonitoringEnabledAllVrfDefault is not None:
      enabled = bmpConfig.locRibMonitoringEnabledAllVrfDefault
      bgpMode[ 'Bgp.config' ].addCommand( cmd if enabled else noCmd )
   elif options.saveAll:
      bgpMode[ 'Bgp.config' ].addCommand( defaultCmd )

   def vrfNamesIncludingDefaultVrf():
      yield DEFAULT_VRF
      # yield from vrfConfigDir.vrfConfig
      yield from vrfConfigDir.vrfConfig

   for vrfName in vrfNamesIncludingDefaultVrf():
      monitoringEnabled = bmpConfig.locRibMonitoringEnabled.get( vrfName )
      needSave = options.saveAll or monitoringEnabled is not None
      if not needSave:
         continue
      bgpVrfMode = bgpMode[ RouterBgpVrfConfigMode ].getOrCreateModeInstance(
         vrfName
      )
      if monitoringEnabled is None: # and options.saveAll is True
         bgpVrfMode[ 'Bgp.vrf.config' ].addCommand( defaultCmd )
      else: # monitoredEnabled is either explicit True or False, we need to save it.
         bgpVrfMode[ 'Bgp.vrf.config' ].addCommand(
            cmd if monitoringEnabled else noCmd
         )

def saveBmpAdjRibinVrfFilter( bmpConfig, bgpConfig, asnConfig,
                              vrfConfigDir, root, options ):
   cmd = 'bgp rib received routes monitoring disabled'
   defaultCmd = 'no ' + cmd

   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
      bgpConfig.asNumber, isAsdotConfigured( asnConfig ) ) )

   def vrfNamesIncludingDefaultVrf():
      yield DEFAULT_VRF
      yield from vrfConfigDir.vrfConfig

   for vrfName in vrfNamesIncludingDefaultVrf():
      filtered = vrfName in bmpConfig.ribinExportVrfFilter
      if not filtered and not options.saveAll:
         continue
      bgpVrfMode = bgpMode[ RouterBgpVrfConfigMode ].getOrCreateModeInstance(
         vrfName )
      if filtered:
         bgpVrfMode[ 'Bgp.vrf.config' ].addCommand( cmd )
      else:
         bgpVrfMode[ 'Bgp.vrf.config' ].addCommand( defaultCmd )
