# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import HttpServiceConstants
from CliMode.Capi import CapiMgmtMode, CapiVrfConfigModeBase
import CliSave
from CliSaveBlock import SensitiveCommand
from CliSavePlugin import HttpService
import Tac

CAPI_CONSTANTS = Tac.Type( "HttpService::Constants" )

class CapiConfigMode( CapiMgmtMode, CliSave.Mode ):
   def __init__( self, param ):
      CapiMgmtMode.__init__( self )
      CliSave.Mode.__init__( self, "mgmt-api-http-cmds" )

   def skipIfEmpty( self ):
      return True

CliSave.GlobalConfigMode.addChildMode( CapiConfigMode )
CapiConfigMode.addCommandSequence( 'Mgmt.api-http-cmds' )

class CapiVrfConfigMode( CapiVrfConfigModeBase, CliSave.Mode ):

   def __init__( self, param ):
      CapiVrfConfigModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CapiConfigMode.addChildMode( CapiVrfConfigMode,
                             after=[ 'Mgmt.api-http-cmds' ] )
CapiVrfConfigMode.addCommandSequence( 'Mgmt.api-http-cmds-vrf' )

@CliSave.saver( 'HttpService::Config', 'mgmt/capi/config',
                requireMounts=( 'cli/config', ) )
def saveConfig( config, root, requireMounts, options ):
   mode = root[ CapiConfigMode ].getSingletonInstance()
   currCmds = mode[ 'Mgmt.api-http-cmds' ]

   if not config.useHttpServiceCli:
      for cmd in HttpService.saveCommonHttpServerCmds( config, options ):
         currCmds.addCommand( cmd )

   for cmd in saveCapiConfigCmds( config, options, requireMounts ):
      currCmds.addCommand( cmd )
   saveVrf( config, root, options )

def saveVrf( config, root, options ):
   # Save the VRF in the proper sub-submode
   serviceAclTypeVrfMap = config.serviceAclTypeVrfMap
   aclVrfs = { key.vrfName for key in serviceAclTypeVrfMap.aclName }
   mode = root[ CapiConfigMode ].getSingletonInstance()

   for ( vrf, vrfConfig ) in config.vrfConfig.items():
      if 'http-commands' in vrfConfig.vrfService:
         vrfMode = mode[ CapiVrfConfigMode
                       ].getOrCreateModeInstance( ( vrf, config ) )
         vrfCmd = vrfMode[ 'Mgmt.api-http-cmds-vrf' ]
         vrfCmd.addCommand( 'no shutdown' )

   for vrf in aclVrfs:
      vrfMode = mode[ CapiVrfConfigMode ].getOrCreateModeInstance( ( vrf, config ) )
      vrfCmd = vrfMode[ 'Mgmt.api-http-cmds-vrf' ]
      for aclType in [ 'ip', 'ipv6' ]:
         key = Tac.Value( "Acl::AclTypeAndVrfName", aclType, vrf )
         aclName = ( serviceAclTypeVrfMap and
                     serviceAclTypeVrfMap.aclName.get( key ) )
         if aclName is None or aclName == '':
            continue
         vrfCmd.addCommand( f'{aclType} access-group {aclName}' )

def saveCapiConfigCmds( config, options, requireMounts ):

   cmds = []

   # We can't configure cert/cipher when using HttpService cli so don't save it
   if not config.useHttpServiceCli:
      saveHttpsCert( cmds, options,
                     config.httpsConfig.sslCertificate,
                     config.httpsConfig.sslKey )
      saveHttpsCipher( cmds, options, config.httpsConfig )

   # Add the "no shutdown" last, so we don't continuously restart the
   # server on startup.
   if ( 'http-commands' in config.service and
         config.service[ 'http-commands' ].enabled ):
      cmds.append( "no shutdown" )
   elif options.saveAll:
      cmds.append( "shutdown" )
   cliConfig = requireMounts[ 'cli/config' ]
   if cliConfig.validateOutput:
      cmds.append( "validate-output" )

   saveSessionTimeoutParams( cmds, options, config.sessionTimeout )
   return cmds

def saveSessionTimeoutParams( cmds, options, timeoutValue ):
   if timeoutValue == CAPI_CONSTANTS.defaultSessionTimeout:
      if options.saveAll:
         cmds.append( f"session timeout {timeoutValue} minutes" )
   else:
      cmds.append( f"session timeout {timeoutValue} minutes" )

def saveHttpsCert( cmds, options, cert, key ):
   if not cert and not key:
      if options.saveAll:
         cmds.append( 'no protocol https certificate' )
   else:
      formatStr = 'protocol https certificate\n{}\nEOF\n{}\nEOF'
      cmd = SensitiveCommand( formatStr,
                              cert.rstrip( '\n' ),
                              key.rstrip( '\n' ) )
      cmds.append( cmd )

def sslConfigs( cmds, cmd, config, options, defaultValues, hasConfig ):
   if config:
      cmds.append( cmd % " ".join( config.values() ) )
   elif hasConfig:
      cmds.append( cmd % " ".join( defaultValues ) )

def saveHttpsCipher( cmds, options, httpsConfig ):
   # If options are not configured then nothing is saved. This is to
   # differentiate the case where user has enabled all the options ( vs )
   # has not configured any thing. If no option is configured  we allow
   # openssl to use all the builtin CipherSuites.
   sslConfigs( cmds, "protocol https cipher %s", httpsConfig.ciphers,
               options, HttpServiceConstants.CapiCipherSuites.CIPHERS,
               httpsConfig.hasCipherConfig )
   sslConfigs( cmds, "protocol https key-exchange %s",
               httpsConfig.keyExchange, options,
               HttpServiceConstants.CapiCipherSuites.KEYEXCHANGE,
               httpsConfig.hasKeyExchangeConfig )
   sslConfigs( cmds, "protocol https mac %s", httpsConfig.mac,
               options, HttpServiceConstants.CapiCipherSuites.MAC,
               httpsConfig.hasMacConfig )

