# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
import Tac
import functools
from AclCliLib import dscpAclNames
from MultiRangeRule import multiRangeToCanonicalString
from CliMode.Cbf import ( SrTeColorDscpModeBase, TeCbfModeBase, TeColorDscpModeBase,
                          TeColorTcModeBase, TeColorVrfModeBase,
                          CbfFwdPlaneMplsSubModeBase )
from CliSavePlugin.TeCliSave import RouterGlobalTeMode
from CliSavePlugin.SrTePolicyCliSave import SrTeMode, getOrCreateSrTeModeInst
from Toggles import IpLibToggleLib

_dscpValueToName = {}
for _dscpName, ( _dscpValue, _ ) in dscpAclNames.items():
   _dscpValueToName[ _dscpValue ] = _dscpName

def _dscpMixedTypeCmp( a, b ):
   '''
   Sort objects of the same type in their natural (object-defined) order.

   For strings, make them sort at a lower value than all other types.  For example,
   with [3, 'af12', 1, 'ef'], produce ['af12', 'ef', 1, 3].
   '''
   if type( a ) != type( b ):  # pylint: disable=C0123
      return -1 if isinstance( a, str ) else 1
   return ( a > b ) - ( a < b )

# -------------------------------------------------------------------------------
#  sr-te color-dscp mappings cli save config
# -------------------------------------------------------------------------------

class SrTeColorDscpMode( SrTeColorDscpModeBase, CliSave.Mode ):
   def __init__( self, param ):
      SrTeColorDscpModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

SrTeMode.addChildMode( SrTeColorDscpMode )
SrTeColorDscpMode.addCommandSequence( 'SrTePolicy.color-dscp.config' )

@CliSave.saver( 'TrafficEngineering::CbfConfigMap',
                'te/segmentrouting/cbf/color-dscp/config',
                requireMounts=( 'te/segmentrouting/srtepolicy/staticconfig', ) )
def saveSrTeCbfColorDscpMappings( config, root, requireMounts, options ):
   # do not create our mode unless SrTePolicy is enabled since it's our parent
   # otherwise we'd enable SrTePolicy implicitly.
   if not requireMounts[ 'te/segmentrouting/srtepolicy/staticconfig' ].enabled:
      return
   p = getOrCreateSrTeModeInst( root )
   colorDscpMappingsMode = p[ SrTeColorDscpMode ].getSingletonInstance()
   cmds = colorDscpMappingsMode[ 'SrTePolicy.color-dscp.config' ]
   cmdFormat = 'color {color} dscp {valuesStr}'
   for color in sorted( config.colorToValue ):
      dscpList = config.colorToValue[ color ]
      if dscpList.isDefault:
         cmds.addCommand( f'color {color} default' )
         continue
      haveSymbolic = any( dscpList.cliSaveUseSymbolic_i.values() )
      if haveSymbolic:
         dscpValues = []
         for value in dscpList.value:
            if dscpList.cliSaveUseSymbolic_i.get( value, False ):
               dscpValues.append( _dscpValueToName[ value.dscp ] )
            else:
               dscpValues.append( value.dscp )
         # Use the _dscpMixedTypeCmp comparison function so that we show the
         # symbolic DSCP values before the raw integer values.  This choice
         # of ordering is arbitrary.
         dscpValuesStr = ' '.join(
            map( str, sorted( dscpValues,
                              key=functools.cmp_to_key( _dscpMixedTypeCmp ) ) ) )
      else:
         dscpValuesStr = multiRangeToCanonicalString(
            [ v.dscp for v in dscpList.value ] )
      cmds.addCommand( cmdFormat.format( color=color, valuesStr=dscpValuesStr ) )

# -------------------------------------------------------------------------------
#  te color-dscp mappings cli save config
# -------------------------------------------------------------------------------

class TeCbfMode( TeCbfModeBase, CliSave.Mode ):
   def __init__( self, param ):
      TeCbfModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

RouterGlobalTeMode.addChildMode( TeCbfMode,
                                 after=[ 'TrafficEngineering.global' ] )
TeCbfMode.addCommandSequence( 'TrafficEngineering.cbf.config' )

class TeColorDscpMappingsMode( TeColorDscpModeBase, CliSave.Mode ):
   def __init__( self, param ):
      TeColorDscpModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

TeCbfMode.addChildMode( TeColorDscpMappingsMode,
                        after=[ 'TrafficEngineering.cbf.config' ] )
TeColorDscpMappingsMode.addCommandSequence(
                        'TrafficEngineering.cbf.color-dscp.config' )

@CliSave.saver( 'TrafficEngineering::CbfConfigMap',
                'te/cbf/color-dscp/config',
                requireMounts=( 'te/config', ) )
def saveTeCbfColorDscpMappings( teColorDscpConfig, root, requireMounts, options ):
   teConfig = requireMounts[ 'te/config' ]
   if teConfig.enabled == teConfig.enabledDefault:
      return
   teMode = root[ RouterGlobalTeMode ].getSingletonInstance()
   teCbfMode = teMode[ TeCbfMode ].getSingletonInstance()
   colorDscpMappingsMode = teCbfMode[ TeColorDscpMappingsMode ].\
      getSingletonInstance()
   cmds = colorDscpMappingsMode[ 'TrafficEngineering.cbf.color-dscp.config' ]
   cmdFormat = 'color {color} dscp {valuesStr}'
   for color in sorted( teColorDscpConfig.colorToValue ):
      dscpList = teColorDscpConfig.colorToValue[ color ]
      if dscpList.isDefault:
         cmds.addCommand( f'color {color} default' )
         continue
      haveSymbolic = any( dscpList.cliSaveUseSymbolic_i.values() )
      if haveSymbolic:
         dscpValues = []
         for value in dscpList.value:
            if dscpList.cliSaveUseSymbolic_i.get( value, False ):
               dscpValues.append( _dscpValueToName[ value.dscp ] )
            else:
               dscpValues.append( value.dscp )
         # Use the _dscpMixedTypeCmp comparison function so that we show the
         # symbolic DSCP values before the raw integer values.  This choice
         # of ordering is arbitrary.
         dscpValuesStr = ' '.join(
            map( str, sorted( dscpValues,
                              key=functools.cmp_to_key( _dscpMixedTypeCmp ) ) ) )
      else:
         dscpValuesStr = multiRangeToCanonicalString(
            [ value.dscp for value in dscpList.value ] )
      cmds.addCommand( cmdFormat.format( color=color, valuesStr=dscpValuesStr ) )

# -------------------------------------------------------------------------------
#  color-tc mappings cli save config
# -------------------------------------------------------------------------------

class TeColorTcMappingsMode( TeColorTcModeBase, CliSave.Mode ):
   def __init__( self, param ):
      TeColorTcModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

TeCbfMode.addChildMode( TeColorTcMappingsMode,
                        after=[ 'TrafficEngineering.cbf.config' ] )
TeColorTcMappingsMode.addCommandSequence(
   'TrafficEngineering.cbf.color-tc.config' )

@CliSave.saver( 'TrafficEngineering::CbfConfigMap',
                'te/cbf/color-tc/config',
                requireMounts=( 'te/config', ) )
def saveCbfColorTcMappings( teColorTcConfig, root, requireMounts, options ):
   teConfig = requireMounts[ 'te/config' ]
   if teConfig.enabled == teConfig.enabledDefault:
      return
   teMode = root[ RouterGlobalTeMode ].getSingletonInstance()
   teCbfMode = teMode[ TeCbfMode ].getSingletonInstance()
   colorTcMappingsMode = teCbfMode[ TeColorTcMappingsMode ].getSingletonInstance()
   cmds = colorTcMappingsMode[ 'TrafficEngineering.cbf.color-tc.config' ]
   cmdFormat = 'color {color} traffic-class {valuesStr}'
   for color in sorted( teColorTcConfig.colorToValue ):
      tcList = teColorTcConfig.colorToValue[ color ]
      if tcList.isDefault:
         cmds.addCommand( f'color {color} default' )
         continue

      tcValuesStr = multiRangeToCanonicalString(
         [ val.tc for val in tcList.value ] )
      cmds.addCommand( cmdFormat.format( color=color, valuesStr=tcValuesStr ) )

# -------------------------------------------------------------------------------
#  color-vrf mappings cli save config
# -------------------------------------------------------------------------------

class TeColorVrfMappingsMode( TeColorVrfModeBase, CliSave.Mode ):
   def __init__( self, param ):
      TeColorVrfModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

TeCbfMode.addChildMode( TeColorVrfMappingsMode,
                        after=[ 'TrafficEngineering.cbf.config' ] )
TeColorVrfMappingsMode.addCommandSequence(
   'TrafficEngineering.cbf.color-vrf.config' )

@CliSave.saver( 'TrafficEngineering::CbfConfigMap',
                'te/cbf/color-vrf/config',
                requireMounts=( 'te/config', ) )
def saveCbfColorVrfMappings( teColorVrfConfig, root, requireMounts, options ):
   teConfig = requireMounts[ 'te/config' ]
   if teConfig.enabled == teConfig.enabledDefault:
      return
   teMode = root[ RouterGlobalTeMode ].getSingletonInstance()
   teCbfMode = teMode[ TeCbfMode ].getSingletonInstance()
   colorVrfMappingsMode = teCbfMode[ TeColorVrfMappingsMode ].\
      getSingletonInstance()
   cmds = colorVrfMappingsMode[ 'TrafficEngineering.cbf.color-vrf.config' ]
   cmdFormat = 'color {color} vrf {vrfNames}'

   for color in sorted( teColorVrfConfig.colorToValue ):
      vrfNamesList = teColorVrfConfig.colorToValue[ color ]
      if vrfNamesList.isDefault:
         cmds.addCommand( f'color {color} vrf default' )
         continue
      vrfNamesStr = ' '.join( [ val.vrf
                                for val in sorted( vrfNamesList.value ) ] )
      cmds.addCommand( cmdFormat.format( color=color, vrfNames=vrfNamesStr ) )

# -------------------------------------------------------------------------------
#  cbf-mpls and cbf-ecmp cli save config
# -------------------------------------------------------------------------------

class CbfFwdPlaneMplsSubMode( CbfFwdPlaneMplsSubModeBase, CliSave.Mode ):
   def __init__( self, param ):
      CbfFwdPlaneMplsSubModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

if IpLibToggleLib.toggleMplsCbfLdpOverRsvpTeEnabled():
   TeCbfMode.addChildMode( CbfFwdPlaneMplsSubMode,
                           after=[ 'TrafficEngineering.cbf.config' ] )
CbfFwdPlaneMplsSubMode.addCommandSequence( 'TrafficEngineering.cbf.mpls.config' )

@CliSave.saver( 'TrafficEngineering::Config', 'te/config',
                requireMounts=( 'te/config', ) )
def saveGlobalConfig( teConfig, root, requireMounts, options ):
   teConfig = requireMounts[ 'te/config' ]
   if teConfig.enabled == teConfig.enabledDefault:
      return
   mode = root[ RouterGlobalTeMode ].getSingletonInstance()
   cbfMode = mode[ TeCbfMode ].getSingletonInstance()
   subCmds = cbfMode[ 'TrafficEngineering.cbf.config' ]

   if teConfig.cbfEcmpEndpoints:
      subCmds.addCommand( 'cbf ecmp endpoints multiple' )
   elif options.saveAll:
      subCmds.addCommand( 'no cbf ecmp endpoints multiple' )

   if IpLibToggleLib.toggleMplsCbfLdpOverRsvpTeEnabled():
      cbfMplsMode = cbfMode[ CbfFwdPlaneMplsSubMode ].getSingletonInstance()
      subCmds = cbfMplsMode[ 'TrafficEngineering.cbf.mpls.config' ]

      if teConfig.cbfMplsColorDefFallback != \
         teConfig.cbfMplsColorDefFallbackDefault:
         subCmds.addCommand( 'color default fallback highest' )
      elif options.saveAll:
         subCmds.addCommand( 'no color default fallback highest' )
      if teConfig.cbfMplsTunnelingLdp != teConfig.cbfMplsTunnelingLdpDefault:
         subCmds.addCommand( 'tunneling ldp' )
      elif options.saveAll:
         subCmds.addCommand( 'no tunneling ldp' )

      if IpLibToggleLib.toggleMplsCbfSrOverRsvpTeEnabled():
         if teConfig.cbfMplsTunnelingIsisSr != \
            teConfig.cbfMplsTunnelingIsisSrDefault:
            subCmds.addCommand( 'tunneling segment-routing' )
         elif options.saveAll:
            subCmds.addCommand( 'no tunneling segment-routing' )
