#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import CliSave
import Tac
from CliMode.Classification import ( AppTrafficRecModeBase, AppProfileModeBase,
                                     FieldSetPrefixModeBase, CategoryModeBase,
                                     FieldSetL4PortModeBase, AppModeBase,
                                     AppTrafficRecConfigMode )
from ClassificationLib import ( numericalRangeToRangeString, getProtocolNumToNameMap,
                                numericalRangeToSet, rangeSetToNumericalRange )
from TeCliLib import dscpAclNames
from CliSavePlugin.IntfCliSave import IntfConfigMode
from Toggles.ClassificationToggleLib import toggleAppDscpMatchEnabled
import os

sysname = None

def isDpiSupportedOnPlatform( entMib ):
   return ( ( entMib.root.modelName == 'vEOS' or
              entMib.root.vendorType == 'Caravan' ) if entMib.root else
            ( os.getenv( 'SIMULATION_CLASSIFICATION' ) ) )

class AppRecognitionSaveMode( AppTrafficRecModeBase, CliSave.Mode ):
   def __init__( self, param ):
      AppTrafficRecModeBase.__init__( self )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

CliSave.GlobalConfigMode.addChildMode( AppRecognitionSaveMode,
                                       after=[ IntfConfigMode ] )
AppRecognitionSaveMode.addCommandSequence( 'Classification.AppRecognition' )

class FieldSetL4PortSaveMode( FieldSetL4PortModeBase, CliSave.Mode ):
   def __init__( self, param ):
      FieldSetL4PortModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

AppRecognitionSaveMode.addChildMode( FieldSetL4PortSaveMode )
FieldSetL4PortSaveMode.addCommandSequence( 'Classification.FieldSetL4Port' )

class FieldSetIpPrefixSaveMode( FieldSetPrefixModeBase, CliSave.Mode ):
   def __init__( self, param ):
      FieldSetPrefixModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

AppRecognitionSaveMode.addChildMode( FieldSetIpPrefixSaveMode )
FieldSetIpPrefixSaveMode.addCommandSequence( 'Classification.FieldSetIpPrefix' )

def makeAppSaveMode( afToken ):
   class AppSaveMode( AppModeBase, CliSave.Mode ):
      def __init__( self, param ):
         AppModeBase.__init__( self, param, afToken )
         CliSave.Mode.__init__( self, param )

   return AppSaveMode

AppSaveModeIpv4 = makeAppSaveMode( 'ipv4' )
AppSaveModeL4 = makeAppSaveMode( 'l4' )

_afToSaveMode = { 'ipv4': AppSaveModeIpv4,
                  'bothIpv4AndIpv6': AppSaveModeL4 }
for mode in _afToSaveMode.values():
   AppRecognitionSaveMode.addChildMode( mode )
   mode.addCommandSequence( 'Classification.App' )

class AppProfileSaveMode( AppProfileModeBase, CliSave.Mode ):
   def __init__( self, param ):
      AppProfileModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

AppRecognitionSaveMode.addChildMode( AppProfileSaveMode )
AppProfileSaveMode.addCommandSequence( 'Classification.AppProfile' )

classifConstants = Tac.Type( "Classification::ClassificationConstants" )
DEFAULT_CATEGORY = classifConstants.defaultCategoryName

class CategorySaveMode( CategoryModeBase, CliSave.Mode ):
   def __init__( self, param ):
      CategoryModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

# Category save block should come before AppProfile save block,
# otherwise during save and reload, 'category <name>' will be considered as a
# command within application profile block (since application profile also accepts
# 'category <name>' command) and not as a category configuration command.
AppRecognitionSaveMode.addChildMode( CategorySaveMode,
                                     before=[ AppProfileSaveMode ],
                                     after=[ AppSaveModeIpv4, AppSaveModeL4 ] )
CategorySaveMode.addCommandSequence( 'Classification.Category' )

def isDefaultCategoryConfig( categoryName, appName, serviceName,
                             appRecConfig ):
   appConfig = appRecConfig.app.get( appName )
   # Ipv4 app not yet created. Default category is general for ipv4 apps.
   if not appConfig:
      return categoryName == 'general'

   if serviceName == 'all':
      for defCat in appConfig.defaultServiceCategory.values():
         if defCat != categoryName:
            return False
   else:
      if ( serviceName not in appConfig.defaultServiceCategory ) or \
         ( appConfig.defaultServiceCategory[ serviceName ] != categoryName ):
         return False
   return True

dscpValueToName = {}
for dscpName, ( dscpValue, _ ) in dscpAclNames.items():
   dscpValueToName[ dscpValue ] = dscpName

@CliSave.saver( 'Classification::AppRecognitionConfig',
                'classification/app-recognition/config',
                requireMounts=( 'classification/app-recognition/fieldset',
                                'hardware/entmib',
                                'cli/config' ) )
def saveAppRecognitionConfig( entity, root, requireMounts, options ):
   fieldSetConfig = requireMounts[ 'classification/app-recognition/fieldset' ]
   appRecogMode = root[ AppRecognitionSaveMode ].getSingletonInstance()
   for fieldSetName in sorted( fieldSetConfig.fieldSetL4Port ):
      param = ( "app", "l4-port", fieldSetName, AppTrafficRecConfigMode )
      cmds = appRecogMode[ FieldSetL4PortSaveMode ].getOrCreateModeInstance(
         param )[ 'Classification.FieldSetL4Port' ]
      fieldSetL4PortCfg = fieldSetConfig.fieldSetL4Port[ fieldSetName ]

      if fieldSetL4PortCfg.currCfg and fieldSetL4PortCfg.currCfg.ports:
         cmds.addCommand( numericalRangeToRangeString(
            fieldSetL4PortCfg.currCfg.ports ) )

   for fieldSetName in sorted( fieldSetConfig.fieldSetIpPrefix ):
      fieldSetIpPrefixCfg = fieldSetConfig.fieldSetIpPrefix[ fieldSetName ]
      param = ( "app", fieldSetIpPrefixCfg.af,
                fieldSetName, AppTrafficRecConfigMode )
      cmds = appRecogMode[ FieldSetIpPrefixSaveMode ].getOrCreateModeInstance(
         param )[ 'Classification.FieldSetIpPrefix' ]
      if fieldSetIpPrefixCfg.currCfg and fieldSetIpPrefixCfg.currCfg.prefixes:
         prefixes = sorted( prefix.stringValue for prefix in
                            fieldSetIpPrefixCfg.currCfg.prefixes )
         cmdStr = " ".join( prefixes )
         cmds.addCommand( cmdStr )

   for appName, appConfig in entity.app.items():
      if appConfig.readonly or appConfig.defaultApp:
         continue
      cmds = appRecogMode[ _afToSaveMode[ appConfig.af ] ].getOrCreateModeInstance(
         appName )[ 'Classification.App' ]
      if appConfig.srcPrefixFieldSet:
         cmds.addCommand( 'source prefix field-set %s' %
                          appConfig.srcPrefixFieldSet )
      if appConfig.dstPrefixFieldSet:
         cmds.addCommand( 'destination prefix field-set %s' %
                          appConfig.dstPrefixFieldSet )
      if appConfig.srcPortFieldSet or \
         appConfig.dstPortFieldSet:
         protoSet = numericalRangeToSet( appConfig.proto )
         cmd = 'protocol'
         if 6 in protoSet:
            cmd += ' tcp'
         if 17 in protoSet:
            cmd += ' udp'
         if appConfig.srcPortFieldSet:
            srcFieldSet = appConfig.srcPortFieldSet
            cmd += ' source port field-set %s' % srcFieldSet
         if appConfig.dstPortFieldSet:
            dstFieldSet = appConfig.dstPortFieldSet
            cmd += ' destination port field-set %s' % dstFieldSet
         cmds.addCommand( cmd )
      elif appConfig.proto:
         protoNoToName = getProtocolNumToNameMap()
         protoSet = numericalRangeToSet( appConfig.proto )
         for proto in sorted( protoSet ):
            if proto in protoNoToName:
               cmds.addCommand( 'protocol %s' % protoNoToName[ proto ] )
               protoSet.remove( proto )
         if protoSet:
            numRange = rangeSetToNumericalRange( protoSet,
                                                 "Classification::ProtocolRange" )
            protoStr = numericalRangeToRangeString( numRange )
            cmds.addCommand( 'protocol %s' % protoStr )
      if appConfig.dscp and toggleAppDscpMatchEnabled():
         dscpSet = sorted( numericalRangeToSet( appConfig.dscp ) )
         haveSymbolic = any( appConfig.dscpSymbolic.values() )
         if haveSymbolic:
            dscpNames = []
            dscpValues = []
            for dscp in dscpSet:
               if appConfig.dscpSymbolic[ dscp ]:
                  dscpNames.append( dscpValueToName[ dscp ] )
               else:
                  dscpValues.append( dscp )
            dscpNameStr = ' '.join( sorted( dscpNames ) )
            numRange = rangeSetToNumericalRange( dscpValues,
                                                 "Classification::DscpRangeConfig" )
            dscpValueStr = numericalRangeToRangeString( numRange )
            dscpNameStr = dscpNameStr + ' ' if dscpValueStr else dscpNameStr
            dscpCompleteStr = dscpNameStr + dscpValueStr
         else:
            numRange = rangeSetToNumericalRange( dscpSet,
                                                 "Classification::DscpRangeConfig" )
            dscpCompleteStr = numericalRangeToRangeString( numRange )
         cmds.addCommand( 'dscp %s' % dscpCompleteStr )

   for appProfileName, appProfile in entity.appProfile.items():
      cmds = appRecogMode[ AppProfileSaveMode ].getOrCreateModeInstance(
         appProfileName )[ 'Classification.AppProfile' ]
      for appName in appProfile.app:
         # "service" needs to be sorted manually until we support ordered sets
         appService = \
            sorted( appProfile.app[ appName ].service )
         if 'all' in appService:
            cmds.addCommand( 'application %s' % appName )
         else:
            for serviceName in appService:
               cmd = f'application {appName} service {serviceName}'
               cmds.addCommand( cmd )

      for appName in sorted(
              entity.appProfile[ appProfileName ].appTransport ):
         cmds.addCommand( 'application %s transport' % appName )

      for categoryName in appProfile.category:
         # "service" needs to be sorted manually until we support ordered sets
         categoryService = sorted(
            appProfile.category[ categoryName ].service )
         if 'all' in categoryService:
            cmds.addCommand( 'category %s' % categoryName )
         else:
            for serviceName in categoryService:
               cmd = f'category {categoryName} service {serviceName}'
               cmds.addCommand( cmd )

   # pylint: disable-msg=R1702
   saveAll = options.saveAll or options.saveAllDetail

   if not saveAll:
      for category in sorted( entity.category ):
         cmds = []
         catConfig = entity.category[ category ]
         for app in sorted( catConfig.appService ):
            services = catConfig.appService[ app ].service
            for service in sorted( services ):
               if isDefaultCategoryConfig( category, app, service, entity ):
                  continue
               cmd = f'application {app}' if service == 'all' else \
                     f'application {app} service {service}'
               cmds.append( cmd )

         if ( cmds or not catConfig.defaultCategory or
              CliSave.hasComments( f"category-{category}", requireMounts ) ):
            appRecMode = root[ AppRecognitionSaveMode ].getSingletonInstance()
            catMode = appRecMode[ CategorySaveMode ].getOrCreateModeInstance(
               category )[ 'Classification.Category' ]
            for cmd in cmds:
               catMode.addCommand( cmd )
   else: # For saveAll, show both configured and default category assignments
      entityMib = requireMounts[ 'hardware/entmib' ]
      if not isDpiSupportedOnPlatform( entityMib ):
         return

      appCategoryResult = {}
      categories = {}

      # Process the category config. If category is configured for an (app, service)
      # then that will be the category assigned to that (app, service)
      for cat, catConfig in entity.category.items():
         categories[ cat ] = {}
         for app in catConfig.appService:
            services = catConfig.appService[ app ].service
            appCategoryResult[ app ] = {}
            for service in services:
               appCategoryResult[ app ][ service ] = cat

      # Process default category config. If there is no configured category
      # for an (app, service), then default category config will be assigned
      # to that (app, service )
      for app, appConfig in entity.app.items():
         if app in appCategoryResult:
            if 'all' in appCategoryResult[ app ]:
               continue
         else:
            appCategoryResult[ app ] = {}

         for service, defCat in appConfig.defaultServiceCategory.items():
            if app in appCategoryResult and service in appCategoryResult[ app ]:
               continue
            appCategoryResult[ app ][ service ] = defCat

      # Build a collection keyed by category so that the running-config
      # could be generated sorted first by category, then app and service
      for app, serviceCat in appCategoryResult.items():
         for service, cat in serviceCat.items():
            if cat not in categories:
               categories[ cat ] = {}
            if app not in categories[ cat ]:
               categories[ cat ][ app ] = []
            categories[ cat ][ app ].append( service )

      for cat in sorted( categories ):
         catMode = appRecogMode[ CategorySaveMode ].getOrCreateModeInstance(
               cat )[ 'Classification.Category' ]
         for app in sorted( categories[ cat ] ):
            services = categories[ cat ][ app ]
            for service in sorted( services ):
               cmd = f'application {app}' if service == 'all' else \
                     f'application {app} service {service}'
               catMode.addCommand( cmd )

def Plugin( entMan ):
   global sysname
   sysname = entMan.sysname_
