#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import ClbCliLib
from CliMode.Clb import ( ClbMode, ClbFlowLearningMode, ClbPortGroupMode,
                          ClbMatchPolicyMode )
import CliSave
from CliSavePlugin.IntfCliSave import IntfConfigMode
from Intf import IntfRange

class ClbConfigMode( ClbMode, CliSave.Mode ):
   def __init__( self, param ):
      ClbMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

class ClbFlowLearningConfigMode( ClbFlowLearningMode, CliSave.Mode ):
   def __init__( self, param ):
      ClbFlowLearningMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class ClbHostPortGroupConfigMode( ClbPortGroupMode, CliSave.Mode ):
   def __init__( self, param ):
      ClbPortGroupMode.__init__( self, "intfGroupTypeHost", param )
      CliSave.Mode.__init__( self, param )

class ClbUplinkPortGroupConfigMode( ClbPortGroupMode, CliSave.Mode ):
   def __init__( self, param ):
      ClbPortGroupMode.__init__( self, "intfGroupTypeUplink", param )
      CliSave.Mode.__init__( self, param )

class ClbMatchPolicyConfigMode( ClbMatchPolicyMode, CliSave.Mode ):
   def __init__( self, param ):
      ClbMatchPolicyMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( ClbConfigMode, after=[ IntfConfigMode ] )
ClbConfigMode.addCommandSequence( 'Clb.config' )

ClbConfigMode.addChildMode( ClbFlowLearningConfigMode, after=[ 'Clb.config' ] )
ClbFlowLearningConfigMode.addCommandSequence( 'Clb.flow.config' )

ClbConfigMode.addChildMode( ClbHostPortGroupConfigMode, after=[ 'Clb.config' ] )
ClbHostPortGroupConfigMode.addCommandSequence( 'Clb.portGroup.config' )

ClbConfigMode.addChildMode( ClbUplinkPortGroupConfigMode,
                            after=[ ClbHostPortGroupConfigMode ] )
ClbUplinkPortGroupConfigMode.addCommandSequence( 'Clb.portGroup.config' )

ClbConfigMode.addChildMode( ClbMatchPolicyConfigMode )
ClbMatchPolicyConfigMode.addCommandSequence( 'Clb.trafficPolicy.config' )

def addPortGroupCommonCommands( cmds, portGroup, options ):
   intfNames = IntfRange.intfListToCanonical( portGroup.member )
   if intfNames:
      if portGroup.useInterfaceToken:
         token = "interface"
      else:
         token = "member"
      cmds.addCommand( f"{token} {intfNames[0]}" )
   elif options.saveAll:
      cmds.addCommand( "no member" )

@CliSave.saver( 'Clb::Config', 'clb/config',
                requireMounts=( 'clb/hardware/capabilities', ) )
def saveConfig( entity, root, requireMounts, options ):
   clbMode = root[ ClbConfigMode ].getSingletonInstance()
   cmds = clbMode[ 'Clb.config' ]
   saveAll = ( options.saveAll and
               requireMounts[ 'clb/hardware/capabilities' ].clbSupported )
   if entity.forwardingMode == 'routed':
      cmd = 'forwarding type routed'
   elif entity.forwardingMode == 'bridgedVxlan':
      cmd = 'forwarding type bridged encapsulation vxlan'
   else:
      cmd = ''

   if cmd:
      # for now, force "ipv4"
      if entity.forwardingModeCliTokenIpv4:
         cmd += " ipv4"
   elif saveAll:
      cmd = "no forwarding type"

   if cmd:
      cmds.addCommand( cmd )

   if entity.destinationGroupingType != 'destGroupTypeDefault':
      cmd = 'destination grouping'
      if entity.destinationGroupingType == 'destGroupTypeBgpFieldSet':
         cmd += ' bgp field-set'
      elif entity.destinationGroupingType == 'destGroupTypeVtep':
         cmd += ' vtep'
      else:
         assert entity.destinationGroupingType == 'destGroupTypePrefix'
         cmd += f' prefix length {entity.destinationGroupingPrefixLen}'
      cmds.addCommand( cmd )

   if entity.loadBalanceMethod == "methodFlowRoundRobin":
      cmds.addCommand( "load-balance method flow round-robin" )
   elif options.saveAll:
      cmds.addCommand( "no load-balance method" )

   if entity.flowMatchCriteria != entity.flowMatchCriteriaDefault or saveAll:
      if entity.flowMatchCriteria.encapType == "encapTypeVxlan":
         encapType = "vxlan"
      else:
         encapType = "none"
      ipProtocol = entity.flowMatchCriteria.ipProtocol
      cmds.addCommand( f'flow match encapsulation {encapType} {ipProtocol}' )

   for name, pg in entity.hostIntfGroup.items():
      portGroupMode = \
            clbMode[ ClbHostPortGroupConfigMode ].getOrCreateModeInstance( name )
      portGroupCmds = portGroupMode[ 'Clb.portGroup.config' ]
      addPortGroupCommonCommands( portGroupCmds, pg, options )
      if pg.flowLimit != pg.flowLimitDefault:
         portGroupCmds.addCommand( f"flow limit {pg.flowLimit}" )
      elif saveAll:
         portGroupCmds.addCommand( "no flow limit" )

      if( pg.flowWriteLengthThreshold != pg.flowWriteLengthThresholdDefault or
          saveAll ):
         portGroupCmds.addCommand(
            f"balance factor {pg.flowWriteLengthThreshold}" )
      if pg.flowWarningThreshold != pg.flowWarningThresholdDefault or saveAll:
         portGroupCmds.addCommand( f'flow warning {pg.flowWarningThreshold}' )
      cmd = ClbCliLib.getActionCommand( pg.flowExhaustionAction )
      if cmd:
         portGroupCmds.addCommand( f"flow exhaustion {cmd}" )

   for name, pg in entity.uplinkIntfGroup.items():
      portGroupMode = \
            clbMode[ ClbUplinkPortGroupConfigMode ].getOrCreateModeInstance( name )
      portGroupCmds = portGroupMode[ 'Clb.portGroup.config' ]
      addPortGroupCommonCommands( portGroupCmds, pg, options )

   if entity.flowMonitor:
      cmds.addCommand( 'flow monitor' )
   elif saveAll:
      cmds.addCommand( 'no flow monitor' )

   if entity.flowSource == 'flowSourceLearning':
      snoopMode = clbMode[ ClbFlowLearningConfigMode ].getSingletonInstance()
      # add more commands to snoop mod
      snoopCmds = snoopMode[ 'Clb.flow.config' ]

      if entity.flowAgingTime != entity.flowAgingTimeDefault or saveAll:
         snoopCmds.addCommand( f'aging timeout {entity.flowAgingTime} seconds' )

      if entity.globalFlowLearnLimit != entity.globalFlowLearnLimitDefault:
         snoopCmds.addCommand( f'limit {entity.globalFlowLearnLimit}' )
      elif saveAll:
         snoopCmds.addCommand( 'no limit' )

   elif saveAll:
      cmds.addCommand( "no flow source" )

   for name, tp in entity.flowMatchPolicy.items():
      tcMode = clbMode[ ClbMatchPolicyConfigMode ].getOrCreateModeInstance( name )
      tcCmds = tcMode[ 'Clb.trafficPolicy.config' ]
      cmd = ClbCliLib.getMatchPolicyCommand( tp )
      if cmd:
         tcCmds.addCommand( cmd )

   if entity.flowExhaustionMatchPolicy:
      cmds.addCommand( f"flow exhaustion match {entity.flowExhaustionMatchPolicy}" )
   # This should be uncommented if we unhide the command
#   elif options.saveAll:
#      cmds.addCommand( "no flow exhaustion match" )
