# Copyright (c) 2006, 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from functools import total_ordering
import CliSave
from CliMode.Alias import AliasMode

CliSave.GlobalConfigMode.addCommandSequence( 'Cli.config' )

quote = '"{}"'.format

@total_ordering
class AliasConfigMode( AliasMode, CliSave.Mode ):
   aliasOrder = {}

   def __init__( self, param ):
      AliasMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def __eq__( self, other ): # TODO: commentKey?
      # This is used to sort the aliases when saving to a config.
      # Regex aliases have order which is stored in `aliasOrder`.
      # Other aliases get -1 for index, so we also compare their name (`param_`).
      return ( ( self.aliasOrder.get( self.param_, -1 ), self.param_ ) ==
               ( self.aliasOrder.get( other.param_, -1 ), other.param_ ) )

   def __lt__( self, other ):
      # This is used to sort the aliases when saving to a config.
      # Regex aliases have order which is stored in `aliasOrder`.
      # Other aliases get -1 for index, so we also compare their name (`param_`).
      return ( ( self.aliasOrder.get( self.param_, -1 ), self.param_ ) <
               ( self.aliasOrder.get( other.param_, -1 ), other.param_ ) )

CliSave.GlobalConfigMode.addChildMode( AliasConfigMode, after=[ 'Cli.config' ] )
AliasConfigMode.addCommandSequence( 'Cli.config.alias' )

@CliSave.saver( 'Cli::Config', 'cli/config' )
def saveConfig( entity, root, requireMounts, options ):
   cmds = root[ 'Cli.config' ]
   saveAll = options.saveAll
   if entity.prompt != entity.promptDefault or saveAll:
      cmds.addCommand( 'prompt %s' % entity.prompt )
   # Do not print "no terminal length|width" in "show running all"
   if entity.terminalLengthOverride != entity.terminalLengthOverrideDefault:
      cmds.addCommand( 'terminal length %s' % entity.terminalLengthOverride )
   if entity.terminalWidthOverride != entity.terminalWidthOverrideDefault:
      cmds.addCommand( 'terminal width %s' % entity.terminalWidthOverride )

   if not entity.commandConfirmation:
      cmds.addCommand( 'terminal confirmation disabled' )
   elif saveAll:
      cmds.addCommand( 'no terminal confirmation disabled' )

   if entity.commandsPath:
      cmds = root[ 'Cli.config' ]
      cmds.addCommand( 'alias commands path %s' % entity.commandsPath )
   elif saveAll:
      cmds = root[ 'Cli.config' ]
      cmds.addCommand( 'no alias commands path' )

   if entity.activityLockSyslogMsg:
      cmds = root[ 'Cli.config' ]
      cmds.addCommand( 'activity-lock-monitor' )

   for alias in entity.alias.values():
      commentKey = 'alias-' + alias.cmdAlias
      # We skip empty aliases with no command.
      # Everything is discarded.
      if not alias.originalCmd:
         pass
      # We save one line aliases in the original one-line format.
      # Convert into multi-line alias if there is any comment for a single
      # command alias.
      elif ( ( len( alias.originalCmd ) == 1 ) and
             ( commentKey not in entity.comment ) ):
         cmds = root[ 'Cli.config' ]
         cmds.addCommand(
            'alias %s %s' %
            ( alias.cmdAlias, list( alias.originalCmd.values() )[ 0 ] ) )
      # Multi-line aliases, we save everything.
      else:
         mode = root[ AliasConfigMode ].getOrCreateModeInstance( alias.cmdAlias )
         cmds = mode[ 'Cli.config.alias' ]
         for seq, oCmd in alias.originalCmd.items():
            if options.showNoSeqNum:
               cmds.addCommand( f'{oCmd}' )
            else:
               cmds.addCommand( f'{seq} {oCmd}' )

   # Regex aliases.
   if entity.regexAlias:
      for index, reAlias in entity.regexAlias.items():
         # We want to preseve order from Sysdb,
         # so we store the indices in the save mode.
         AliasConfigMode.aliasOrder[ quote( reAlias.cmdAlias ) ] = index
         if reAlias.originalCmd:
            # We save all regex aliases as multi-line format or we lose order,
            # because some would go in Cli.config, and some in Cli.config.alias.
            alias = quote( reAlias.cmdAlias )
            mode = root[ AliasConfigMode ].getOrCreateModeInstance( alias )
            cmds = mode[ 'Cli.config.alias' ]
            for seq, oCmd in reAlias.originalCmd.items():
               if options.showNoSeqNum:
                  cmds.addCommand( f'{oCmd}' )
               else:
                  cmds.addCommand( f'{seq} {oCmd}' )

def timerValueDisplay( timerValueSecs ):
   if timerValueSecs is None:
      return None

   hrs = timerValueSecs // 3600
   mins = ( timerValueSecs % 3600 ) // 60
   secs = timerValueSecs - ( hrs * 3600 ) - ( mins * 60 )
   return f'{hrs}:{mins:02d}:{secs:02d}'

@CliSave.saver( 'Cli::Session::CliConfig', 'cli/session/input/config' )
def saveCliSessionConfig( entity, root, requireMounts, options ):
   cmds = root[ 'Cli.config' ]
   saveAll = options.saveAll or options.saveAllDetail
   if entity.maxSavedSessions != 1: # TODO: use defaults from .tac file
      cmds.addCommand( 'service configuration session max completed %s'
                        % entity.maxSavedSessions )
   elif saveAll:
      cmds.addCommand( 'no service configuration session max completed' )

   if entity.maxOpenSessions != 5:
      cmds.addCommand( 'service configuration session max pending %s'
                        % entity.maxOpenSessions )
   elif saveAll:
      cmds.addCommand( 'no service configuration session max pending' )

   if not entity.configTerminalEnabled:
      cmds.addCommand( 'service configuration terminal disabled' )
   elif saveAll:
      cmds.addCommand( 'no service configuration terminal disabled' )

   if entity.mergeOnCommit != entity.defaultMergeOnCommit:
      cmds.addCommand( 'service configuration session commit merge' )
   elif saveAll:
      cmds.addCommand( 'no service configuration session commit merge' )

   if ( entity.saveToStartupConfigOnCommit !=
        entity.defaultSaveToStartupConfigOnCommit ):
      cmds.addCommand( 'service configuration session commit save startup-config' )
   elif saveAll:
      cmds.addCommand( 'no service configuration session '
                       'commit save startup-config' )

   if ( entity.loggingTerminal !=
        entity.defaultLoggingTerminal ):
      cmds.addCommand( 'service configuration session logging terminal' )
   elif saveAll:
      cmds.addCommand( 'no service configuration session logging terminal' )

   if entity.defaultCommitTimer:
      timer = timerValueDisplay( int( entity.defaultCommitTimer ) )
      cmds.addCommand( f'service configuration session commit timer default '
                       f'{timer} interactive' )
   elif saveAll:
      cmds.addCommand(
            'no service configuration session commit timer default interactive' )
