# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import AclCliLib
import CliSave
import Tac
from CliMode.ConnectivityMonitor import ConnectivityMonitorBaseMode, \
   ConnectivityMonitorHostMode, ConnectivityMonitorVrfMode, \
   ConnectivityMonitorClientMode
from CliSavePlugin.IntfCliSave import IntfConfigMode
from Toggles.ConnectivityMonitorToggleLib import (
      toggleCMTcpProbeEnabled,
      toggleCMIcmpPingCountEnabled,
)

defaultVrf = "default"

class ConnMonitorConfigMode( ConnectivityMonitorBaseMode, CliSave.Mode ):
   def __init__( self, param ):
      ConnectivityMonitorBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

ConnMonitorConfigMode.addCommandSequence( 'ConnectivityMonitor.config' )
ConnMonitorConfigMode.addCommandSequence( 'ConnectivityMonitor.config.intfs' )
CliSave.GlobalConfigMode.addChildMode( ConnMonitorConfigMode,
                                       after=[ IntfConfigMode ] )

class ConnMonitorHostConfigMode( ConnectivityMonitorHostMode, CliSave.Mode ):
   def __init__( self, param ):
      ConnectivityMonitorHostMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

ConnMonitorConfigMode.addCommandSequence( 'ConnectivityMonitor.config.hosts' )
ConnMonitorHostConfigMode.addCommandSequence( 'ConnectivityMonitor.config.host' )
ConnMonitorHostConfigMode.addCommandSequence(
                                          'ConnectivityMonitor.config.vrf.host' )
ConnMonitorConfigMode.addChildMode( ConnMonitorHostConfigMode,
                                    after=[ 'ConnectivityMonitor.config.intfs' ],
                                    before=[ 'ConnectivityMonitor.config.hosts' ] )
ConnMonitorHostConfigMode.addCommandSequence(
                                          'ConnectivityMonitor.config.client.host' )

class ConnMonitorVrfConfigMode( ConnectivityMonitorVrfMode, CliSave.Mode ):
   def __init__( self, param ):
      ConnectivityMonitorVrfMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

ConnMonitorVrfConfigMode.addCommandSequence( 'ConnectivityMonitor.config.vrf' )
ConnMonitorVrfConfigMode.addCommandSequence( 'ConnectivityMonitor.config.vrf.intfs' )
ConnMonitorVrfConfigMode.addChildMode( ConnMonitorHostConfigMode,
                                 after=[ 'ConnectivityMonitor.config.vrf.intfs' ] )
# Add ConnMonitorVrfConfigMode as a childMode to ConnMonitorConfigMode
ConnMonitorConfigMode.addChildMode( ConnMonitorVrfConfigMode )

class ConnMonitorClientConfigMode( ConnectivityMonitorClientMode, CliSave.Mode ):
   def __init__( self, param ):
      ConnectivityMonitorClientMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

ConnMonitorClientConfigMode.addCommandSequence( 'ConnectivityMonitor.config.client' )
ConnMonitorClientConfigMode.addChildMode( ConnMonitorHostConfigMode )
ConnMonitorConfigMode.addChildMode( ConnMonitorClientConfigMode,
                                    after=[ 'ConnectivityMonitor.config.hosts' ] )

def saveHostConfig( hostConfig, root, cmConfig, clientName ):
   parentMode = root[ ConnMonitorConfigMode ].getSingletonInstance()
   vrfName = hostConfig.key.vrfName
   if vrfName != defaultVrf:
      parentMode = parentMode[ ConnMonitorVrfConfigMode ].getOrCreateModeInstance(
         vrfName )
   if clientName:
      parentMode = parentMode[ ConnMonitorClientConfigMode ].getOrCreateModeInstance(
         clientName )
   mode = parentMode[ ConnMonitorHostConfigMode ].getOrCreateModeInstance(
      ( hostConfig.key.hostName, vrfName, clientName ) )
   cmds =  mode[ 'ConnectivityMonitor.config.host' ]
   if vrfName != defaultVrf:
      cmds = mode[ 'ConnectivityMonitor.config.vrf.host' ]
   if clientName:
      cmds = mode[ 'ConnectivityMonitor.config.client.host' ]
   if hostConfig.description != "":
      cmds.addCommand( 'description %s' % hostConfig.description )
   if hostConfig.intfSetType.key.setName != "":
      if cmConfig.vrf[ vrfName ].defaultIntfSetType != hostConfig.intfSetType:
         if hostConfig.intfSetType.useSrcIp:
            localIntfStr = 'local-interfaces %s address-only'
         else:
            localIntfStr = 'local-interfaces %s'
         cmds.addCommand( localIntfStr % hostConfig.intfSetType.key.setName )
   if not hostConfig.icmpConfig.ipAddr.isAddrZero:
      cmds.addCommand( 'ip %s' % hostConfig.icmpConfig.ipAddr.stringValue )
   if hostConfig.icmpConfig.hostname:
      if hostConfig.icmpConfig.addressFamily != \
               Tac.Type( 'Arnet::AddressFamily' ).ipunknown:
         cmds.addCommand( 'ip hostname %s resolution %s' %
               ( hostConfig.icmpConfig.hostname,
                 hostConfig.icmpConfig.addressFamily ) )
      else:
         cmds.addCommand( 'ip hostname %s' % ( hostConfig.icmpConfig.hostname ) )
   if hostConfig.icmpConfig.pingSize != hostConfig.icmpConfig.pingSizeDefault:
      cmds.addCommand( 'icmp echo size %d' % hostConfig.icmpConfig.pingSize )
   if hostConfig.icmpConfig.icmpDscp != \
      hostConfig.icmpConfig.icmpDscpDefault:
      if hostConfig.icmpConfig.icmpDscpNameUsed:
         # Convert DSCP value to ACL name
         dscpName = AclCliLib.dscpNameFromValue( hostConfig.icmpConfig.icmpDscp )
         cmds.addCommand( 'icmp echo qos dscp %s' % dscpName )
      else:
         cmds.addCommand( 'icmp echo qos dscp %d' % hostConfig.icmpConfig.icmpDscp )
   if toggleCMIcmpPingCountEnabled():
      if hostConfig.icmpConfig.pingCount != \
            hostConfig.icmpConfig.pingCountDefault:
         cmds.addCommand( 'icmp count %d' % hostConfig.icmpConfig.pingCount )
   if toggleCMTcpProbeEnabled():
      if hostConfig.icmpConfig.icmpEnabled != \
            hostConfig.icmpConfig.icmpEnabledDefault:
         cmds.addCommand( 'icmp echo disabled' )
      if hostConfig.tcpConfig.useDport:
         port = hostConfig.tcpConfig.dport
         cmds.addCommand( f'tcp port {port}' )
   if hostConfig.httpConfig.url != "":
      cmds.addCommand( 'url %s' % hostConfig.httpConfig.url )
   if hostConfig.nameServerGroup != "":
      if hostConfig.nameServerGroup != "default":
         nameServerGroup = \
               hostConfig.nameServerGroup.partition( "internal-ns-" )[ 2 ]
         cmds.addCommand( 'name-server group %s' % nameServerGroup )
      else:
         cmds.addCommand( 'name-server group %s' % hostConfig.nameServerGroup )

def saveIntfSet( intfSet, root ):
   mode = root[ ConnMonitorConfigMode ].getSingletonInstance()
   cmds = mode[ 'ConnectivityMonitor.config.intfs' ]
   vrfName = intfSet.key.vrfName
   if vrfName != defaultVrf:
      mode = mode[ ConnMonitorVrfConfigMode ].getOrCreateModeInstance( vrfName )
      cmds = mode[ 'ConnectivityMonitor.config.vrf.intfs' ]
   cmd = 'interface set %s ' % intfSet.key.setName
   cmd += ','.join( sorted( intfSet.intf ) )
   cmds.addCommand( cmd )

def saveVrfConfig( vrfConfig, root ):
   mode = root[ ConnMonitorConfigMode ].getSingletonInstance()
   intfCmds = mode[ 'ConnectivityMonitor.config.intfs' ]
   if vrfConfig.vrfName != defaultVrf:
      mode = mode[ ConnMonitorVrfConfigMode ].getOrCreateModeInstance(
         vrfConfig.vrfName )
      intfCmds = mode[ 'ConnectivityMonitor.config.vrf.intfs' ]
   if vrfConfig.description != '':
      intfCmds.addCommand( 'description %s' % vrfConfig.description )
   if vrfConfig.defaultIntfSetType.key.setName != "":
      if vrfConfig.defaultIntfSetType.useSrcIp:
         localIntfStr = 'local-interfaces %s address-only default'
      else:
         localIntfStr = 'local-interfaces %s default'
      intfCmds.addCommand( localIntfStr %
                           vrfConfig.defaultIntfSetType.key.setName )

@CliSave.saver( 'ConnectivityMonitor::ConfigDir',
                'connectivityMonitor/clientConfigDir/default',
                requireMounts=( 'connectivityMonitor/config', ) )
def saveConfig( config, root, requireMounts, options ):
   cmConfig = requireMounts[ 'connectivityMonitor/config' ]
   for intfSetKey in sorted( cmConfig.intfSet ):
      saveIntfSet( cmConfig.intfSet[ intfSetKey ], root )
   for vrfName in sorted( cmConfig.vrf ):
      saveVrfConfig( cmConfig.vrf[ vrfName ], root )
   for hostKey in sorted( config.hostConfig ):
      saveHostConfig( config.hostConfig[ hostKey ], root, cmConfig, None )

   mode = root[ ConnMonitorConfigMode ].getSingletonInstance()
   cmds = mode[ 'ConnectivityMonitor.config' ]

   if config.nameServerGroup != config.nameServerGroupDefault:
      nameServerGroup = config.nameServerGroup.partition( "internal-ns-" )[ 2 ]
      cmds.addCommand( 'name-server group %s' % nameServerGroup )
   elif options.saveAll:
      cmds.addCommand( 'no name-server group' )
   if config.icmpDscp != config.icmpDscpDefault:
      if config.icmpDscpNameUsed:
         # Convert DSCP value to ACL name
         dscpName = AclCliLib.dscpNameFromValue( config.icmpDscp )
         cmds.addCommand( 'icmp echo qos dscp %s' % dscpName )
      else:
         cmds.addCommand( 'icmp echo qos dscp %d' % config.icmpDscp )
   elif options.saveAll:
      cmds.addCommand( 'no icmp echo qos dscp' )

   if config.probeInterval != config.probeIntervalDefault:
      cmds.addCommand( 'interval %d' % config.probeInterval )
   elif options.saveAll:
      cmds.addCommand( 'no interval' )

   if toggleCMIcmpPingCountEnabled():
      if config.pingCount != config.pingCountDefault:
         cmds.addCommand( 'icmp count %d' % config.pingCount )
      elif options.saveAll:
         cmds.addCommand( 'no icmp count' )

   if config.probeLossThreshold:
      cmds.addCommand( 'loss-threshold %d probes' % config.probeLossThreshold )
   elif options.saveAll:
      cmds.addCommand( 'no loss-threshold' )

   if cmConfig.enabled:
      cmds.addCommand( 'no shutdown' )
   elif options.saveAll:
      cmds.addCommand( 'shutdown' )

@CliSave.saver( 'Tac::Dir', 'connectivityMonitor/clientConfigDir/clients',
                requireMounts=( 'connectivityMonitor/config', ) )
def saveClientConfig( config, root, requireMounts, options ):
   cmConfig = requireMounts[ 'connectivityMonitor/config' ]
   parentMode = root[ ConnMonitorConfigMode ].getSingletonInstance()
   for client, clientConfig in config.items():
      if clientConfig.name == 'default':
         # Default client will be include in global Cm config mode
         continue
      mode = parentMode[ ConnMonitorClientConfigMode ].\
         getOrCreateModeInstance( client )
      cmds = mode[ 'ConnectivityMonitor.config.client' ]
      for hostKey in sorted( clientConfig.hostConfig ):
         saveHostConfig(
               clientConfig.hostConfig[ hostKey ], root, cmConfig,
               client )

      if clientConfig.nameServerGroup != clientConfig.nameServerGroupDefault:
         nameServerGroup = (
            clientConfig.nameServerGroup.partition( "internal-ns-" )[ 2 ] )
         cmds.addCommand( 'name-server group %s' % nameServerGroup )
      elif options.saveAll:
         cmds.addCommand( 'no name-server group' )
      if clientConfig.icmpDscp != clientConfig.icmpDscpDefault:
         if clientConfig.icmpDscpNameUsed:
            # Convert DSCP value to ACL name
            dscpName = AclCliLib.dscpNameFromValue( clientConfig.icmpDscp )
            cmds.addCommand( 'icmp echo qos dscp %s' % dscpName )
         else:
            cmds.addCommand( 'icmp echo qos dscp %d' % clientConfig.icmpDscp )
      elif options.saveAll:
         cmds.addCommand( 'no icmp echo qos dscp' )

      if clientConfig.probeInterval != clientConfig.probeIntervalDefault:
         cmds.addCommand( 'interval %d' % clientConfig.probeInterval )
      elif options.saveAll:
         cmds.addCommand( 'no interval' )

      if toggleCMIcmpPingCountEnabled():
         if clientConfig.pingCount != clientConfig.pingCountDefault:
            cmds.addCommand( 'icmp count %d' % clientConfig.pingCount )
         elif options.saveAll:
            cmds.addCommand( 'no icmp count' )

      if clientConfig.probeLossThreshold:
         cmds.addCommand( 'loss-threshold %d probes' %
                          clientConfig.probeLossThreshold )
      elif options.saveAll:
         cmds.addCommand( 'no loss-threshold' )
