# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Tac
import CliSave
from CliSavePlugin.Controllerdb import ( CvxConfigMode,
                                         controllerConfigPath,
                                         getClusterName )
from CliSavePlugin.ControllerClient import ( MgmtCvxConfigMode,
                                             mgmtControllerConfigPath )
from CliMode.CvxDebugService import CvxDebugServiceMode, MgmtCvxDebugServiceMode

class CvxDebugSaveMode( CvxDebugServiceMode, CliSave.Mode ):
   def __init__( self, param ):
      CvxDebugServiceMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

class MgmtCvxDebugSaveMode( MgmtCvxDebugServiceMode, CliSave.Mode ):
   def __init__( self, param ):
      MgmtCvxDebugServiceMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

CvxConfigMode.addChildMode( CvxDebugSaveMode, after=[ 'cvx' ] )
CvxDebugSaveMode.addCommandSequence( "ControllerDebugServer.config" )

MgmtCvxConfigMode.addChildMode( MgmtCvxDebugSaveMode, after=[ 'Mgmt.cvx' ] )
MgmtCvxDebugSaveMode.addCommandSequence( "ControllerDebugClient.config" )

def _addCmdsToCvxDebugMode( cmds, root, cvxClusterName ):
   if cmds:
      mode = root[ CvxConfigMode ].getOrCreateModeInstance(
         CvxConfigMode.modeName( cvxClusterName ) )
      debugMode = mode[ CvxDebugSaveMode ]
      debugMode = debugMode.getSingletonInstance()
      debugMode = debugMode[ "ControllerDebugServer.config" ]
      for cmd in cmds:
         debugMode.addCommand( cmd )

def _addCmdsToMgmtCvxDebugMode( cmds, root, cvxClusterName ):
   if cmds:
      mode = root[ MgmtCvxConfigMode ].getOrCreateModeInstance(
         MgmtCvxConfigMode.modeName( cvxClusterName ) )
      debugMode = mode[ MgmtCvxDebugSaveMode ]
      debugMode = debugMode.getSingletonInstance()
      debugMode = debugMode[ "ControllerDebugClient.config" ]
      for cmd in cmds:
         debugMode.addCommand( cmd )

def _parseWhitelists( cfg ):
   cmds = []
   # whitelists must be created before you can add elements to them
   for wln in cfg.versionWhiteList:
      wl = cfg.versionWhiteList[ wln ]
      cmds.append( "whitelist version name %s" % wln )
      if len( wl.allowedMembers.keys() ):
         cmds.append( "whitelist name {} add {}".format( wln,
            ",".join( map( str, wl.allowedMembers.keys() ) ) ) )

   for wln in cfg.switchWhiteList:
      wl = cfg.switchWhiteList[ wln ]
      cmds.append( "whitelist switch name %s" % wln )
      if len( wl.allowedMembers.keys() ):
         cmds.append( "whitelist name {} add {}".format( wln,
            ",".join( wl.allowedMembers.keys() ) ) )

   return cmds

# pylint: disable-msg=C0322 # pylint: disable=bad-option-value
@CliSave.saver( "ControllerDebug::Config", "controller/debug/config",
                requireMounts=( controllerConfigPath, ) )
def saveControllerDebugServerConfig( cfg, root, requireMounts, options ):
   saveAll = options.saveAll

   cmds = []
   if cfg.enabled or saveAll:
      cmds.append( "no shutdown" )

   updateIntervalChanged = cfg.updateInterval != 1 
   if updateIntervalChanged or saveAll:
      cmds.append( 'interval %d' % int( cfg.updateInterval ) )

   updateStatsIntervalChanged = cfg.statsCollectInterval != 1
   if updateStatsIntervalChanged or saveAll:
      cmds.append( 'stats interval %f' % float( cfg.statsCollectInterval ) )

   if cfg.pingEnabled:
      cmds.append( "pingpong" )
   elif saveAll:
      cmds.append( "no pingpong" )

   cmds += _parseWhitelists( cfg )
   cvxClusterName = getClusterName( requireMounts[ controllerConfigPath ] )
   _addCmdsToCvxDebugMode( cmds, root, cvxClusterName )
   
@CliSave.saver( "ControllerDebug::Config", "mgmt/controller/debug/config",
                requireMounts=( mgmtControllerConfigPath, ) )
def saveControllerDebugClientConfig( cfg, root, requireMounts, options ):
   saveAll = options.saveAll

   cmds = []
   if cfg.enabled or saveAll:
      cmds.append( "no shutdown" )

   updateIntervalChanged = cfg.updateInterval != 1
   if updateIntervalChanged or saveAll:
      cmds.append( 'interval %d' % int( cfg.updateInterval ) )

   if cfg.pingEnabled:
      cmds.append( "pingpong" )
   elif saveAll:
      cmds.append( "no pingpong" )

   cvxClusterName = requireMounts[ mgmtControllerConfigPath ].clusterName
   _addCmdsToMgmtCvxDebugMode( cmds, root, cvxClusterName )

@CliSave.saver( "Controller::OverrideServiceConfigDir", 
                "controller/debug/overrideService",
                requireMounts=( controllerConfigPath, ) )
def saveControllerDebugVersion( cfg, root, requireMounts, options ):
   cmds = []
   if "ControllerDebug" in cfg.overrideService:
      debugService = cfg.overrideService["ControllerDebug"]
      for version in debugService.versionConfig.version:
         cmds.append( "version %d" % int(version) )

   cvxClusterName = getClusterName( requireMounts[ controllerConfigPath ] )
   _addCmdsToCvxDebugMode( cmds, root, cvxClusterName )

@CliSave.saver( "Controller::OverrideServiceConfigDir",
                "mgmt/controller/debug/overrideService",
                requireMounts=( mgmtControllerConfigPath, ) )
def saveClientDebugVersion( cfg, root, requireMounts, options ):
   cmds = []
   if "ControllerDebug" in cfg.overrideService:
      debugService = cfg.overrideService["ControllerDebug"]
      for version in debugService.versionConfig.version:
         cmds.append( "version %d" % int(version) )
   cvxClusterName = requireMounts[ mgmtControllerConfigPath ].clusterName
   _addCmdsToMgmtCvxDebugMode( cmds, root, cvxClusterName )

def _buildMPString( mpCmd, cfg ):
   cmds = []
   for mpObj in cfg.values():
      cmd = f"{mpCmd} src {mpObj.srcPath}"
      cmd += " version %s" % mpObj.versionWhitelist
      cmd += " target %s" % mpObj.targetPath
      if ( mpObj.switchWhitelist ): # pylint: disable=superfluous-parens
         cmd += " switch %s" % mpObj.switchWhitelist
      cmds.append( cmd )
   return cmds

def _buildEntityString( cfg ):
   cmds = []
   for entity in cfg.entityConfig.values():
      cmds.append( "entity create %s" % entity.name )
      for index, value in entity.stringBuf.items():
         cmd = "entity name %s index %d value %s" % \
               ( entity.name, index, value )
         cmds.append( cmd )
   for dirName in cfg.subdirName:
      cmds.append( "dir path %s" % dirName )
   return cmds

@CliSave.saver( "ControllerDebug::ControllerDebugServerCliConfig",
                "controller/debug/mpconfig",
                requireMounts=( controllerConfigPath, ) )
def saveControllerDebugMountPublishConfig( cfg, root, requireMounts, options ):

   cmds = _buildEntityString( cfg.cdsDataDirConfig ) + \
          _buildMPString( "mount", cfg.cdsMountConfig ) + \
          _buildMPString( "publish", cfg.cdsPublishConfig )

   cvxClusterName = getClusterName( requireMounts[ controllerConfigPath ] )
   _addCmdsToCvxDebugMode( cmds, root, cvxClusterName )

@CliSave.saver( "ControllerDebug::ControllerDebugClientCliConfig",
                "mgmt/controller/debug/mpconfig",
                requireMounts=( mgmtControllerConfigPath, ) )
def saveDebugClientConfig( cfg, root, requireMounts, options ):
   cmds = _buildEntityString( cfg.cdcDataDirConfig )
   cvxClusterName = requireMounts[ mgmtControllerConfigPath ].clusterName
   _addCmdsToMgmtCvxDebugMode( cmds, root, cvxClusterName )

