# Copyright (c) 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
import EthIntfUtil
from CliSavePlugin import IntfCliSave
from DcbxTypes import cliSel, applicationTable, algorithmNames


CliSave.GlobalConfigMode.addCommandSequence( 'Dcbx.config' )
IntfCliSave.IntfConfigMode.addCommandSequence( 'Dcbx.config' )

@CliSave.saver( 'Dcbx::Config', 'dcb/dcbx/config',
                requireMounts=( 'interface/config/eth/phy/slice', ) )
def saveDcbxCliConfig( entity, root, requireMounts, options ):
   # First save the global configuration.
   cmds = root[ 'Dcbx.config' ]
   saveAll = options.saveAll

   # Dcbx classification rule is based on a protocol selector (i.e. ether,
   # tcp-sctp, tcp-sctp-udp, and udp) and a protocol ID (i.e. ethertype 
   # or port id). Multiple protocol entries can be grouped into an application,
   # e.g. 'iscsi' contains both tcp-sctp 860 and tcp-sctp 3260. User is 
   # allowed to enter a rule based on application. But internally we 
   # generate multiple rules, one for each protocol tupple. When displaying
   # "show run" we do the reverse. We parse all the rules to see if some
   # can be grouped into an application, and only display them as a single
   # rule.

   reverseApplicationTable = {}
   for appName, selProtos in applicationTable.items():
      for selProto in selProtos:
         assert selProto not in reverseApplicationTable
         reverseApplicationTable[ selProto ] = appName

   applicationEntry = {   app: []  for app in applicationTable  }
   applicationMatch = {   app: False  for app in applicationTable  }
   outputList = []

   def applicationPriorityCommand( entry ):
      # pylint: disable-next=consider-using-f-string
      return 'dcbx application %s %d priority %d' % (
         cliSel( entry.sel ), entry.protocolId, entry.priority )

   if len( entity.applicationPriorityEntry ) == 0 and saveAll:
      # display default
      cmds.addCommand( 'no dcbx application' )

   # In the first iteration we try to parse the rules and group them
   # by application. We generate the command on the fly into a temporary
   # buffer. We will display them after we complete application 
   # matching. Please note that entity.applicationPriorityEntry is
   # already sorted. 
   for entry in entity.applicationPriorityEntry.values():
      selProto = ( entry.sel, entry.protocolId )
      if selProto in reverseApplicationTable:
         application = reverseApplicationTable[ selProto ]
         applicationEntry[ application ].append( entry )
      else:
         application = ''
      outputList.append( ( cliSel( entry.sel ), application,
                           applicationPriorityCommand( entry ) ) )

   for app, entries in applicationEntry.items():
      if len( entries ) == len( applicationTable[ app ] ) and \
            all(  entry.priority == entries[ 0 ].priority
                 for entry in entries  ):
         # We can use the application name, instead of listing each entry
         # separately.
         applicationMatch[ app ] = True
         entry = entries[ 0 ]
         # pylint: disable-next=consider-using-f-string
         outputList.append( (app, '', 'dcbx application %s priority %d' % 
                             ( app, entry.priority ) ) )

   # In the second iteration we bypass those protocol entries that
   # can be grouped into an application entry. The application entries
   # are appended to the end of the output buffer. We want sort the 
   # entires so that protocol/application name show in alphabatic order,
   # e.g. 'iscsi' before 'tcp-sctp'. Phython sort is stable, i.e. it
   # preserves existing order for protocol entries.
   for _, application, cliString in sorted ( 
         outputList, key=lambda output: output[ 0 ] ):
      if application == '' or not applicationMatch[ application ]:
         cmds.addCommand( cliString )

   # Now save ETS specific configuration
   infoFields = [ ( ' ', entity.etsConfigInfo ) ]
   infoFields.append( ( ' recommendation ', entity.etsRecommendationInfo ) )

   for cliRecommendation, info in infoFields:
      if not info:
         continue

      for c in range(0, 8):
         tc = info.cosToTrafficClass[ c ]
         if tc or saveAll:
            # pylint: disable-next=consider-using-f-string
            cmds.addCommand( "dcbx ets%sqos map cos %d traffic-class %d" %
               ( cliRecommendation, c, tc ) )

      for t in range(0, 8):
         bw = info.trafficClassBandwidth[ t ]
         if bw or saveAll:
            # pylint: disable-next=consider-using-f-string
            cmds.addCommand( "dcbx ets%straffic-class %d bandwidth %d" %
                              ( cliRecommendation, t, bw ) )

         algo = info.trafficClassAlgorithm.get( t )
         if algo is not None or saveAll:
            if algo is None:
               algo = 255

            # Use the algorithm name if specified
            if t in info.trafficClassAlgorithmName:
               algo = algorithmNames[ algo ]

            # pylint: disable-next=consider-using-f-string
            cmds.addCommand( "dcbx ets%straffic-class %d algorithm %s" %
                           ( cliRecommendation, t, str( algo ) ) )
   
   # Then save the port-specific configuration.
   # Get the port names based on the command type
   if saveAll:
      phyIntfConfigDir = requireMounts[ 'interface/config/eth/phy/slice' ]
      cfgPortNames = EthIntfUtil.allPhyEthernetInterfaceNames( phyIntfConfigDir )
   else:
      cfgPortNames = entity.portEnabled
   
   for portName in cfgPortNames:
      mode = entity.portEnabled.get( portName )
      if mode == "modeIeee":
         cmd = 'dcbx mode ieee'
      elif mode == "modeCee":
         cmd = 'dcbx mode cee'
      elif saveAll:
         cmd = 'no dcbx mode'
      else:
         continue
      mode = root[ IntfCliSave.IntfConfigMode ].getOrCreateModeInstance( portName )
      cmds = mode[ 'Dcbx.config' ]
      cmds.addCommand( cmd )

