# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave, Tracing, AclLib
from CliSavePlugin.IntfCliSave import IntfConfigMode
from CliMode.DpAcl import ( HardwareAclDuringUpdateMode,
                            HardwareAclIngressSharingMode,
                            HardwareAclEgressIp6SharingMode,
                            HardwareAclEgressIpSharingOnMode,
                            HardwareAclEgressIpSharingOffMode,
                            HardwareAclEgressLogRecirculationMode,
                            HardwareAclEgressIpLogRecirculationMode )

__defaultTraceHandle__ = Tracing.Handle( 'AclCliSave' )

class HardwareAclActionDuringUpdate( HardwareAclDuringUpdateMode,
                                           CliSave.Mode ):
   def __init__( self, param ):
      HardwareAclDuringUpdateMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class HardwareAclIngressSharing( HardwareAclIngressSharingMode, CliSave.Mode ):
   def __init__( self, param ):
      HardwareAclIngressSharingMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class HardwareAclEgressIp6Sharing( HardwareAclEgressIp6SharingMode, CliSave.Mode ):
   def __init__( self, param ):
      HardwareAclEgressIp6SharingMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class HardwareAclEgressIpSharingOn( HardwareAclEgressIpSharingOnMode, CliSave.Mode ):
   def __init__( self, param ):
      HardwareAclEgressIpSharingOnMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class HardwareAclEgressIpSharingOff( HardwareAclEgressIpSharingOffMode,
                                     CliSave.Mode ):
   def __init__( self, param ):
      HardwareAclEgressIpSharingOffMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class HardwareAclEgressIpLogRecirculation( HardwareAclEgressIpLogRecirculationMode,
                                           CliSave.Mode ):
   def __init__( self, param ):
      HardwareAclEgressIpLogRecirculationMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class HardwareAclEgressLogRecirculation( HardwareAclEgressLogRecirculationMode,
                                         CliSave.Mode ):
   def __init__( self, param ):
      HardwareAclEgressLogRecirculationMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class HardwareRouterAclExcludeMlagPeerLink( CliSave.Mode ):
   def enterCmd( self ):
      return 'hardware access-list router-acl exclude mlag peer-link'

IntfConfigMode.addCommandSequence( 'Acl.intf', after=[ 'Lag.lagConfig' ] )

CliSave.GlobalConfigMode.addChildMode( HardwareAclActionDuringUpdate,
                                       before=[ IntfConfigMode ] )
HardwareAclActionDuringUpdate.addCommandSequence( 'Acl.knob' )

CliSave.GlobalConfigMode.addChildMode( HardwareAclIngressSharing,
                                       before=[ IntfConfigMode ] )
HardwareAclIngressSharing.addCommandSequence( 'Acl.ingressRaclSharing' )

CliSave.GlobalConfigMode.addChildMode( HardwareAclEgressIp6Sharing,
                                       before=[ IntfConfigMode ] )
HardwareAclEgressIp6Sharing.addCommandSequence( 'Acl.egressIp6RaclSharing' )

CliSave.GlobalConfigMode.addChildMode( HardwareAclEgressIpSharingOn,
                                       before=[ IntfConfigMode ] )
HardwareAclEgressIpSharingOn.addCommandSequence( 'Acl.egressIpRaclSharingOn' )

CliSave.GlobalConfigMode.addChildMode( HardwareAclEgressIpSharingOff,
                                       before=[ IntfConfigMode ] )
HardwareAclEgressIpSharingOff.addCommandSequence( 'Acl.egressIpRaclSharingOff' )

CliSave.GlobalConfigMode.addChildMode( HardwareRouterAclExcludeMlagPeerLink,
                                       before=[ IntfConfigMode ] )
HardwareRouterAclExcludeMlagPeerLink.addCommandSequence(
                                       'Acl.excludeMlagPeerLink' )

CliSave.GlobalConfigMode.addChildMode( HardwareAclEgressIpLogRecirculation,
                                       before=[ IntfConfigMode ] )
HardwareAclEgressIpLogRecirculation.addCommandSequence(
       'Acl.egressIpAclRecirculation' )

CliSave.GlobalConfigMode.addChildMode( HardwareAclEgressLogRecirculation,
                                       before=[ IntfConfigMode ] )
HardwareAclEgressLogRecirculation.addCommandSequence(
       'Acl.egressAclRecirculation' )

CliSave.GlobalConfigMode.addCommandSequence( 'Acl.deepInsp',
                                             before=[ IntfConfigMode ] )

CliSave.GlobalConfigMode.addCommandSequence( 'Acl.Ipv6Icmp6Match',
                                             before=[ IntfConfigMode ] )

CliSave.GlobalConfigMode.addCommandSequence( 'Acl.EgressRoutedInterfaceAclSharing',
                                             before=[ IntfConfigMode ] )

CliSave.GlobalConfigMode.addCommandSequence( 
      'Acl.ingressIpv4Ipv6SecurityAclSharingSupported', 
      before=[ IntfConfigMode ] )

CliSave.GlobalConfigMode.addCommandSequence( 
      'Acl.ingressIpv4Ipv6MirroringAclSharingSupported', 
      before=[ IntfConfigMode ] )

CliSave.GlobalConfigMode.addCommandSequence( 
      'Acl.ingressIpv4Ipv6QosAclSharingSupported', 
      before=[ IntfConfigMode ] )

CliSave.GlobalConfigMode.addCommandSequence( 
      'Acl.ingressIpv6SecurityAclKeyWidthMode', 
      before=[ IntfConfigMode ] )

CliSave.GlobalConfigMode.addCommandSequence( 
      'Acl.ingressIpv6QosAclKeyWidthMode', 
      before=[ IntfConfigMode ] )

CliSave.GlobalConfigMode.addCommandSequence( 
      'Acl.ingressMirrorAclKeyWidthMode', 
      before=[ IntfConfigMode ] )

@CliSave.saver( 'Acl::IntfConfig', 'acl/intf/config/cli',
                requireMounts=( 'acl/config/cli',
                                'acl/intf/config/input/secure-monitor' ),
                secureMonitor=True )
def saveDpConfig( entity, root, requireMounts, options ):
   allAclConfig = requireMounts[ 'acl/config/cli' ]
   if options.secureMonitor:
      entity = requireMounts[ 'acl/intf/config/input/secure-monitor' ]
   # per-interface ACL config
   for aclType in AclLib.aclTypes:
      typeConfig = entity.config[ aclType ]
      for direction in AclLib.aclDirections:
         config = typeConfig.intf[ direction ].intf
         for intfName, aclName in config.items():
            aclConfType = allAclConfig.config.get( aclType )
            if aclConfType:
               aclConf = aclConfType.acl.get( aclName )
               if aclConf and aclConf.dynamic:
                  continue
            # only create the mode if it has non-default configuration
            mode = root[ IntfConfigMode ].getOrCreateModeInstance( intfName )
            cmds = mode[ 'Acl.intf' ]
            # pylint: disable-next=consider-using-f-string
            cmds.addCommand( "{} access-group {} {}".format(
               aclType.lower( ), aclName, direction ) )

@CliSave.saver( 'Acl::ParamConfig', 'acl/paramconfig',
                requireMounts=( 'acl/status/all', ) )
def saveAclKnobConfig( entity, root, requireMounts, options ):
   status = requireMounts[ 'acl/status/all' ]
   if entity.permitDuringAclUpdateConfig == \
      AclLib.ActionDuringAclUpdate.actionDuringAclUpdatePermit :
      # CLI value is permit
      if status.dpDefaultActionDuringAclUpdate == \
         AclLib.ActionDuringAclUpdate.actionDuringAclUpdatePermit :
         if options.saveAll:
            # default behaviour permit then only saveAll
            root[ HardwareAclActionDuringUpdate ].getOrCreateModeInstance(
               HardwareAclDuringUpdateMode.PERMIT )
      else:
         # default behaviour not permit
         root[ HardwareAclActionDuringUpdate ].getOrCreateModeInstance(
            HardwareAclDuringUpdateMode.PERMIT )
   elif entity.permitDuringAclUpdateConfig == \
        AclLib.ActionDuringAclUpdate.actionDuringAclUpdateDeny :
      # CLI value is deny
      if status.dpDefaultActionDuringAclUpdate == \
         AclLib.ActionDuringAclUpdate.actionDuringAclUpdateDeny :
         if options.saveAll:
            # default behaviour deny then only saveAll
            root[ HardwareAclActionDuringUpdate ].getOrCreateModeInstance(
               HardwareAclDuringUpdateMode.DENY )
      else:
         # default behaviour not deny
         root[ HardwareAclActionDuringUpdate ].getOrCreateModeInstance(
            HardwareAclDuringUpdateMode.DENY )
   elif ( ( entity.permitDuringAclUpdateConfig == \
          AclLib.ActionDuringAclUpdate.actionDuringAclUpdateUnset ) and \
          options.saveAll ) :
      # CLI value is unset
      if status.dpDefaultActionDuringAclUpdate == \
         AclLib.ActionDuringAclUpdate.actionDuringAclUpdateUnset :
         # default behaviour unset
         root[ HardwareAclActionDuringUpdate ].getOrCreateModeInstance(
            HardwareAclDuringUpdateMode.UNSET )
      elif status.dpDefaultActionDuringAclUpdate == \
           AclLib.ActionDuringAclUpdate.actionDuringAclUpdateDeny :
         # default behaviour deny
         root[ HardwareAclActionDuringUpdate ].getOrCreateModeInstance(
            HardwareAclDuringUpdateMode.DENY )
      elif status.dpDefaultActionDuringAclUpdate == \
           AclLib.ActionDuringAclUpdate.actionDuringAclUpdatePermit :
         # default behaviour permit
         root[ HardwareAclActionDuringUpdate ].getOrCreateModeInstance(
            HardwareAclDuringUpdateMode.PERMIT )

@CliSave.saver( 'Acl::ParamConfig', 'acl/paramconfig' )
def saveIngressRaclSharingConfig( entity, root, requireMounts, options ):
   if entity.ingressRaclSharing:
      root[ HardwareAclIngressSharing ].getOrCreateModeInstance( entity.name )

@CliSave.saver( 'Acl::ParamConfig', 'acl/paramconfig' )
def saveEgressIp6RaclSharingConfig( entity, root, requireMounts, options ):
   if entity.egressIp6RaclSharing:
      root[ HardwareAclEgressIp6Sharing ].getOrCreateModeInstance( entity.name )

@CliSave.saver( 'Acl::ParamConfig', 'acl/paramconfig',
                requireMounts = ( 'acl/status/all', ) )
def saveEgressIpRaclSharingConfig( entity, root, requireMounts, options ):
   status = requireMounts[ 'acl/status/all' ]
   if not entity.egressIpRaclSharing:
      root[ HardwareAclEgressIpSharingOff ].getOrCreateModeInstance( entity.name )
   else:
      # Check for platform compatibility.
      if status.dpEgressRaclSharingSupported and options.saveAll:
         root[ HardwareAclEgressIpSharingOn ].getOrCreateModeInstance( entity.name )

@CliSave.saver( 'Acl::ParamConfig', 'acl/paramconfig' )
def saveExcludeMlagPeerLinkConfig( entity, root, requireMounts, options ):
   if entity.excludeMlagPeerLink:
      root[ HardwareRouterAclExcludeMlagPeerLink ].getOrCreateModeInstance(
                                                                     entity.name )

@CliSave.saver( 'Acl::ParamConfig', 'acl/paramconfig' )
def saveEgressIpAclRecirculationConfig( entity, root, requireMounts, options ):
   if entity.egressIpAclLogRecirc:
      root[ HardwareAclEgressIpLogRecirculation ].getOrCreateModeInstance(
                                                                     entity.name )

@CliSave.saver( 'Acl::ParamConfig', 'acl/paramconfig' )
def saveEgressAclRecirculationConfig( entity, root, requireMounts, options ):
   if entity.egressAclLogRecirc:
      root[ HardwareAclEgressLogRecirculation ].getOrCreateModeInstance(
                                                                     entity.name )

@CliSave.saver( 'Acl::ParamConfig', 'acl/paramconfig' )
def saveAclDeepInspConfig( entity, root, requireMounts, options ):
   if not entity.deepInspSkipL2 and not entity.deepInspSkipL4 and not \
          options.saveAll:
      return
   cmd = root[ 'Acl.deepInsp' ]
   if entity.deepInspSkipL2:
      # pylint: disable-next=consider-using-f-string
      cmd.addCommand( 'deep-inspection payload l2 skip %d' % entity.deepInspSkipL2 )
   elif options.saveAll:
      cmd.addCommand( 'no deep-inspection payload l2 skip' )
   if entity.deepInspSkipL4:
      # pylint: disable-next=consider-using-f-string
      cmd.addCommand( 'deep-inspection payload l4 skip %d' % entity.deepInspSkipL4 )
   elif options.saveAll:
      cmd.addCommand( 'no deep-inspection payload l4 skip' )

@CliSave.saver( 'Acl::ParamConfig', 'acl/paramconfig' )
def saveImplicitIcmp6RulesTypeConfig( entity, root, requireMounts, options ):
   cmd = root[ 'Acl.Ipv6Icmp6Match' ]
   if entity.permitIcmp6TypesConfig == AclLib.Icmp6RuleType.neighborDiscovery:
      cmd.addCommand( 'hardware access-list ipv6 implicit-permit icmpv6 ' + \
                      'neighbor-discovery' )
   elif entity.permitIcmp6TypesConfig == AclLib.Icmp6RuleType.all:
      cmd.addCommand( 'hardware access-list ipv6 implicit-permit icmpv6 all' )
   elif entity.permitIcmp6TypesConfig == AclLib.Icmp6RuleType.none:
      cmd.addCommand( 'hardware access-list ipv6 implicit-permit icmpv6 disabled' )
   elif options.saveAll:
      cmd.addCommand( 'default hardware access-list ipv6 implicit-permit ' + \
                      'icmpv6' )

@CliSave.saver( 'Acl::ParamConfig', 'acl/paramconfig' )
def saveEgressRoutedInterfaceAclSharingConfig( entity, root, requireMounts,
                                               options ):
   cmd = root[ 'Acl.EgressRoutedInterfaceAclSharing' ]
   command = "hardware access-list ipv4 egress resource sharing routed-interfaces"
   if entity.egressRoutedInterfaceAclSharing:
      cmd.addCommand( command )
   elif options.saveAll:
      cmd.addCommand( "no " + command )

@CliSave.saver( 'Acl::ParamConfig', 'acl/paramconfig' )
def saveIngressIpv4Ipv6SecurityAclSharing( entity, root, requireMounts, options ):
   cmd = root[ 'Acl.ingressIpv4Ipv6SecurityAclSharingSupported' ]
   if entity.ingressIpv4Ipv6PaclSharingSupported:
      cmd.addCommand( 'hardware access-list resource sharing ipv4-ipv6 ' + \
                      'security-acl in' )
   elif options.saveAll:
      cmd.addCommand( 'no hardware access-list resource sharing ipv4-ipv6' + \
                      ' security-acl in' )

@CliSave.saver( 'Acl::ParamConfig', 'acl/paramconfig' )
def saveIngressIpv4Ipv6MirroringAclSharing( entity, root, requireMounts, options ):
   cmd = root[ 'Acl.ingressIpv4Ipv6MirroringAclSharingSupported' ]
   if entity.ingressIpv4Ipv6MirrorAclSharingSupported:
      cmd.addCommand( 'hardware access-list resource sharing ipv4-ipv6 ' + \
                      'mirroring-policy in' )
   elif options.saveAll:
      cmd.addCommand( 'no hardware access-list resource sharing ipv4-ipv6' + \
                      ' mirroring-policy in' )

@CliSave.saver( 'Acl::ParamConfig', 'acl/paramconfig' )
def saveIngressIpv4Ipv6QosAclSharingMode( entity, root, requireMounts, options ):
   cmd = root[ 'Acl.ingressIpv4Ipv6QosAclSharingSupported' ]
   if entity.ingressIpv4Ipv6QosAclSharingSupported:
      cmd.addCommand( 'hardware access-list resource sharing ipv4-ipv6 ' + \
                      'qos-policy in' )
   elif options.saveAll:
      cmd.addCommand( 'no hardware access-list resource sharing ipv4-ipv6' + \
                      ' qos-policy in' )

@CliSave.saver( 'Acl::ParamConfig', 'acl/paramconfig' )
def saveIngressMirroringAclKeyWidthMode( entity, root, requireMounts, options ):
   cmd = root[ 'Acl.ingressMirrorAclKeyWidthMode' ]
   if entity.ingressMirrorAclKeyWidthMode == \
         AclLib.TcamBankSharingMode.bankSharingModeNarrow:
      cmd.addCommand( 'hardware access-list ipv4-ipv6 mirroring-policy ' + \
            'key-width narrow in' )
   elif options.saveAll:
      cmd.addCommand( 'no hardware access-list ipv4-ipv6 mirroring-policy ' + \
                      ' key-width in' )

@CliSave.saver( 'Acl::ParamConfig', 'acl/paramconfig' )
def saveIngressIpv6SecurityAclKeyWidthMode( entity, root, requireMounts, options ):
   cmd = root[ 'Acl.ingressIpv6SecurityAclKeyWidthMode' ]
   if entity.ingressIpv6SecurityAclKeyWidthMode == \
         AclLib.TcamBankSharingMode.bankSharingModeNarrow:
      cmd.addCommand( 'hardware access-list ipv6 security-acl ' + \
            'key-width narrow in' )
   elif options.saveAll:
      cmd.addCommand( 'no hardware access-list ipv6 security-acl ' + \
                      ' key-width in' )

@CliSave.saver( 'Acl::ParamConfig', 'acl/paramconfig' )
def saveIngressIpv6QosAclKeyWidthMode( entity, root, requireMounts, options ):
   cmd = root[ 'Acl.ingressIpv6QosAclKeyWidthMode' ]
   if entity.ingressIpv6QosAclKeyWidthMode == \
         AclLib.TcamBankSharingMode.bankSharingModeNarrow:
      cmd.addCommand( 'hardware access-list ipv6 qos-policy ' + \
            'key-width narrow in' )
   elif options.saveAll:
      cmd.addCommand( 'no hardware access-list ipv6 qos-policy ' + \
                      ' key-width in' )

