# Copyright (c) 2006-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

#-------------------------------------------------------------------------------
# This module implements saving the Ebra CLI, including VLAN-specific and
# non-VLAN-specific commands.
#-------------------------------------------------------------------------------
import Tac, Ethernet, CliSave
import EbraLib
# pylint: disable-next=consider-using-from-import
import CliSavePlugin.IntfCliSave as IntfCliSave
import Vlan
import Tracing
from CliSavePlugin.IntfCliSave import IntfConfigMode
import EthIntfUtil
from CliMode.Vlan import VlanMode
from CliMode.VlanEtree import VlanEtreeMode
import Intf.IntfRange
from EbraLib import defaultTpid, egressVlanXlateKeyIfBoth
import MultiRangeRule
import CliExtensions
import Toggles.EbraToggleLib
from CliSavePlugin import EthIntfCliSave # pylint: disable=ungrouped-imports
from CliSavePlugin.IntfCliSave import subintfMtuCliSaveHook

traceDetail = Tracing.trace2

# CLI hook to return the command for label entry unicast mac addresses in
# bridging config.
labelEntryHook = CliExtensions.CliHook()

SubIntfId = Tac.Type( 'Arnet::SubIntfId' )

isSubIntf = SubIntfId.isSubIntfId

subintfMtuCliSaveHook.addExtension( isSubIntf )

#-------------------------------------------------------------------------------
# Object used for saving commands in "config-vlan" mode.
#-------------------------------------------------------------------------------
class VlanConfigMode( VlanMode, CliSave.Mode ):
   mergeRange = True

   def __init__( self, param ):
      VlanMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def instanceKey( self ):
      return self.param_[ 0 ]

   def canMergeRange( self ):
      # Can merge vlan>=2
      return self.instanceKey() > 1

   @staticmethod
   def enterRangeCmd( modes ):
      return "vlan " + MultiRangeRule.multiRangeToCanonicalString( 
         x.instanceKey() for x in modes )

class VlanEtreeConfigMode( VlanEtreeMode, CliSave.Mode ):
   def __init__( self, param ):
      VlanEtreeMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def instanceKey( self ):
      return self.param_

VlanConfigMode.addChildMode( VlanEtreeConfigMode )
VlanEtreeConfigMode.addCommandSequence( 'Ebra.vlanEtree' )

CliSave.GlobalConfigMode.addChildMode( VlanConfigMode, before=[ IntfConfigMode ] )
VlanConfigMode.addCommandSequence( 'Ebra.vlan' )

IntfConfigMode.addCommandSequence( 'Ebra.switchport',
   before=[ 'Arnet.l2l3barrier' ], after=[ EthIntfCliSave.EthIntfCmdSeq ] )
IntfConfigMode.addCommandSequence( 'Ebra.vlanIntf', after=[ 'Ebra.switchport' ] )
CliSave.GlobalConfigMode.addCommandSequence( 'Ebra.agingTime',
                                             after=[ IntfConfigMode ] )
CliSave.GlobalConfigMode.addCommandSequence( 'Ebra.staticMac',
                                             after=[ 'Ebra.agingTime' ] )
CliSave.GlobalConfigMode.addCommandSequence( 'Ebra.internalVlanAllocationPolicy' )
CliSave.GlobalConfigMode.addCommandSequence( 'Ebra.ieeeReservedForwarding',
                                             after=[ 'Ebra.staticMac' ] )

CliSave.GlobalConfigMode.addCommandSequence( 'Ebra.pausePassThrough' )
CliSave.GlobalConfigMode.addCommandSequence( 'Ebra.sourceCheckCPUMac' )
CliSave.GlobalConfigMode.addCommandSequence( 'Ebra.sourceMacInvalid' )
CliSave.GlobalConfigMode.addCommandSequence( 'Ebra.vlanMacFlushAggregationDisabled' )
CliSave.GlobalConfigMode.addCommandSequence( 'Ebra.macIpv4McMismatchDropEnabled' )
CliSave.GlobalConfigMode.addCommandSequence( 'Ebra.forwardingMode' )
CliSave.GlobalConfigMode.addCommandSequence( 'Ebra.l3McastForwardingMode' )
CliSave.GlobalConfigMode.addCommandSequence( 'Ebra.macBasedVlanAssignmentScope' )
CliSave.GlobalConfigMode.addCommandSequence( 'Ebra.defaultSwitchportMode' )
CliSave.GlobalConfigMode.addCommandSequence( 'Ebra.defaultSwitchportPhoneVlan' )
CliSave.GlobalConfigMode.addCommandSequence( 'Ebra.defaultSwitchportPhoneCos' )
CliSave.GlobalConfigMode.addCommandSequence( 'Ebra.defaultSwitchportPhoneTrnkUntag' )
CliSave.GlobalConfigMode.addCommandSequence( 'Ebra.defaultSwitchportPhoneAclBypass' )
CliSave.GlobalConfigMode.addCommandSequence( 'Ebra.defaultSwitchportPhoneTrustMode' )
CliSave.GlobalConfigMode.addCommandSequence( 'Ebra.switchportVlanTagValidation' )
CliSave.GlobalConfigMode.addCommandSequence( 'Ebra.switchportLlcValidation' )
CliSave.GlobalConfigMode.addCommandSequence( 'Ebra.vlanDynamicRange', 
                                             after=[ 'Ebra.agingTime' ] )
CliSave.GlobalConfigMode.addCommandSequence( 'Ebra.vlanTpids' )

#-------------------------------------------------------------------------------
# Saves the state of an Interface::VlanIntfConfigDir object.
#-------------------------------------------------------------------------------
# pylint: disable-msg=C0322 # pylint: disable=bad-option-value
# XXX Workaround for a pylint bug with decorators.
@CliSave.saver( 'Interface::VlanIntfConfigDir', 'interface/config/eth/vlan',
                requireMounts=( 'interface/status/all', 'routerMac/hwCapability' ) )
def saveVlanIntfConfig( entity, root, requireMounts, options ):
   for intfConfig in entity.intfConfig.values():
      # Save the baseclass (Arnet::IntfConfig) attributes.
      if intfConfig.source != "cli":
         # don't save dynamic SVIs
         continue
      if options.intfFilter and intfConfig.intfId not in options.intfFilter:
         continue
      IntfCliSave.saveIntfConfig( intfConfig, root, requireMounts, options )
      saveAll = options.saveAll
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intfConfig.intfId )
      cmds = mode[ 'Ebra.vlanIntf' ]
      if not intfConfig.autoState:
         cmds.addCommand( 'no autostate' )
      elif saveAll:
         cmds.addCommand( 'autostate' )

      hwCaps = requireMounts[ 'routerMac/hwCapability' ]
      if hwCaps.intfRouterMacSupported:
         if intfConfig.addr != '00:00:00:00:00:00':
            rmac = Ethernet.convertMacAddrCanonicalToDisplay( intfConfig.addr )
            cmds.addCommand( 'mac-address router %s' % rmac )
         elif saveAll:
            cmds.addCommand( 'no mac-address router' )

#-------------------------------------------------------------------------------
# Given a vlanConfig obj, returns the cmds for the vlan
#-------------------------------------------------------------------------------
 
def getVlanConfig( entity, saveAll ):
   cmds = []
   if entity.configuredName != entity.defaultName or saveAll:
      cmds.append( 'name %s' % entity.configuredName )

   if not entity.macLearning:
      cmds.append( 'no mac address learning' )
   elif entity.maxLearnedDynamicMac > 0:
      cmds.append( 'mac address learning local limit %d hosts' %
                    entity.maxLearnedDynamicMac )
   elif saveAll:
      cmds.append( 'mac address learning' ) 

   def isValidAgingTime( age ):
      return 10 <= age <= 1000000
   
   if entity.agingTime and not isValidAgingTime( entity.agingTime ):
      cmds.append( 'bridge mac-address-table aging disabled' )
   elif entity.agingTime:
      cmds.append( 'bridge mac-address-table aging timeout %s seconds' %
            entity.agingTime )
   elif saveAll:
      cmds.append( 'default bridge mac-address-table aging' )
   
   if entity.adminState == 'suspended':
      cmds.append( 'state suspend' )
   elif saveAll:
      cmds.append( 'state active' )

   if entity.floodsetExpandedVlanId:
      cmds.append( 'floodset expanded vlan %d' % entity.floodsetExpandedVlanId )

   if entity.l2MissPolicyConfig:
      policy = entity.l2MissPolicyConfig
      if saveAll:
         if policy.l2UcMissPolicy == 'flood':
            cmds.append( 'mac address forwarding unicast miss action flood' )
         if policy.l2McMissPolicy == 'flood':
            cmds.append( 'mac address forwarding multicast miss action flood' )
            
      if policy.l2UcMissPolicy != 'flood':
         cmds.append( 'mac address forwarding unicast miss action %s' %
                                                   policy.l2UcMissPolicy )
      if policy.l2McMissPolicy != 'flood':
         cmds.append( 'mac address forwarding multicast miss action %s' %
                                                   policy.l2McMissPolicy )

   elif saveAll:
      cmds.append( 'mac address forwarding unicast miss action flood' )
      cmds.append( 'mac address forwarding multicast miss action flood' )

   for i in sorted( entity.trunkGroup ):
      cmds.append( 'trunk group %s' % i )

   if Toggles.EbraToggleLib.toggleEvpnVxlanOCSupportEnabled():
      if entity.macVrf != '':
         cmds.append( 'mac-vrf %s' % entity.macVrf )
      elif saveAll:
         cmds.append( 'no mac-vrf' )
         
   return cmds

# -------------------------------------------------------------------------------
# Saves the e-tree specific config for a list of vlans
# -------------------------------------------------------------------------------

def saveVlanEtreeConfig( vlanMode, vlanString, vlanEntity, saveAll ):
   EtreeRole = Tac.Type( 'Bridging::EtreeRole' )
   if vlanEntity.etreeRole == EtreeRole.etreeRoleLeaf:
      etreeMode = vlanMode[ VlanEtreeConfigMode ].getOrCreateModeInstance(
            vlanString )
      cmds = etreeMode[ 'Ebra.vlanEtree' ]
      EtreeRemoteLeafInstallMode = Tac.Type( 'Bridging::EtreeRemoteLeafInstallMode' )
      if vlanEntity.etreeRemoteLeafInstallMode == \
            EtreeRemoteLeafInstallMode.installDrop:
         cmds.addCommand( 'remote leaf host drop' )
      elif saveAll:
         # If the toggle is disabled, there's no user-facing way to set the remote
         # leaf drop mode on a VLAN, so no need to gate the generation of the
         # positive case.  However, if the toggle is disabled we don't want to
         # generate a full config that can't be loaded by the same release with
         # the same toggle settings, so gate the generation of the negative case
         if Toggles.EbraToggleLib.toggleEtreeRemoteLeafInstallModeEnabled():
            cmds.addCommand( 'no remote leaf host drop' )

#-------------------------------------------------------------------------------
# Saves the config for a list of vlans
#-------------------------------------------------------------------------------

def saveVlanListConfig( root, vlanList, vlanEnabled, vlanEntity, cmdList, saveAll ):
   # The first vlan in the vlan list is included as the first parameter
   # to ensure that the vlan lists are sorted numerically
   vlanString = Vlan.vlanSetToCanonicalString( vlanList )
   mode = root[ VlanConfigMode ].getOrCreateModeInstance( ( vlanList[0],
                                                            vlanString,
                                                            vlanEnabled ) )
   cmds = mode[ 'Ebra.vlan' ]
   for cmd in cmdList:
      cmds.addCommand( cmd )

   if vlanEnabled:
      saveVlanEtreeConfig( mode, vlanString, vlanEntity, saveAll )

#-------------------------------------------------------------------------------
# Saves the config of Bridging::VlanConfig objects
#-------------------------------------------------------------------------------
 
def saveVlanConfig( entity, root, requireMounts, options ):
   # Save the VLAN configuration.  Note that VLANs that are in 'inactive' state
   # are not explicitly saved as all the configuration for such VLANs must be at
   # their default values.
   # There are a couple of exceptions to deal with for vlan 1.  It is created
   # by default, so we don't save the 'vlan 1' command unless the name or
   # state attributes are non-default.  Also, if it is deleted (i.e. state is
   # 'inactive', we have to save the 'no vlan 1' command.
   # If inactive vlans have non-default MAC address table
   # configuration (aging time and/or configured hosts) then they will be
   # recreated implicitly by the "mac address-table" commands saved below.
   #
   # This function iterates through the vlan objects, and outputs vlan range
   # commands, when a list of vlans have the same list of commands.
   # If multiple vlan ranges have the same set of commands, then vlan ranges are
   # displayed in the same line, separated by commas. However, if there are one or
   # more vlans in between the vlan ranges, with a different set of commands,
   # then the corresponding vlan ranges are not combined into a single line.

   # vlan 1 is handled as a special case, before the for loop
   EtreeRole = Tac.Type( 'Bridging::EtreeRole' )
   vlanEntity = entity.vlanConfig.get( 1 )
   traceDetail( "Processing vlan 1" )
   if vlanEntity is None:
      # Set the enabled flag to False, so that the "no vlan 1" is saved 
      saveVlanListConfig( root, [ 1 ], False, None, [], options.saveAll )
   else:
      cmds = getVlanConfig( vlanEntity, options.saveAll )
      # Since e-tree is a submode, we also need to save the VLAN 1 config if the
      # e-tree role has a non-default value
      if ( cmds or CliSave.hasComments( 'vlan-1', requireMounts ) or
            vlanEntity.etreeRole != EtreeRole.etreeRoleRoot ):
         saveVlanListConfig( root, [ 1 ], True, vlanEntity, cmds, options.saveAll )

   for v in sorted( entity.vlanConfig, key=int ):
      if v == 1:
         continue
      
      traceDetail( "Processing vlan", v )
      vlanEntity = entity.vlanConfig[ v ]
      cmds = getVlanConfig( vlanEntity, options.saveAll )

      traceDetail( "vlan cfg", cmds )
      saveVlanListConfig( root, [ v ], True, vlanEntity, cmds, options.saveAll )

#-------------------------------------------------------------------------------
# Saves the state of ieee reserved mac forward <address|group> entries
#-------------------------------------------------------------------------------
def saveIeeeReservedMacForward( entity, root, requireMounts, options ):
   cmds = root[ 'Ebra.ieeeReservedForwarding' ]

   def saveAddrList():
      allowedMacList = EbraLib.ieeeReservedAddressConfigList()
      if options.saveAll:
         fwdMacList = allowedMacList
      else:
         fwdMacList = list( entity.ieeeReservedForwarding )
      fwdMacList.sort()
      for mac in fwdMacList:
         assert fwdMacList is allowedMacList or mac in allowedMacList, \
                'Invalid IEEE reserved forwarding entry "%s"' % mac
         if mac in entity.ieeeReservedForwarding:
            fwd = entity.ieeeReservedForwarding[ mac ]
         else:
            fwd = False
         assert options.saveAll or fwd is True, \
                'Ebra.ieeeReservedForwarding set entry values must be True'
         no = 'no ' if not fwd else ''
         token = EbraLib.ieeeReservedAddressConfigToToken( mac )
         cmds.addCommand( no + 'mac address-table reserved forward ' + token )

   def saveGroupList():
      # cfgStatus: build a map of possible configurations and their configured
      # values
      cfgStatus = {}
      for cfg in EbraLib.ieeeReservedGroupConfigList():
         cfgStatus[ cfg ] = False

      # The CLI plugin that writes entries to entity.ieeeReservedForwarding uses same
      # EbraLib static mapping, so we don't actually need to check the values that
      # have been written to the hwCaps.ieeeReservedMacForwardGroup collection.
      for groupMacAddr in EbraLib.ieeeReservedGroupConfigList():
         # The platform only needs to see the first address in a given
         # group of addresses to forward for address group configurations
         # e.g., FocalPointV2
         #   01:80:c2:00:00:04: Forward 0180.c200.0004-0d,0f
         if groupMacAddr in entity.ieeeReservedForwarding:
            cfgStatus[ groupMacAddr ] = True

      for cfg in sorted( cfgStatus ):
         if cfgStatus[ cfg ] or options.saveAll:
            no = 'no ' if not cfgStatus[ cfg ] else ''
            token = EbraLib.ieeeReservedGroupConfigToToken( cfg )
            cmds.addCommand( no + 'mac address-table reserved forward ' + token )

   hwCaps = requireMounts[ 'bridging/hwcapabilities' ]
   if hwCaps.ieeeReservedMacForwardAddrSupported:
      saveAddrList()
   elif hwCaps.ieeeReservedMacForwardGroup:
      saveGroupList()


#-------------------------------------------------------------------------------
# Saves the state of global Bridging Config.
#-------------------------------------------------------------------------------
# pylint: disable-msg=C0322 # pylint: disable=bad-option-value
# XXX Workaround for a pylint bug with decorators.
@CliSave.saver( 'Bridging::Input::CliConfig', 'bridging/input/config/cli',
                requireMounts=( 'cli/config',
                                'bridging/config',
                                'bridging/hwcapabilities',
                                'hwEpoch/status' ) )
def saveBridgingConfig( entity, root, requireMounts, options ):
   saveBridgingNonIntfConfig( entity, root, requireMounts, options )
   saveBridgingIntfConfig( entity, root, requireMounts, options )

def saveBridgingNonIntfConfig( entity, root, requireMounts, options ):
   # Save config outside interface modes
   if options.intfFilter:
      return

   hwCaps = requireMounts[ 'bridging/hwcapabilities' ]

   saveVlanConfig( entity, root, requireMounts, options )
   # Save the MAC address table configuration.
   if entity.hostAgingTime != 300 or options.saveAll:
      cmds = root[ 'Ebra.agingTime' ]
      cmds.addCommand( 'mac address-table aging-time %d' %
                       ( entity.hostAgingTime ) )

   # Save the default switchport mode.
   if entity.defaultSwitchportMode != 'access' or options.saveAll:
      cmds = root[ 'Ebra.defaultSwitchportMode' ]
      cmds.addCommand( 'switchport default mode %s' %
                       ( entity.defaultSwitchportMode ) )

   # Save the default switchport phone vlan and traffic class
   cmds = root[ 'Ebra.defaultSwitchportPhoneVlan' ]
   if entity.defaultPhoneVlan:
      cmds.addCommand( 'switchport default phone vlan %d' %
                     ( entity.defaultPhoneVlan ) )
   elif options.saveAll:
      cmds.addCommand( 'no switchport default phone vlan' )

   cmds = root[ 'Ebra.defaultSwitchportPhoneCos' ]
   if entity.defaultPhoneCos != 5:
      cmds.addCommand( 'switchport default phone cos %d' %
                     ( entity.defaultPhoneCos ) )
   elif options.saveAll:
      cmds.addCommand( 'switchport default phone cos 5' )

   cmds = root[ 'Ebra.defaultSwitchportPhoneTrnkUntag' ]
   if entity.defaultPhoneTrunkTagMode == 'phoneTrunkUntagged':
      cmds.addCommand( 'switchport default phone trunk untagged' )
   elif entity.defaultPhoneTrunkTagMode == 'phoneVlanUntagged':
      # Save the switchport default phone untagged phone
      cmds.addCommand( 'switchport default phone trunk untagged phone' )
   elif entity.defaultPhoneTrunkTagMode == 'phoneTrunkTagged':
      cmds.addCommand( 'switchport default phone trunk tagged' )
   elif entity.defaultPhoneTrunkTagMode == 'phoneVlanTagged':
      cmds.addCommand( 'switchport default phone trunk tagged phone' )
   elif options.saveAll:
      cmds.addCommand( 'no switchport default phone trunk' )

   if Toggles.EbraToggleLib.togglePerInterfaceMbvaCliEnabled():
      if hwCaps.macBasedVlanAssignmentScopeSupported:
         cmds = root[ 'Ebra.macBasedVlanAssignmentScope' ]
         if entity.macBasedVlanAssignmentScope == 'mbvaScopeInterface':
            cmds.addCommand( 'mac based vlan assignment scope interface' )
         elif options.saveAll:
            cmds.addCommand( 'mac based vlan assignment scope global' )

   cmds = root[ 'Ebra.defaultSwitchportPhoneAclBypass' ]
   if entity.defaultPhoneAclBypass:
      cmds.addCommand( 'switchport default phone access-list bypass' )
   elif options.saveAll:
      cmds.addCommand( 'no switchport default phone access-list bypass' )

   TrustMode = Tac.Type( 'Qos::TrustMode' )
   cmds = root[ 'Ebra.defaultSwitchportPhoneTrustMode' ]
   if entity.defaultPhoneTrustMode != TrustMode.invalid:
      if entity.defaultPhoneTrustMode != TrustMode.untrusted:
         cmds.addCommand( 'switchport default phone qos trust %s' % 
                          entity.defaultPhoneTrustMode )
      elif options.saveAll:
         cmds.addCommand( 'no switchport default phone qos trust' )

   cmds = root[ 'Ebra.switchportVlanTagValidation' ]
   if entity.vlanTagValidationEnabled:
      cmds.addCommand( 'switchport vlan tag validation' )
   elif options.saveAll:
      cmds.addCommand( 'no switchport vlan tag validation' )

   cmds = root[ 'Ebra.switchportLlcValidation' ]
   if entity.llcValidationEnabled:
      cmds.addCommand( 'switchport ethernet llc validation' )
   elif options.saveAll:
      cmds.addCommand( 'no switchport ethernet llc validation' )

   # Save the pause frame pass through configuration.
   if entity.pausePassThrough or options.saveAll:
      cmds = root[ 'Ebra.pausePassThrough' ]
      if entity.pausePassThrough:
         cmds.addCommand( 'mac pause-frame pass-through' )
      else:
         cmds.addCommand( 'no mac pause-frame pass-through' )

   if entity.macIpv4McMismatchDropEnabled or options.saveAll:
      cmds = root[ 'Ebra.macIpv4McMismatchDropEnabled' ]
      if entity.macIpv4McMismatchDropEnabled:
         cmds.addCommand( 'mac multicast destination-check ipv4 unicast action '
                          'drop' )
      elif hwCaps.macIpv4McMismatchDropSupported:
         cmds.addCommand( 'mac multicast destination-check ipv4 unicast action '
                          'forward' )

   if not entity.sourceCheckCPUMac or options.saveAll:
      cmds = root[ 'Ebra.sourceCheckCPUMac' ]
      if not entity.sourceCheckCPUMac:
         cmds.addCommand( 'mac source-check system-mac default disabled' )
      elif hwCaps.invalidAndBridgeSmacPktSwitchSupported:
         cmds.addCommand( 'mac source-check system-mac default' )

   if not entity.sourceMacInvalid or options.saveAll:
      cmds = root[ 'Ebra.sourceMacInvalid' ]
      if not entity.sourceMacInvalid:
         cmds.addCommand( 'mac source-check invalid default disabled' )
      elif hwCaps.invalidAndBridgeSmacPktSwitchSupported:
         cmds.addCommand( 'mac source-check invalid default' )

   cmds = root[ 'Ebra.vlanMacFlushAggregationDisabled' ]
   if entity.vlanMacFlushAggregationDisabled:
      cmds.addCommand( 'mac address-table flushing interface vlan '
                       'aggregation disabled' )
   elif options.saveAll:
      cmds.addCommand( 'no mac address-table flushing interface vlan '
                       'aggregation disabled' )

   if entity.dynVlanRange != '1-4094' or options.saveAll:
      cmds = root[ 'Ebra.vlanDynamicRange' ]
      cmds.addCommand( 'vlan dynamic range %s' %
                       (  entity.dynVlanRange ) )

   # Save the forwarding mode.
   # Since we can't depend on the forwarding agent to be up and have
   # filled in the list of supported modes, and since we must consistantly
   # report the forwarding mode at all points in the boot process, we cannot
   # use the supported forwarding modes to tailor the output. We must instead
   # blindly output what is in the cli entity. 
   SwitchForwardingMode = Tac.Type( "Bridging::SwitchForwardingMode" )
   
   fmCmds = root[ 'Ebra.forwardingMode' ]
   mode = entity.forwardingMode
   if mode != SwitchForwardingMode.defaultMode or options.saveAll:
      if mode == SwitchForwardingMode.storeAndForward:
         fmCmds.addCommand( 'switch forwarding-mode store-and-forward' )
      elif mode == SwitchForwardingMode.cutThrough:
         fmCmds.addCommand( 'switch forwarding-mode cut-through' )
      elif mode == SwitchForwardingMode.defaultMode:
         fmCmds.addCommand( 'default switch forwarding-mode' )
      else:
         assert False, "Forwarding mode is not a valid value."

   # Save the multicast forwarding mode override if not set to defaultMode
   fmCmds = root[ 'Ebra.l3McastForwardingMode' ]
   mode = entity.l3McastForwardingMode
   if mode == SwitchForwardingMode.storeAndForward:
      fmCmds.addCommand( 'switch forwarding-mode store-and-forward multicast')
   elif mode != SwitchForwardingMode.defaultMode:
      assert False, "Multicast forwarding mode is not a valid value."

   # Save the vlan internal allocation policy.
   cmds = root[ 'Ebra.internalVlanAllocationPolicy' ]
   if ( entity.internalVlanAllocationPolicy != 'ascending' or
        entity.internalVlanRange != Tac.Value( "Bridging::InternalVlanRange" ) ):
      cmd = 'vlan internal order '
      if entity.internalVlanAllocationPolicy == 'ascending':
         cmd += 'ascending'
      else:
         cmd += 'descending'
      if entity.internalVlanRange != Tac.Value( "Bridging::InternalVlanRange" ):
         cmd += ' range %d %d' % ( entity.internalVlanRange.vlanBegin, 
                  entity.internalVlanRange.vlanEnd )
      cmds.addCommand( cmd )
   elif options.saveAll:
      # default case should always be saved in new format
      cmds.addCommand( "vlan internal order ascending" )

   saveIeeeReservedMacForward( entity, root, requireMounts, options )

   # Save the VLAN TPIDs.
   # The TPIDs are sorted for consistent output.
   vlanTpidsCmds = root[ 'Ebra.vlanTpids' ]
   for vlanTpid in sorted( entity.vlanTpid ):
      vlanTpidsCmds.addCommand( f"vlan tpid 0x{vlanTpid:04x}" )

def saveBridgingIntfConfig( entity, root, requireMounts, options ):
   # Save config under interface modes
   bridgingConfig = requireMounts[ 'bridging/config' ]
   if options.saveAll or entity.defaultSwitchportMode == 'routed':
      # Get all eligible switchports, since '[no] switchport' is always displayed.
      cfgIntfNames = EthIntfUtil.allSwitchportNames( requireMounts,
                                                     includeEligible=True,
                                                     includeSubIntf=True )
   else:
      cfgIntfNames = entity.switchIntfConfig

   for intfName in cfgIntfNames:
      if options.intfFilter and intfName not in options.intfFilter:
         continue

      intfConfig = entity.switchIntfConfig.get( intfName )
      if not intfConfig:
         if options.saveAll or entity.defaultSwitchportMode == 'routed':
            intfConfig = Tac.newInstance( 'Bridging::Input::SwitchIntfConfig', 
                                          intfName, 'access' )
            if entity.defaultSwitchportMode == 'routed':
               intfConfig.enabled = False
         else:
            continue
      saveSwitchIntfConfig( intfConfig, bridgingConfig, root, options,
                             entity, entity.defaultSwitchportMode )

#-------------------------------------------------------------------------------
# Saves the state of an Bridging::SwitchIntfConfig object.
#-------------------------------------------------------------------------------
def saveSwitchIntfConfig( entity, bridgingConfig, root, options,
                          bridgingInputCli, defaultSwitchportMode ):
   mode = root[ IntfConfigMode ].getOrCreateModeInstance( entity.intfId )
   cmds = mode[ 'Ebra.switchport' ]
   saveAll = options.saveAll
   saveAllDetail = options.saveAllDetail
   addSwitchIntfConfigCmds( cmds, entity, bridgingConfig, saveAll, saveAllDetail,
                             bridgingInputCli, defaultSwitchportMode )

def addSwitchIntfConfigCmds( cmds, entity, bridgingConfig, saveAll, saveAllDetail,
                             bridgingInputCli, defaultSwitchportMode=None ):

   # Display defaults on all ports for saveAllDetail and on L2 ports for saveAll. 
   saveDefault = saveAllDetail or ( saveAll and entity.enabled )

   if Tac.Type( 'Arnet::SubIntfId' ).isSubIntfId( entity.intfId ):
      if entity.l2SubIntfVlan.nameOrId == 'name':
         cmds.addCommand( 'vlan name %s' % entity.l2SubIntfVlan.vlanName )
      elif entity.l2SubIntfVlan.nameOrId == 'id':
         cmds.addCommand( 'vlan id %d' % entity.l2SubIntfVlan.vlanId )
      elif saveDefault:
         # Note that this is really emitted only during saveAllDetail as bridging is
         # enabled on an L2 sub-interface only if vlan is configured.
         cmds.addCommand( 'no vlan id' )

      # after vlan is set
      if entity.splitHorizonGroup != 'default':
         cmds.addCommand( 'split-horizon group %s' % entity.splitHorizonGroup )

      if not entity.macLearningEnabled:
         cmds.addCommand( 'mac address learning disabled' )
      elif saveDefault:
         cmds.addCommand( 'default mac address learning disabled' )

      # Sub interface does not support rest of the switchport configs.
      return

   if entity.accessVlan != 1 or saveDefault:
      cmds.addCommand( 'switchport access vlan %s' % entity.accessVlan )
   
   if entity.trunkNativeVlan != 1 or saveDefault:
      if entity.trunkNativeVlan == 0:
         cmds.addCommand( 'switchport trunk native vlan tag' )
      else:
         cmds.addCommand( 'switchport trunk native vlan %s'
                          % entity.trunkNativeVlan )

   if entity.phoneVlan:
      cmds.addCommand( 'switchport phone vlan %d' % ( entity.phoneVlan )  )
   elif saveDefault:
      cmds.addCommand( 'no switchport phone vlan'  )

   if entity.phoneTrunkTagMode == 'phoneTrunkUntagged':
      cmds.addCommand( 'switchport phone trunk untagged' )
   elif entity.phoneTrunkTagMode == 'phoneTrunkTagged':
      cmds.addCommand( 'switchport phone trunk tagged' )
   elif entity.phoneTrunkTagMode == 'phoneVlanUntagged':
      cmds.addCommand( 'switchport phone trunk untagged phone' )
   elif entity.phoneTrunkTagMode == 'phoneVlanTagged':
      cmds.addCommand( 'switchport phone trunk tagged phone' )
   elif saveDefault:
      cmds.addCommand( 'no switchport phone trunk' )
   
   VlanMappingDirection = Tac.Type( 'Bridging::VlanMappingDirection' )
   ingressReqEnabled = entity.vlanMappingRequired.get(
         VlanMappingDirection.vlanMappingIngress, False )
   egressReqEnabled = entity.vlanMappingRequired.get(
         VlanMappingDirection.vlanMappingEgress, False )

   if ingressReqEnabled:
      cmds.addCommand( 'switchport vlan translation in required' )

   if egressReqEnabled:
      cmds.addCommand( 'switchport vlan translation out required' )

   if saveDefault:
      # ingress direction
      if not ingressReqEnabled:
         cmds.addCommand( 'no switchport vlan translation in required' )

      # egress direction
      if not egressReqEnabled:
         cmds.addCommand( 'no switchport vlan translation out required' )

   VlanTagFormat = Tac.Type( 'Bridging::VlanTagFormat' )
   if entity.vlanTagFormatDrop.get( VlanTagFormat.untagged, False ) and \
         entity.vlanTagFormatDrop.get( VlanTagFormat.priority, False ):
      cmds.addCommand( 'switchport dot1q vlan tag required' )
   elif entity.vlanTagFormatDrop.get( VlanTagFormat.tagged, False ):
      cmds.addCommand( 'switchport dot1q vlan tag disallowed' )
   elif saveDefault:
      cmds.addCommand( 'no switchport dot1q vlan tag' )


   if entity.trunkAllowedVlans == '':
      cmds.addCommand( 'switchport trunk allowed vlan none' )
   elif entity.trunkAllowedVlans != '1-4094' or saveDefault:
      cmds.addCommand( 'switchport trunk allowed vlan %s' % 
                       entity.trunkAllowedVlans )
   if entity.switchportMode not in ( 'access', 'tap', 'tool', 'tap-tool' ) or \
         saveDefault:
      modeMapping = dict( access='access', 
                          trunk='trunk',
                          dot1qTunnel='dot1q-tunnel',
                          tap='tap',
                          tool='tool',
                          tapTool='tap-tool'
                          )
      # pylint: disable-next=superfluous-parens
      assert( entity.switchportMode in modeMapping )
      if entity.phoneTrunk:
         cmds.addCommand( 'switchport mode trunk phone' )
      else:
         cmds.addCommand( 'switchport mode %s' % modeMapping[entity.switchportMode] )

   # Include default value in case of saveAll for route port too
   if entity.vlanTpid != defaultTpid or saveAll:
      cmds.addCommand(
            'switchport dot1q ethertype 0x%x' % ( entity.vlanTpid ) )
  
   if not entity.macLearningEnabled:
      cmds.addCommand( 'no switchport mac address learning' )
   elif saveDefault:
      cmds.addCommand( 'switchport mac address learning' )

   def isValidStaleAge( age ):
      # pylint: disable-next=chained-comparison
      return age == 1 or ( age >= 10 and age <= 1000000 )

   if entity.agingTime and not isValidStaleAge( entity.agingTime ):
      cmds.addCommand( 'bridge mac-address-table aging timeout disabled' )
   elif entity.agingTime:
      cmds.addCommand( 'bridge mac-address-table aging timeout %s seconds' %
            entity.agingTime )
   elif saveDefault:
      cmds.addCommand( 'default bridge mac-address-table aging timeout' )

   if entity.vlanForwardingMode == "vlanForwardingModeAllConfiguredVlans":
      cmds.addCommand( 'switchport vlan forwarding accept all' )
   elif saveDefault:
      cmds.addCommand( 'no switchport vlan forwarding accept all' )

   for i in sorted( entity.trunkGroup):
      cmds.addCommand( 'switchport trunk group %s' % i )

   # Consider defaultSwitchportMode from bridgingConfig if it has not been
   # passed to the method. 
   defSwitchportMode = bridgingConfig.defaultSwitchportMode if \
         defaultSwitchportMode is None else defaultSwitchportMode
   # '[no] switchport' is always displayed for saveAll.
   if not entity.enabled:
      cmds.addCommand( 'no switchport' )
   elif saveAll or defSwitchportMode == 'routed':
      cmds.addCommand( 'switchport' )

   if entity.ingressNativeVlanXlate != 0:
      cmds.addCommand( 'encapsulation dot1q vlan %s' % ( 
                                                 entity.ingressNativeVlanXlate ) )
   elif saveAll:
      cmds.addCommand( 'no encapsulation dot1q vlan' )

   if entity.sourceportFilterMode == 'sourceportFilterDisabled':
      cmds.addCommand( 'switchport source-interface tx' )
   elif entity.sourceportFilterMode == 'sourceportFilterMulticastDisabled':
      cmds.addCommand( 'switchport source-interface tx multicast' )
   elif saveAll:
      cmds.addCommand( 'default switchport source-interface tx' )

   if entity.l3SourceportDrop:
      cmds.addCommand( 'forwarding l3 source-interface drop' )
   # FIXME: do not print for 'saveAll' for now since not all switchports
   # ( notably LAG ) support it.

   def handleIngressVlanMap( maps, egressSkip, xlateKey, xlateMapping ):
      assert not xlateMapping.innerVid
      if xlateMapping.tunnel:
         assert not xlateKey.innerVid

      direction = 'in'
      egressXlateKey = egressVlanXlateKeyIfBoth( entity.egressVlanXlate,
                                                 xlateKey, xlateMapping )
      if egressXlateKey is not None:
         direction = ''
         egressSkip.add( ( egressXlateKey.outerVid, egressXlateKey.innerVid ) )

      maps = maps.setdefault( direction, {} )
      maps = maps.setdefault( xlateMapping.tunnel, {} )
      maps = maps.setdefault( xlateMapping.vlanId, {} )
      maps = maps.setdefault( xlateMapping.deQinQXlate, {} )
      maps = maps.setdefault( xlateKey.innerVid, set() )
      assert xlateKey.outerVid not in maps
      maps.add( xlateKey.outerVid )

   def handleEgressVlanMap( maps, egressSkip, xlateKey, xlateMapping ):
      if xlateMapping.tunnel and not xlateMapping.allInnerVid:
         assert xlateKey.innerVid == xlateMapping.vlanId
      else:
         assert not xlateKey.innerVid

      direction = 'out'
      if ( xlateKey.outerVid, xlateKey.innerVid ) not in egressSkip:
         maps = maps.setdefault( direction, {} )
         maps = maps.setdefault( xlateMapping.tunnel, {} )
         maps = maps.setdefault( xlateMapping.innerVid, {} )
         maps = maps.setdefault( xlateMapping.allInnerVid, {} )
         if xlateMapping.tunnel:
            if xlateMapping.allInnerVid:
               maps[ xlateKey.outerVid ] = True
            else:
               maps = maps.setdefault( xlateKey.outerVid, set() )
               maps.add( xlateMapping.vlanId )
         else:
            maps = maps.setdefault( xlateMapping.vlanId, set() )
            assert xlateKey.outerVid not in maps
            maps.add( xlateKey.outerVid )

   def addInOrBothCmd( direction, tunnel, maps ):
      for newVid, maps1 in sorted( maps.items() ):
         for deQinQXlate, maps2 in sorted( maps1.items() ):
            for origInnerVid, origVid in sorted( maps2.items() ):
               directionStr = ' in' if direction else direction
               newVidStr = newVid
               if direction == '' and not tunnel:
                  assert len( origVid ) == 1
                  origVidStr = '%d' % ( list( origVid )[ 0 ] )
               else:
                  origVidStr = Vlan.vlanSetToCanonicalString( list( origVid ) )
               if tunnel:
                  assert not origInnerVid
                  dot1qTunnelOrInnerStr = ' dot1q-tunnel'
               elif origInnerVid:
                  if deQinQXlate:
                     dot1qTunnelOrInnerStr = ' inner {} network'.format(
                           origInnerVid )
                  else:
                     dot1qTunnelOrInnerStr = ' inner %d' % origInnerVid
               else:
                  dot1qTunnelOrInnerStr = ''
               cmds.addCommand( 'switchport vlan translation%s %s%s %s' % (
                  directionStr, origVidStr, dot1qTunnelOrInnerStr, newVidStr ) )

   def addOutCmd( direction, tunnel, maps ):
      assert direction == 'out'
      for newInnerVid, maps1 in sorted( maps.items() ):
         for allInnerVlan, maps2 in sorted( maps1.items() ):
            for key, val in sorted( maps2.items() ):
               if tunnel and not allInnerVlan:
                  assert not newInnerVid
                  origVidStr = '%d' % key
                  newVidStr = Vlan.vlanSetToCanonicalString( list( val ) )
               elif tunnel and allInnerVlan:
                  origVidStr = '%d' % key
                  newVidStr = 'all'
               else:
                  origVidStr = Vlan.vlanSetToCanonicalString( list( val ) )
                  newVidStr = '%d' % key
               dot1qTunnelStr = ' dot1q-tunnel' if tunnel else ''
               newInnerVidStr = ' inner %d' % newInnerVid if newInnerVid else ''
               cmds.addCommand( 'switchport vlan translation out %s%s %s%s' % (
                  origVidStr, dot1qTunnelStr, newVidStr, newInnerVidStr ) )

   maps = {}
   egressSkip = set()

   for xlateKey, xlateMapping in entity.ingressVlanXlate.items():
      handleIngressVlanMap( maps, egressSkip, xlateKey, xlateMapping )

   for xlateKey, xlateMapping in entity.egressVlanXlate.items():
      handleEgressVlanMap( maps, egressSkip, xlateKey, xlateMapping )

   for direction, maps1 in sorted( maps.items() ):
      for tunnel, maps2 in sorted( maps1.items() ):
         func = addOutCmd if direction == 'out' else addInOrBothCmd
         func( direction, tunnel, maps2 )

#-------------------------------------------------------------------------------
# Saves the state of an Bridging::FdbConfig.
#-------------------------------------------------------------------------------
@CliSave.saver( 'Bridging::Input::FdbConfig', 'bridging/input/config/cli' )
def saveFdbConfig( entity, root, requireMounts, options ):
   cmds = root[ 'Ebra.staticMac' ]

   # Unicast addresses
   hostTable = entity.configuredHost
   addresses = hostTable.values()
   labelExtensions = labelEntryHook.extensions()
   assert len( labelExtensions ) <= 1, \
      'There should be only one label extension hook for unicast address'
   for addr in sorted( addresses, key=lambda x: x.address ):
      addrStr = Ethernet.convertMacAddrCanonicalToDisplay( addr.address )
      forwardingEligibleStr = ''
      if addr.forwardingEligible:
         forwardingEligibleStr = 'eligibility forwarding'
      if addr.intf == '':
         if addr.dropMode == 'dropModeDst':
            intfsOrDropStr = 'drop address destination'
         else:
            intfsOrDropStr = 'drop'
      else:
         intfsOrDropStr = 'interface %s' % addr.intf
      if not Tac.Value( 'Arnet::MplsLabel', addr.label ).isValid():
         cmds.addCommand(
            f'mac address-table static {addrStr} '
            f'vlan {entity.vlanId} {intfsOrDropStr} '
            f'{forwardingEligibleStr}'
         )
      else:
         cmd = labelExtensions[ 0 ]( addrStr, entity.vlanId, addr.label, addr.intf )
         cmds.addCommand( cmd )

   # Multicast addresses
   groupTable = entity.ethGroup
   addresses = groupTable.values()
   for addr in sorted( addresses, key=lambda x: x.addr ):
      addrStr = Ethernet.convertMacAddrCanonicalToDisplay( addr.addr )
      printCmd = 'mac address-table static %s vlan %s interface' % \
                 ( addrStr, entity.vlanId )
      if len( addr.intf ) == 1:
         printIntfs = addr.intf
      else:
         printIntfs = Intf.IntfRange.intfListToCanonical( addr.intf )
      for intf in printIntfs:
         cmds.addCommand( printCmd + ' ' + intf )

#-------------------------------------------------------------------------------
# Saves the state of an Interface::SubIntfConfig object.
#-------------------------------------------------------------------------------
# pylint: disable-msg=C0322 # pylint: disable=bad-option-value
# XXX Workaround for a pylint bug with decorators.
@CliSave.saver( 'Interface::SubIntfConfig', 'interface/config/subintf',
                attrName = 'intfConfig',
                requireMounts=( 'interface/status/all', 'routerMac/hwCapability' ) )
def saveSubIntfConfig( entity, root, requireMounts, options ):
   # Save the baseclass (Arnet::IntfConfig) attributes.
   IntfCliSave.saveIntfConfig( entity, root, requireMounts, options )
   saveAll = options.saveAll
   mode = root[ IntfConfigMode ].getOrCreateModeInstance( entity.intfId )
   cmds = mode[ EthIntfCliSave.EthIntfCmdSeq ]
   outerTag = int( entity.dot1qEncap.outerTag )
   innerTag = int( entity.dot1qEncap.innerTag )
   Dot1qEncapMode = Tac.Type( "Interface::SubIntfDot1qEncapConfigMode" )
   legacyMode = ( entity.dot1qEncapConfigMode ==
                  Dot1qEncapMode.subIntfDot1qEncapConfigLegacy )
   if outerTag and legacyMode:
      cmd = 'encapsulation dot1q vlan %d' % outerTag
      if innerTag:
         cmd += ' inner %d' % innerTag
      cmds.addCommand( cmd )
   elif saveAll:
      cmds.addCommand( 'no encapsulation dot1q vlan' )

   hwCaps = requireMounts[ 'routerMac/hwCapability' ]
   if hwCaps.intfRouterMacSupported:
      if entity.addr != '00:00:00:00:00:00':
         rmac = Ethernet.convertMacAddrCanonicalToDisplay( entity.addr )
         cmds.addCommand( 'mac-address router %s' % rmac )
      elif saveAll:
         cmds.addCommand( 'no mac-address router' )
