#!/usr/bin/env python3
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import CliSave
from CliMode.Eoam import EoamProfileModeBase, LinkMonitoringModeBase
from CliMode.Eoam import EoamConfigModeBase
from CliSavePlugin import IntfCliSave
from EoamTypes import tacEoamAction
from EoamTypes import errorTypeToEnumMap, enumToActionMap, enumToIntervalMap
import EthIntfUtil
import Tac, Tracing
t0 = Tracing.trace0

CliSave.GlobalConfigMode.addCommandSequence( 'Eoam.config' )
IntfCliSave.IntfConfigMode.addCommandSequence( 'Eoam.config' )

class EoamConfigMode( EoamConfigModeBase, CliSave.Mode ):
   def __init__( self, param ):
      EoamConfigModeBase.__init__( self )
      CliSave.Mode.__init__( self, self.longModeKey )

   def skipIfEmpty( self ):
      return True

class EoamProfileConfigMode( EoamProfileModeBase, CliSave.Mode ):
   def __init__( self, param ):
      EoamProfileModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, self.longModeKey )

class LinkMonitoringConfigMode( LinkMonitoringModeBase, CliSave.Mode ):
   def __init__( self, param ):
      LinkMonitoringModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, self.longModeKey )

   def skipIfEmpty( self ):
      return True

CliSave.GlobalConfigMode.addChildMode( EoamConfigMode )
EoamConfigMode.addChildMode( EoamProfileConfigMode )
EoamProfileConfigMode.addCommandSequence( 'Eoam.profile' )
EoamProfileConfigMode.addChildMode( LinkMonitoringConfigMode,
                                    after=[ 'Eoam.profile' ] )
LinkMonitoringConfigMode.addCommandSequence( 'EoamProfile.linkMonitoring' )

@CliSave.saver( 'Eoam::Config', 'eoam/config',
                requireMounts=( 'interface/config/eth/phy/slice',
                              ) )
def saveConfig( entity, root, requireMounts, options ):
   cmds = root[ 'Eoam.config' ]

   saveAll = options.saveAll
   if saveAll:
      phyIntfConfigDir = requireMounts[ 'interface/config/eth/phy/slice' ]
      phyIntfNames = EthIntfUtil.allPhyEthernetInterfaceNames( phyIntfConfigDir )
      cfgIntfNames = set( entity.intfToProfileName )
      cfgIntfNames.update( phyIntfNames )
   else:
      cfgIntfNames = entity.intfToProfileName

   intfToProfileName = entity.intfToProfileName

   def getAppliedEoamProfileName( intfName ):
      # Only one eoamProfile will be active at a time
      # Get that from intfToProfileName
      intToProfileMap = entity.intfToProfileName
      if intfName in intToProfileMap:
         return intToProfileMap[ intfName ]
      return None

   for intfName in cfgIntfNames:
      if intfName in intfToProfileName:
         eoamProfileName = getAppliedEoamProfileName( intfName )
         intfMode = root[ IntfCliSave.IntfConfigMode ].getOrCreateModeInstance(
            intfName )
         cmds = intfMode[ 'Eoam.config' ]
         if eoamProfileName:
            cmds.addCommand( 'monitor ethernet oam profile %s' % (
               eoamProfileName ) )

   eoamConfigMode = root[ EoamConfigMode ].getSingletonInstance()
   for profileName, eoamProfile in sorted( entity.eoamProfile.items() ):
      eoamProfileMode = eoamConfigMode[ EoamProfileConfigMode ].\
         getOrCreateModeInstance( profileName )
      mode = eoamProfileMode[ LinkMonitoringConfigMode ].\
         getOrCreateModeInstance( profileName )
      cmds = mode[ 'EoamProfile.linkMonitoring' ]
      if eoamProfile.recoveryTimeoutSecs:
         cmds.addCommand( f"recovery-time {eoamProfile.recoveryTimeoutSecs}" )
      elif saveAll:
         cmds.addCommand( "default recovery-time" )
      for errorType in [ 'fcs', 'symbol' ]:
         linkMonitorConfig = eoamProfile.linkMonitorConfig.get(
               errorTypeToEnumMap[ errorType ] )
         if linkMonitorConfig and linkMonitorConfig.threshold:
            cmds.addCommand( f"{errorType} threshold {linkMonitorConfig.threshold}" )
         elif saveAll:
            cmds.addCommand( f"{errorType} threshold 100" )
         if ( linkMonitorConfig and linkMonitorConfig.action !=
              tacEoamAction.actionLog ):
            cmds.addCommand( f"{errorType} action "
                             f"{enumToActionMap[ linkMonitorConfig.action ]}" )
         elif saveAll:
            cmds.addCommand( f"{errorType} action log" )
         if linkMonitorConfig and linkMonitorConfig.period.val:
            cmds.addCommand( f"{errorType} period {linkMonitorConfig.period.val} "
                             f"{enumToIntervalMap[ linkMonitorConfig.period.unit ]}"
                            )
         elif saveAll:
            cmds.addCommand( f"{errorType} period 10 sec " )

