# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import CliSave
import Tac
from BgpLib import (
   vpnAfTypeMapInv,
)
from CliMode.Evpn import ( EthernetSegmentMode,
      EvpnMulticastMode,
      EvpnMulticastAfMode )
from CliSavePlugin.IntfCliSave import IntfConfigMode
from CliSavePlugin.RoutingBgpCliSave import (
   RouterBgpBaseConfigMode,
   RouterBgpBaseAfConfigMode,
   RouterBgpVrfConfigMode,
   RouterBgpVrfAfConfigMode,
   saveAfConfigCallbackDict )
from CliToken.RoutingBgpConstants import evpnNexthopSelfSupportNlris
from RouteMapLib import isAsdotConfigured
from Toggles import (
   BgpCommonToggleLib,
   EvpnToggleLib,
   McastVpnLibToggleLib,
   ArBgpToggleLib as arBgpTl,
)

ethIntfId = Tac.Type( 'Arnet::EthIntfId' )
portChannelIntfId = Tac.Type( 'Arnet::PortChannelIntfId' )
dfElectionHoldTimeDefault = \
      Tac.Value( 'Evpn::DFElectionHoldTime' ).defaultHoldTime
dfElectionDelayTimeDefault = 0
multihomeModeToRedundancyMap = \
      { 'multihomeModeAllActive' : 'all-active',
        'multihomeModeSingleActive' : 'single-active' }
defaultRt = Tac.Value( 'Arnet::EthAddr' )

class EthernetSegmentConfigSaveMode( EthernetSegmentMode, CliSave.Mode ):
   def __init__( self, param ):
      EthernetSegmentMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

# 'evpn ethernet-segment' can be configured under L2 interface or
# 'address-family evpn' mode
IntfConfigMode.addChildMode( EthernetSegmentConfigSaveMode )
EthernetSegmentConfigSaveMode.addCommandSequence( 'Evpn.intf' )
RouterBgpBaseAfConfigMode.addChildMode( EthernetSegmentConfigSaveMode )
EthernetSegmentConfigSaveMode.addCommandSequence( 'Bgp.afConfig.evpn' )

class EvpnMulticastConfigMode( EvpnMulticastMode, CliSave.Mode ):
   def __init__( self, param ):
      EvpnMulticastMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

class EvpnMulticastAfConfigMode( EvpnMulticastAfMode, CliSave.Mode ):
   def __init__( self, afAndVrfName ):
      af = afAndVrfName[ 0 ]
      vrfName = afAndVrfName[ 1 ]
      EvpnMulticastAfMode.__init__( self, af, vrfName )
      CliSave.Mode.__init__( self, af )

if McastVpnLibToggleLib.toggleMcastVpnTransitEnabled():
   RouterBgpVrfConfigMode.addChildMode( EvpnMulticastConfigMode,
         after=[ RouterBgpVrfAfConfigMode ] )
   EvpnMulticastConfigMode.addCommandSequence( 'Evpn.mcast' )

   EvpnMulticastConfigMode.addChildMode( EvpnMulticastAfConfigMode )
   EvpnMulticastAfConfigMode.addCommandSequence( 'Evpn.mcast.afConfig.ipv4' )
   EvpnMulticastAfConfigMode.addCommandSequence( 'Evpn.mcast.afConfig.ipv6' )

def allEthernetSegmentIntfNames( ethernetSegmentConfig, includeEligible=False,
                                 requireMounts=None ):
   # if includeEligible, then all Ethernet and Port-Channel interfaces are
   # include; otherwise, only include interface with this feature configured
   # For example:
   # int Ethernet1
   #    no switchport
   #    evpn ethernet-segment
   #       esi ...
   # int Ethernet2
   #    ...
   # 
   # Result:
   # "show running-config all" -- display configured esi for Ethernet1
   # "show running-config all detail" -- display configured esi and default for
   # Ethernet1 and Ethernet2
   assert requireMounts
   intfConfigDir = requireMounts[ 'interface/config/all' ]
   intfNames = []
   for intfName in intfConfigDir.intfConfig:
      if portChannelIntfId.isPortChannelIntfId( intfName ) or \
            ethIntfId.isEthIntfId( intfName ):
         if includeEligible:
            intfNames.append( intfName )
         elif intfName in ethernetSegmentConfig.intfConfig:
            intfNames.append( intfName )
   # WAN interconnect ethernet segment attached to interface 'Vxlan1'
   if 'Vxlan1' in ethernetSegmentConfig.intfConfig:
      intfNames.append( 'Vxlan1' )
   return intfNames

def saveEthernetSegmentConfigHelper( cmds, config, saveAll, isDomainConfig=False ):
   if not config.esid.none:
      cmds.addCommand( 'identifier %s' % config.esid.value )
   elif ( EvpnToggleLib.toggleEvpnType1EsiEnabled() and
          config.esiType.type == 'type1' ):
      cmds.addCommand( 'identifier auto lacp' )
   elif saveAll:
      cmds.addCommand( 'no identifier' )

   if config.esName and ( config.intfId != config.esName or saveAll ):
      cmds.addCommand( 'name %s' % config.esName )

   if config.multihomeMode != config.multihomeModeDefault:
      cmds.addCommand( 'redundancy %s' %
                       multihomeModeToRedundancyMap[ config.multihomeMode ] )
   elif saveAll:
      cmds.addCommand( 'redundancy %s' %
                 multihomeModeToRedundancyMap[ config.multihomeModeDefault ] )

   if config.dfElectionAlgorithm == 'dfaHrw':
      cmds.addCommand( 'designated-forwarder election algorithm hrw' )
   elif config.dfElectionAlgorithm == 'dfaPreference':
      dp = ' dont-preempt' if config.dfElectionDontPreempt else ''
      cmds.addCommand( 'designated-forwarder election algorithm preference '
                       '%d%s' % ( config.dfElectionPreference, dp ) )
   elif saveAll:
      cmds.addCommand( 'designated-forwarder election algorithm modulus' )

   if config.holdTime != dfElectionHoldTimeDefault and \
      config.dfElectionDelayTime != dfElectionDelayTimeDefault:
      cmds.addCommand( 'designated-forwarder election hold-time %s'
                       ' subsequent-hold-time %d' %
                       ( config.holdTime,
                         config.dfElectionDelayTime * 1000 ) )
   elif config.holdTime != dfElectionHoldTimeDefault:
      cmds.addCommand( 'designated-forwarder election hold-time %s' %
                       config.holdTime )
   elif config.dfElectionDelayTime != dfElectionDelayTimeDefault:
      cmds.addCommand( 'designated-forwarder election hold-time %s'
                       ' subsequent-hold-time %d' %
                       ( dfElectionHoldTimeDefault,
                         config.dfElectionDelayTime * 1000 ) )
   elif saveAll:
      cmds.addCommand( 'no designated-forwarder election hold-time' )

   if config.dfElectionCandidateReachableOnly:
      cmds.addCommand(
         'designated-forwarder election candidate reachability required' )
   elif saveAll:
      cmds.addCommand(
         'no designated-forwarder election candidate reachability required' )

   if config.encapBumFilterHoldTime != 0:
      cmds.addCommand( 'mpls tunnel flood filter time %d' %
                       ( config.encapBumFilterHoldTime * 1000 ) )
   elif saveAll:
      cmds.addCommand( 'no mpls tunnel flood filter time' )

   if config.sharedEsiLabelIndex != 0:
      cmds.addCommand( 'mpls shared index %d' % config.sharedEsiLabelIndex )
   elif saveAll:
      cmds.addCommand( 'no mpls shared index' )

   if config.esImportRouteTarget != defaultRt.stringValue:
      cmds.addCommand( 'route-target import %s' % config.esImportRouteTarget )
   elif saveAll:
      cmds.addCommand( 'no route-target import' )

   # When dual-encap toggle is enabled, isDomainConfig should be set to true for
   # GW ESI.  In single-encap GW, intf should be Vxlan1 for GW ESI.
   if isDomainConfig or config.intfId == 'Vxlan1':

      if EvpnToggleLib.toggleEvpnDciNoAliasingEnabled():
         # Take care of aliasing knob
         if config.aliasingEnabled.isSet and \
            config.aliasingEnabled.value:
            cmds.addCommand( "aliasing" )
         elif saveAll:
            cmds.addCommand( "no aliasing" )

      if EvpnToggleLib.toggleEvpnInterconnectEsDisabledEnabled():
         # disabled command should be applicable to domain ES config only
         if config.disabled:
            cmds.addCommand( 'disabled' )
         elif saveAll:
            cmds.addCommand( 'no disabled' )

@CliSave.saver( 'Evpn::InterconnectEsCliConfigDir', 'evpn/interface/domainConfig',
      requireMounts=( 'routing/bgp/config',
                     'routing/bgp/asn/config' ) )
def saveDomainEthernetSegmentConfig( ethernetSegmentDomainConfig,
                                     root, requireMounts, options ):
   cmds = {}
   saveAll = options.saveAll
   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   if not bgpConfig.asNumber:
      return
   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]

   ethernetSegmentDomainCliConfigs = [
         ( 'all', ethernetSegmentDomainConfig.allDomainIEsConfig ),
         ( 'remote', ethernetSegmentDomainConfig.remoteDomainIEsConfig ),
         ( 'local', ethernetSegmentDomainConfig.localDomainIEsConfig ),
         ]
   for ( domain, domainConfig ) in ethernetSegmentDomainCliConfigs:
      if domainConfig is None:
         continue
      bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
            bgpConfig.asNumber, isAsdotConfigured( asnConfig ), ) )
      bgpAfMode = bgpMode[ RouterBgpBaseAfConfigMode
            ].getOrCreateModeInstance( 'evpn' )
      ethernetSegmentMode = bgpAfMode[ EthernetSegmentConfigSaveMode
            ].getOrCreateModeInstance( 'domain %s' % domain )
      cmds = ethernetSegmentMode[ 'Bgp.afConfig.evpn' ]
      saveEthernetSegmentConfigHelper( cmds, domainConfig, saveAll,
                                       isDomainConfig=True )

@CliSave.saver( 'Evpn::EthernetSegmentCliConfigDir', 'evpn/interface/config',
      requireMounts=( 'interface/config/all', 'routing/bgp/config',
                      'routing/bgp/asn/config' ) )
def saveEthernetSegmentConfig( ethernetSegmentConfig, root, requireMounts, options ):
   configuredIntfNames = []
   saveAll = options.saveAll
   if options.saveAllDetail:
      configuredIntfNames = \
         allEthernetSegmentIntfNames( ethernetSegmentConfig, includeEligible=True,
                                      requireMounts=requireMounts )
   elif options.saveAll:
      configuredIntfNames = \
         allEthernetSegmentIntfNames( ethernetSegmentConfig,
                                      requireMounts=requireMounts )
   else:
      configuredIntfNames = list( ethernetSegmentConfig.intfConfig )
   for intf in configuredIntfNames:
      if intf in ( 'Vxlan1', 'MplsTrunk1' ):
         return
      intfMode = root[ IntfConfigMode ].getOrCreateModeInstance( intf )
      ethernetSegmentMode = intfMode[ EthernetSegmentConfigSaveMode
            ].getSingletonInstance()
      cmds = ethernetSegmentMode[ 'Evpn.intf' ]
      intfConfig = ethernetSegmentConfig.intfConfig.get( intf, None )
      if not intfConfig:
         if saveAll:
            intfConfig = Tac.newInstance( 'Evpn::EthernetSegmentCliConfig', intf )
         else:
            continue
      saveEthernetSegmentConfigHelper( cmds, intfConfig, saveAll )

@CliSave.saver( 'Routing::Bgp::MacPlugin::EvpnConfig', 'routing/bgp/evpnconfig',
                requireMounts = ( 'routing/bgp/config',
                                  'routing/bgp/asn/config', ) )
def saveEvpnHostFlapConfig( hostFlapConfig, root, requireMounts, options ):
   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]

   if bgpConfig.asNumber == 0:
      return

   window = hostFlapConfig.duplicateTimeout
   threshold = hostFlapConfig.duplicateThreshold
   expiry = hostFlapConfig.duplicateExpiry
   isDefault = ( window == hostFlapConfig.duplicateTimeoutDefault and
                 threshold == hostFlapConfig.duplicateThresholdDefault and
                 expiry == hostFlapConfig.duplicateExpiryDefault )
   if isDefault and not options.saveAll:
      return

   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
         bgpConfig.asNumber, isAsdotConfigured( asnConfig ), ) )
   bgpAfMode = bgpMode[ RouterBgpBaseAfConfigMode
         ].getOrCreateModeInstance( 'evpn' )

   if window == 0 and threshold == 0:
      cmd = 'no host-flap detection'
   else:
      if expiry:
         if window != hostFlapConfig.duplicateTimeoutDefault or \
            threshold != hostFlapConfig.duplicateThresholdDefault:
            cmd = 'host-flap detection window %s threshold %s expiry timeout %s '\
                  'seconds' % ( window, threshold, expiry )
         else:
            cmd = 'host-flap detection expiry timeout %s seconds' % expiry
      else:
         cmd = 'host-flap detection window %s threshold %s' % \
               ( window, threshold )

   bgpAfMode[ 'Bgp.afConfig.evpn' ].addCommand( cmd )

@CliSave.saver( 'Routing::Bgp::MacPlugin::EvpnConfig', 'routing/bgp/evpnconfig',
                requireMounts=( 'routing/bgp/config',
                                  'routing/bgp/asn/config', ) )
def saveEvpnSmetAdvertisedByConfig( evpnConfig, root, requireMounts, options ):
   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]

   if bgpConfig.asNumber == 0:
      return

   smetAdvertisedByAll = evpnConfig.smetAdvertisedByAll
   isDefault = smetAdvertisedByAll == evpnConfig.smetAdvertisedByAllDefault
   if isDefault and not options.saveAll:
      return

   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
         bgpConfig.asNumber, isAsdotConfigured( asnConfig ), ) )
   bgpAfMode = bgpMode[ RouterBgpBaseAfConfigMode
         ].getOrCreateModeInstance( 'evpn' )

   if isDefault and options.saveAll:
      cmd = 'route type smet advertised-by designated-forwarder'
   else:
      cmd = 'route type smet advertised-by all'
   bgpAfMode[ 'Bgp.afConfig.evpn' ].addCommand( cmd )

@CliSave.saver( 'Routing::Bgp::MacPlugin::EvpnConfig', 'routing/bgp/evpnconfig',
                requireMounts=( 'routing/bgp/config', 'routing/bgp/asn/config' ) )
def saveEvpnSmetProxyInterDomainConfig( evpnConfig, root, requireMounts, options ):
   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   if not McastVpnLibToggleLib.toggleOismGatewayEnabled():
      return

   if bgpConfig.asNumber == 0:
      return
   if evpnConfig.smetProxyInterDomain:
      cmd = 'route type smet proxy inter-domain'
   elif options.saveAll:
      cmd = 'no route type smet proxy inter-domain'
   else:
      cmd = ''

   if not cmd:
      return
   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
         bgpConfig.asNumber, isAsdotConfigured( asnConfig ), ) )
   bgpAfMode = bgpMode[ RouterBgpBaseAfConfigMode ].getOrCreateModeInstance( 'evpn' )

   bgpAfMode[ 'Bgp.afConfig.evpn' ].addCommand( cmd )


@CliSave.saver( 'Routing::Bgp::MacPlugin::EvpnConfig', 'routing/bgp/evpnconfig',
                requireMounts=( 'routing/bgp/config',
                                  'routing/bgp/asn/config', ) )
def saveEvpnRemoteRefreshConfig( evpnConfig, root, requireMounts, options ):
   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]

   if bgpConfig.asNumber == 0:
      return

   remoteRefreshDelay = evpnConfig.remoteRefreshDelay
   inPlaceUpdateEnabled = evpnConfig.inPlaceUpdateEnabled
   isDefault = remoteRefreshDelay == evpnConfig.remoteRefreshDelayDefault and \
               inPlaceUpdateEnabled == evpnConfig.inPlaceUpdateEnabledDefault
   if isDefault and not options.saveAll:
      return

   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
         bgpConfig.asNumber, isAsdotConfigured( asnConfig ), ) )
   bgpAfMode = bgpMode[ RouterBgpBaseAfConfigMode
         ].getOrCreateModeInstance( 'evpn' )

   if isDefault:
      cmd = 'no layer-2 fec in-place update'
   elif remoteRefreshDelay == evpnConfig.remoteRefreshDelayDefault and \
        not options.saveAll:
      cmd = 'layer-2 fec in-place update'
   else:
      cmd = 'layer-2 fec in-place update timeout %d seconds' % remoteRefreshDelay
   bgpAfMode[ 'Bgp.afConfig.evpn' ].addCommand( cmd )

@CliSave.saver( 'Routing::Bgp::MacPlugin::EvpnConfig', 'routing/bgp/evpnconfig',
                requireMounts=( 'routing/bgp/config',
                                  'routing/bgp/asn/config', ) )
def saveEvpnRouteImportOverlayIndexConfig( evpnConfig, root, requireMounts,
                                           options ):
   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]

   if bgpConfig.asNumber == 0:
      return

   isDefault = \
         evpnConfig.overlayIndexEnabled == evpnConfig.overlayIndexEnabledDefault
   if isDefault and not options.saveAll:
      return

   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
         bgpConfig.asNumber, isAsdotConfigured( asnConfig ), ) )
   bgpAfMode = bgpMode[ RouterBgpBaseAfConfigMode
         ].getOrCreateModeInstance( 'evpn' )

   cmd = 'route import overlay-index gateway'
   if isDefault:
      cmd = 'no ' + cmd
   bgpAfMode[ 'Bgp.afConfig.evpn' ].addCommand( cmd )

@CliSave.saver( 'Routing::Bgp::MacPlugin::EvpnConfig', 'routing/bgp/evpnconfig',
                requireMounts=( 'routing/bgp/config',
                                  'routing/bgp/asn/config', ) )
def saveEvpnUmrConfig( evpnConfig, root, requireMounts, options ):
   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]

   if ( bgpConfig.asNumber == 0 or not arBgpTl.toggleEvpnUmrEnabled() ):
      return

   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
         bgpConfig.asNumber, isAsdotConfigured( asnConfig ), ) )
   bgpAfMode = bgpMode[ RouterBgpBaseAfConfigMode
         ].getOrCreateModeInstance( 'evpn' )

   if evpnConfig.installUmr:
      bgpAfMode[ 'Bgp.afConfig.evpn' ].addCommand( 'unknown-mac-route install' )
   elif options.saveAll:
      bgpAfMode[ 'Bgp.afConfig.evpn' ].addCommand( 'no unknown-mac-route install' )

   if not evpnConfig.reAdvertiseMacIpIntoLocalDomain:
      bgpAfMode[ 'Bgp.afConfig.evpn' ].addCommand(
         'route-type mac-ip re-advertise domain local disabled' )
   elif options.saveAll:
      bgpAfMode[ 'Bgp.afConfig.evpn' ].addCommand(
         'route-type mac-ip re-advertise domain local' )

@CliSave.saver( 'Evpn::EthernetSegmentCliConfigDir', 'evpn/interface/config',
                     requireMounts=( 'routing/bgp/config',
                                       'routing/bgp/asn/config', ) )
def saveEvpnEsIpMassWithdrawConfig( ethernetSegmentConfigDir, root,
                                    requireMounts, options ):
   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]

   if bgpConfig.asNumber == 0:
      return

   saveAllOrDetail = options.saveAll or options.saveAllDetail
   if ( not saveAllOrDetail and
        not ethernetSegmentConfigDir.adPerEsExportL3RT and
        not ethernetSegmentConfigDir.adPerEsImportL3RT ):
      return

   exportCmd = 'route export ethernet-segment ip mass-withdraw'
   importCmd = 'route import ethernet-segment ip mass-withdraw'

   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
         bgpConfig.asNumber, isAsdotConfigured( asnConfig ), ) )
   bgpAfMode = bgpMode[ RouterBgpBaseAfConfigMode
         ].getOrCreateModeInstance( 'evpn' )

   if ethernetSegmentConfigDir.adPerEsExportL3RT:
      bgpAfMode[ 'Bgp.afConfig.evpn' ].addCommand( exportCmd )
   elif saveAllOrDetail:
      exportCmd = 'no ' + exportCmd
      bgpAfMode[ 'Bgp.afConfig.evpn' ].addCommand( exportCmd )

   if ethernetSegmentConfigDir.adPerEsImportL3RT:
      bgpAfMode[ 'Bgp.afConfig.evpn' ].addCommand( importCmd )
   elif saveAllOrDetail:
      importCmd = 'no ' + importCmd
      bgpAfMode[ 'Bgp.afConfig.evpn' ].addCommand( importCmd )


def saveEvpnNeighborDefaultNexthopSelf( cmds, config, saveAll ):
   # pylint: disable-next=consider-using-dict-items
   for rType in evpnNexthopSelfSupportNlris:
      rTypeName = evpnNexthopSelfSupportNlris[ rType ][ 'keyword' ]
      attrName = 'nexthopSelfEvpnType%d' % rType
      val = getattr( config, attrName )
      valDefault = getattr( config, attrName + 'Default' )
      if val != valDefault or saveAll:
         if val == 'isTrue':
            cmds.append( 'neighbor default next-hop-self '
                         'received-evpn-routes route-type %s' % rTypeName )
         elif val == 'isInvalid':
            cmds.append( 'no neighbor default next-hop-self '
                         'received-evpn-routes route-type %s' % rTypeName )

      interDomainAttrName = 'nexthopSelfEvpnType%dInterDomain' % rType
      interDomainVal = getattr( config, interDomainAttrName )
      interDomainValDefault = getattr( config, interDomainAttrName + 'Default' )
      if interDomainVal != interDomainValDefault or saveAll:
         if interDomainVal == 'isTrue':
            cmds.append( 'neighbor default next-hop-self '
                         'received-evpn-routes route-type %s '
                         'inter-domain' % rTypeName )
         elif interDomainVal == 'isInvalid':
            cmds.append( 'no neighbor default next-hop-self '
                         'received-evpn-routes route-type %s '
                         'inter-domain' % rTypeName )

def saveEvpnConfig( bgpConfig, af, saveAll=False, saveAllDetail=False ):
   cmds = []
   saveAllOrDetail = saveAll or saveAllDetail
   if bgpConfig.domainIdEvpn != bgpConfig.domainIdEvpnDefault:
      cmds.append( 'domain identifier %s' % bgpConfig.domainIdEvpn.toStrepCli() )
   elif saveAllOrDetail:
      cmds.append( 'no domain identifier' )

   if EvpnToggleLib.toggleEvpnMacIpDomainPathEnabled():
      if bgpConfig.remoteDomainIdEvpn != bgpConfig.remoteDomainIdEvpnDefault:
         cmds.append( 'domain identifier %s remote' %
                      bgpConfig.remoteDomainIdEvpn.toStrepCli() )
      elif saveAllOrDetail:
         cmds.append( 'no domain identifier remote' )

   if EvpnToggleLib.toggleEvpnAttachDomainPathEnabled():
      if bgpConfig.attachDomainIdEvpn != bgpConfig.attachDomainIdEvpnDefault:
         cmds.append( 'domain identifier %s attach' %
                      bgpConfig.attachDomainIdEvpn.toStrepCli() )
      elif saveAllOrDetail:
         cmds.append( 'no domain identifier attach' )

   if bgpConfig.afEvpnVxlanL3UseNexthopIgpCost != \
      bgpConfig.afEvpnVxlanL3UseNexthopIgpCostDefault:
      cmds.append( 'encapsulation vxlan layer-3 set next-hop igp-cost' )
   elif saveAllOrDetail:
      cmds.append( 'no encapsulation vxlan layer-3 set next-hop igp-cost' )

   if bgpConfig.afEvpnNexthopResolutionDisabled:
      cmds.append( 'next-hop resolution disabled' )
   elif saveAllOrDetail:
      cmds.append( 'no next-hop resolution disabled' )

   if bgpConfig.vpnPruningEnabledEvpn != bgpConfig.vpnPruningEnabledEvpnDefault:
      cmds.append( 'route import match-failure action discard' )
   elif saveAllOrDetail:
      cmds.append( 'no route import match-failure action discard' )

   if BgpCommonToggleLib.toggleEvpnIBgpAsPeCeProtocolEnabled():
      afiSafi = vpnAfTypeMapInv[ 'evpn' ] if 'evpn' in vpnAfTypeMapInv else None
      if afiSafi in bgpConfig.vpnClientIbgpAttributeTunneling and \
         bgpConfig.vpnClientIbgpAttributeTunneling[ afiSafi ]:
         cmds.append( 'vpn client ibgp attribute tunneling' )
      elif saveAllOrDetail:
         cmds.append( 'no vpn client ibgp attribute tunneling' )

   saveEvpnNeighborDefaultNexthopSelf( cmds, bgpConfig, saveAll )

   return cmds

# saveAfConfigCallbackDict would be used to display Evpn specific config in
# "show bgp config active" output.
# For  "show running" the config is displayed through saveEvpnAfConfig with
# CliSave.saver decorator.
saveAfConfigCallbackDict[ 'evpn' ] = saveEvpnConfig

# Save address-family evpn configuration commands. Majority of the
# evpn config commands are defined and cli saved in BgpCommon package.
@CliSave.saver( 'Routing::Bgp::Config', 'routing/bgp/config',
                requireMounts=( 'routing/bgp/asn/config', ) )
def saveEvpnAfConfig( bgpConfig, root, requireMounts, options ):
   if bgpConfig.asNumber == 0:
      return

   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
      bgpConfig.asNumber, isAsdotConfigured( asnConfig ), ) )
   cmds = saveEvpnConfig( bgpConfig, 'evpn', saveAll=options.saveAll,
                          saveAllDetail=options.saveAllDetail )
   if cmds:
      bgpAfMode = bgpMode[ RouterBgpBaseAfConfigMode
      ].getOrCreateModeInstance( 'evpn' )
      for cmd in cmds:
         bgpAfMode[ 'Bgp.afConfig.evpn' ].addCommand( cmd )

def saveEvpnOismGatewayDrConfig( vrfBgpConfig, evpnMcastMode, options ):

   drElectionCmd = "gateway dr election algorithm "

   if vrfBgpConfig.gatewayDrElectionConfigured:

      if vrfBgpConfig.gatewayDrElectionAlgorithm == "dfaDefault":
         drElectionCmd += "modulus"
      elif vrfBgpConfig.gatewayDrElectionAlgorithm == "dfaPreference":
         drElectionCmd += ( "preference %d" %
                            vrfBgpConfig.gatewayDrElectionPreference )
      evpnMcastMode[ 'Evpn.mcast' ].addCommand( drElectionCmd )
   elif options.saveAll or options.saveAllDetail:
      drElectionCmd += "hrw"
      evpnMcastMode[ 'Evpn.mcast' ].addCommand( drElectionCmd )

def saveEvpnIpv4ModeConfig( vrfName, transitConfig, vrfBgpConfig, evpnMcastMode,
      options ):

   transitSave = vrfName in transitConfig.transitVrf
   disabledSave = vrfBgpConfig.afEvpnOismExplicitlyDisabled
   saveAll = options.saveAll or options.saveAllDetail

   evpnMcastIpMode = evpnMcastMode[ EvpnMulticastAfConfigMode ].\
                  getOrCreateModeInstance( ( 'ipv4', vrfName ) )
   cmds = []
   if disabledSave:
      cmds.append( 'disabled' )
   elif saveAll:
      cmds.append( 'no disabled' )

   if transitSave:
      cmds.append( 'transit' )
   elif saveAll:
      cmds.append( 'no transit' )

   for cmd in cmds:
      evpnMcastIpMode[ 'Evpn.mcast.afConfig.ipv4' ].addCommand( cmd )

def saveEvpnIpv6ModeConfig( vrfName, vrfBgpConfig, evpnMcastMode, options ):

   enabledSave = vrfBgpConfig.afIpv6EvpnOismEnabled
   disabledSave = vrfBgpConfig.afIpv6EvpnOismExplicitlyDisabled
   saveAll = options.saveAll or options.saveAllDetail

   if not saveAll and not enabledSave and not disabledSave:
      return

   evpnMcastIpv6Mode = evpnMcastMode[ EvpnMulticastAfConfigMode ].\
                       getOrCreateModeInstance( ( 'ipv6', vrfName ) )

   cmds = []
   if enabledSave and saveAll:
      cmds.append( 'no disabled' )
   elif disabledSave or saveAll:
      cmds.append( 'disabled' )

   for cmd in cmds:
      evpnMcastIpv6Mode[ 'Evpn.mcast.afConfig.ipv6' ].addCommand( cmd )

def saveEvpnVrfModeConfig( vrfName, vrfBgpConfig, requireMounts, bgpVrfMode,
                           hasComment, options ):

   transitConfig = requireMounts[ 'routing/multicast/evpn/config' ]
   hwCapability = requireMounts[ 'bridging/hwcapabilities' ]

   v4ModeEnabled = vrfBgpConfig.afEvpnOismEnabled
   v4ModeExplicitDisabled = vrfBgpConfig.afEvpnOismExplicitlyDisabled
   v6ModeEnabled = vrfBgpConfig.afIpv6EvpnOismEnabled
   v6ModeExplicitDisabled = vrfBgpConfig.afIpv6EvpnOismExplicitlyDisabled
   gatewayDrElectionConfigured = vrfBgpConfig.gatewayDrElectionConfigured
   transitEnabled = transitConfig.transitVrf

   # Create evpn multicast submode if
   # 1. V4 Oism is enabled / explicit disabled OR
   # 2. V6 Oism is enabled / explicit disabled OR
   # 3. gateway DR election algorithm is configured OR
   # 4. transit is enabled
   # 5. we have a comment for the mode
   saveEvpnMcastMode = ( v4ModeEnabled or v4ModeExplicitDisabled or
                         v6ModeEnabled or v6ModeExplicitDisabled or
                         gatewayDrElectionConfigured or transitEnabled or
                         hasComment )
   if saveEvpnMcastMode:
      evpnMcastMode = bgpVrfMode[ EvpnMulticastConfigMode ].\
         getOrCreateModeInstance( vrfName )
      saveEvpnOismGatewayDrConfig( vrfBgpConfig, evpnMcastMode, options )
      saveEvpnIpv4ModeConfig( vrfName, transitConfig, vrfBgpConfig, evpnMcastMode,
                              options )
      if hwCapability.vxlanMcastIpv6OverlaySupported:
         saveEvpnIpv6ModeConfig( vrfName, vrfBgpConfig, evpnMcastMode, options )
   elif options.saveAll or options.saveAllDetail:
      bgpVrfMode[ 'Bgp.vrf.config' ].addCommand( 'no evpn multicast' )

@CliSave.saver( 'Routing::Bgp::VrfConfigDir', 'routing/bgp/vrf/config',
                requireMounts=( 'routing/bgp/config',
                                'routing/bgp/asn/config',
                                'routing/multicast/evpn/config',
                                'bridging/hwcapabilities',
                                'cli/config' ) )
def saveEvpnVrfConfig( vrfConfigDir, root, requireMounts, options ):
   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   if bgpConfig.asNumber == 0:
      return
   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
      bgpConfig.asNumber, isAsdotConfigured( asnConfig ), ) )

   for vrfName in vrfConfigDir.vrfConfig:
      vrfBgpConfig = vrfConfigDir.vrfConfig[ vrfName ]
      bgpVrfMode = bgpMode[ RouterBgpVrfConfigMode ].\
         getOrCreateModeInstance( vrfName )
      # Save it in new format as
      # evpn multicast
      #    addr-family ipv4
      #       transit
      #       ... 
      #    addr-family ipv6
      hasComment = CliSave.hasComments( f'evpn-mcast-vrf-{vrfName}', requireMounts )
      if ( McastVpnLibToggleLib.toggleMcastVpnTransitEnabled() and \
           options.saveAll or options.saveAllDetail or
           not vrfBgpConfig.evpnOismLegacy or
           hasComment ):
         saveEvpnVrfModeConfig( vrfName, vrfBgpConfig, requireMounts,
                                bgpVrfMode, hasComment, options )
      else:
         # Legacy format
         if vrfBgpConfig.afEvpnOismEnabled:
            bgpVrfMode[ 'Bgp.vrf.config' ].addCommand( 'evpn multicast' )
         elif options.saveAll or options.saveAllDetail:
            bgpVrfMode[ 'Bgp.vrf.config' ].addCommand( 'no evpn multicast' )

@CliSave.saver( 'Evpn::EthernetSegmentCliConfigDir', 'evpn/interface/config',
                     requireMounts=( 'routing/bgp/config',
                                       'routing/bgp/asn/config', ) )
def saveEsImportAutoRouteTargetConfig( ethernetSegmentConfigDir, root, requireMounts,
                                       options ):
   bgpConfig = requireMounts[ 'routing/bgp/config' ]
   asnConfig = requireMounts[ 'routing/bgp/asn/config' ]
   if not EvpnToggleLib.toggleEvpnType1EsiEnabled() or not bgpConfig.asNumber:
      return

   saveAllOrDetail = options.saveAll or options.saveAllDetail
   esImportAutoRT = ethernetSegmentConfigDir.esImportAutoRT
   isDefault = ( esImportAutoRT == ethernetSegmentConfigDir.esImportAutoRTDefault )
   if ( not saveAllOrDetail and isDefault ):
      return

   bgpMode = root[ RouterBgpBaseConfigMode ].getOrCreateModeInstance( (
      bgpConfig.asNumber, isAsdotConfigured( asnConfig ), ) )
   bgpAfMode = bgpMode[ RouterBgpBaseAfConfigMode
      ].getOrCreateModeInstance( 'evpn' )

   cmd = 'route type ethernet-segment route-target auto'
   if EvpnToggleLib.toggleEvpnType1EsiEnabled():
      if ethernetSegmentConfigDir.esImportAutoRT:
         bgpAfMode[ 'Bgp.afConfig.evpn' ].addCommand( cmd )
      elif saveAllOrDetail:
         cmd = 'no ' + cmd
         bgpAfMode[ 'Bgp.afConfig.evpn' ].addCommand( cmd )
