#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
import FabricIntfLib
import CliSavePlugin.IntfCliSave as IntfCliSave
from CliSavePlugin.IntfCliSave import IntfConfigMode
from CliSavePlugin.EthIntfCliSave import (
   saveLoopbackMode,
   saveSpeed,
)

IntfConfigCmdSeq = 'FabricIntf.fabricIntf'
IntfConfigMode.addCommandSequence( IntfConfigCmdSeq,
                                   after=[ 'EthIntf.ethIntf' ] )

@CliSave.saver(
   'Interface::FabricIntfConfigDir',
   FabricIntfLib.Constants.intfConfigPath(),
   requireMounts = ( # Paths required by the base IntfCli save infrastructure
                     'interface/status/all',
                     'interface/config/global',
                     'interface/profile/config',
                     # Paths required to determine fabric interfaces can be saved.
                     FabricIntfLib.Constants.fabricCapabilitiesCliPath() ) )
def saveFabricIntfConfig( fabricIntfConfigDir, root, requireMounts, options ):
   # Capabilities are usually managed by the platform's central agent. This means
   # that on startup there may be a delay where saving the running configuration does
   # not show any fabric interfaces.
   fabricCliCapabilities = requireMounts[
      FabricIntfLib.Constants.fabricCapabilitiesCliPath() ]
   if not fabricCliCapabilities.configIfModeSupported:
      return

   for entity in fabricIntfConfigDir.intfConfig.values():
      if options.intfFilter and entity.intfId not in options.intfFilter:
         continue

      saveAll = options.saveAll

      IntfCliSave.saveIntfConfig( entity, root, requireMounts, options )

      mode = root[ IntfConfigMode ].getOrCreateModeInstance( entity.intfId )
      cmds = mode[ IntfConfigCmdSeq ]

      saveLoopbackMode( entity, cmds, saveAll )
      saveSpeed( entity, cmds, saveAll )

      # Unlike "enabled" config, fabric intf default is always unisolated.
      # We only need to save the config if the fabric is isolated.
      if entity.adminIsolated:
         cmds.addCommand( 'isolated' )
