# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

#-------------------------------------------------------------------------------
# This module implements the CliSave code for FHRP commands
#-------------------------------------------------------------------------------

import CliSave
from CliSavePlugin.IntfCliSave import IntfConfigMode
from FhrpUtils import intfSupportsVarp, getVrrpEncryptionKey
from IpLibConsts import DEFAULT_VRF
import IpUtils
import ReversibleSecretCli
import Tac
import functools

#-------------------------------------------------------------------------------
# save vrrp config under separate interfaces
#-------------------------------------------------------------------------------

authTypeEnum = Tac.Type( 'Routing::Fhrp::Lib::VrrpAuthType' )
IntfConfigMode.addCommandSequence( 'Vrrp.intf', after=[ 'Ira.ipIntf' ] )

def saveVrrpConfig( entity, root, options, requireMounts ):
   if options.intfFilter and entity.parent.intfId not in options.intfFilter:
      return
   mode = root[ IntfConfigMode ].getOrCreateModeInstance( entity.parent.intfId )
   cmds = mode[ 'Vrrp.intf' ]
   saveAll = options.saveAll
   securityConfig = requireMounts[ 'mgmt/security/config' ]

   if entity.priority != entity.priorityDefault or saveAll:
      cmds.addCommand( 'vrrp %s priority-level %s' % ( entity.vrId,
                                                       entity.priority ) )

   if entity.advInterval != entity.advIntervalDefault or saveAll:
      # it is stored in centiseconds
      advIntervalSeconds = entity.advInterval // 100
      cmds.addCommand( 'vrrp %s advertisement interval %s' % ( entity.vrId,
                                                               advIntervalSeconds ) )

   if entity.advMismatchDrop != entity.advMismatchDropDefault or \
      entity.advMismatchLog != entity.advMismatchLogDefault or saveAll:
      if entity.advMismatchDrop and entity.advMismatchLog:
         cmds.addCommand( 'vrrp %s advertisement mismatch action drop log' %
                          ( entity.vrId, ) )
      elif entity.advMismatchDrop:
         cmds.addCommand( 'vrrp %s advertisement mismatch action drop' %
                          ( entity.vrId, ) )
      elif entity.advMismatchLog:
         cmds.addCommand( 'vrrp %s advertisement mismatch action log' %
                          ( entity.vrId, ) )
      else:
         cmds.addCommand( 'no vrrp %s advertisement mismatch action' %
                          ( entity.vrId, ) )

   if entity.arpOrNAInterval != entity.arpOrNAIntervalDefault or saveAll:
      arpOrNAInterval = entity.arpOrNAInterval // 100
      cmds.addCommand( 'vrrp %s mac-address advertisement-interval %s' %
                       ( entity.vrId, arpOrNAInterval ) )

   if entity.preempt != entity.preemptDefault or saveAll:
      if entity.preempt:
         cmds.addCommand( 'vrrp %s preempt' % ( entity.vrId, ) )
      else:
         cmds.addCommand( 'no vrrp %s preempt' % ( entity.vrId, ) )

   delayCmd = ''
   if entity.preemptDelay != entity.preemptDelayDefault or saveAll:
      delaySeconds = entity.preemptDelay // 100
      delayCmd = 'minimum %s' % delaySeconds

   reloadCmd = ''
   if entity.preemptReloadDelay != entity.preemptReloadDelayDefault or saveAll:
      reloadSeconds = entity.preemptReloadDelay // 100
      reloadCmd = 'reload %s' % reloadSeconds

   if delayCmd or reloadCmd:
      spacing = ' ' if delayCmd and reloadCmd else ''
      cmds.addCommand( 'vrrp %s preempt delay %s%s%s' % ( entity.vrId, delayCmd,
                                                          spacing, reloadCmd ) )

   if entity.reloadDelay != entity.reloadDelayDefault or saveAll:
      reloadSeconds = entity.reloadDelay
      cmds.addCommand( 'vrrp %s timers delay reload %s' % ( entity.vrId,
                                                            reloadSeconds ) )

   if ( entity.authType != entity.authTypeDefault ) or saveAll:
      if entity.authType == authTypeEnum.vrrpAuthNone:
         cmds.addCommand( 'no vrrp %s peer authentication' % ( entity.vrId, ) )
      else:
         authTypeStr = ( 'text' if entity.authType == authTypeEnum.vrrpAuthPasswd
                         else 'ietf-md5 key-string' )
         uniqueKey = getVrrpEncryptionKey( entity.parent.intfId )
         cmd = ReversibleSecretCli.getCliSaveCommand(
            f'vrrp {entity.vrId} peer authentication {authTypeStr} {{}}',
            securityConfig,
            entity.authKey,
            uniqueKey,
            'DES' )
         cmds.addCommand( cmd )

   if entity.primaryAddr != entity.primaryAddrDefault or saveAll:
      cmds.addCommand( 'vrrp %s ipv4 %s' % ( entity.vrId, entity.primaryAddr ) )
   if entity.secondaryAddrs:
      for addr in sorted( entity.secondaryAddrs,
                          key=functools.cmp_to_key( IpUtils.compareIpAddress ) ):
         cmds.addCommand( 'vrrp %s ipv4 %s secondary' % ( entity.vrId, addr ) )

   if entity.ip6Addrs:
      for addr in sorted( entity.ip6Addrs,
                          key=functools.cmp_to_key( IpUtils.compareIp6Address ) ):
         cmds.addCommand( 'vrrp %s ipv6 %s' % ( entity.vrId, addr ) )
   elif saveAll:
      cmds.addCommand( 'vrrp %s ipv6 ::' % entity.vrId )

   if entity.description != entity.descriptionDefault:
      cmds.addCommand( 'vrrp %s session description %s' % ( entity.vrId,
                                                            entity.description ) )
   elif saveAll:
      cmds.addCommand( 'no vrrp %s session description' % entity.vrId )

   if entity.adminEnabledState != entity.adminEnabledStateDefault or saveAll:
      if entity.adminEnabledState == 'shutdown':
         cmds.addCommand( 'vrrp %s disabled' % ( entity.vrId, ) )
      else:
         cmds.addCommand( 'no vrrp %s disabled' % ( entity.vrId, ) )

   for trackedObjectName in entity.vrrpTrackedObject:
      trackedObject = entity.vrrpTrackedObject[ trackedObjectName ]
      if trackedObject.trackPriorityDecrement:
         cmds.addCommand( 'vrrp %s tracked-object %s decrement %s' %
                          ( entity.vrId, trackedObject.trackedObjectName,
                            trackedObject.trackPriorityDecrement ) )
      if trackedObject.trackActionIsShutdown:
         cmds.addCommand( 'vrrp %s tracked-object %s shutdown' %
                          ( entity.vrId, trackedObject.trackedObjectName ) )

   if entity.bfdPeerAddrV4 != entity.bfdPeerAddrV4Default:
      cmds.addCommand( 'vrrp %s bfd ip %s' %
                       ( entity.vrId, entity.bfdPeerAddrV4 ) )
   elif saveAll:
      cmds.addCommand( 'no vrrp %s bfd ip' % entity.vrId )

   if entity.bfdPeerAddrV6 != entity.bfdPeerAddrV6Default:
      cmds.addCommand( 'vrrp %s bfd ipv6 %s' %
                       ( entity.vrId, entity.bfdPeerAddrV6 ) )
   elif saveAll:
      cmds.addCommand( 'no vrrp %s bfd ipv6' % entity.vrId )
   if entity.version != entity.versionDefault or saveAll:
      cmds.addCommand( 'vrrp %s ipv4 version %s' %
                       ( entity.vrId, entity.version ) )
   if entity.ipv4ExcludePseudoChecksum != \
      entity.ipv4ExcludePseudoChecksumDefault or saveAll:
      if entity.ipv4ExcludePseudoChecksum:
         cmds.addCommand( 'vrrp %s ipv4 checksum pseudo-header exclude' %
                          entity.vrId )
      else:
         cmds.addCommand( 'no vrrp %s ipv4 checksum pseudo-header exclude' %
                          entity.vrId )

@CliSave.saver( 'Routing::Fhrp::VrrpIntfConfig', 'routing/fhrp/config',
                requireMounts=( 'mgmt/security/config', 'routing/fhrp/config', ) )
def saveVrrpConfigPerIntf( entity, root, requireMounts, options ):
   for vr in sorted( entity.virtualRouterConfig.values(),
                     key=lambda v:v.vrId ):
      saveVrrpConfig( vr, root, options, requireMounts )

#-------------------------------------------------------------------------------
# save vrrp config under global config mode
#-------------------------------------------------------------------------------

CliSave.GlobalConfigMode.addCommandSequence( 'Ip.Vrrp',
                                             after=[ IntfConfigMode ] )

@CliSave.saver( 'Routing::Fhrp::VrrpConfig', 'routing/fhrp/config' )
def saveGlobalVrrpConfig( entity, root, requireMounts, options ):
   saveAll = options.saveAll
   cmds = root[ 'Ip.Fhrp' ]
   if entity.ipv6ForceIpv4MacPrefix:
      cmds.addCommand( 'vrrp ipv6 mac range ipv4' )
   elif saveAll:
      cmds.addCommand( 'no vrrp ipv6 mac range ipv4' )

#-------------------------------------------------------------------------------
# save varp config under separate interfaces
#-------------------------------------------------------------------------------

IntfConfigMode.addCommandSequence( 'Varp.intf', after=[ 'Ira.ipIntf' ] )

@CliSave.saver( 'Routing::Fhrp::VarpIntfConfig', 'routing/fhrp/varp/config' )
def saveVarpConfigPerIntf( entity, root, requireMounts, options ):
   if options.intfFilter and entity.intfId not in options.intfFilter:
      return
   mode = root[ IntfConfigMode ].getOrCreateModeInstance( entity.intfId )
   cmds = mode[ 'Varp.intf' ]

   if entity.virtualIpAddr:
      for addr in entity.virtualIpAddr:
         if addr.len == 32:
            addr = addr.address
         cmds.addCommand( 'ip virtual-router address %s' % addr )
   elif options.saveAll and intfSupportsVarp( entity.intfId ):
      cmds.addCommand( 'no ip virtual-router address' )

   if entity.virtualIp6Addr:
      for addr in entity.virtualIp6Addr:
         cmds.addCommand( 'ipv6 virtual-router address %s' % addr )
   elif options.saveAll and intfSupportsVarp( entity.intfId ):
      cmds.addCommand( 'no ipv6 virtual-router address' )

   for trackedObjectName in entity.varpTrackedObject:
      trackedObject = entity.varpTrackedObject[ trackedObjectName ]
      if trackedObject.trackActionIsShutdown:
         cmds.addCommand( 'ip virtual-router track %s shutdown' %
               trackedObject.trackedObjectName )

   if entity.useVirtualMac:
      cmds.addCommand( "mac address virtual-router" )
   elif options.saveAll and intfSupportsVarp:
      cmds.addCommand( "no mac address virtual-router" )

@CliSave.saver( 'Ip::Config', 'ip/config' )
def saveIpVirtualConfig( entity, root, requireMounts, options ):
   for intfName in entity.ipIntfConfig:
      if options.intfFilter and intfName not in options.intfFilter:
         continue
      intfConfig = entity.ipIntfConfig.get( intfName )
      if not intfSupportsVarp( intfConfig.intfId ):
         continue

      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intfConfig.intfId )
      cmds = mode[ 'Varp.intf' ]
      if intfConfig.virtualAddrWithMask.address != '0.0.0.0':
         cmds.addCommand( "ip address virtual %s/%d" %
                             ( intfConfig.virtualAddrWithMask.address,
                               intfConfig.virtualAddrWithMask.len ) )
         for addr in intfConfig.virtualSecondaryWithMask:
            cmds.addCommand( "ip address virtual %s/%d secondary" %
                                ( addr.address, addr.len ) )

@CliSave.saver( 'Ip6::Config', 'ip6/config' )
def saveIp6VirtualConfig( entity, root, requireMounts, options ):
   for intfName in entity.intf:
      if options.intfFilter and intfName not in options.intfFilter:
         continue
      intfConfig = entity.intf.get( intfName )
      if not intfSupportsVarp( intfConfig.intfId ):
         continue

      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intfConfig.intfId )
      cmds = mode[ 'Varp.intf' ]
      for addrWithMask in intfConfig.virtualAddr:
         cmds.addCommand( "ipv6 address virtual %s/%d" %
                           ( addrWithMask.address, addrWithMask.len ) )

#-------------------------------------------------------------------------------
# save varp config under global config mode
#-------------------------------------------------------------------------------

CliSave.GlobalConfigMode.addCommandSequence( 'Ip.Varp', after=[ IntfConfigMode ] )

@CliSave.saver( 'Routing::Fhrp::VarpConfig', 'routing/fhrp/varp/config' )
def saveVarpConfig( entity, root, requireMounts, options ):
   saveAll = options.saveAll
   cmds = root[ 'Ip.Varp' ]
   if entity.virtualMac != entity.virtualMacDefault:
      mask = entity.virtualMacWithMask.mask
      if mask != entity.virtualMacMaskDefault:
         cmds.addCommand( 'ip virtual-router mac-address %s mask %s'
                          % ( entity.virtualMac, mask ) )
      else:
         cmds.addCommand( 'ip virtual-router mac-address %s'
                          % entity.virtualMac )
   elif saveAll:
      cmds.addCommand( 'ip virtual-router mac-address %s'
                       % entity.virtualMacDefault )

   if entity.advtInterval != entity.advtIntervalDefault or saveAll:
      cmds.addCommand( 'ip virtual-router mac-address advertisement-interval %s'
                       % entity.advtInterval )

   if entity.subnetRoutes:
      cmds.addCommand( 'ip virtual-router address subnet-routes' )

   for vrf in sorted( entity.borrowedIpAddr ):
      ipAddr = entity.borrowedIpAddr[ vrf ]
      if vrf == DEFAULT_VRF:
         cmds.addCommand( 'ip address virtual source-nat address %s' % ipAddr )
      else:
         cmds.addCommand( 'ip address virtual source-nat vrf %s address %s' %
                              ( vrf, ipAddr ) )

   for vrf in sorted( entity.borrowedIp6Addr ):
      ip6Addr = entity.borrowedIp6Addr[ vrf ]
      if vrf == DEFAULT_VRF:
         cmds.addCommand( 'ipv6 address virtual source-nat address %s' % ip6Addr )
      else:
         cmds.addCommand( 'ipv6 address virtual source-nat vrf %s address %s' %
                              ( vrf, ip6Addr ) )

#-------------------------------------------------------------------------------
# save fhrp config under global config mode
#-------------------------------------------------------------------------------

CliSave.GlobalConfigMode.addCommandSequence( 'Ip.Fhrp',
      after=[ IntfConfigMode ] )

@CliSave.saver( 'Routing::Fhrp::Config', 'routing/fhrp/config' )
def saveFhrpConfig( entity, root, requireMounts, options ):
   saveAll = options.saveAll
   cmds = root[ 'Ip.Fhrp' ]
   if entity.acceptMode == 'enabled':
      cmds.addCommand( 'ip fhrp accept-mode' )
   elif saveAll:
      cmds.addCommand( 'no ip fhrp accept-mode' )
