# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliMode.FlexEncap import EncapsulationBaseMode
import CliSave
from CliSavePlugin import EthIntfCliSave
from CliSavePlugin.IntfCliSave import IntfConfigMode
from TypeFuture import TacLazyType

Dot1qEncapMode = TacLazyType( "Interface::SubIntfDot1qEncapConfigMode" )
FieldType = TacLazyType( "Ebra::FlexEncapField::FieldType" )
FlexEncapConstants = TacLazyType( 'Ebra::FlexEncapConstants' )
SpecType = TacLazyType( "Ebra::FlexEncapSpec::SpecType" )

class EncapsulationConfigMode( EncapsulationBaseMode, CliSave.Mode ):
   def __init__( self, intfName ):
      EncapsulationBaseMode.__init__( self, intfName )
      CliSave.Mode.__init__( self, intfName )

IntfConfigMode.addChildMode( EncapsulationConfigMode,
                             after=[ EthIntfCliSave.EthIntfCmdSeq ] )
EncapsulationConfigMode.addCommandSequence( "Encapsulation.config" )

def tagRangeStr( flexEncapRange ):
   tagStr = ''
   for tag in flexEncapRange.values():
      if tagStr == '':
         tagStr = tag.stringValue()
      else:
         tagStr += ', ' + tag.stringValue()
   return tagStr

def tpidStr( flexEncapField ):
   if flexEncapField.type == FieldType.userDefinedTpid:
      return f"tpid 0x{flexEncapField.tpidValue:04x}"
   else:
      assert flexEncapField.type in { FieldType.dot1ad, FieldType.dot1q }
      return str( flexEncapField.type )

def singleTagStr( outer ):
   return f' {tpidStr(outer)} {tagRangeStr(outer.range)}'

def doubleTagStr( outer, inner ):
   assert outer.type in { FieldType.client, FieldType.dot1q,
                          FieldType.dot1ad, FieldType.userDefinedTpid }
   assert inner.type in { FieldType.client, FieldType.dot1q,
                          FieldType.dot1ad, FieldType.userDefinedTpid }
   if inner.type == FieldType.client:
      innerStr = "client"
   elif outer.tpidValue == inner.tpidValue:
      innerStr = tagRangeStr( inner.range )
   else:
      innerStr = f"{tpidStr( inner )} {tagRangeStr( inner.range )}"
   return \
      f' {tpidStr( outer )} outer {tagRangeStr( outer.range )} inner {innerStr}'


def entryStr( clientId, entry ):
   client = entry.client
   network = entry.network
   # Client side cmd
   cmd = 'client'
   if clientId != FlexEncapConstants.defaultClientId:
      cmd += f' {clientId}'
   if client.type == SpecType.unmatched:
      cmd += ' unmatched'
   elif client.type == SpecType.untagged:
      cmd += ' untagged'
   elif client.inner.type in ( FieldType.dot1q,
                               FieldType.dot1ad,
                               FieldType.userDefinedTpid ):
      cmd += doubleTagStr( client.outer, client.inner )
   else:
      cmd += singleTagStr( client.outer )
   # Network side cmd.
   # Note: "client unmatched" implies "network client".  Prevent it from entering
   # into configuration as "client unmatched network client".
   if ( network.type != SpecType.anyOrNone and
         ( client.type != SpecType.unmatched or network.type != SpecType.client ) ):
      cmd += ' network'
      if network.type == SpecType.client:
         cmd += ' client'
      elif network.type == SpecType.clientInner:
         cmd += ' client inner'
      elif network.type == SpecType.untagged:
         cmd += ' untagged'
      elif network.inner.type in ( FieldType.dot1q, FieldType.dot1ad,
                                   FieldType.userDefinedTpid, FieldType.client ):
         cmd += doubleTagStr( network.outer, network.inner )
      else:
         cmd += singleTagStr( network.outer )
   return cmd

@CliSave.saver( "Interface::SubIntfConfig", "interface/config/subintf",
                requireMounts=( "interface/status/all", ) )
def saveIntfEncapsulationConfig( entity, root, requireMounts, options ):
   intfId = entity.intfId
   mode = root[ IntfConfigMode ].getOrCreateModeInstance( intfId )
   cmds = mode[ EthIntfCliSave.EthIntfCmdSeq ]
   # We can unconditionally create the mode instance because we have defined
   # skipIfEmpty. This means the submode will be printed even if only comments are
   # configured.
   encapMode = mode[ EncapsulationConfigMode ].getOrCreateModeInstance( intfId )
   flexEncapMode = ( entity.dot1qEncapConfigMode ==
                     Dot1qEncapMode.subIntfDot1qEncapConfigFlexEncap )
   if flexEncapMode:
      cmds = encapMode[ "Encapsulation.config" ]
      if entity.flexEncap:
         # Add the default client ID to the list of encap commands first.
         if FlexEncapConstants.defaultClientId in entity.flexEncap.entry:
            entry = entity.flexEncap.entry[ FlexEncapConstants.defaultClientId ]
            cmd = entryStr( FlexEncapConstants.defaultClientId, entry )
            cmds.addCommand( cmd )
         # Add non-default client IDs in sorted order to the list of encap commands.
         for clientId in sorted( entity.flexEncap.entry ):
            if clientId == FlexEncapConstants.defaultClientId:
               # Skip default client ID it was added above.
               continue
            entry = entity.flexEncap.entry[ clientId ]
            cmd = entryStr( clientId, entry )
            cmds.addCommand( cmd )
   elif options.saveAll:
      cmds.addCommand( 'no encapsulation vlan' )
