# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tracing
from CliMode.FlowWatcher import (
   MonitorSecurityAwakeModeBase,
   NdrModeBase,
   NucleusModeBase
)
import CliSave
from CliSavePlugin.IntfCliSave import IntfConfigMode
from CliSavePlugin.Security import SecurityConfigMode
from FlowWatcherCliUtil import (
   FlowTableSize,
   IpfixCollectorPort,
   MpId,
   NucleusPort,
)

traceHandle = Tracing.Handle( 'FlowWatcherCliSave' )
t0 = traceHandle.trace0
t1 = traceHandle.trace1

class MonitorSecurityAwakeCliSaveMode( MonitorSecurityAwakeModeBase, CliSave.Mode ):
   def __init__( self, param ):
      t1( 'MonitorSecurityAwakeCliSaveMode:', param )
      MonitorSecurityAwakeModeBase.__init__( self )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

class NdrSaveMode( NdrModeBase, CliSave.Mode ):
   def __init__( self, param ):
      t1( 'NdrSaveMode: ', param )
      NdrModeBase.__init__( self )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

class NucleusSaveMode( NucleusModeBase, CliSave.Mode ):
   def __init__( self, param ):
      t1( 'NucluesSaveMode: ', param )
      NucleusModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

CliSave.GlobalConfigMode.addChildMode( MonitorSecurityAwakeCliSaveMode,
                                       after=[ IntfConfigMode,
                                               SecurityConfigMode ] )
MonitorSecurityAwakeCliSaveMode.addCommandSequence( 'FlowWatcher.config' )
MonitorSecurityAwakeCliSaveMode.addCommandSequence( 'FlowWatcher.trailer',
                                                   after=[ 'FlowWatcher.config' ] )
MonitorSecurityAwakeCliSaveMode.addChildMode( NucleusSaveMode,
                                              before=[ 'FlowWatcher.trailer' ] )
NucleusSaveMode.addCommandSequence( 'FlowWatcher.nucleusConfig' )
MonitorSecurityAwakeCliSaveMode.addChildMode(
   NdrSaveMode, before=[ 'FlowWatcher.trailer' ] )
NdrSaveMode.addCommandSequence( 'FlowWatcher.ndrConfig' )

@CliSave.saver( 'FlowWatcher::Config', 'flowwatcher/config' )
def saveConfig( entity, root, requireMounts, options ):
   t0( 'saveConfig ' )
   detail = options.saveAll or options.saveAllDetail
   mode = root[ MonitorSecurityAwakeCliSaveMode ].getOrCreateModeInstance( None )
   cmds = mode[ 'FlowWatcher.config' ]

   if entity.topic != entity.topicDefault or detail:
      cmds.addCommand( 'topic %s' % entity.topic )

   if entity.monitorPointId != MpId.mpIdDefault or detail:
      cmds.addCommand( 'monitor-point identifier %s' % entity.monitorPointId )

   if entity.flowTableSize != FlowTableSize.sizeDefault or detail:
      cmds.addCommand( 'flow table size %d entries' % entity.flowTableSize )

   if entity.ipfixCollectorPort != IpfixCollectorPort.ipfixPortDefault or detail:
      cmds.addCommand( 'ipfix listener port %d' % entity.ipfixCollectorPort )

   for nucleus, nucleusConfig in entity.nucleus.items():
      t1( 'saveConfig nucleus:', nucleus )
      nucleusMode = mode[ NucleusSaveMode ].getOrCreateModeInstance( nucleus )
      nucleusCmds = nucleusMode[ 'FlowWatcher.nucleusConfig' ]

      if nucleusConfig.localIntf != nucleusConfig.localIntfDefault:
         nucleusCmds.addCommand( 'local interface %s' % nucleusConfig.localIntf )
      elif detail:
         nucleusCmds.addCommand( 'no local interface' )

      hnp = nucleusConfig.nucleusHostAndPort
      if hnp.hostname != '':
         fmtStr = 'destination %s' % hnp.hostname
         if detail or hnp.port != NucleusPort.nucleusPortDefault:
            fmtStr += ' port %d' % hnp.port
         nucleusCmds.addCommand( fmtStr )
      elif detail:
         nucleusCmds.addCommand( 'no destination' )

      if nucleusConfig.sslProfile != nucleusConfig.sslProfileDefault:
         nucleusCmds.addCommand( 'ssl profile %s' % nucleusConfig.sslProfile )
      elif detail:
         nucleusCmds.addCommand( 'no ssl profile' )

   if entity.ndrConfig:
      t1( 'saveConfig ndrConfig' )
      ndrMode = mode[ NdrSaveMode ].getOrCreateModeInstance( None )
      ndrCmds = ndrMode[ 'FlowWatcher.ndrConfig' ]
      if entity.ndrConfig.unknownUdp:
         ndrCmds.addCommand( 'unknown-udp' )
      elif detail:
         ndrCmds.addCommand( 'no unknown-udp' )
   elif detail:
      cmds.addCommand( 'no ndr application' )

@CliSave.saver( 'FlowWatcher::Config', 'flowwatcher/config' )
def saveConfigTrailer( entity, root, requireMounts, options ):
   detail = options.saveAll or options.saveAllDetail

   if entity.enabled or detail:
      t1( 'saveConfigTrailer' )
      cmd = 'no disabled' if entity.enabled else 'disabled'
      mode = root[ MonitorSecurityAwakeCliSaveMode ].getOrCreateModeInstance( None )
      mode[ 'FlowWatcher.trailer' ].addCommand( cmd )

@CliSave.saver( 'FlowWatcher::Config', 'flowwatcher/config' )
def saveThreadConfig( entity, root, requireMounts, options ):
   threadConfig = entity.threadConfig
   if threadConfig is not None:
      cmds = root[ 'Agent.config' ]
      mode = 'single' if threadConfig.singleThreaded else 'dpi tx'
      cmds.addCommand( f'agent FlowWatcher threads {mode}' )
