# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AclLib
import IpLibTypes
import CliSave
from CliSavePlugin.AclCliSave import CpConfigMode
from CliSavePlugin.IraTcpCliSave import saveTcpMssConfig
import Tracing

__defaultTraceHandle__ = Tracing.Handle( 'AclCliSave' )

@CliSave.saver( 'Acl::Input::CpConfig', 'acl/cpconfig/cli',
                requireMounts = ( 'ip/vrf/config',
                   'acl/paramconfig' ) )
def saveCpConfig( entity, root, requireMounts, options ):
   saveAll = options.saveAll
   paramConfig = requireMounts[ 'acl/paramconfig' ]
   allVrfConfig = requireMounts[ "ip/vrf/config" ]

   # collect all the VRFs and remove duplicates
   vrfNames = set( allVrfConfig.vrf )
   vrfNames.update( entity.cpConfig[ 'ip' ].globalCpAcl )
   vrfNames.update( entity.cpConfig[ 'ipv6' ].globalCpAcl )
   vrfNames.add( AclLib.defaultVrf )
   vrfNames = sorted( vrfNames, key=IpLibTypes.vrfNameSortKey )

   def saveCpAcl( aclType, mode ):
      cpAclDefault = entity.cpConfig[ aclType ].globalCpAclDefault if entity.\
            cpConfig[ aclType ].globalCpAclDefault else paramConfig.cpAclNameDefault
      for vrf in vrfNames:
         cpAclName = entity.cpConfig[ aclType ].globalCpAcl.get( vrf, cpAclDefault )
         if ( cpAclName != cpAclDefault ) or saveAll:
            # only create the mode if it has non-default configuration
            if mode is None:
               mode = root[ CpConfigMode ].getSingletonInstance()
            cmds = mode[ 'Acl.cp' ]
            if cpAclName:
               if vrf != AclLib.defaultVrf:
                  # pylint: disable-next=consider-using-f-string
                  cmds.addCommand( "%s access-group %s vrf %s in" %
                                                ( aclType, cpAclName, vrf ) )
               else:
                  cmds.addCommand( f"{aclType} access-group {cpAclName} in" )
            else:
               if vrf != AclLib.defaultVrf:
                  # pylint: disable-next=consider-using-f-string
                  cmds.addCommand( "no %s access-group vrf %s in" %
                                                ( aclType, vrf ) )
               else:
                  # pylint: disable-next=consider-using-f-string
                  cmds.addCommand( "no %s access-group in" % aclType )

   def saveGlobalCpAclDefault( aclType, mode ):
      cmd = None
      globalCpAclDefault = entity.cpConfig[ aclType ].globalCpAclDefault
      if globalCpAclDefault:
         cmd = f"{aclType} access-group ingress default {globalCpAclDefault}"
      elif saveAll:
         # pylint: disable-next=consider-using-f-string
         cmd = "no %s access-group ingress default" % aclType
      if cmd is not None:
         # only create the mode if it's needed
         if mode is None:
            mode = root[ CpConfigMode ].getSingletonInstance()
         mode[ 'Acl.cp' ].addCommand( cmd )

   mode = None

   config4 = entity.cpConfig[ 'ip' ].tcpMssConfig
   config6 = entity.cpConfig[ 'ipv6' ].tcpMssConfig
   cmd = saveTcpMssConfig( saveAll, config4, config6 )
   if cmd is not None:
      # only create the mode if it's needed
      mode = root[ CpConfigMode ].getSingletonInstance()
      mode[ 'Acl.cp' ].addCommand( cmd )

   for aclType in ( 'ip', 'ipv6' ):
      saveGlobalCpAclDefault( aclType, mode )
      saveCpAcl( aclType, mode )
