#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AuthnUserPriorityCli
import CliSave
from CliMode.GribiMode import MgmtGribiMode, GribiTransportMode
from IpLibConsts import DEFAULT_VRF
import Toggles.gribiToggleLib

class MgmtGribiSaveMode( MgmtGribiMode, CliSave.Mode ):
   def __init__( self, param ):
      MgmtGribiMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

class GribiTransportSaveMode( GribiTransportMode, CliSave.Mode ):
   def __init__( self, param ):
      GribiTransportMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

CliSave.GlobalConfigMode.addChildMode( MgmtGribiSaveMode )
MgmtGribiSaveMode.addCommandSequence( "Mgmt.gribi" )

MgmtGribiSaveMode.addChildMode( GribiTransportSaveMode )
GribiTransportSaveMode.addCommandSequence( "Mgmt.gribi.transport" )

@CliSave.saver( "Gribi::Config", "mgmt/gribi/config" )
def saveGribi( gribiConfig, root, requireMounts, options ):
   parentMode = root[ MgmtGribiSaveMode ].getSingletonInstance()
   if gribiConfig.endpoints:
      for name in gribiConfig.endpoints:
         endpoint = gribiConfig.endpoints.get( name )
         break
      mode = parentMode[ GribiTransportSaveMode ].getOrCreateModeInstance(
         endpoint.name )
      cmds = mode[ "Mgmt.gribi.transport" ]

      if not endpoint.enabled:
         cmds.addCommand( "shutdown" )
      elif options.saveAll:
         cmds.addCommand( "no shutdown" )

      if endpoint.serviceAcl != '':
         cmds.addCommand( "ip access-group %s" % endpoint.serviceAcl )
      elif options.saveAll:
         cmds.addCommand( "no ip access-group" )

      if endpoint.serviceAclV6 != '':
         cmds.addCommand( "ipv6 access-group %s" % endpoint.serviceAclV6 )
      elif options.saveAll:
         cmds.addCommand( "no ipv6 access-group" )

      if endpoint.port != endpoint.portDefault or options.saveAll:
         cmds.addCommand( "port %s" % endpoint.port )

      if endpoint.dscp != 0 or options.saveAll:
         cmds.addCommand( "qos dscp %s" % endpoint.dscp )

      if endpoint.sslProfile != '':
         cmds.addCommand( "ssl profile %s" % endpoint.sslProfile )
      elif options.saveAll:
         cmds.addCommand( "no ssl profile" )
      if Toggles.gribiToggleLib.toggleGribiAccountingRequestsEnabled():
         if endpoint.accountingRequests:
            cmds.addCommand( "accounting requests" )
         elif options.saveAll:
            cmds.addCommand( "no accounting requests" )

      if Toggles.gribiToggleLib.toggleGribiAUPEnabled():
         if ( endpoint.authnUsernamePriority.values() !=
               AuthnUserPriorityCli.defaultAuthnUserPriority ) or options.saveAll:
            tokens = [ AuthnUserPriorityCli.authnUsernamePriorityToCLI[ v.source ]
                  for v in endpoint.authnUsernamePriority.values() ]
            cmds.addCommand( "authentication username priority %s" %
                  " ".join( tokens ) )
      else:
         if endpoint.certUsernameAuthn:
            cmds.addCommand( "certificate username authentication" )
         elif options.saveAll:
            cmds.addCommand( "no certificate username authentication" )

      if endpoint.vrfName != DEFAULT_VRF or options.saveAll:
         cmds.addCommand( f'vrf {endpoint.vrfName}' )
