#!/usr/bin/env python3
# Copyright (c) 2008, 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import CliSave, Tracing
import Arnet
import Tac
import IgmpSnoopingLib
from CliSavePlugin.IntfCliSave import IntfConfigMode

t0 = Tracing.trace0

__defaultTraceHandle__ = Tracing.Handle( 'IgmpSnoopingCliSave' )

CliSave.GlobalConfigMode.addCommandSequence( 'IgmpSnooping.global' )
CliSave.GlobalConfigMode.addCommandSequence( 'IgmpSnooping.proxy' )
IntfConfigMode.addCommandSequence( 'IgmpSnooping.intfConfig' )

defaults = Tac.Type( "Igmp::Snooping::Proxy::Defaults" )

def saveVlanQuerierConfig( cmds, vlanId, querierConfig, saveAll ):

   if not querierConfig:
      return

   if querierConfig.enabledConfigured == 'vlanQuerierEnabled':
      cmds.addCommand( 'ip igmp snooping vlan %s querier' %vlanId )
   elif querierConfig.enabledConfigured == 'vlanQuerierDisabled':
      cmds.addCommand( 'no ip igmp snooping vlan %s querier' %vlanId )
   elif saveAll:
      cmds.addCommand( 'default ip igmp snooping vlan %s querier' % vlanId )

   if querierConfig.isIpAddrConfigured:
      cmds.addCommand( 'ip igmp snooping vlan %s querier address %s' %(
        vlanId, querierConfig.ipAddrConfigured ) )
   elif saveAll:
      cmds.addCommand( 'no ip igmp snooping vlan %s querier address' %  vlanId )
       
   if querierConfig.queryIntervalConfigured != 0.0:
      cmds.addCommand( 'ip igmp snooping vlan %s querier query-interval %d' %(
         vlanId, querierConfig.queryIntervalConfigured ) )
   elif saveAll:
      cmds.addCommand( 'no ip igmp snooping vlan %s querier query-interval' %  
         vlanId )

   if querierConfig.queryResponseIntervalConfigured != 0:
      cmds.addCommand( 'ip igmp snooping vlan %s querier max-response-time %d' %(
        vlanId, querierConfig.queryResponseIntervalConfigured ) )
   elif saveAll:
      cmds.addCommand( 'no ip igmp snooping vlan %s querier max-response-time' %
        vlanId )

   if querierConfig.lastMemberQueryIntervalConfigured != 0:
      cmds.addCommand( ( 'ip igmp snooping vlan %s querier '
                         'last-member-query-interval %d' ) %
                       ( vlanId, querierConfig.lastMemberQueryIntervalConfigured ) )
   elif saveAll:
      cmds.addCommand( ( 'no ip igmp snooping vlan %s querier '
                         'last-member-query-interval' ) % vlanId )

   if querierConfig.lastMemberQueryCountConfigured != 0:
      cmds.addCommand( ( 'ip igmp snooping vlan %s querier '
                         'last-member-query-count %d' ) %
                       ( vlanId, querierConfig.lastMemberQueryCountConfigured ) )
   elif saveAll:
      cmds.addCommand( ( 'no ip igmp snooping vlan %s querier '
                         'last-member-query-count' ) % vlanId )     

   if querierConfig.startupQueryIntervalConfigured != 0:
      cmds.addCommand( ( 'ip igmp snooping vlan %s querier '
                         'startup-query-interval %d' ) %
                       ( vlanId, querierConfig.startupQueryIntervalConfigured ) )
   elif saveAll:
      cmds.addCommand( ( 'no ip igmp snooping vlan %s querier '
                         'startup-query-interval' ) % vlanId )

   if querierConfig.startupQueryCountConfigured != 0:
      cmds.addCommand( ( 'ip igmp snooping vlan %s querier '
                         'startup-query-count %d' ) %
                       ( vlanId, querierConfig.startupQueryCountConfigured ) )
   elif saveAll:
      cmds.addCommand( ( 'no ip igmp snooping vlan %s querier '
         'startup-query-count' ) % vlanId )     

   if querierConfig.querierVersionConfigured != 'igmpVersionUnknown':
      querierVersion = int( querierConfig.querierVersionConfigured[ -1 ] )
      cmds.addCommand( 'ip igmp snooping vlan %s querier version %d' % \
            ( vlanId, querierVersion ) )
              
   elif saveAll:
      cmds.addCommand( 'no ip igmp snooping vlan %s querier version' % vlanId )

   if querierConfig.mtuConfigured != 0:
      cmds.addCommand (
            'ip igmp snooping vlan %s querier mtu %s' % \
                  ( vlanId, querierConfig.mtuConfigured ) )
      # Default is not displayed in saveAll because this is a hidden command.

def saveGlobalQuerierConfig( cmds, querierConfig, saveAll ):
   if not querierConfig:
      return
   if querierConfig.enabled:
      cmds.addCommand( 'ip igmp snooping querier' )
   elif saveAll:
      cmds.addCommand( 'no ip igmp snooping querier' )
   if querierConfig.isIpAddrConfigured:
      cmds.addCommand( 'ip igmp snooping querier address %s' % querierConfig.ipAddr )
   elif saveAll:
      cmds.addCommand( 'no ip igmp snooping querier address' )
   if querierConfig.queryInterval != querierConfig.queryIntervalDefault or saveAll:
      cmds.addCommand( 'ip igmp snooping querier query-interval %d'
                      % querierConfig.queryInterval )
   if querierConfig.queryResponseInterval != \
         querierConfig.queryResponseIntervalDefault or saveAll:
      cmds.addCommand( 'ip igmp snooping querier max-response-time %d'
                      % querierConfig.queryResponseInterval )

   if querierConfig.lastMemberQueryInterval != \
         querierConfig.lastMemberQueryIntervalDefault or saveAll:
      cmds.addCommand( ( 'ip igmp snooping querier last-member-query-interval %d' ) %
                      ( querierConfig.lastMemberQueryInterval ) )

   if querierConfig.lastMemberQueryCount != 0:
      cmds.addCommand( ( 'ip igmp snooping querier last-member-query-count %d' ) %
                      ( querierConfig.lastMemberQueryCount ) )
   elif saveAll:
      cmds.addCommand( 'no ip igmp snooping querier last-member-query-count' )     

   if querierConfig.startupQueryInterval != 0:
      cmds.addCommand( ( 'ip igmp snooping querier startup-query-interval %d' ) %
                      ( querierConfig.startupQueryInterval ) )
   elif saveAll:
      cmds.addCommand( 'no ip igmp snooping querier startup-query-interval' )      

   if querierConfig.startupQueryCount != 0:
      cmds.addCommand( ( 'ip igmp snooping querier startup-query-count %d' ) %
                      ( querierConfig.startupQueryCount ) )
   elif saveAll:
      cmds.addCommand( 'no ip igmp snooping querier startup-query-count' )     
     
   if querierConfig.querierVersion != querierConfig.querierVersionDefault:
      cmds.addCommand( 'ip igmp snooping querier version %d' %
                      int( querierConfig.querierVersion[ -1 ] ) )
   elif saveAll:
      cmds.addCommand( 'no ip igmp snooping querier version' )
     
   if querierConfig.mtu != querierConfig.mtuDefault:
      cmds.addCommand (
         'ip igmp snooping querier mtu %s' % querierConfig.mtu )
      # Default is not displayed in saveAll because this is a hidden command.     

def getBridgingVlans( requireMounts ):
   # save the config for vlans which are active and/or has non-default config.
   bridgingInputCli = requireMounts[ 'bridging/input/config/cli' ]
   return IgmpSnoopingLib.activeVlans( bridgingInputCli )

@CliSave.saver( 'Bridging::IgmpSnooping::Config', 'bridging/igmpsnooping/config',
                requireMounts = ( 'bridging/input/config/cli', ) )
def saveConfig( entity, root, requireMounts, options ):
   cmds = root[ 'IgmpSnooping.global' ]
   if entity.vlanDefaultEnabled != True: # pylint: disable=singleton-comparison
      cmds.addCommand( 'no ip igmp snooping' )
   elif options.saveAll:
      cmds.addCommand( 'ip igmp snooping' )

   # pylint: disable-next=singleton-comparison
   if entity.globalReportFloodingEnabled == True:
      cmds.addCommand( 'ip igmp snooping report-flooding' )
   elif options.saveAll:
      cmds.addCommand( 'no ip igmp snooping report-flooding' )

   for switchPort in Arnet.sortIntf( entity.switchPort ):
      cmds.addCommand ( 'ip igmp snooping report-flooding switch-port %s'\
            % ( switchPort ) )
   
   if entity.robustness != entity.robustnessDefault or options.saveAll:
      cmds.addCommand (
         'ip igmp snooping robustness-variable %u' %entity.robustness )

   if entity.restartQueryInterval != 0:
      cmds.addCommand (
         'ip igmp snooping restart query-interval %u' %entity.restartQueryInterval )
   elif options.saveAll:
      cmds.addCommand (
         'no ip igmp snooping restart query-interval' )

   if len( entity.intfRestartQueryTime ) == 0:
      if options.saveAll:
         cmds.addCommand( "ip igmp snooping interface-restart-query 2000" )
   else:
      queryTimes = [ "%d " % qt for qt in entity.intfRestartQueryTime.values() ]
      cmdStr = "".join( queryTimes )

      cmds.addCommand( "ip igmp snooping interface-restart-query %s" % cmdStr )

   vlanIds = set( entity.vlanConfig )
   if options.saveAll:
      vlanIds.update( getBridgingVlans( requireMounts ) )

   if not entity.immediateLeave:
      cmds.addCommand ( 'no ip igmp snooping fast-leave' )
   elif options.saveAll:
      cmds.addCommand (
         'ip igmp snooping fast-leave' )

   for vlanId in sorted( vlanIds, key=int ):
      vlanConfig = entity.vlanConfig.get( vlanId )
      if not vlanConfig:
         if options.saveAll:
            vlanConfig = Tac.newInstance( "Bridging::IgmpSnooping::VlanConfig",
                                          vlanId )
         else:
            continue

      if vlanConfig.enabled == 'vlanStateDisabled':
         cmds.addCommand ( 'no ip igmp snooping vlan %s' %vlanId )
      elif options.saveAll:
         cmds.addCommand ( 'default ip igmp snooping vlan %s' %vlanId )

      for routerIntf in Arnet.sortIntf( vlanConfig.routerIntf ):
         cmds.addCommand( 'ip igmp snooping vlan %s multicast-router interface %s' %
                          ( vlanId, routerIntf ) )
      for groupAddr in sorted( vlanConfig.ipGroup ):
         ipGroup = vlanConfig.ipGroup[ groupAddr ]
         for hostIntf in Arnet.sortIntf( ipGroup.intf ):
            cmds.addCommand ( 'ip igmp snooping vlan %s member %s interface %s'
                              % ( vlanId, groupAddr, hostIntf ) )
      vlanQuerierConfig = vlanConfig.vlanQuerierConfig
      if not vlanQuerierConfig and options.saveAll:
         vlanQuerierConfig = Tac.newInstance(
            "Bridging::IgmpSnooping::VlanQuerierConfig" )

      saveVlanQuerierConfig( cmds, vlanId, vlanQuerierConfig, options.saveAll )

      if vlanConfig.maxGroups < vlanConfig.unlimitedGroups:
         cmds.addCommand( 'ip igmp snooping vlan %s max-groups %s' % ( vlanId,
                          vlanConfig.maxGroups ) )
      elif options.saveAll:
         cmds.addCommand( 'no ip igmp snooping vlan %s max-groups' % vlanId )

      if vlanConfig.immediateLeave == "immediateLeaveDisabled":
         cmds.addCommand ( 'no ip igmp snooping vlan %s fast-leave' % vlanId )
      elif vlanConfig.immediateLeave == "immediateLeaveEnabled":
         cmds.addCommand ( 'ip igmp snooping vlan %s fast-leave' % vlanId )
      elif options.saveAll:
         cmds.addCommand (
            'default ip igmp snooping vlan %s fast-leave' % vlanId )

      if vlanConfig.reportFloodingConfig == 'vlanReportFloodingEnabled':
         cmds.addCommand (
            'ip igmp snooping vlan %s report-flooding' % vlanId )

      for switchPort in Arnet.sortIntf( vlanConfig.switchPort ):
         cmds.addCommand ( 'ip igmp snooping vlan %s report-flooding switch-port %s'\
               % ( vlanId, switchPort ) )

   saveGlobalQuerierConfig( cmds, entity.globalQuerierConfig, options.saveAll )
   
   for intfConfig in entity.intfConfig.values():
      mode = root[ IntfConfigMode ].getOrCreateModeInstance( intfConfig.intfId )
      cmds = mode[ 'IgmpSnooping.intfConfig' ]
      if intfConfig.profileName != "":
         cmds.addCommand( 'ip igmp snooping filter %s' % intfConfig.profileName )

def saveProxyVlanConfig( entity, vlanId, root, saveAll ):
   if vlanId in entity.vlanProxyState:
      # pylint: disable-next=singleton-comparison
      if entity.vlanProxyState[ vlanId ] == True:
         root.addCommand( 'ip igmp snooping vlan %s proxy' % vlanId )
      else:
         root.addCommand( 'no ip igmp snooping vlan %s proxy' % vlanId )
   elif saveAll:
      root.addCommand( 'default ip igmp snooping vlan %s proxy' % vlanId )

@CliSave.saver( 'Igmp::Snooping::Proxy::Config',
                'bridging/igmpsnooping/proxy/config', 
                requireMounts = ( 'bridging/input/config/cli', ) )
def saveProxyConfig( entity, root, requireMounts, options ):
   cmds = root[ 'IgmpSnooping.proxy' ]
   saveAll = options.saveAll
   if entity.globalEnabled:
      cmds.addCommand( 'ip igmp snooping proxy' )
   elif saveAll:
      cmds.addCommand( 'no ip igmp snooping proxy' )

   # Save VlanConfig
   vlanIds = set( entity.vlanProxyState )
   if saveAll:
      vlanIds.update( getBridgingVlans( requireMounts ) )

   for vlanId in vlanIds:
      saveProxyVlanConfig( entity, vlanId, cmds, saveAll )

