#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import CliSave
from CliSavePlugin import IntfCliSave
from CliSavePlugin.IntfCliSave import IntfConfigMode
from CliSavePlugin.SamplePolicyCliSave import SamplePolicySaveMode, \
      SamplePolicySaver
from CliSavePlugin.TrafficPolicyCliSave import ( MatchRuleSaveMode, ActionsSaveMode )
from CliMode.InbandTelemetry import ( FEATURE, InbandTelemetryModeBase,
                                      ProfileModeBase )
from InbandTelemetryTypes import tacCollectorType, tacPortProfileType
from Toggles.InbandTelemetryCommonToggleLib import (
   toggleFeatureInbandTelemetrySamplePolicyEnabled,
   toggleFeatureInbandTelemetryVxlanEnabled )

class InbandTelemetryConfigMode( InbandTelemetryModeBase, CliSave.Mode ):
   def __init__( self, param ):
      InbandTelemetryModeBase.__init__( self )
      CliSave.Mode.__init__( self, self.longModeKey )

CliSave.GlobalConfigMode.addCommandSequence( 'InbandTelemetry.Config' )
IntfCliSave.IntfConfigMode.addCommandSequence( 'InbandTelemetry.Config' )

CliSave.GlobalConfigMode.addChildMode( InbandTelemetryConfigMode,
                                       after=[ IntfConfigMode ] )
InbandTelemetryConfigMode.addCommandSequence( 'InbandTelemetry.Config' )

class ProfileConfigMode( ProfileModeBase, CliSave.Mode ):
   def __init__( self, param ):
      ProfileModeBase.__init__( self, param )
      CliSave.Mode.__init__( self, self.longModeKey )

if toggleFeatureInbandTelemetrySamplePolicyEnabled():
   InbandTelemetryConfigMode.addChildMode( SamplePolicySaveMode,
                                           after=[ 'InbandTelemetry.Config' ] )
   SamplePolicySaveMode.addCommandSequence( 'InbandTelemetry.Policy' )

   SamplePolicySaveMode.addChildMode( MatchRuleSaveMode )
   MatchRuleSaveMode.addCommandSequence( 'IntSamplePolicy.MatchRule' )

   ActionsSaveMode.addCommandSequence( 'IntSamplePolicy.Actions' )
   InbandTelemetryConfigMode.addChildMode( ProfileConfigMode,
                                           after=[ SamplePolicySaveMode ] )
else:
   InbandTelemetryConfigMode.addChildMode( ProfileConfigMode,
                                           after=[ 'InbandTelemetry.Config' ] )

ProfileConfigMode.addCommandSequence( 'InbandTelemetry.profile' )

class IntSamplePolicySaver( SamplePolicySaver ):
   trafficPoliciesMode = InbandTelemetryConfigMode
   defaultActionsMode = None

   def __init__( self, entity, root, requireMounts, options, feature, commentKey ):
      SamplePolicySaver.__init__( self, entity, root, requireMounts, options,
                                  feature, commentKey )
      self.policyMapType = 'inband-sample-policy'

   def defaultActionsModeCmds( self, defaultActionsMode ):
      raise NotImplementedError

   def matchModeCmds( self, matchMode ):
      return matchMode[ 'IntSamplePolicy.MatchRule' ]

   def actionModeCmds( self, actionsMode ):
      return actionsMode[ 'IntSamplePolicy.Actions' ]

@CliSave.saver( 'InbandTelemetry::Config', 'inbandtelemetry/config',
                requireMounts=( 'cli/config', 'inbandtelemetry/hwCapability',
                                'inbandtelemetry/samplePolicies/input/cli',
                              ) )
def saveConfig( entity, root, requireMounts, options ):
   hwCapability = requireMounts[ 'inbandtelemetry/hwCapability' ]
   saveAll = options.saveAll

   for intfName in entity.intfToEdgePortProfiles:
      profileName = entity.intfToEdgePortProfiles[ intfName ]
      intfMode = root[ IntfCliSave.IntfConfigMode ].getOrCreateModeInstance(
         intfName )
      intfCmds = intfMode[ 'InbandTelemetry.Config' ]
      intfCmds.addCommand( 'telemetry inband profile edge %s' % profileName )
   for intfName in entity.intfToCorePortProfiles:
      profileName = entity.intfToCorePortProfiles[ intfName ]
      intfMode = root[ IntfCliSave.IntfConfigMode ].getOrCreateModeInstance(
         intfName )
      intfCmds = intfMode[ 'InbandTelemetry.Config' ]
      intfCmds.addCommand( 'telemetry inband profile core %s' % profileName )
   if toggleFeatureInbandTelemetryVxlanEnabled():
      for intfName in entity.intfToVxlanPortProfiles:
         profileName = entity.intfToVxlanPortProfiles[ intfName ]
         intfMode = root[ IntfCliSave.IntfConfigMode ].getOrCreateModeInstance(
            intfName )
         intfCmds = intfMode[ 'InbandTelemetry.Config' ]
         intfCmds.addCommand( 'telemetry inband profile vxlan %s' % profileName )

   def addCommandIntProfile( profile ):
      profileName = profile.name
      if profile.type == tacPortProfileType.IntEdge:
         profileType = 'edge'
      elif profile.type == tacPortProfileType.IntCore:
         profileType = 'core'
      else:
         profileType = 'vxlan'
      param = ( profileName, profile.type )
      cmdList = []
      isDefaultProfile = profileName == 'default'
      if profile:
         # display default edge/core profiles only with saveAll
         if not saveAll and isDefaultProfile and profileType != 'vxlan':
            return
         profileMode = intConfigMode[ ProfileConfigMode ].\
                       getOrCreateModeInstance( param )
         if profileType == 'edge':
            if profile.sampleRate != 1048576 or saveAll:
               cmdList.append( 'ingress sample rate %d' % profile.sampleRate )
            if profile.samplePolicy != '':
               cmdList.append( 'ingress sample policy %s'
                               % profile.samplePolicy )
            if not profile.egressDrop:
               cmdList.append( 'egress drop disabled' )
            elif saveAll:
               cmdList.append( 'default egress drop disabled' )
         if profileType == 'vxlan':
            if profile.sampleRate != 1048576 or saveAll:
               cmdList.append( 'ingress sample rate %d' % profile.sampleRate )
         if profile.collectorName:
            collectionName = profile.collectorName
            collection = entity.collectors[ collectionName ]
            # TODO: when collector log is supported uncomment the below
            # if collection.type == tacCollectorType.CollectorLog:
            #   cmdList.append( 'egress collection log' )
            if collection.type == tacCollectorType.CollectorFlowTracker:
               if profileType == 'vxlan':
                  cmdList.append( 'ingress collection flow tracker %s'
                        % collectionName )
               else:
                  cmdList.append( 'egress collection flow tracker %s'
                        % collectionName )
         if cmdList:
            cmdSeq = profileMode[ 'InbandTelemetry.profile' ]
            for cmd in cmdList:
               cmdSeq.addCommand( cmd )

   def isDefaultIntParam( param ):
      return bool( ( param.deviceId == param.defaultDeviceId ) and
                   ( param.probeMarker == param.defaultProbeMarker ) and
                   ( not param.probeMarkerDisabled ) and
                   ( not param.udpDstPort1Enabled ) and
                   ( not param.udpDstPort2Enabled ) and
                   ( param.probeProtocol == param.defaultProbeProtocol ) )

   def isDefaultConfig( entity ):
      policiesCliConfig = requireMounts[ 'inbandtelemetry/samplePolicies/input/cli' ]
      if len( entity.corePortProfiles ) == 1 and \
         len( entity.edgePortProfiles ) == 1 and \
         not entity.vxlanPortProfiles and \
         isDefaultIntParam( entity.intParam ) and \
         not policiesCliConfig.pmapType.pmapInput and \
         not CliSave.hasComments( 'mon-telemetry-inband', requireMounts ) and \
         not entity.enable:
         # TODO: when collector log is supported uncomment the below
         # entity.collectors[ 'DefaultLog' ].maxPktsToBuffer == 64:
         return True
      return False

   if hwCapability.inbandTelemetrySupported and \
      ( not isDefaultConfig( entity ) or saveAll ):
      intConfigMode = root[ InbandTelemetryConfigMode ].getOrCreateModeInstance(
         None )
      intCmds = intConfigMode[ 'InbandTelemetry.Config' ]
      if entity.enable:
         intCmds.addCommand( 'no disabled' )
      elif saveAll:
         intCmds.addCommand( 'disabled' )
      if entity.intParam.deviceId != entity.intParam.defaultDeviceId or saveAll:
         intCmds.addCommand( 'device-id %s' % entity.intParam.deviceId )

      markerA = entity.intParam.probeMarker >> 32
      markerB = entity.intParam.probeMarker - ( markerA << 32 )
      if hwCapability.defaultIntDetectionMethod == \
            Tac.Type( "InbandTelemetry::IntDetectionMethod" ).ProbeMarkerBased:
         if entity.intParam.probeMarker != entity.intParam.defaultProbeMarker or \
            saveAll:
            hexA = hex( markerA )[ : -1 ] \
                  if 'L' in hex( markerA ) else hex( markerA )
            hexB = hex( markerB )[ : -1 ] \
                  if 'L' in hex( markerB ) else hex( markerB )
            if entity.intParam.probeMarkerDisabled:
               intCmds.addCommand( 'no probe marker' )
            else:
               intCmds.addCommand( f'probe marker {hexA} {hexB}' )
         IntNativeVer = Tac.Type( "InbandTelemetry::IntVersion" ).VersionNativeV1
         if hwCapability.defaultIntVersion == IntNativeVer:
            cmd = ''
            if entity.intParam.udpDstPort1Enabled or saveAll:
               port = entity.intParam.udpDstPort1
               cmd = 'probe marker udp destination port %s' % ( port )
            if entity.intParam.udpDstPort2Enabled or saveAll:
               port = entity.intParam.udpDstPort2
               cmd += ' %s' % ( port )
            if cmd:
               intCmds.addCommand( cmd )

      else:
         if entity.intParam.probeProtocol != entity.intParam.defaultProbeProtocol or\
               saveAll:
            intCmds.addCommand( 'probe marker ip protocol %s'
                                % entity.intParam.probeProtocol )
      if toggleFeatureInbandTelemetrySamplePolicyEnabled():
         cliDumper = IntSamplePolicySaver( requireMounts[
            'inbandtelemetry/samplePolicies/input/cli' ],
                                           root, requireMounts, options, FEATURE,
                                           'mon-telemetry-inband' )
         cliDumper.save()
      # TODO: when collector log is supported uncomment the below
      #logCollection = entity.collectors[ 'DefaultLog' ]
      # if logCollection.maxPktsToBuffer != 64:
      #   intCmds.addCommand( 'egress collection log %d' %
      #                    logCollection.maxPktsToBuffer )
      # elif saveAll:
      #   intCmds.addCommand( 'egress collection log 64' )

      portProfilesList = [ entity.corePortProfiles, entity.edgePortProfiles ]
      if toggleFeatureInbandTelemetryVxlanEnabled():
         portProfilesList.append( entity.vxlanPortProfiles )
      for portProfiles in portProfilesList:
         for profile in sorted( portProfiles ):
            addCommandIntProfile( portProfiles[ profile ] )
