#!/usr/bin/env python3
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave
from CliSave import escapeFormatString
from CliMode.InfluxTelemetry import InfluxTelemetryBaseMode
from CliMode.InfluxTelemetry import DestinationInfluxdbMode
from CliMode.InfluxTelemetry import SourceSocketMode
from CliSavePlugin.Security import mgmtSecurityConfigPath
from CliSavePlugin.Security import SecurityConfigMode
from IpLibConsts import DEFAULT_VRF
import ReversibleSecretCli

class InfluxTelemetryConfigMode( InfluxTelemetryBaseMode,
                                 CliSave.Mode ):
   def __init__( self, param ):
      InfluxTelemetryBaseMode.__init__( self )
      CliSave.Mode.__init__( self, self.longModeKey )

InfluxTelemetryConfigMode.addCommandSequence( 'TelegrafMgr.config' )
# CLI save block 'Mgmt.Security' need to be loaded first
# for password encryption configuration in InfluxTelemetry
CliSave.GlobalConfigMode.addChildMode( InfluxTelemetryConfigMode,
                                       after=[ SecurityConfigMode ] )

class DestinationInfluxdbConfigMode( DestinationInfluxdbMode,
                                     CliSave.Mode ):
   def __init__( self, param ):
      DestinationInfluxdbMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

DestinationInfluxdbConfigMode.addCommandSequence(
   'TelegrafMgr.config.dest.influxdb' )
InfluxTelemetryConfigMode.addChildMode( DestinationInfluxdbConfigMode )

class SourceSocketConfigMode( SourceSocketMode,
                                        CliSave.Mode ):
   def __init__( self, param ):
      SourceSocketMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

SourceSocketConfigMode.addCommandSequence(
   'TelegrafMgr.config.source.socket' )
InfluxTelemetryConfigMode.addChildMode( SourceSocketConfigMode )

def saveDestConfig( root, destName, destConfig, requireMounts, options ):
   securityConfig = requireMounts[ mgmtSecurityConfigPath ]
   parentMode = root[ InfluxTelemetryConfigMode ].getOrCreateModeInstance(
      None )
   mode = parentMode[ DestinationInfluxdbConfigMode ].getOrCreateModeInstance(
      destName )
   cmds = mode[ 'TelegrafMgr.config.dest.influxdb' ]
   if destConfig.url:
      cmds.addCommand( f'url {destConfig.url}' )
   if destConfig.databaseName:
      cmds.addCommand( f'database name {destConfig.databaseName}' )
   if destConfig.retentionPolicyName != 'default':
      cmds.addCommand( 'retention policy {}'.format(
         destConfig.retentionPolicyName ) )
   if destConfig.vrfName != DEFAULT_VRF:
      cmds.addCommand( f'vrf {destConfig.vrfName}' )
   if destConfig.username:
      formatStr = f'username {escapeFormatString(destConfig.username)} password {{}}'
      cmd = ReversibleSecretCli.getCliSaveCommand( formatStr,
                                                   securityConfig,
                                                   destConfig.password )
      cmds.addCommand( cmd )

def saveSourceSocketConfig( root, socketName, socketConfig ):
   parentMode = root[ InfluxTelemetryConfigMode ].getOrCreateModeInstance(
      None )
   mode = parentMode[ SourceSocketConfigMode ].getOrCreateModeInstance(
      socketName )
   cmds = mode[ 'TelegrafMgr.config.source.socket' ]
   if socketConfig.url:
      cmds.addCommand( f'url {socketConfig.url}' )
   if socketConfig.connectionLimit:
      cmds.addCommand(
         f'connection limit {socketConfig.connectionLimit}' )

@CliSave.saver( 'TelegrafMgr::TelegrafMgrConfig', 'telegrafMgr/config',
                requireMounts=( mgmtSecurityConfigPath, ) )
def saveConfig( entity, root, requireMounts, options ):
   for destName, destConfig in entity.destinationInfluxdbConfig.items():
      saveDestConfig( root, destName, destConfig, requireMounts, options )
   for socketName, socketConfig in entity.sourceSocketConfig.items():
      saveSourceSocketConfig( root, socketName, socketConfig )

   mode = root[ InfluxTelemetryConfigMode ].getOrCreateModeInstance( None )
   cmds = mode[ 'TelegrafMgr.config' ]
   for key, value in entity.globalTags.items():
      if key and value:
         cmds.addCommand( f'tag global {key} {value}' )

   for group, status in entity.sourceGroups.items():
      if group == 'standard':
         # standard by default is on, so it's logic is inverted from other groups
         if status == 'disabled':
            cmds.addCommand( 'source group standard disabled' )
         elif options.saveAllDetail or options.saveAll:
            cmds.addCommand( 'no source group standard disabled' )
      else:
         if status == 'enabled':
            cmds.addCommand( 'source group %s' % group )
         elif options.saveAllDetail or options.saveAll:
            cmds.addCommand( 'no source group %s' % group )
