# Copyright (c) 2006-2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements the framework for saving interface configuration.
#-------------------------------------------------------------------------------

import Tac
import CliSave, Arnet
import CliSaveBlock
from CliMode.Intf import IntfMode
from CliMode.InterfaceDefaults import InterfaceDefaultsMode
import CliExtensions

subintfMtuCliSaveHook = CliExtensions.CliHook()

# The "profile" command in the interfaces should be saved before interface commands.
# Since we don't have a way of saying "this should be the first command" without
# changing all the command sequences, mangle the name to make it more likely to be
# the first.
INTF_PROFILE_CMD_SEQ_NAME = '!!!!!!!!EthIntf.profile'

#-------------------------------------------------------------------------------
# Object used for saving commands in "config-if" mode.
#-------------------------------------------------------------------------------
class IntfConfigMode( IntfMode, CliSave.Mode ):
   __slots__ = ( 'saveOrder_', 'inactive_', 'unconnected_', 'profileSupported_',
                 'useProfileOverrideCmd_' )

   orderMap_ = {
      'Port-Channel': 20,
      'Recirc-Channel': 20,
   }

   def instanceKey( self ):
      '''Used for sorting Mode instances.'''
      return ( self.saveOrder_, Arnet.intfNameKey( self.param_ ) )

   def __init__( self, param ):
      """The interface name is looked up in orderMap; if the key
      matches a prefix of the interface name, the given save order
      is used (lower is earlier).  If there is no match, the default
      order value of 50 is used."""
      self.saveOrder_ = 50
      for k, v in self.orderMap_.items():
         if param.startswith( k ):
            self.saveOrder_ = v
            break
      # This is just like the short name in the CliPlugin
      shortName = IntfMode.getShortname( param )
      # Whether this interface is an inactive interface lane.
      self.inactive_ = False
      self.unconnected_ = False
      # this speeds up things a bit
      self.profileSupported_ = False
      self.useProfileOverrideCmd_ = True
      IntfMode.__init__( self, param, shortName )
      CliSave.Mode.__init__( self, param )

   def processProfileCmds( self, pcmds ):
      # Based on profile cmds, update our save blocks to reflect the
      # real running-config.
      # Note the first save block should be a CommandSequence with
      # the 'profile' command itself
      firstSb = self.saveBlocks_[ 0 ]
      assert isinstance( firstSb, CliSaveBlock.CommandSequence )
      assert firstSb.name_ == INTF_PROFILE_CMD_SEQ_NAME
      if not firstSb.commands_:
         # This happens when localRootsOnly=True and
         # we have not touched the profile command.
         return
      assert firstSb.commands_[ 0 ].startswith( 'profile ' )
      overriddenPcmds = list( pcmds )
      for sb in self.saveBlocks_[ 1 : ]:
         if isinstance( sb, CliSaveBlock.CommandSequence ):
            for idx, cmd in enumerate( sb.commands_ ):
               if cmd in overriddenPcmds:
                  # Slight efficiency trick, check marker in CommandSequence.write()
                  sb.commands_[ idx ] = CliSaveBlock.SKIP_COMMAND_MARKER
                  overriddenPcmds.remove( cmd )
      for cmd in overriddenPcmds:
         if cmd.startswith( 'default ' ):
            continue
         if self.useProfileOverrideCmd_:
            cmd = 'profile override ' + cmd
         else:
            if cmd.startswith( 'no ' ):
               cmd = cmd[ 3 : ]
            # insert default commands after the 'profile' command
            cmd = 'default ' + cmd
         self.saveBlocks_[ 0 ].commands_.append( cmd )

   def expandMode( self, param ):
      profileCmds = self.getIntfProfileCmds( param )
      if profileCmds and not param.options.showProfileExpanded:
         self.processProfileCmds( profileCmds )

   def getIntfProfileCmds( self, param ):
      if self.profileSupported_:
         profileConfig = param.getSessionEntity( 'interface/profile/config' )
         intfToProfileCfg = profileConfig.intfToProfile.get( self.param_ )
         if intfToProfileCfg:
            return intfToProfileCfg.profileCmdsApplied.splitlines()
      return []

   def hideInactive( self, param ):
      if not self.inactive_:
         return False
      globalInactiveExpose = param.getSessionEntity(
            'interface/config/global' ).exposeInactiveLanes
      return not globalInactiveExpose and self.emptyCmds( param )

   def hideUnconnected( self, param ):
      if not self.unconnected_:
         return False
      globalUnconnectedExpose = param.getSessionEntity(
         'interface/config/global' ).exposeUnconnectedLanes
      return ( not globalUnconnectedExpose
               and self.emptyCmds( param ) )

CliSave.GlobalConfigMode.addChildMode( IntfConfigMode )
IntfConfigMode.addCommandSequence( 'Arnet.intf' )
IntfConfigMode.addCommandSequence( INTF_PROFILE_CMD_SEQ_NAME )
# 'Arnet.l2l3barrier' makes sure that l3 things comes after l2.
IntfConfigMode.addCommandSequence( 'Arnet.l2l3barrier' )

#------------------------------------------------------------------------------
# Helper methods
#-----------------------------------------------------------------------------

# Returns True if the interface supports routing.
# if includeEligible is set then this method returns True even for interfaces which
# are 'capable' of routing, irrespective of their configured state, e.g. For ethernet
# interface configured as switchport, this method returns True, if includeEligible
# is set.
def _isRoutingInterface( intfName, requireMounts, includeEligible=False ):
   intfStatusDir = requireMounts[ 'interface/status/all' ]
   intfStatus = intfStatusDir.intfStatus.get( intfName )
   if not intfStatus:
      return False

   # TODO use RoutingLib/RoutingIntfUtils allRoutingProtocolIntfNames instead?
   if intfName.startswith( 'Fabric' ):
      return False

   if includeEligible:
      # XXX. This is ugly. We should check some attribute to figure out if
      # the interface is capable of supporting routing or not. But nothing is
      # available right now. All interfaces are capable of supporting routing. So
      # returning True as default.
      return True
   else:
      return intfStatus.forwardingModel == 'intfForwardingModelRouted' 

# Returns the string for the load-interval command
def getLoadIntervalCommandStr( saveAll, li, extra='' ):
   if saveAll:
      if li.useDefault:
         return 'default load-interval' + extra
      else:
         # pylint: disable-next=consider-using-f-string
         return 'load-interval' + extra + ( ' %d' % li.val )
   elif not li.useDefault:
      # pylint: disable-next=consider-using-f-string
      return 'load-interval' + extra + ( ' %d' % li.val )
   return None

IntfEnabledState = Tac.Type( 'Interface::IntfEnabledState' )

def isSubIntf( intfId ):
   for extension in subintfMtuCliSaveHook.extensions():
      return extension( intfId )
#-------------------------------------------------------------------------------
# Saves the state of the baseclass attributes of an object derived from
# Intf::IntfConfig.  This function should be called from any other CliSave
# function that saves that state of a subclass of Intf::IntfConfig.
#-------------------------------------------------------------------------------
def saveIntfConfig( entity, root, requireMounts, options ):
   mode = root[ IntfConfigMode ].getOrCreateModeInstance( entity.intfId )
   cmds = mode[ 'Arnet.intf' ]
   saveAll = options.saveAll
   saveAllDetail = options.saveAllDetail

   intfStatusDir = requireMounts[ 'interface/status/all' ]
   intfGlobalConfig = requireMounts.get( 'interface/config/global' )
   profileConfig = requireMounts.get( 'interface/profile/config' )
   if profileConfig:
      mode.useProfileOverrideCmd_ = profileConfig.useOverrideCmd

   intfStatus = intfStatusDir.intfStatus.get( entity.intfId )
   if intfStatus:
      mode.inactive_ = not intfStatus.active

   mode.unconnected_ = entity.prettyName.startswith( ( 'Un', 'Ue' ) )
   mode.profileSupported_ = entity.prettyName.startswith( ( 'Et', 'Po' ) )

   if entity.description != '':
      # pylint: disable-next=consider-using-f-string
      cmds.addCommand( 'description %s' % entity.description )
   elif saveAll:
      cmds.addCommand( 'no description' )

   # Deciding whether to save the enabled state is a bit tricky, because
   # the default can vary across different interface types.  If there is
   # no defaultConfig attached to the IntfConfig or if the defaultConfig
   # doesn't initialize its enabledStateLocal atribute, then we fall all
   # the way back to the IntfConfig::enabledDefault constAttr.  Thankfully,
   # the IntfConfig.enabled attribute encapsulates most of this.  We just
   # have to figure out which default value to compare against, as this is
   # not as well encapsulated.  To really get this, make sure that you look
   # through the innards of IntfConfig::enabled, IntfConfig::enabledState,
   # and the IntfEnabledState enum (where the default value is
   # 'unknownEnabledState'.
   enabled = entity.adminEnabled
   if entity.defaultConfig:
      enabledDefault = entity.defaultConfig.adminEnabled
   else:
      enabledDefault = entity.enabledDefault

   if ( enabled != enabledDefault or saveAll or
        ( intfGlobalConfig and intfGlobalConfig.defaultEthernetShutdown and
          entity.intfId.startswith( 'Ethernet' ) ) ):
      if enabled:
         cmds.addCommand( 'no shutdown' )
      else:
         cmds.addCommand( 'shutdown' )

   loadIntervalCmdStr = getLoadIntervalCommandStr( saveAll, entity.loadInterval )
   if loadIntervalCmdStr is not None:
      cmds.addCommand( loadIntervalCmdStr )
   
   # If saveAll, display the default only on current L3 interfaces.
   # If saveAllDetail, display the default on all L3 capable interfaces.
   if ( entity.mtu and entity.l3MtuConfigured ):
      # If we have configured an L3 mtu for this interface, include it,
      # regardless of whether it matches the current default or not.
      # pylint: disable-next=consider-using-f-string
      cmds.addCommand( 'mtu %d' % entity.mtu )
   elif ( saveAll and _isRoutingInterface( entity.intfId, requireMounts,
                                           includeEligible=saveAllDetail ) ):
      # In saveAll for interfaces where there was no mtu configured, include
      # 'no mtu' so when applying this config it will remove any previous
      # interface mtu and track the default (regardless of what the global default
      # is currently).
      cmds.addCommand( 'no mtu' )

   # Interface default configurations should be used whenever possible. If they are
   # not available the legacy behavior of "using global" is preserved.
   #
   # As an example: Fabric interfaces depending on the system will either default to
   #                "on" or will follow the old "useGlobal"
   #
   # TODO BUG892322: This technically introduces a race condition for the fabric
   #                 interfaces which diverge from "useGlobal" as during system
   #                 initialization they can momentarily result in a running config
   #                 diff.
   defaultLinkStatus = entity.defaultConfig.linkStatusLogging if \
                       entity.defaultConfig else 'useGlobal'

   if entity.linkStatusLogging != defaultLinkStatus or saveAll:
      if entity.linkStatusLogging == 'on':
         cmds.addCommand( 'logging event link-status' )
      elif entity.linkStatusLogging == 'off':
         cmds.addCommand( 'no logging event link-status' )
      elif entity.linkStatusLogging == 'useGlobal':
         cmds.addCommand( 'logging event link-status use-global' )

#-------------------------------------------------------------------------------
# Save the state of the GlobalIntfConfig
#-------------------------------------------------------------------------------

CliSave.GlobalConfigMode.addCommandSequence( 'Log.linkStatus' )

def saveLogFacilityConfig( entity, root, options ):
   cmds = root[ 'Log.linkStatus' ]
   if entity.linkStatusLogging == 'off':
      cmds.addCommand( 'no logging event link-status global' )
   elif options.saveAll:
      cmds.addCommand( 'logging event link-status global' )

CliSave.GlobalConfigMode.addCommandSequence( 'Interface.loadInterval' )
  
def saveGlobalLoadInterval( entity, root, options ):
   cmds = root[ 'Interface.loadInterval' ]
   loadIntervalCmdStr = getLoadIntervalCommandStr( options.saveAll, 
                                                   entity.loadInterval,
                                                   ' default' )
   if loadIntervalCmdStr is not None:
      cmds.addCommand( loadIntervalCmdStr )

CliSave.GlobalConfigMode.addCommandSequence(
   'Interface.inactiveIntfPortIdAllocationEnabled' )

def saveInactiveIntfPortIdAllocationEnabled( entity, root, options ):
   cmds = root[ 'Interface.inactiveIntfPortIdAllocationEnabled' ]
   if not entity.inactiveIntfPortIdAllocationEnabled:
      cmds.addCommand( 'service interface inactive port-id allocation disabled' )
   elif options.saveCleanConfig:
      cmds.addCommand( 'no service interface inactive port-id allocation disabled' )

CliSave.GlobalConfigMode.addCommandSequence( 'Interface.exposeInactiveLanes' )
  
def saveExposeInactiveLanes( entity, root, options ):
   cmds = root[ 'Interface.exposeInactiveLanes' ]
   if entity.exposeInactiveLanes:
      cmds.addCommand( 'service interface inactive expose' )
   elif options.saveAll:
      cmds.addCommand( 'no service interface inactive expose' )

CliSave.GlobalConfigMode.addCommandSequence( 'Interface.exposeUnconnectedLanes' )
  
def saveExposeUnconnectedLanes( entity, root, options ):
   cmds = root[ 'Interface.exposeUnconnectedLanes' ]
   if entity.exposeUnconnectedLanes:
      cmds.addCommand( 'service interface unconnected expose' )
   elif options.saveAll:
      cmds.addCommand( 'no service interface unconnected expose' )

# pylint: disable-msg=C0322 # pylint: disable=bad-option-value
# XXX Workaround for a pylint bug with decorators.
@CliSave.saver( 'Interface::GlobalIntfConfig', 'interface/config/global' )
def saveGlobalIntfConfig( entity, root, requireMounts, options ):
   saveLogFacilityConfig( entity, root, options )
   saveGlobalLoadInterval( entity, root, options )
   saveInactiveIntfPortIdAllocationEnabled( entity, root, options )
   saveExposeInactiveLanes( entity, root, options )
   saveExposeUnconnectedLanes( entity, root, options )

#-------------------------------------------------------------------------------
# Object used for saving commands in "interface-defaults" mode.
#-------------------------------------------------------------------------------
class InterfaceDefaultsConfigMode( InterfaceDefaultsMode, CliSave.Mode ):

   def __init__( self, param ):
      InterfaceDefaultsMode.__init__( self )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

CliSave.GlobalConfigMode.addChildMode( InterfaceDefaultsConfigMode,
                                       before=[ IntfConfigMode ] )
InterfaceDefaultsConfigMode.addCommandSequence( 'Interface.defaults' )

@CliSave.saver( 'Interface::GlobalIntfConfig', 'interface/config/global' )
def saveGlobalMtu( entity, root, requireMounts, options ):
   mode = root[ InterfaceDefaultsConfigMode ].getSingletonInstance()
   cmd = mode[ 'Interface.defaults' ]
   if entity.l3Mtu and entity.l3Mtu != entity.l3MtuDefault:
      # pylint: disable-next=consider-using-f-string
      cmd.addCommand( 'mtu %d' % entity.l3Mtu )

# This function is registered with the CliSave module.
# It provides an alternative to 'interface/config/all'
# For CLI save plug-ins while in a configuration session.
# This is necessary since the reactors that maintain
# that collection will not react until the session is
# committed.

def syntheticRequireMountGenerator( sysdbRoot, sessionRoot, pathHasSessionPrefix ):
   synthIntfDict = {}
   walk = Tac.newInstance( "Interface::CliSaveIntfConfigWalk" )
   intfConfigPath = 'interface/config'
   walk.root = sysdbRoot[ intfConfigPath ]
   rootName = sysdbRoot.fullName( intfConfigPath )
   for entity in walk.match.values():
      path = entity.fullName.replace( rootName + "/", "", 1 )
      if not pathHasSessionPrefix( path ):
         synthIntfDict[ entity.prettyName ] = entity
   intfConfig = sessionRoot.entity.get( "interface/config" )
   if intfConfig:
      walk = Tac.newInstance( "Interface::CliSaveIntfConfigWalk" )
      walk.root = intfConfig
      for entity in walk.match.values():
         synthIntfDict[ entity.prettyName ] = entity
   return synthIntfDict

CliSave.registerSyntheticRequireMountGenerator(
   'interface/config/all',
   syntheticRequireMountGenerator )
