# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Toggles.IpLockingToggleLib as ILTL

from Arnet import IntfId
import CliSave
import EthIntfUtil
import Tracing
from CliMode.IpLocking import (
      AddressLockingBaseMode,
      InterfaceAddressLockingBaseMode,
      VlanAddressLockingBaseMode
)
from CliSavePlugin.IntfCliSave import IntfConfigMode
from CliSavePlugin.EbraCliSave import VlanConfigMode
from IpLockingLib import (
      intfIsEthAndSupportsIpLocking,
      intfIsLag
      )
import Tac
import Vlan

t0 = Tracing.trace0

class AddressLockingConfigMode( AddressLockingBaseMode, CliSave.Mode ):
   def __init__( self, param ):
      AddressLockingBaseMode.__init__( self )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

class EthIntfAddressLockingConfigMode( InterfaceAddressLockingBaseMode,
                                       CliSave.Mode ):
   def __init__( self, param ):
      InterfaceAddressLockingBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

class LagIntfAddressLockingConfigMode( InterfaceAddressLockingBaseMode,
                                       CliSave.Mode ):
   def __init__( self, param ):
      InterfaceAddressLockingBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

class VlanAddressLockingConfigMode( VlanAddressLockingBaseMode,
                                    CliSave.Mode ):
   def __init__( self, param ):
      VlanAddressLockingBaseMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

CliSave.GlobalConfigMode.addChildMode( AddressLockingConfigMode,
                                       before=[ IntfConfigMode ] )
IntfConfigMode.addCommandSequence( 'IpLocking.switchport',
                                   after=[ 'Ebra.switchport' ] )
IntfConfigMode.addChildMode( EthIntfAddressLockingConfigMode )
EthIntfAddressLockingConfigMode.addCommandSequence(
      'IpLocking.ethIntfAddressLocking' )
VlanConfigMode.addChildMode( VlanAddressLockingConfigMode )
VlanAddressLockingConfigMode.addCommandSequence( 'IpLocking.vlanAddressLocking' )
IntfConfigMode.addCommandSequence( 'IpLocking.switchportLag',
                                   after=[ 'IpLocking.switchport' ] )
IntfConfigMode.addChildMode( LagIntfAddressLockingConfigMode )
LagIntfAddressLockingConfigMode.addCommandSequence(
      'IpLocking.lagIntfAddressLocking' )

AddressLockingConfigMode.addCommandSequence( 'IpLocking.config' )

def getIpLockingVlanCmds( ipLockingConfig, vlanId ):
   cmds = []
   vlanConfig = ipLockingConfig.vlan.get( vlanId, None )
   if vlanConfig:
      if vlanConfig.ipv4:
         cmds.append( 'address-family ipv4' )
      if vlanConfig.ipv6:
         cmds.append( 'address-family ipv6' )
      if vlanConfig.ipv4LockedAddressEnforcementDisabled:
         cmds.append( 'locked-address ipv4 enforcement disabled' )
   return cmds

def saveVlanConfig( root, vlanId, cmdList ):
   vlanString = Vlan.vlanSetToCanonicalString( [ vlanId ] )
   vlanMode = root[ VlanConfigMode ].getOrCreateModeInstance( ( vlanId,
                                                                vlanString,
                                                                True ) )
   vlanAddressLockingMode = (
         vlanMode[ VlanAddressLockingConfigMode ].getOrCreateModeInstance(
                                                                vlanString ) )
   vlanAddressLockingCmds = vlanAddressLockingMode[ 'IpLocking.vlanAddressLocking' ]
   for cmd in cmdList:
      vlanAddressLockingCmds.addCommand( cmd )

def saveIpLockingVlanConfig( ipLockingConfig, root, requireMounts, options ):
   bridgingCliConfig = requireMounts[ 'bridging/input/config/cli' ]
   # bridgingConfig.vlanConfig contains all vlans which have been configured via
   # CLI using the `(config)# vlan <VLAN>` command even if no other configuration
   # is made inside the `config-vl-<VLAN>` mode.
   bridgingVlanConfigList = bridgingCliConfig.vlanConfig
   for vlanId in sorted( bridgingVlanConfigList, key=int ):
      cmds = getIpLockingVlanCmds( ipLockingConfig, vlanId )
      if cmds:
         saveVlanConfig( root, vlanId, cmds )

def saveIpLockingLagIntfConfigHelper( intfConfig, cmds ):
   if intfConfig:
      if intfConfig.ipv4Disabled:
         cmds.addCommand( 'address-family ipv4 disabled' )
   else:
      cmds.addCommand( 'no address locking' )

def saveIpLockingEthIntfConfigHelper( intfConfig, cmds, cmdPrefix ):
   if intfConfig:
      if cmdPrefix == 'address-family':
         if intfConfig.ipv4:
            cmds.addCommand( 'address-family ipv4' )
         if intfConfig.ipv6:
            cmds.addCommand( 'address-family ipv6' )
         if intfConfig.ipv4DeniedAddr:
            for addr in intfConfig.ipv4DeniedAddr.keys():
               cmds.addCommand( f'deny {addr}' )
         if intfConfig.ipv4Disabled:
            cmds.addCommand( 'address-family ipv4 disabled' )
         if intfConfig.ipv6Disabled:
            cmds.addCommand( 'address-family ipv6 disabled' )
         if intfConfig.ipv4LockedAddressEnforcementDisabled:
            cmds.addCommand( 'locked-address ipv4 enforcement disabled' )
      elif cmdPrefix == 'address locking':
         af = 'ipv4' if intfConfig.ipv4 else ''
         af += ' ' if af and intfConfig.ipv6 else ''
         af += 'ipv6' if intfConfig.ipv6 else ''
         cmds.addCommand( f'address locking {af}' )
   else:
      cmds.addCommand( f'no {cmdPrefix}' )

def saveIpLockingLagIntfAddressLockingConfig( intf, intfConfig, root, requireMounts,
                                              options ):
   intfAddressLockingMode = (
         root[ LagIntfAddressLockingConfigMode ].getOrCreateModeInstance( intf ) )
   cmds = intfAddressLockingMode[ 'IpLocking.lagIntfAddressLocking' ]
   saveIpLockingLagIntfConfigHelper( intfConfig, cmds )

def saveIpLockingEthIntfAddressLockingConfig( intf, intfConfig, root, requireMounts,
                                              options ):
   intfAddressLockingMode = (
         root[ EthIntfAddressLockingConfigMode ].getOrCreateModeInstance( intf ) )
   cmds = intfAddressLockingMode[ 'IpLocking.ethIntfAddressLocking' ]
   saveIpLockingEthIntfConfigHelper( intfConfig, cmds, 'address-family' )

def saveIpLockingIntfConfig( ipLockingConfig, root, requireMounts, options ):
   saveAll = options.saveAll
   saveAllDetail = options.saveAllDetail

   # turn generator into a list as it can be iterated twice
   allIntfNames = list( EthIntfUtil.allSwitchportNames( requireMounts,
                                                        includeEligible=True ) )
   # include all interface address locking modes with comments
   intfsWithComments = set( intf for intf in allIntfNames if CliSave.hasComments(
      f"if-{IntfId( intf ).shortName}-address-locking",
      requireMounts ) )

   # Get the list of interfaces on which config needs to be displayed.
   if saveAllDetail:
      cfgIntfNames = allIntfNames
   elif saveAll:
      cfgIntfNames = set( ipLockingConfig.interface )
      switchportNames = EthIntfUtil.allSwitchportNames( requireMounts )
      cfgIntfNames.update( switchportNames )
      cfgIntfNames.update( intfsWithComments )
   elif intfsWithComments:
      cfgIntfNames = set( ipLockingConfig.interface ) | intfsWithComments
   else:
      cfgIntfNames = ipLockingConfig.interface

   for intf in cfgIntfNames:
      if intfIsEthAndSupportsIpLocking( intf ):
         intfMode = root[ IntfConfigMode ].getOrCreateModeInstance( intf )
         cmds = intfMode[ 'IpLocking.switchport' ]
         intfConfig = ipLockingConfig.interface.get( intf, None )
         if ( intfConfig and intf not in ipLockingConfig.useOldSyntax or
              intf in intfsWithComments ):
            saveIpLockingEthIntfAddressLockingConfig( intf, intfConfig, intfMode,
                                                      requireMounts, options )
         else:
            saveIpLockingEthIntfConfigHelper( intfConfig, cmds, 'address locking' )
      elif intfIsLag( intf ):
         intfMode = root[ IntfConfigMode ].getOrCreateModeInstance( intf )
         cmds = intfMode[ 'IpLocking.switchportLag' ]
         intfConfig = ipLockingConfig.interface.get( intf, None )
         saveIpLockingLagIntfAddressLockingConfig( intf, intfConfig, intfMode,
                                                   requireMounts, options )

@CliSave.saver( 'IpLocking::CliConfig', 'iplocking/cliConfig',
                requireMounts=( 'bridging/config',
                                'bridging/input/config/cli',
                                'cli/config' ) )
def saveAddressLockingConfig( ipLockingConfig, root, requireMounts, options ):
   saveIpLockingIntfConfig( ipLockingConfig, root, requireMounts, options )
   saveIpLockingVlanConfig( ipLockingConfig, root, requireMounts, options )
   saveAll = options.saveAll
   saveAllDetail = options.saveAllDetail

   if ipLockingConfig.configured == ipLockingConfig.configuredDefault \
         and not saveAllDetail:
      return

   mode = root[ AddressLockingConfigMode ].getSingletonInstance()
   cmds = mode[ 'IpLocking.config' ]

   if ipLockingConfig.disabled != ipLockingConfig.disabledDefault:
      cmds.addCommand( 'disabled' )
   elif saveAll:
      cmds.addCommand( 'no disabled' )

   if ipLockingConfig.localInterface != ipLockingConfig.localInterfaceDefault:
      cmds.addCommand( f'local-interface {ipLockingConfig.localInterface}' )
   elif saveAll:
      cmds.addCommand( 'no local-interface' )

   if ipLockingConfig.dhcpV4Server:
      for ds, proto in ipLockingConfig.dhcpV4Server.items():
         if proto == 'leaseQuery':
            cmds.addCommand( f'dhcp server ipv4 {ds}' )
      for ds, proto in ipLockingConfig.dhcpV4Server.items():
         if proto == 'arIpQuery':
            cmds.addCommand( f'dhcp server ipv4 {ds} protocol arista-query' )
   elif saveAll:
      cmds.addCommand( 'no dhcp server ipv4' )

   if ILTL.toggleIpLockingArIpEnabled():
      if ipLockingConfig.dhcpV6Server:
         # we have to sort manually until TACC supports ordered set
         for ds in sorted( ipLockingConfig.dhcpV6Server ):
            cmds.addCommand(
                  f'dhcp server ipv6 {ds} protocol arista-query' )
      elif saveAll:
         cmds.addCommand( 'no dhcp server ipv6' )

   for staticLease in ipLockingConfig.staticLease.values():
      cmds.addCommand( f'lease {staticLease.ip} mac {staticLease.mac}' )

   if ipLockingConfig.queryTimeoutInfo.timeout:
      retryInterval = ipLockingConfig.queryTimeoutInfo.retryInterval
      timeout = ipLockingConfig.queryTimeoutInfo.timeout
      cmds.addCommand( f'lease query retry interval {retryInterval} '
                       f'timeout {timeout}' )
   elif saveAll:
      cmds.addCommand( 'no lease query retry' )

   if ( ipLockingConfig.macAgingCleanUp !=
        ipLockingConfig.macAgingCleanUpDefault ):
      cmds.addCommand( 'locked-address expiration mac disabled' )
   elif saveAll:
      cmds.addCommand( 'no locked-address expiration mac disabled' )

   if ipLockingConfig.ipv4LockedAddressEnforcementDisabled:
      cmds.addCommand( 'locked-address ipv4 enforcement disabled' )
   elif saveAll:
      cmds.addCommand( 'no locked-address ipv4 enforcement disabled' )

   if ipLockingConfig.ipv6LockedAddressEnforcementDisabled:
      cmds.addCommand( 'locked-address ipv6 enforcement disabled' )
   elif saveAll:
      cmds.addCommand( 'no locked-address ipv6 enforcement disabled' )
