# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSave, Tac, Tracing
from CliMode.Ipsec import IkePolicyMode, IpsecProfileMode, IpsecEntropyMode
from CliMode.Ipsec import IpSecurityMode, SecurityAssociationMode
from CliMode.Ipsec import IpsecIkeCryptoSuiteCliMode
from CliMode.Ipsec import IpsecSaCryptoSuiteCliMode
from CliMode.Ipsec import SecurityAssociationVxlanMode
from CliMode.Ipsec import KeyControllerMode
from CliSavePlugin.IntfCliSave import IntfConfigMode
from CliSavePlugin.Security import SecurityConfigMode
from CliSavePlugin.Security import mgmtSecurityConfigPath
import Toggles.IpsecToggleLib
from Toggles.AclToggleLib import toggleIpsecServiceAclEnabled
from Toggles.AclToggleLib import toggleIpsecServiceAclVrfEnabled
from Toggles.IpsecToggleLib import toggleMultiCryptoEnabled
from Toggles.IpsecToggleLib import togglePfConnectionLossEnabled
from Toggles.PolicyMapToggleLib import togglePolicyBasedVpnEnabled
from IpLibConsts import DEFAULT_VRF
import ReversibleSecretCli
from TypeFuture import TacLazyType

traceHandle = Tracing.Handle( 'IpsecCli' )
t0 = traceHandle.trace0

IpsecAuthTypeEnum = Tac.Type( "Ipsec::Ike::IpsecAuthType" )
IpsecEspAlgorithmEnum = Tac.Type( "Ipsec::IpsecEspAlgorithm" )
IpsecHmacAlgorithmEnum = Tac.Type( "Ipsec::IpsecHmacAlgorithm" )
IpsecDhGroupEnum = Tac.Type( "Ipsec::IpsecDhGroup" )
IpsecEncapProtocolEnum = Tac.Type( "Ipsec::Ike::IpsecEncapProtocol" )
IpsecFlowParallelization = Tac.Type( 'Ipsec::Ike::FlowParallelization' )
EncapSrcPortRange = Tac.Type( "Ipsec::Ike::EncapSrcPortRange" )
IpsecSaType = Tac.Type( "Ipsec::IpsecSaType" )

ipsecCapabilitiesStatusPath = 'ipsec/capabilities/status'

encapProtocolMap = {
   IpsecEncapProtocolEnum.udp : "udp",
   }

dhGroupMap = { 
         IpsecDhGroupEnum.IpsecDhGroupNone: 'none',
         IpsecDhGroupEnum.modp768 : '1',
         IpsecDhGroupEnum.modp1024 : '2',
         IpsecDhGroupEnum.modp1536 : '5',
         IpsecDhGroupEnum.modp2048 : '14',
         IpsecDhGroupEnum.modp3072 : '15',
         IpsecDhGroupEnum.modp4096 : '16',
         IpsecDhGroupEnum.modp6144 : '17',
         IpsecDhGroupEnum.ecp256 : '19',
         IpsecDhGroupEnum.ecp384 : '20',
         IpsecDhGroupEnum.ecp521 : '21',
         IpsecDhGroupEnum.modp2048s256 : '24'
         }

ipsecEncryptionMap = { IpsecEspAlgorithmEnum.des : '3des',
      IpsecEspAlgorithmEnum.aes128 : 'aes128',
      IpsecEspAlgorithmEnum.aes192 : 'aes192',
      IpsecEspAlgorithmEnum.aes256 : 'aes256',
      IpsecEspAlgorithmEnum.aes128gcm64 : 'aes128gcm64',
      IpsecEspAlgorithmEnum.aes128gcm128 : 'aes128gcm128',
      IpsecEspAlgorithmEnum.aes256gcm128 : 'aes256gcm128',
      IpsecEspAlgorithmEnum.nullesp : 'null', }

ipsecIntegrityMap = {
      IpsecHmacAlgorithmEnum.md5: 'md5',
      IpsecHmacAlgorithmEnum.sha1: 'sha1',
      IpsecHmacAlgorithmEnum.sha256: 'sha256',
      IpsecHmacAlgorithmEnum.sha384: 'sha384',
      IpsecHmacAlgorithmEnum.sha512: 'sha512',
      IpsecHmacAlgorithmEnum.nullhash: 'null',
      IpsecHmacAlgorithmEnum.nonehash: 'none',
      }

ipsecAuthTypeMap = { IpsecAuthTypeEnum.pre_share: 'pre-share',
                     IpsecAuthTypeEnum.rsa_sig: 'rsa-sig',
                     IpsecAuthTypeEnum.pki: 'pki', }

class IpsecMode ( IpSecurityMode, CliSave.Mode ):
   def __init__( self, param ):
      IpSecurityMode.__init__( self )
      CliSave.Mode.__init__( self, 1 )

class IkePolicyConfigMode ( IkePolicyMode, CliSave.Mode ):
   def __init__( self, param ):
      IkePolicyMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class IpsecSaMode ( SecurityAssociationMode, CliSave.Mode ):
   def __init__( self, param ):
      SecurityAssociationMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class IpsecProfileConfigMode ( IpsecProfileMode, CliSave.Mode ):
   def __init__( self, param ):
      IpsecProfileMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class IpsecEntropyUdpConfigMode ( IpsecEntropyMode, CliSave.Mode ):
   def __init__( self, param ):
      IpsecEntropyMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return False

class IpsecKeyControllerConfigMode ( KeyControllerMode, CliSave.Mode ):
   def __init__( self, param ):
      KeyControllerMode.__init__( self )
      CliSave.Mode.__init__( self, 1 )

class IpsecSaVxlanMode ( SecurityAssociationVxlanMode, CliSave.Mode ):
   def __init__( self, param ):
      SecurityAssociationVxlanMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class IpsecIkeCryptoSuiteConfigMode ( IpsecIkeCryptoSuiteCliMode, CliSave.Mode ):
   def __init__( self, param ):
      IpsecIkeCryptoSuiteCliMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

class IpsecSaCryptoSuiteConfigMode ( IpsecSaCryptoSuiteCliMode, CliSave.Mode ):
   def __init__( self, param ):
      IpsecSaCryptoSuiteCliMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

def saveHostIdentity( prefix, cmds, hostIdentity, saveAll ):
   if hostIdentity.containsIp():
      cmds.addCommand( f"{prefix} {hostIdentity.ip.stringValue}" )
   elif hostIdentity.containsFqdn():
      cmds.addCommand( f"{prefix} fqdn {hostIdentity.fqdn}" )
   elif saveAll:
      cmds.addCommand( f"no {prefix}" )

LifetimeUnit = TacLazyType( 'Ipsec::LifetimeUnit' )

#Save the IKE config 
def saveIpsecIsakmpConfig( mode, cmds, ipsecConfig, policy, 
                           root, options ):
   saveAll = options.saveAll
   ikeParams = policy.ikeParams
   factor = 60 if ikeParams.ikeLifetimeUnit == LifetimeUnit.minutes else 3600
   if ikeParams.integrity != ikeParams.integrityDefault or saveAll:
      integrity = str( ikeParams.integrity )
      cmds.addCommand( f'integrity {integrity}' )
   if ikeParams.auth != ikeParams.authDefault or saveAll:
      cmds.addCommand( f'authentication {ipsecAuthTypeMap[ikeParams.auth]}' )
   if ikeParams.ikeLifetime != ikeParams.ikeLifetimeDefault or saveAll:
      cmds.addCommand( f'ike-lifetime {int(ikeParams.ikeLifetime / factor)} '
            f'{ikeParams.ikeLifetimeUnit}' )
   if ikeParams.encryption != ikeParams.encryptionDefault or saveAll:
      encryption = ipsecEncryptionMap[ ikeParams.encryption ]
      cmds.addCommand( f'encryption {encryption}' )
   if ikeParams.dhGroup != ikeParams.dhGroupDefault or saveAll:
      cmds.addCommand( f'dh-group {dhGroupMap[ikeParams.dhGroup]}' )
   if ikeParams.version != ikeParams.versionDefault or saveAll:
      cmds.addCommand( f'version {ikeParams.version}' )
   if ikeParams.localId or saveAll:
      saveHostIdentity( 'local-id', cmds, ikeParams.localId, saveAll )
   if ikeParams.remoteId or saveAll:
      saveHostIdentity( 'remote-id', cmds, ikeParams.remoteId, saveAll )
   if toggleMultiCryptoEnabled():
      for ikeSuite in policy.ikeCryptoSuite.values():
         cmds.addCommand( f'crypto suite {ikeSuite.cryptoSuiteName}' )
   if Toggles.IpsecToggleLib.toggleIpsecRemoteIpAnyEnabled():
      if ikeParams.remoteIpAny != ikeParams.remoteIpAnyDefault:
         cmds.addCommand( 'remote-ip any' )
      elif saveAll:
         cmds.addCommand( 'no remote-ip any' )

#Save Security Associations
def saveIpsecSaConfig( mode, cmds, ipsecConfig, sa, root, options ):

   saveAll = options.saveAll
   saParams = sa.saParams
   factor = 60 if saParams.saLifetimeUnit == LifetimeUnit.minutes else 3600
   if saParams.espAes != saParams.espAesDefault or saveAll:
      cmds.addCommand( f'esp encryption {ipsecEncryptionMap[saParams.espAes]}' )
   if saParams.espSha != saParams.espShaDefault or saveAll:
      espSha = str( saParams.espSha )
      if espSha == 'nullhash':
         espSha = 'null'
      cmds.addCommand( f'esp integrity {espSha}' )
   if saParams.saLifetime != saParams.saLifetimeDefault or saveAll:
      cmds.addCommand( f'sa lifetime {int(saParams.saLifetime / factor)} '
            f'{saParams.saLifetimeUnit}' )
   if saParams.packetLimit != saParams.packetLimitDefault or saveAll:
      cmds.addCommand(
         f'sa lifetime {int( saParams.packetLimit / 1000 )} thousand-packets' )
   if saParams.byteLimit != saParams.byteLimitDefault or saveAll:
      if saParams.byteLimitUnit == 'MegaBytes':
         cmds.addCommand(
            f'sa lifetime {int( saParams.byteLimit / ( 1024 ** 2 ) )} megabytes' )
      elif saParams.byteLimitUnit == 'GigaBytes':
         cmds.addCommand(
            f'sa lifetime {int( saParams.byteLimit / ( 1024 ** 3 ) )} gigabytes' )
   if saParams.pfsGroup != IpsecDhGroupEnum.IpsecDhGroupNone:
      cmds.addCommand( f'pfs dh-group {dhGroupMap[saParams.pfsGroup]}' )
   if saParams.replayWindowSize:
      if saParams.replayWindowSize != saParams.replayWindowSizeDefault:
         cmds.addCommand( f'anti-replay window {saParams.replayWindowSize}' )
      elif saveAll:
         cmds.addCommand( 'anti-replay detection' )
   else:
      cmds.addCommand( 'no anti-replay detection' )
   if toggleMultiCryptoEnabled():
      for saSuite in sa.saCryptoSuite.values():
         cmds.addCommand( f'crypto suite {saSuite.cryptoSuiteName}' )

   if togglePolicyBasedVpnEnabled():
      if saParams.saIdleTimeout != saParams.saIdleTimeoutDefault:
         cmds.addCommand( f'sa idle-timeout {saParams.saIdleTimeout} seconds' )
      elif saveAll:
         cmds.addCommand( 'no sa idle-timeout' )

#Adds the Ipsec Flow Entropy UDP mode as a child mode of the Ipsec Profile mode.
IpsecProfileConfigMode.addChildMode( IpsecEntropyUdpConfigMode )
IpsecEntropyUdpConfigMode.addCommandSequence( 'Ipsec.Entropy.udp.config' )

#Save Ipsec flow entropy UDP mode
def saveIpsecEntropyUdp(
      parentMode, parentCmds, ipsecCapabilities, flowParallelization, saveAll ):
   if not ipsecCapabilities.encapSrcPortRangeSupported:
      return

   encapSrcPortRange = flowParallelization.encapSrcPortRange
   if encapSrcPortRange == EncapSrcPortRange():
      parentCmds.addCommand( 'no flow entropy udp' )
      return

   entropyMode = parentMode[ IpsecEntropyUdpConfigMode ].getOrCreateModeInstance(
         ( parentMode.param_, 'udp' ) )

   if encapSrcPortRange != IpsecFlowParallelization().dynamicPortRange:
      cmds = entropyMode[ 'Ipsec.Entropy.udp.config' ]
      cmds.addCommand( f'port source offset {encapSrcPortRange.offset} '
            f'length {encapSrcPortRange.length}' )

#Save Ipsec profile
def saveIpsecProfile( mode, cmds, ipsecConfig, profile,
                      root, requireMounts, options ):
   saveAll = options.saveAll

   if profile.ikePolicyName:
      cmds.addCommand( f'ike-policy {profile.ikePolicyName}' )

   if profile.securityAssocName:
      cmds.addCommand( f'sa-policy {profile.securityAssocName}' )

   profileParams = profile.profileParams
   if profileParams.connectionType != profileParams.connectionTypeDefault:
      cmds.addCommand( f'connection {profileParams.connectionType}' )
   elif saveAll:
      cmds.addCommand( f'connection {profileParams.connectionTypeDefault}' )

   securityConfig = requireMounts[ mgmtSecurityConfigPath ]
   if profile.ikeSharedKey:
      cmd = ReversibleSecretCli.getCliSaveCommand( 'shared-key {}',
                                                   securityConfig,
                                                   profile.ikeSharedKey )
      cmds.addCommand( cmd )
   elif saveAll:
      cmds.addCommand( 'no shared-key' )

   if profileParams.pkiProfile != profileParams.pkiProfileDefault:
      cmds.addCommand( f"pki-profile {profileParams.pkiProfile}" )
   elif saveAll:
      cmds.addCommand( "no pki-profile" )

   if profileParams.dpd.action != 'none' :
      dpd = profileParams.dpd
      cmds.addCommand( f"dpd {dpd.interval} {dpd.timeout} {dpd.action}" )
   elif saveAll:
      cmds.addCommand( 'no dpd' )

   fp = profileParams.flowParallelization
   if fp.encapProtocol != IpsecEncapProtocolEnum.no_encap:
      cmds.addCommand( 'flow parallelization encapsulation '
         f'{encapProtocolMap[fp.encapProtocol]}' )
   elif saveAll:
      cmds.addCommand( 'no flow parallelization encapsulation' )

   ipsecCapabilities = requireMounts[ 'ipsec/capabilities/status' ]
   saveIpsecEntropyUdp( mode, cmds, ipsecCapabilities, fp, options.saveAll )

   if profileParams.mode != profileParams.modeDefault:
      cmds.addCommand( f'mode {profileParams.mode}' )
   elif saveAll:
      cmds.addCommand( f'mode {profileParams.modeDefault}' )

#Save Ipsec key controller
def saveIpsecKeyController( mode, cmds, ipsecConfig, root, options ):
   saveAll = options.saveAll

   keyControllerCfg = ipsecConfig.keyController

   if keyControllerCfg.profileName != '':
      cmds.addCommand( f'profile {keyControllerCfg.profileName}' )
   elif saveAll:
      cmds.addCommand( 'no profile' )

   if keyControllerCfg.dhGroup != keyControllerCfg.dhGroupDefault \
         or saveAll:
      cmds.addCommand( f'dh-group {dhGroupMap[ keyControllerCfg.dhGroup ]}' )

   if keyControllerCfg.dhLifetime != keyControllerCfg.dhLifetimeDefault \
         or saveAll:
      cmds.addCommand( f'lifetime {int(keyControllerCfg.dhLifetime / 3600)} hours' )

   if togglePfConnectionLossEnabled() and \
      ( keyControllerCfg.holdTime != keyControllerCfg.holdTimeDefault or saveAll ):
      cmds.addCommand( f'holdtime {int(keyControllerCfg.holdTime / 3600)} hours' )

#Save Security Associations for Ipsec Vxlan
def saveIpsecVxlanSaConfig( mode, cmds, ipsecConfig, sa, root, options ):
   saveAll = options.saveAll
   saParams = sa.saParams
   if saParams.vxlanEncap:
      cmds.addCommand( 'encapsulation vxlan security' )
   elif saParams.vxlanEncapTagEsp:
      cmds.addCommand( 'encapsulation vxlan security tag esp' )
   elif saveAll:
      cmds.addCommand( 'no encapsulation vxlan security' )
   if saParams.unprotectedTraffic:
      cmds.addCommand( 'traffic unprotected allow' )
   elif saveAll:
      cmds.addCommand( 'no traffic unprotected allow' )
   if saParams.saLifetime != saParams.saLifetimeDefault or saveAll:
      factor = 60 if saParams.saLifetimeUnit == LifetimeUnit.minutes else 3600
      cmds.addCommand( f'sa lifetime {int( saParams.saLifetime / factor)} \
            {saParams.saLifetimeUnit}' )
   if saParams.replayWindowSize:
      if saveAll:
         cmds.addCommand( 'anti-replay detection' )
   else:
      cmds.addCommand( 'no anti-replay detection' )

CliSave.GlobalConfigMode.addChildMode( IpsecMode, 
                                       before=[ IntfConfigMode ],
                                       after=[ SecurityConfigMode ] )
IpsecMode.addCommandSequence('ip.security')

#Adds the Ike mode as a child mode  of the global config mode.
IpsecMode.addChildMode( IkePolicyConfigMode, 
                        before=[ IpsecProfileConfigMode ] )
IkePolicyConfigMode.addCommandSequence( 'Ipsec.IkePolicy.config' )

@CliSave.saver( 'Ipsec::Ike::Config', 'ipsec/ike/config' )
def saveIsakmpConfig( entity, root, requireMounts, options ):

   for policyName in entity.ikePolicy:
      policy = entity.ikePolicy.get( policyName )
      parent = root[ IpsecMode ].getOrCreateModeInstance( 1 )
      parentMode = parent[ IkePolicyConfigMode ].\
                        getOrCreateModeInstance( policyName )

      cmds = parentMode[ 'Ipsec.IkePolicy.config' ]
      saveIpsecIsakmpConfig( parentMode, cmds, entity, policy, 
                             root, options )

IpsecMode.addChildMode( IpsecSaMode, 
                        before=[ IpsecProfileConfigMode ] )
IpsecSaMode.addCommandSequence( 'Ipsec.Sa.config' )

@CliSave.saver( 'Ipsec::Ike::Config', 'ipsec/ike/config' )
def saveSaConfig( entity, root, requireMounts, options ):
   for name in entity.securityAssoc:
      sa = entity.securityAssoc.get( name )
      if sa.saParams.saType != IpsecSaType.IpsecKeyMgmtSa:
         continue
      parent = root[ IpsecMode ].getOrCreateModeInstance( 1 )
      parentMode = parent[ IpsecSaMode ].getOrCreateModeInstance( name )

      cmds = parentMode[ 'Ipsec.Sa.config' ]
      saveIpsecSaConfig( parentMode, cmds, entity, sa, 
                         root, options )

#Adds the Ipsec Profile mode as a child mode of the global config mode.
IpsecMode.addChildMode( IpsecProfileConfigMode, 
                        after = [ IkePolicyConfigMode ],
                        before = [ IpsecKeyControllerConfigMode ] )
IpsecProfileConfigMode.addCommandSequence( 'Ipsec.Profile.config' )

@CliSave.saver( 'Ipsec::Ike::Config', 'ipsec/ike/config',
                requireMounts=( mgmtSecurityConfigPath,
                                ipsecCapabilitiesStatusPath, ) )
def saveIpsecProfileConfig( entity, root, requireMounts, options ):

   for name in entity.ipsecProfile:
      profile = entity.ipsecProfile.get( name )
      parent = root[ IpsecMode ].getOrCreateModeInstance( 1 )
      parentMode = parent[ IpsecProfileConfigMode ].\
                     getOrCreateModeInstance( name )

      cmds = parentMode[ 'Ipsec.Profile.config' ]
      saveIpsecProfile( parentMode, cmds, entity, profile, 
                        root, requireMounts, options )

#Adds the Ipsec Key Controller mode as a child mode of the global config mode.
IpsecMode.addChildMode( IpsecKeyControllerConfigMode, 
                        after = [ IpsecProfileConfigMode ] )
IpsecKeyControllerConfigMode.addCommandSequence( 'Ipsec.KeyController.config' )


@CliSave.saver( 'Ipsec::Ike::Config', 'ipsec/ike/config' )
def saveIpsecKeyControllerConfig( entity, root, requireMounts, options ):
   if not entity.keyController.configured:
      return
   parent = root[ IpsecMode ].getOrCreateModeInstance( 1 )
   parentMode = parent[ IpsecKeyControllerConfigMode ].\
                getOrCreateModeInstance( 1 )

   cmds = parentMode[ 'Ipsec.KeyController.config' ]
   saveIpsecKeyController( parentMode, cmds, entity,
                           root, options )

#Adds the Ipsec Vxlan Sa mode as a child mode of the global config mode.
IpsecMode.addChildMode( IpsecSaVxlanMode, 
                        before=[ IpsecProfileConfigMode ] )
IpsecSaVxlanMode.addCommandSequence( 'Ipsec.Sa.config' )

@CliSave.saver( 'Ipsec::Ike::Config', 'ipsec/ike/config' )
def saveVxlanSaConfig( entity, root, requireMounts, options ):
   if not Toggles.IpsecToggleLib.toggleTunnelSecAssocEnabled():
      return
   for name in entity.securityAssoc:
      sa = entity.securityAssoc.get(name)
      if sa.saParams.saType != IpsecSaType.IpsecVxlanSa:
         continue
      parent = root[ IpsecMode ].getOrCreateModeInstance( 1 )
      parentMode = parent[ IpsecSaVxlanMode ].getOrCreateModeInstance( name )
      cmds = parentMode[ 'Ipsec.Sa.config' ]
      saveIpsecVxlanSaConfig( parentMode, cmds, entity, sa, 
                         root, options )

@CliSave.saver( 'Ipsec::Ike::Config', 'ipsec/ike/config' )
def saveFipsRestrictionsConfig( entity, root, requireMounts, options ):
   if entity.fipsRestrictions != entity.fipsRestrictionsDefault or \
      options.saveAll:
      mode = root[ IpsecMode ].getOrCreateModeInstance( 1 )
      cmds = mode[ 'ip.security' ]
      if entity.fipsRestrictions:
         cmds.addCommand( 'fips restrictions' )
      elif options.saveAll:
         cmds.addCommand('no fips restrictions' )

@CliSave.saver( 'Ipsec::Ike::Config', 'ipsec/ike/config',
                requireMounts=( ipsecCapabilitiesStatusPath, ) )
def saveUntaggedAllowedConfig( entity, root, requireMounts, options ):
   ipsecCapabilities = requireMounts[ ipsecCapabilitiesStatusPath ]
   if ipsecCapabilities.untaggedAllowed:
      return
   if entity.untaggedAllowed != entity.untaggedAllowedDefault or \
      options.saveAll:
      mode = root[ IpsecMode ].getOrCreateModeInstance( 1 )
      cmds = mode[ 'ip.security' ]
      if entity.untaggedAllowed:
         cmds.addCommand( 'ethernet untagged allowed' )
      elif options.saveAll:
         cmds.addCommand( 'no ethernet untagged allowed' )

@CliSave.saver( 'Ipsec::Ike::Config', 'ipsec/ike/config',
                requireMounts=( 'acl/cpconfig/cli', ) )
def saveIpAccessGroupConfig( entity, root, requireMounts, options ):
   if toggleIpsecServiceAclEnabled():
      ipsecAclName = None
      espAclName = None
      aclCpConfig = requireMounts[ 'acl/cpconfig/cli' ]
      serviceAclConfig = aclCpConfig.cpConfig[ 'ip' ].serviceAcl

      for vrfName, serviceAclVrfConfig in serviceAclConfig.items():
         serviceAclIpsecConfig = serviceAclVrfConfig.service.get( 'ipsec' )
         serviceAclEspConfig = serviceAclVrfConfig.service.get( 'esp' )
         if serviceAclIpsecConfig:
            ipsecAclName = serviceAclIpsecConfig.aclName
         if serviceAclEspConfig:
            espAclName = serviceAclEspConfig.aclName
         if ipsecAclName and espAclName and ipsecAclName == espAclName:
            mode = root[ IpsecMode ].getOrCreateModeInstance( 1 )
            cmds = mode[ 'ip.security' ]
            if toggleIpsecServiceAclVrfEnabled():
               if vrfName == DEFAULT_VRF:
                  cmds.addCommand( f'ip access-group {ipsecAclName}' )
               else:
                  vrfArg = "vrf " + vrfName
                  cmds.addCommand( f'ip access-group {ipsecAclName} {vrfArg}' )
            else:
               cmds.addCommand( f'ip access-group {ipsecAclName}' )
         elif ipsecAclName != espAclName:
            t0( 'IPSEC:', ipsecAclName, 'and ESP:', espAclName,
                'chain config mismatch' )
         elif options.saveAll:
            mode = root[ IpsecMode ].getOrCreateModeInstance( 1 )
            cmds = mode[ 'ip.security' ]
            cmds.addCommand( 'no ip access-group' )

@CliSave.saver( 'Ipsec::Ike::Config', 'ipsec/ike/config',
                 requireMounts=( ipsecCapabilitiesStatusPath, ) )
def saveHwEncyptionConfig( entity, root, requireMounts, options ):
   ipsecCapabilities = requireMounts[ ipsecCapabilitiesStatusPath ]
   if not ipsecCapabilities.hwCryptoSupported:
      return
   if entity.hwCryptoEnabled != ipsecCapabilities.hwCryptoByDefault or \
      options.saveAll:
      mode = root[ IpsecMode ].getOrCreateModeInstance( 1 )
      cmds = mode[ 'ip.security' ]
      if not entity.hwCryptoEnabled:
         cmds.addCommand( 'hardware encryption disabled' )
      elif options.saveAll:
         # On platforms that support HW crypto, the command is disabled by default
         cmds.addCommand( 'no hardware encryption disabled' )

def registerMultiCryptoCommands():
   if not toggleMultiCryptoEnabled():
      return

   # Adds the Ike crypto suite mode as a child mode of the IPsec config mode.
   IpsecMode.addChildMode( IpsecIkeCryptoSuiteConfigMode,
                        before=[ IkePolicyConfigMode ] )
   IpsecIkeCryptoSuiteConfigMode.addCommandSequence(
                                    'Ipsec.IkeCryptoSuiteConfig.config' )

   # Adds the Sa crypto suite mode as a child mode of the IPsec config mode.
   IpsecMode.addChildMode( IpsecSaCryptoSuiteConfigMode,
                        before=[ IkePolicyConfigMode ] )
   IpsecSaCryptoSuiteConfigMode.addCommandSequence(
                                    'Ipsec.SaCryptoSuiteConfig.config' )

registerMultiCryptoCommands()

@CliSave.saver( 'Ipsec::Ike::Config', 'ipsec/ike/config' )
def saveCryptoSuiteConfig( entity, root, requireMounts, options ):
   if not toggleMultiCryptoEnabled():
      return

   # ike suites
   for ikeSuiteName in entity.ikeCryptoSuiteConfig:
      parent = root[ IpsecMode ].getOrCreateModeInstance( 1 )
      parentMode = parent[ IpsecIkeCryptoSuiteConfigMode ].\
                        getOrCreateModeInstance( ikeSuiteName )
      ikeCmds = parentMode[ 'Ipsec.IkeCryptoSuiteConfig.config' ]
      suite = entity.ikeCryptoSuiteConfig[ ikeSuiteName ].cryptoSuite
      # show active - In the suite mode display only valid suite entry
      if ipsecEncryptionMap[ suite.encryption ] == \
            list( ipsecEncryptionMap.values() )[ 0 ] and \
         ipsecIntegrityMap[ suite.integrity ] == \
            list( ipsecIntegrityMap.values() )[ 0 ] and \
         dhGroupMap[ suite.dhGroup ] == list( dhGroupMap.values() )[ 0 ]:
         continue
      cmd = f"encryption {ipsecEncryptionMap[ suite.encryption ]} "
      cmd += f"integrity {ipsecIntegrityMap[ suite.integrity ]} "
      cmd += f"dh-group {dhGroupMap[ suite.dhGroup ]}"
      ikeCmds.addCommand( cmd )

   # sa suites
   for saSuiteName in entity.saCryptoSuiteConfig:
      parent = root[ IpsecMode ].getOrCreateModeInstance( 1 )
      parentMode = parent[ IpsecSaCryptoSuiteConfigMode ].\
                        getOrCreateModeInstance( saSuiteName )
      saCmds = parentMode[ 'Ipsec.SaCryptoSuiteConfig.config' ]
      suite = entity.saCryptoSuiteConfig[ saSuiteName ].cryptoSuite
      if suite.integrity == IpsecHmacAlgorithmEnum.nonehash:
         cmd = f"encryption {ipsecEncryptionMap[ suite.encryption ]}"
      else:
         cmd = f"encryption {ipsecEncryptionMap[ suite.encryption ]} "
         cmd += f"integrity {ipsecIntegrityMap[ suite.integrity ]}"
      if suite.dhGroup != IpsecDhGroupEnum.IpsecDhGroupNone:
         cmd += f" dh-group {dhGroupMap[ suite.dhGroup ]}"
      saCmds.addCommand( cmd )
