# Copyright (c) 2006-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=superfluous-parens
# pylint: disable=consider-using-f-string

import CliSave
import Tac

from CliMode.Ira import (
   IpHardwareEcmpFecMode,
   NexthopGroupConfigBase,
   RouterKernelMode
)
from Arnet.MplsLib import getLabelStack
from Arnet.MplsLib import labelStackToString
from CliSavePlugin.IntfCliSave import IntfConfigMode
from CliSavePlugin.IraCliSaveCommon import (
      saveOptimizePrefixConfig,
      IPV4,
)
# Need to import this to get the ConfigTag.config config sequence
import CliSavePlugin.ConfigTagCliSave # pylint: disable=unused-import
from IpLibConsts import DEFAULT_VRF
from IpLibTypes import ProtocolAgentModelType as ProtoAgentModel
from RoutingIntfUtils import allIpIntfNames

from RoutingConsts import (
      defaultStaticRouteName,
      defaultStaticRoutePreference,
      defaultStaticRouteTag,
      defaultStaticRouteMetric,
      defaultStaticRouteWeight
)
from NexthopGroupConsts import (
      EntryCounterType,
      NexthopGroupType,
      ipTunnelNexthopGroupTypes,
      greKeyIngressIntf,
      GreKeyType,
)
from NexthopGroupUtils import (
      nexthopGroupCliString,
      nexthopGroupCliStringToTacType,
)
from TypeFuture import TacLazyType
import Toggles.IraToggleLib
from Toggles.RoutingLibToggleLib import toggleFlexibleEcmpJ2Enabled

ipIntfCmdSeq = 'Ira.ipIntf'
IntfConfigMode.addCommandSequence( ipIntfCmdSeq,
   after=[ 'Arnet.intf', 'Arnet.l2l3barrier' ] )
# The service routing command must appear in config before the
# interface configuration as there are interface specific commands
# starting with "service". When "service routing" is parsed while in
# inteface configuration mode it results in an ambiguous command
# error (see BUG163576).
CliSave.GlobalConfigMode.addCommandSequence( 'Ira.serviceRouting',
                                             before=[ IntfConfigMode ] )
CliSave.GlobalConfigMode.addCommandSequence( 'Ira.routes',
                                             after=[ 'ConfigTag.config',
                                                IntfConfigMode ] )
CliSave.GlobalConfigMode.addCommandSequence( 'Ira.routing',
                                             after=[ IntfConfigMode ] )

tristateBoolEnum = Tac.Type( "Ip::TristateBool" )
IpGenAddrTac = TacLazyType( 'Arnet::IpGenAddr' )
TunnelPayloadType = TacLazyType( "L3::Tunnel::PayloadType" )
UdpPortConfigMode = TacLazyType( "L3::Tunnel::UdpPortConfigMode" )

#---------------------------------------------------------------------------------
# Cli savers
#---------------------------------------------------------------------------------

routingConsts = Tac.Value( "Ira::RoutingConsts", defaultStaticRoutePreference,
                           defaultStaticRouteTag, defaultStaticRouteName,
                           defaultStaticRouteMetric, defaultStaticRouteWeight )

@CliSave.saver( 'Routing::Config', 'routing/config' )
def saveRoutingAttr( entity, root, requireMounts, options ):
   if options.intfFilter:
      return
   saveRoutingAttribute( entity, root, options, vrfName=None )

@CliSave.saver( 'Routing::VrfConfigDir', 'routing/vrf/config' )
def saveVrfRoutingAttr( entity, root, requireMounts, options ):
   if options.intfFilter:
      return
   for vrfname in sorted( entity.vrfConfig ):
      saveRoutingAttribute( entity.vrfConfig[ vrfname ], root,
                            options, vrfname )

def saveRoutingAttribute( entity, root, options, vrfName ):
   vrfString = ' vrf %s' % vrfName if vrfName else ''
   if entity.routing:
      if entity.addresslessForwarding == tristateBoolEnum.isTrue:
         root[ 'Ira.routing' ].addCommand( "ip routing ipv6 interfaces %s" %
               vrfString )
      else:
         root[ 'Ira.routing' ].addCommand( "ip routing%s" % vrfString )
   elif options.saveCleanConfig:
      root[ 'Ira.routing' ].addCommand( "no ip routing%s" % vrfString )

@CliSave.saver( 'Routing::RouteListConfig', 'routing/routelistconfig',
      requireMounts=( 'configTag/configTagIdState', ),
      commandTagSupported=True )
def saveRoutingTable( entity, root, requireMounts, options ):
   if options.intfFilter:
      return
   saveStaticRoutes( entity, root, requireMounts, options, vrfName=None )

@CliSave.saver( 'Routing::VrfRouteListConfigDir', 'routing/vrf/routelistconfig',
      requireMounts=( 'configTag/configTagIdState', ),
      commandTagSupported=True )
def saveVrfRoutingTables( entity, root, requireMounts, options ):
   if options.intfFilter:
      return
   for vrfname in sorted( entity.vrfConfig ):
      saveStaticRoutes( entity.vrfConfig[ vrfname ], root,
                        requireMounts, options, vrfname )

def saveStaticRoutes( entity, root, requireMounts, options, vrfName ):
   saveAll = options.saveAll
   commandTagIdState = requireMounts[ 'configTag/configTagIdState' ]
   vrfString = ' vrf %s' % vrfName if vrfName else ''
   routeCmd = 'ip route%s ' % vrfString
   vrfName = vrfName.strip() if vrfName else 'default'
   saver = Tac.Value( "Ira::StaticRoutesCliSaver" )
   cmds = saver.save( entity, saveAll, routeCmd, routingConsts,
         vrfName, commandTagIdState )
   for cmd in cmds.split( '\n' ):
      if cmd != '':
         root[ 'Ira.routes' ].addCommand( cmd )

@CliSave.saver( 'Routing::Config', 'routing/config' )
def saveIpIcmpRedirectConfig( entity, root, requireMounts, options ):
   saveAll = options.saveAll
   if saveAll and entity.sendRedirects:
      root[ 'Ira.routing' ].addCommand( "ip icmp redirect")
   elif not entity.sendRedirects:
      root[ 'Ira.routing' ].addCommand( "no ip icmp redirect")

@CliSave.saver( 'Routing::Config', 'routing/config' )
def saveIpIcmpSrcIntfConfigDefault( entity, root, requireMounts, options ):
   saveIpIcmpSrcIntfConfig( entity, root, options, vrfName=None )

@CliSave.saver( 'Routing::Config', 'routing/vrf/config', attrName = 'vrfConfig' )
def saveIpIcmpSrcIntfConfigVrf( entity, root, requireMounts, options ):
   saveIpIcmpSrcIntfConfig( entity, root, options, entity.name )

# requires that entity be a Routing::Config (as opposed to working on
# Routing::ConfigGen)
def saveIpIcmpSrcIntfConfig( entity, root, options, vrfName ):
   if vrfName:
      vrfString = ' vrf %s' % vrfName
   else:
      vrfString = ''
   saveAll = options.saveAll
   if( entity.icmpSrcIntf != "" ):
      root[ 'Ira.routing' ].addCommand( "ip icmp source-interface %s%s" %
                                        ( entity.icmpSrcIntf, vrfString ) )
   elif( saveAll ):
      root[ 'Ira.routing' ].addCommand( "no ip icmp source-interface%s" %
                                        vrfString )

@CliSave.saver( 'Routing::Config', 'routing/config' )
def saveIpSoftwareDropIpOptions( entity, root, requireMounts, options ):
   if entity.ipOptionsSoftwareDropEnabled:
      root[ 'Ira.routing' ].addCommand(
         'ip software forwarding options action drop' )
   elif options.saveAll:
      root[ 'Ira.routing' ].addCommand(
         'no ip software forwarding options action drop' )

@CliSave.saver( 'Routing::Hardware::Config', 'routing/hardware/config',
     requireMounts=( 'routing/hardware/statuscommon',
                     'routing/hardware/status' ) )
def saveFibFilterConfig( entity, root, requireMounts, options ):
   routingHwStatus = requireMounts[ 'routing/hardware/status' ]
   if not routingHwStatus.fibSuppressionSupported:
      return
   saveAll = options.saveAll
   if( entity.fibSuppression ):
      root[ 'Ira.routing' ].addCommand(
                            "ip fib compression redundant-specifics filter" )
   elif( saveAll ):
      root[ 'Ira.routing' ].addCommand(
                            "no ip fib compression redundant-specifics filter" )

@CliSave.saver( 'Ip::Config', 'ip/config',
                requireMounts=( 'interface/config/all', 'interface/status/all',
                                'l3/intf/config', 'routing/hardware/statuscommon',
                                'routing/hardware/status' ) )
def saveIpConfig( entity, root, requireMounts, options ):
   routingHwStatus = requireMounts[ 'routing/hardware/status' ]
   # Interface config
   if options.saveAllDetail:
      cfgIntfNames = allIpIntfNames( requireMounts, includeEligible=True )
   elif options.saveAll:
      # IP configuration is allowed on switchport interfaces as well.
      # Display config on all ip interfaces and switchports with non-default
      # config.
      ipIntfNames = allIpIntfNames( requireMounts )
      cfgIntfNames = set( ipIntfNames )
      cfgIntfNames.update( entity.ipIntfConfig )
   else:
      cfgIntfNames = entity.ipIntfConfig

   for intfName in cfgIntfNames:
      if options.intfFilter and intfName not in options.intfFilter:
         continue
      intfConfig = entity.ipIntfConfig.get( intfName )
      if not intfConfig:
         if options.saveAll:
            intfConfig = Tac.newInstance( 'Ip::IpIntfConfig', intfName )
            intfConfig.l3Config = Tac.newInstance( 'L3::Intf::Config', intfName )
         else:
            continue

      saveIpIntfConfig( intfConfig, root, options, routingHwStatus )

def saveIpIntfConfig( entity, root, options, routingHwStatus ):
   saveAll = options.saveAll
   if entity.intfId.startswith( "Internal" ):
      # Internal interface configuration. Abort
      # BUG944 - need a more general way of doing this
      return

   mode = root[ IntfConfigMode ].getOrCreateModeInstance( entity.intfId )
   cmds = mode[ 'Ira.ipIntf' ]

   if entity.vrf != DEFAULT_VRF:
      assert entity.vrf != ''
      cmds.addCommand( "vrf %s" % entity.vrf )

   if Toggles.IraToggleLib.toggleIpv4AddresslessForwardingEnabled():
      if entity.dpRoutingAddressRequiredDisabled:
         cmds.addCommand( "ip routing address required disabled" )
      elif saveAll:
         cmds.addCommand( "no ip routing address required" )

   if entity.proxyArpEnabled:
      cmd = ( 'ip proxy-arp allow-default' if entity.proxyArpAllowDefaultRoute else
              'ip proxy-arp' )
      cmds.addCommand( cmd )
   elif saveAll:
      # proxy arp is disabled by default
      cmds.addCommand( 'no ip proxy-arp' )

   if entity.localProxyArpEnabled:
      cmds.addCommand( 'ip local-proxy-arp' )
   elif saveAll:
      cmds.addCommand( 'no ip local-proxy-arp' )

   if entity.gratuitousArpAccepted:
      cmds.addCommand( 'arp gratuitous accept' )
   elif saveAll:
      cmds.addCommand( 'no arp gratuitous accept' )

   if entity.proxyMacVirtualMacEnabled:
      cmds.addCommand( 'ip arp proxy reply virtual-router mac-address' )
   elif saveAll:
      cmds.addCommand( 'no ip arp proxy reply virtual-router mac-address' )

   if entity.addrSource == 'manual':
      if entity.addrWithMask.address != '0.0.0.0':
         cmds.addCommand( "ip address %s/%d" % ( entity.addrWithMask.address,
                                                 entity.addrWithMask.len ) )
      elif entity.unnumberedIntfId:
         cmds.addCommand( 'ip address unnumbered %s' % entity.unnumberedIntfId )
      elif saveAll:
         cmds.addCommand( "no ip address" )
   elif entity.addrSource == 'dhcp':
      cmds.addCommand( "ip address dhcp" )

   # Secondary addresses are not supported with DHCP at this moment
   if entity.addrSource != 'dhcp':
      for a in entity.secondaryWithMask:
         cmds.addCommand( "ip address %s/%d secondary" % ( a.address, a.len ) )

   if entity.addrSource == 'dhcp':
      if entity.defaultRouteSource == 'dhcp':
         cmds.addCommand( "dhcp client accept default-route" )
      elif saveAll:
         cmds.addCommand( "no dhcp client accept default-route" )

   if entity.urpf.mode != 'disable':
      if entity.urpf.mode == 'strict':
         cmd = 'ip verify unicast source reachable-via rx'
      elif entity.urpf.mode == 'loose':
         cmd = 'ip verify unicast source reachable-via any'
         if not entity.urpf.allowDefaultRoute and \
               Toggles.IraToggleLib.toggleUrpfIgnoreDefaultRouteEnabled() and \
               routingHwStatus.urpfLooseNonDefaultSupported:
            cmd += ' non-default'
         if entity.urpf.lookupVrf != "" and \
               Toggles.IraToggleLib.toggleSplitVrfUrpfResolutionEnabled() and \
               routingHwStatus.urpfLooseLookupVrfSupported:
            cmd += ' lookup-vrf ' + entity.urpf.lookupVrf
      else:
         # TODO BUG642564: Use entity.urpf.allowDefaultRoute under the strict mode
         # instead of strictDefault.
         cmd = 'ip verify unicast source reachable-via rx allow-default'

      if entity.urpf.acl != "":
         cmd += ' exception-list ' + entity.urpf.acl
      cmds.addCommand( cmd )
   elif saveAll:
      cmds.addCommand( "no ip verify unicast" )

   intfId = entity.intfId
   if ( intfId.startswith( "Ethernet" ) or
      intfId.startswith( "Port-Channel" ) or
      intfId.startswith( "Vlan" ) or
      intfId.startswith( "Test" ) ):
      if entity.directedBroadcastEnabled:
         cmds.addCommand( "ip directed-broadcast" )
      elif saveAll:
         cmds.addCommand( "no ip directed-broadcast" )

   if not entity.attachedRoutes:
      cmds.addCommand( 'no ip attached-routes' )
   elif saveAll:
      cmds.addCommand( 'ip attached-routes' )

@CliSave.saver( 'Routing::Hardware::Config', 'routing/hardware/config',
                requireMounts=( 'routing/hardware/statuscommon',
                                'routing/hardware/status' ) )
def saveResilientEcmpConfig( entity, root, requireMounts, options ):
   routingHwStatus = requireMounts[ 'routing/hardware/status' ]
   if not routingHwStatus.resilientEcmpSupported:
      return
   cmds = root['Ira.routing']
   if not entity.resilientEcmpPrefix and not entity.policyBasedResilientEcmp:
      if options.saveAll or options.saveAllDetail:
         cmds.addCommand( 'no ip hardware fib ecmp resilience' )
      return
   for prefix, ecmpConfig in entity.resilientEcmpPrefix.items() :
      orderedKw = ' ordered' if ecmpConfig.orderedVias else ''
      cmds.addCommand(
         "ip hardware fib ecmp resilience %s capacity %d redundancy %d%s"
            % ( prefix.stringValue , ecmpConfig.capacity, \
                   ecmpConfig.redundancy, orderedKw ) )
   if entity.policyBasedResilientEcmp:
      # ordered keyword is not supported for policy based resilient ecmp
      ecmpConfig = entity.policyBasedResilientEcmp
      cmds.addCommand(
         "ip hardware fib ecmp resilience marked capacity %d redundancy %d"
         % ( ecmpConfig.capacity, ecmpConfig.redundancy ) )

@CliSave.saver( 'Routing::Hardware::DlbConfig', 'routing/hardware/dlb/config',
                requireMounts=( 'routing/hardware/statuscommon',
                                'routing/hardware/status' ) )
def saveDlbConfig( entity, root, requireMounts, options ):
   routingHwStatus = requireMounts[ 'routing/hardware/status' ]
   dlbCfg = Tac.newInstance( "Routing::Hardware::DlbConfig", "DlbConfig" )
   if not routingHwStatus.dlbEcmpSupported:
      return
   cmds = root[ 'Ira.routing' ]
   if entity.globalDlbEcmpEnable:
      cmds.addCommand( 'ip hardware fib load-balance distribution dynamic' )
      if entity.selectiveDlbIpv4Acl:
         cmds.addCommand( 'ip hardware fib load-balance distribution dynamic '
                          'ipv4 access-group %s' % entity.selectiveDlbIpv4Acl )
      if entity.selectiveDlbIpv6Acl:
         cmds.addCommand( 'ip hardware fib load-balance distribution dynamic '
                          'ipv6 access-group %s' % entity.selectiveDlbIpv6Acl )

      memberSelModeMap = {
         0: 'timer',
         2: 'always',
      }
      if ( entity.portAssignmentMode != dlbCfg.portAssignmentMode ):
         cmds.addCommand( 'ip hardware fib load-balance distribution dynamic '
                          'member-selection optimal %s' %
                          memberSelModeMap[ entity.portAssignmentMode ] )

      if ( entity.inactivityDuration != dlbCfg.inactivityDuration ):
         cmds.addCommand( 'ip hardware fib load-balance distribution dynamic '
                          'inactivity-threshold %d microseconds' %
                          entity.inactivityDuration )
      if ( entity.samplingPeriod != dlbCfg.samplingPeriod ):
         cmds.addCommand( 'ip hardware fib load-balance distribution dynamic '
                          'sampling-period %d microseconds' %
                          entity.samplingPeriod )

      if ( entity.portLoadingWeight != dlbCfg.portLoadingWeight ):
         cmds.addCommand( 'ip hardware fib load-balance distribution dynamic '
                          'average-traffic-weight %d' % entity.portLoadingWeight )

      if ( entity.randomSelectionControlSeed != dlbCfg.randomSelectionControlSeed ):
         cmds.addCommand( 'ip hardware fib load-balance distribution dynamic '
                          'seed %d' % entity.randomSelectionControlSeed )

      if ( entity.flowSetSize != dlbCfg.flowSetSize ):
         cmds.addCommand( 'ip hardware fib load-balance distribution dynamic '
                          'flow-set-size %d' % entity.flowSetSize )

   elif options.saveAll or options.saveAllDetail:
      cmds.addCommand( 'no ip hardware fib load-balance distribution dynamic '
                       'ipv4 access-group' )
      cmds.addCommand( 'no ip hardware fib load-balance distribution dynamic '
                       'ipv6 access-group' )
      cmds.addCommand( 'no ip hardware fib load-balance distribution dynamic '
                       'sampling-period' )
      cmds.addCommand( 'no ip hardware fib load-balance distribution dynamic '
                       'member-selection' )
      cmds.addCommand( 'no ip hardware fib load-balance distribution dynamic '
                       'inactivity-threshold' )
      cmds.addCommand( 'no ip hardware fib load-balance distribution dynamic '
                       'average-traffic-weight' )
      cmds.addCommand( 'no ip hardware fib load-balance distribution dynamic seed' )
      cmds.addCommand( 'no ip hardware fib load-balance distribution dynamic '
                       'flow-set-size' )
      cmds.addCommand( 'no ip hardware fib load-balance distribution' )

@CliSave.saver( 'Routing::Hardware::Config', 'routing/hardware/config',
      requireMounts=( 'routing/config', 'routing/hardware/statuscommon',
                       'routing/hardware/status' ) )
def saveIpRoutingHardwareConfig( entity, root, requireMounts, options ):
   saveAll = options.saveAll
   routingConfig = requireMounts[ 'routing/config' ]
   routingHwStatus = requireMounts[ 'routing/hardware/status' ]
   if not entity.icmpUnreachable:
      cmd = 'ip icmp rate-limit-unreachable 0'
      root[ 'Ira.routing' ].addCommand( cmd )
   elif ( saveAll and routingConfig.routing ) or options.saveAllDetail:
      cmd = 'no ip icmp rate-limit-unreachable 0'
      root[ 'Ira.routing' ].addCommand( cmd )

   saveOptimizePrefixConfig( entity, root, options, IPV4, routingHwStatus )

   # pylint: disable-next=unnecessary-comprehension
   disabledVrfs = [ vrf for vrf in entity.optimizeDisabledVrf ]
   if disabledVrfs:
      cmd = 'ip hardware fib optimize disable-vrf ' + ' '.join( disabledVrfs )
      root[ 'Ira.routing' ].addCommand( cmd )
   elif saveAll or options.saveAllDetail:
      cmd = 'no ip hardware fib optimize disable-vrf'
      root[ 'Ira.routing' ].addCommand( cmd )

class NexthopGroupConfigSaveMode( NexthopGroupConfigBase, CliSave.Mode ):
   def __init__( self, ngParam ):
      nexthopGroupName, nexthopGroupType = ngParam
      NexthopGroupConfigBase.__init__( self, nexthopGroupName, nexthopGroupType )
      CliSave.Mode.__init__( self, nexthopGroupName )

   def instanceKey( self ):
      return self.nexthopGroupName_

CliSave.GlobalConfigMode.addCommandSequence( 'NexthopGroup.global',
                                             after=[ IntfConfigMode ] )
CliSave.GlobalConfigMode.addChildMode( NexthopGroupConfigSaveMode,
                                       after=[ 'NexthopGroup.global' ] )
NexthopGroupConfigSaveMode.addCommandSequence( 'NexthopGroup.cfg' )

def getNh( ng, idx ):
   destIpIntf = ng.entry[ idx ].destinationIpIntf
   nh = destIpIntf.destIp.stringValue
   if destIpIntf.destIp.isLinkLocal:
      nh = "%s%%%s" % ( nh, destIpIntf.intfId )
   return nh

@CliSave.saver( 'Routing::NexthopGroup::ConfigInput',
                'routing/nexthopgroup/input/cli' )
def saveNexthopGroupConfig( entity, root, requireMounts, options ):
   if entity.connectivityMonitorClient:
      root[ 'NexthopGroup.global' ].addCommand(
            'nexthop-group monitor connectivity client %s'
            % entity.connectivityMonitorClient )
   for nexthopGroupName in sorted( entity.nexthopGroup ):
      ng = entity.nexthopGroup[ nexthopGroupName ]
      if ng.dynamic:
         continue
      mode = root[ NexthopGroupConfigSaveMode ].getOrCreateModeInstance(
         ( nexthopGroupName, nexthopGroupCliString( ng.type ) ) )
      cmds = mode[ 'NexthopGroup.cfg' ]
      if ng.size:
         cmds.addCommand( 'size %s ' % ng.size )
      elif options.saveAll:
         cmds.addCommand( 'no size ' )
      if ng.type in { NexthopGroupType.mplsOverGre,
            NexthopGroupType.mplsOverUdp,
            NexthopGroupType.ipv4OverUdp,
            NexthopGroupType.ipv6OverUdp } and ng.tos.valid():
         cmds.addCommand( 'tos %s ' % ng.tos )
      if ng.type in ipTunnelNexthopGroupTypes and ng.ttl:
         cmds.addCommand( 'ttl %s ' % ng.ttl )
      if ng.type in ipTunnelNexthopGroupTypes:
         if not ng.sourceIp.isAddrZero:
            cmds.addCommand( 'tunnel-source %s ' % ng.sourceIp )
         elif ng.intfId != "":
            cmds.addCommand( 'tunnel-source intf %s ' % ng.intfId )
      if ng.entryCounterType == EntryCounterType.unshared:
         entryCounterCmd = 'entry counters unshared'
         if ng.preserveCounterOverConfigChange:
            entryCounterCmd += ' persistent'
         cmds.addCommand( entryCounterCmd )
      if ng.hierarchicalFecsEnabled:
         cmds.addCommand( 'fec hierarchical' )
      if ng.dlbEcmpEnable:
         cmds.addCommand( 'dynamic-load-balancing' )
      elif options.saveAll:
         cmds.addCommand( 'no dynamic-load-balancing' )

      # save GRE key config
      assert ng.greKeyType in ( GreKeyType.noGreKey, GreKeyType.ingressIntf )
      if ng.greKeyType == GreKeyType.ingressIntf:
         cmds.addCommand( 'tunnel-key %s ' % greKeyIngressIntf )

      ngType = nexthopGroupCliStringToTacType( mode.nexthopGroupType )
      size = max( ng.entry, default=-1 ) + 1
      for idx in range( ng.size or size ):
         if idx not in ng.entry:
            continue
         entry = ng.entry[ idx ]
         destIp = entry.destinationIpIntf.destIp
         if not destIp.isAddrZero:
            if ngType == NexthopGroupType.ip:
               if Toggles.IraToggleLib.toggleIpNhgIpv6LlAddressEnabled():
                  nh = getNh( ng, idx )
                  cmds.addCommand( 'entry %s nexthop %s ' % ( idx, nh ) )
               else:
                  cmds.addCommand( 'entry %s nexthop %s ' %
                                   ( idx, destIp.stringValue ) )
            elif ngType == NexthopGroupType.mpls:
               labelStack = getLabelStack( entry.mplsLabelStack )
               labelStackString = labelStackToString( labelStack )
               # pylint: disable-next=use-implicit-booleaness-not-comparison
               if labelStack != []:
                  nh = getNh( ng, idx )
                  cmds.addCommand(
                     'entry %d push label-stack %s nexthop %s' %
                     ( idx, labelStackString, nh ) )
               else:
                  cmds.addCommand(
                     'entry %d nexthop %s' % ( idx, destIp.stringValue ) )
            elif ngType == NexthopGroupType.mplsOverGre:
               labelStack = getLabelStack( entry.mplsLabelStack )
               labelStackString = labelStackToString( labelStack )
               if entry.uniqueSourceIp != IpGenAddrTac():
                  cmds.addCommand(
                  'entry %d push label-stack %s tunnel-destination %s '
                  'tunnel-source %s' %
                  ( idx, labelStackString, destIp.stringValue,
                  entry.uniqueSourceIp.stringValue ) )
               else:
                  cmds.addCommand(
                  'entry %d push label-stack %s tunnel-destination %s' %
                  ( idx, labelStackString, destIp.stringValue ) )
            elif ngType == NexthopGroupType.mplsOverUdp:
               labelStack = getLabelStack( entry.mplsLabelStack )
               labelStackString = labelStackToString( labelStack )
               cmds.addCommand(
                  'entry %d push label-stack %s tunnel-destination %s' %
                  ( idx, labelStackString, destIp.stringValue ) )
            elif ngType in ( NexthopGroupType.gre, NexthopGroupType.dzgre ):
               cmHostCmd = ''
               hostName = entry.connectivityMonitorHost.hostName
               if hostName:
                  cmHostCmd = ' monitor connectivity host %s' % hostName
               if entry.uniqueSourceIp != IpGenAddrTac():
                  cmds.addCommand(
                  'entry %d tunnel-destination %s tunnel-source %s%s' %
                  ( idx, destIp.stringValue,
                  entry.uniqueSourceIp.stringValue, cmHostCmd ) )
               else:
                  cmds.addCommand( 'entry %s tunnel-destination %s%s' % (
                     idx, destIp.stringValue, cmHostCmd ) )
            else:
               cmds.addCommand( 'entry %s tunnel-destination %s ' %
                                ( idx, destIp.stringValue ) )
         elif ng.childNexthopGroupName( idx ):
            cmds.addCommand( 'entry %s nexthop-group %s' %
                                ( idx, ng.childNexthopGroupName( idx ) ) )

@CliSave.saver( 'L3::Config', 'l3/config' )
def saveServiceRoutingConfig( entity, root, requireMounts, options ):
   # Show this command always, even if it is the default
   # Down the lane one day multi-agent will become the default
   # instead of ribd. When that happens the default will be masked
   # again.

   # When ribd is removed only show the service routing protocol model if it's not
   # ribd, since it's not possible for a user to configure the device in single
   # agent mode. This is needed to prevent single agent namespace duts from trying
   # to config replace into single agent mode which is not permitted by the cli.
   model = entity.configuredProtocolAgentModel
   if not ( ( model == ProtoAgentModel.multiAgent and not options.saveCleanConfig )
            or ( model == ProtoAgentModel.ribd and entity.isRibdRemoved ) ):
      cmd = f'service routing protocols model {model}'
      root[ 'Ira.serviceRouting' ].addCommand( cmd )

class RouterKernelCliSaveMode( RouterKernelMode, CliSave.Mode ):
   def __init__( self, vrfName ):
      RouterKernelMode.__init__( self, vrfName )
      CliSave.Mode.__init__( self, vrfName )

CliSave.GlobalConfigMode.addChildMode( RouterKernelCliSaveMode,
                                       after=[ IntfConfigMode ] )

RouterKernelCliSaveMode.addCommandSequence( 'Router.kernel' )

CliSave.GlobalConfigMode.addCommandSequence( 'Tunnel.nexthopgroup.input.cli',
                                             after=[ 'Ira.routing' ] )

@CliSave.saver( 'Tunnel::NexthopGroup::ConfigInput',
                'tunnel/nexthopgroup/input/cli' )
def saveNexthopGroupTunnelConfig( entity, root, requireMounts, options ):
   baseCmd = 'ip tunnel'
   for name in entity.entry:
      cmd = []
      nexthopGroupTunnelConfigEntry = entity.entry[ name ]
      cmd.append( nexthopGroupTunnelConfigEntry.tep.stringValue )
      cmd.append( 'nexthop-group' )
      cmd.append( nexthopGroupTunnelConfigEntry.nhgName )
      if nexthopGroupTunnelConfigEntry.igpPref or \
         nexthopGroupTunnelConfigEntry.igpMetric:
         cmd.append( 'igp-cost' )
         if nexthopGroupTunnelConfigEntry.igpPref:
            cmd.append( 'preference %d' % ( nexthopGroupTunnelConfigEntry.igpPref ) )
         if nexthopGroupTunnelConfigEntry.igpMetric:
            cmd.append( 'metric %d' % ( nexthopGroupTunnelConfigEntry.igpMetric ) )
      root[ 'Tunnel.nexthopgroup.input.cli' ].addCommand(
            baseCmd + ' ' + ' '.join( cmd ) )

@CliSave.saver( 'Ip::IpGlobalConfigParams',
      'ip/globalConfigParams' )
def saveDuplicateAddressDetectionConfig( entity, root, requireMounts, options ):
   if options.saveAll and entity.duplicateAddressDetectionLogging:
      root[ 'Ira.routing' ].addCommand( "no ip address duplicate detection" )
   elif not entity.duplicateAddressDetectionLogging:
      root[ 'Ira.routing' ].addCommand( "ip address duplicate detection disabled" )

CliSave.GlobalConfigMode.addCommandSequence( 'Tunnel.udpDestinationPortConfig' )

@CliSave.saver( 'L3::Tunnel::UdpDestinationPortConfig',
                'l3/tunnel/udpDestinationPortConfig' )
def saveTunnelUdpDestPortConfig( entity, root, requireMounts, options ):
   cmds = root[ 'Tunnel.udpDestinationPortConfig' ]
   baseTunnelCmd = 'tunnel type'
   baseNhgCmd = 'nexthop-group type'
   for payloadType, portConfig in entity.payloadTypeToDestPort.items():
      if portConfig.mode == UdpPortConfigMode.tunnelConfig:
         cmd = [ baseTunnelCmd ]
      else:
         cmd = [ baseNhgCmd ]
      if payloadType == TunnelPayloadType.mpls:
         cmd.append( 'mpls-over-udp' )
      elif payloadType == TunnelPayloadType.ipv4:
         cmd.append( 'ipv4-over-udp' )
      elif payloadType == TunnelPayloadType.ipv6:
         cmd.append( 'ipv6-over-udp' )
      cmd += [ 'udp destination port', str( portConfig.destinationPort ) ]
      cmds.addCommand( ' '.join( cmd ) )

class EcmpFecCliSaveMode( IpHardwareEcmpFecMode, CliSave.Mode ):
   def __init__( self, param ):
      IpHardwareEcmpFecMode.__init__( self, param )
      CliSave.Mode.__init__( self, param )

   def skipIfEmpty( self ):
      return True

CliSave.GlobalConfigMode.addChildMode( EcmpFecCliSaveMode,
                                       after=[ 'Ira.routing' ] )
EcmpFecCliSaveMode.addCommandSequence( 'HierEcmpFec.cfg' )

@CliSave.saver( 'Routing::Hardware::Config', 'routing/hardware/config' )
def saveEcmpFecBankConfig( entity, root, requireMounts, options ):
   if not toggleFlexibleEcmpJ2Enabled():
      # Feature not supported. No need to save any config.
      return

   ecmpFecCfgMode = root[ EcmpFecCliSaveMode ].getSingletonInstance()
   cmds = ecmpFecCfgMode[ 'HierEcmpFec.cfg' ]

   currentConfig = entity.ecmpFecReservedBankCount
   if currentConfig.isInvalid():
      if options.saveAll or options.saveAllDetail:
         # Should not be in HierEcmpFec.cfg
         root[ 'Ira.routing' ].addCommand(
            "no ip hardware fib hierarchical next-hop ecmp" )
      # No need to add invalid config
      return

   cmdFormat = "level {} bank minimum count {}"
   banks = [ currentConfig.level1, currentConfig.level2, currentConfig.level3 ]
   for lvl, bank in enumerate( banks ):
      if bank != 0:
         cmds.addCommand( cmdFormat.format( lvl + 1, bank ) )
